#!/usr/bin/env python
# -*- coding: utf-8 -*-

# pofix - perform string fixups on incoming .po files.
#
# The purpose of this script is to save translators from having to
# apply various string fixes needed before stable release by hand.  It is
# intended to be run on each incoming .po file as the Lord of
# Translations receives it.  However, translators may run it on their
# own .po files to be sure, as a second application will harmlessly do
# nothing.
#
# To use this script, give it one or more paths to .po files as
# command-line arguments.  Each file will be tweaked as needed.
# It should work on Windows and MacOS X as well as Linux, provided
# you have Python installed.
#
# This script will emit a report line for each file it modifies,
# and save a backup copy of the original with extension "-bak".
#
# This script will tell you when it is obsolete.  Run it against all .po
# files in the main Wesnoth tree; when it says none are older than this script,
# it can be discarded (assunming that it has in fact been used to transform
# all incoming .po files in the meantime).
#
# Example usage:
# utils/pofix.py po/wesnoth*/*.po*
# find data/campaigns/ -name '*.cfg' -print0 | xargs -0 utils/pofix.py
#
# To make use of >1 CPU core, you have to rely on xargs. In this sample 10 files
# are handed over to 4 instances of pofix.py:
# ls po/wesnoth*/*.po* | xargs -P 4 -n 10 ./utils/pofix.py
#
#
# Please do make sure to add a comment before any new blocks of conversions
# that states when it was added (current version number is enough) so that
# the file can be cleaned up more easily every now and then.
# Example:
# # conversion added in 1.9.5+svn
# ("foo addwd bar", "foo added bar"),
# # conversion added in 1.9.8+svn
# ("fooba foo", "foobar foo"),

stringfixes = {

"wesnoth" : (
# Convert hyphen-minuses that are being used as minus signs
# to the Unicode minus sign
# conversion added in 1.9.0-svn
(" -25%", " −25%"),
(" -1%", " −1%"),
(" -100%", " −100%"),

# Fix screw up
(" —100%", " −100%"),

# Convert makeshift dashes/straight apostrophes:
# conversion added in 1.9.0-svn
("however - from these", "however — from these"),
("campaign first - click", "campaign first — click"),
("unit type -", "unit type —"),
("experience - 4 for", "experience — 4 for"),
("both worlds - for", "both worlds — for"),
("respected - or simply", "respected — or simply"),
("feared - leader", "feared — leader"),
("- usually in blood - although", "— usually in blood — although"),
("position - although", "position — although"),
("as advisors -", "as advisors —"),
("don't have to - let it", "don't have to — let it"),
("your attacks - they will", "your attacks — they will"),
("upload statistics - Help", "upload statistics — Help"),
("(A) - admin command", "(A) — admin command"),
("(D) - debug only, (N) - network only, (A) - admin only", "(D) — debug only, (N) — network only, (A) — admin only"),
("not empty - duplicate", "not empty — duplicate"),
("Player Info -", "Player Info —"),
("About to upload statistics - Help us make Wesnoth better for you!", "About to upload statistics — Help us make Wesnoth better for you!"),
#the following rule applies to wesnoth/*.po* and to wesnoth-manual/*.po*
("victory objectives - getting", "victory objectives — getting"),

# Straight apostrophes and quotes to curly ones
# conversion added in 1.9.0-svn
("Ga'ash", "Ga’ash"),
("Gart'lo", "Gart’lo"),
("Mar'Ildian", "Mar’Ildian"),
("Marra Di'lek", "Marra Di’lek"),
("Bzz'Kza", "Bzz’Kza"),
("unit's", "unit’s"),
("side's", "side’s"),
("man's", "man’s"),
("player's", "player’s"),
("elf's", "elf’s"),
("turn's", "turn’s"),
("it's best to click the", "it’s best to click the"),
("Don't send", "Don’t send"),
("RACV's", "RACV’s"),
("If you don't", "If you don’t"),
("you don't already own", "you don’t already own"),
("aren't quite as tough", "aren’t quite as tough"),
("units don't incur", "units don’t incur"),
("plague doesn't", "plague doesn’t"),
("mechanical units don't", "mechanical units don’t"),
("I'm Ready", "I’m Ready"),

# Fix capitalization
# conversion added in 1.9.0-svn
("Icelandic translation", "Icelandic Translation"),
("Miscellaneous contributors", "Miscellaneous Contributors"),

# 'Can not' -> 'cannot'
("directory name and can not be installed", "directory name and cannot be installed"),

# More straight to curly conversions
# conversion added in 1.9.0+svn
("Don't make a wish.", "Don’t make a wish."),
("$name's", "$name|’s"),
("$road's", "$road|’s"),
("$login's", "$login|’s"),
("$name|'s turn", "$name|’s turn"),

("if it hasn't moved this", "if it hasn’t moved this"),
("to activate 'delay shroud updates' in the", "to activate ‘delay shroud updates’ in the"),
("update via 'update shroud now' (or the", "update via ‘update shroud now’ (or the"),
("with a 5-4 attack may", "with a 5–4 attack may"),
("with a 9-2 attack can", "with a 9–2 attack can"),
("with a 5-4 attack does", "with a 5–4 attack does"),
("the unit they're fighting. If", "the unit they’re fighting. If"),
("have many 'maximum level'", "have many ‘maximum level’"),
("it is 'on', set an", "it is ‘on’, set an"),
("it is 'off' set a", "it is ‘off’ set a"),
("they won't be kicked", "they won’t be kicked"),
("doesn't work on Undead", "doesn’t work on Undead"),
("within a drake's body enables", "within a drake’s body enables"),
("than the Elves'", "than the Elves’"),
("it's most commonly", "it’s most commonly"),
("it's master. Only", "it’s master. Only"),
("Don't neglect to", "Don’t neglect to"),
("the enemy can't reach your", "the enemy can’t reach your"),
("what you're facing. In", "what you’re facing. In"),
("remain at death's door until", "remain at death’s door until"),
("into an enemy's zone of", "into an enemy’s zone of"),
("Don't move an", "Don’t move an"),
("if you don't have to", "if you don’t have to"),
("unit, but don't attack or", "unit, but don’t attack or"),
("word in Wesnoth's playing screen", "word in Wesnoth’s playing screen"),
("You don't have a", "You don’t have a"),
("Don't ask me", "Don’t ask me"),
("Don't show again", "Don’t show again"),
("with the 'cures'", "with the ‘cures’"),
("You don't have enough", "You don’t have enough"),
("players that don't", "players that don’t"),
("it's -1.", "it’s −1."),

# Correct spelling
# conversion added in 1.9.0+svn
("has a tool-tip", "has a tooltip"),

# correct spelling
# conversion added in 1.9.3+svn
("threadments", "treatments"),
("text='edd'", "text='egg'"),

# Kill ugly hyphenated version
("Re-initialize fonts", "Reinitialize fonts"),

# add the font for Old English
("DejaVuSans.ttf,Andagii.ttf,wqy-zenhei.ttc", "DejaVuSans.ttf,Andagii.ttf,wqy-zenhei.ttc,Junicode-Regular.ttf"),
("DejaVu Sans,Andagii,WenQuanYi Zen Hei", "DejaVu Sans,Andagii,WenQuanYi Zen Hei,Junicode"),

# switch fonts for CJK translations
# added in 1.9.13+svn
("wqy-zenhei.ttc","DroidSansJapanese.ttf,DroidSansFallbackFull.ttf"),
("WenQuanYi Zen Hei","Droid Sans Japanese,Droid Sans Fallback"),

# Fix apostrophes
("SHOW ALL allies'", "SHOW ALL allies’"),
("HIDE ALL allies'", "HIDE ALL allies’"),

# Fix capitalization
("Joystick: number of the cursor x-axis joystick", "Joystick: number of the cursor X-axis joystick"),

# Fix quotation marks
("The nickname '$nick'", "The nickname ‘$nick’"),
),

"wesnoth-anl" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("4p - A New Land", "4p — A New Land"),
("some underground mushroom mines nearby -", "some underground mushroom mines nearby —"),
("A New Land - Help", "A New Land — Help"),
("Our talks are complete -", "Our talks are complete —"),

("some spider's nests in", "some spiders’ nests in"),
("the spider's web. Maybe", "the spider’s web. Maybe"),
("our chances wouldn't nearly be", "our chances wouldn’t nearly be"),
("reduce the enemy's income by", "reduce the enemy’s income by"),
("of the 'Elvish' style villages", "of the ‘Elvish’ style villages"),
("Let's cut you free!", "Let’s cut you free!"),
("'s farms now produce", "’s farms now produce"),
("'s mines now produce", "’s mines now produce"),
),

"wesnoth-aoi" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("was easy to follow - a wide", "was easy to follow — a wide"),
("unmistakable - tree stumps", "unmistakable — tree stumps"),
("question remained - would he", "question remained — would he"),
("this scenario - you must", "this scenario — you must"),
("worse - an orcish", "worse — an orcish"),
# Straight apostrophes and quotes to curly ones
# conversion added in 1.9.0-svn
("Wesmere's furthermost-faring scouts.", "Wesmere’s furthermost-faring scouts."),
("Two days' travel later, the forward scouts", "Two days’ travel later, the forward scouts"),
("Use Elven Scouts and Linaera's power of teleportation", "Use Elven Scouts and Linaera’s power of teleportation"),
("It's of human design...but we", "It’s of human design...but we"),
("Do it. We'll have a guest soon.", "Do it. We’ll have a guest soon."),
("Lord... I'm... I am filled with grief.", "Lord... I’m... I am filled with grief."),
("else we'll never have peace again.", "else we’ll never have peace again."),
("We will come in numbers... (*cough*) You can't imagine..", "We will come in numbers... (*cough*) You can’t imagine.."),
("I'll be waiting... Among the dead...", "I’ll be waiting... Among the dead..."),
("It's done, lord. No-one escaped. No-one", "It’s done, lord. No-one escaped. No-one"),
("we won't find much forage on the march.", "we won’t find much forage on the march."),
("We can't carry on Lord, the men are to tired.", "We can’t carry on Lord, the men are to tired."),
("we'll try again when reinforcements arrive.", "we’ll try again when reinforcements arrive."),
("planning an invasion, I'm sure of it.", "planning an invasion, I’m sure of it."),
("The sun's fully over the horizon.", "The sun’s fully over the horizon."),
("We'll rest a bit on the other side;", "We’ll rest a bit on the other side;"),
("they won't take long to rally.", "they won’t take long to rally."),
("No! This can't be!", "No! This can’t be!"),
("We can't get through, my Lord.", "We can’t get through, my Lord."),
("Never in my life did I dream I'd be bested by mere trolls.", "Never in my life did I dream I’d be bested by mere trolls."),
("We'll wait for reinforcements.", "We’ll wait for reinforcements."),
("Haldric's", "Haldric’s"),
("believe it's a", "believe it’s a"),
("since then they've been", "since then they’ve been"),
("It's hopeless; we've tried everything, and they're still coming back.", "It’s hopeless; we’ve tried everything, and they’re still coming back."),
("There's", "There’s"),
("we're", "we’re"),
("Lord Erlornas didn't drive", "Lord Erlornas didn’t drive"),
("I've been bested, but the combat wasn't fair", "I’ve been bested, but the combat wasn’t fair"),
("I'll have some answers", "I’ll have some answers"),
("let's focus on the task at hand", "let’s focus on the task at hand"),
("We don't want any more undesirables", "We don’t want any more undesirables"),
("Lord... I'm... I am filled with grief", "Lord... I’m... I am filled with grief"),
("else we'll never have peace again", "else we’ll never have peace again"),
("You can't imagine", "You can’t imagine"),
("I'll be waiting", "I’ll be waiting"),
("It's done, lord. No-one escaped. No-one tried to escape. I'm... disturbed", "It’s done, lord. No-one escaped. No-one tried to escape. I’m... disturbed"),
("we'll move out at dawn", "we’ll move out at dawn"),
("we won't find much forage", "we won’t find much forage"),
("We can't carry on Lord", "We can’t carry on Lord"),
("we'll try again when reinforcements arrive", "we’ll try again when reinforcements arrive"),
("I'm sure of it", "I’m sure of it"),
("The sun's fully over the horizon", "The sun’s fully over the horizon"),
("We'll rest a bit on the other side", "We’ll rest a bit on the other side"),
("they won't take long to rally", "they won’t take long to rally"),
("No! This can't be!", "No! This can’t be!"),
("We can't get through", "We can’t get through"),
("I dream I'd be bested", "I dream I’d be bested"),
("We'll wait for reinforcements", "We’ll wait for reinforcements"),
("not frequented even by Wesmere's", "not frequented even by Wesmere’s"),
("Two days' travel later", "Two days’ travel later"),
("Linaera's power", "Linaera’s power"),
("It's of human design", "It’s of human design"),
("We'll have a guest soon", "We’ll have a guest soon"),
("Without Linaera's help", "Without Linaera’s help"),
("The Ka'lian has deliberated", "The Ka’lian has deliberated"),
("they're tired and afraid", "they’re tired and afraid"),
("I'm... disturbed", "I’m... disturbed"),

# Fixed spelling
("no a chance to issue", "not a chance to issue"),
("the men are to tired", "the men are too tired"),

# Added a missing "a"
("but temporary solution", "but a temporary solution"),

# Added a missing comma
("Soon after Erlornas died the elven party", "Soon after Erlornas died, the elven party"),
),

"wesnoth-did" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("A Small Favor -", "A Small Favor —"),
("running away - my horsemen", "running away — my horsemen"),
("build by armies", "build my armies"),

# Convert straight apostrophes/quotation marks
# conversion added in 1.9.0-svn
("Kreg'a'shar", "Kreg’a’shar"),
("Parthyn's", "Parthyn’s"),
("orcs'", "orcs’"),
("'Allow me to introduce", "“Allow me to introduce"),
("town for a few days.'", "town for a few days.”"),
("'Surely you know that", "“Surely you know that"),
("only in dark magic.'", "only in dark magic.”"),
("You won't truly banish", "You won’t truly banish"),
("I've no wish to", "I’ve no wish to"),
("you've come", "you’ve come"),
("I won't spare", "I won’t spare"),
("Three days' travel", "Three days’ travel"),
("T'shar", "T’shar"),
("Don't say", "Don’t say"),
("it's ridiculous", "it’s ridiculous"),
("I don't see any. Maybe it's", "I don’t see any. Maybe it’s"),
("'zed'", "‘zee’"), # Use the American spelling; philosopher's quotes are being used here
("So, I've finally", "So, I’ve finally"),
("he's threatening", "he’s threatening"),
("It's time he learned", "It’s time he learned"),
("I've been itching", "I’ve been itching"),
("I'm ready", "I’m ready"),
("transformation they'll begin", "transformation they’ll begin"),
("I won't go down", "I won’t go down"),
("I won't see them", "I won’t see them"),
("orc's", "orc’s"),
("'The spells of necromancy", "“The spells of necromancy"),
("spirit world.'", "spirit world.”"),
("'To become a lich, one must first die.'", "“To become a lich, one must first die.”"),
("Malin's", "Malin’s"),
("I've just got", "I’ve just got"),
("We'll see", "We’ll see"),
("when they didn't", "when they didn’t"),
("You can't", "You can’t"),
("What's in it", "What’s in it"),
("Karres's", "Karres’s"),
("Let's get", "Let’s get"),
("bats won't stand", "bats won’t stand"),
("I'm eager to", "I’m eager to"),
("if ye dinna' want tae be a walking pile o'", "if ye dinna’ want tae be a walking pile o’"),
("they don't understand", "they don’t understand"),
("I've got the rest", "I’ve got the rest"),
("Gron'r Hronk", "Gron’r Hronk"),
("K'rrlar Oban", "K’rrlar Oban"),
("doesn't look very", "doesn’t look very"),
("lake's", "lake’s"),
("'They are quite useful in battle,'", "“They are quite useful in battle,”"),
("'but none of them have even a tenth of your potential power.'", "“but none of them have even a tenth of your potential power.”"),
("P'Gareth", "P’Gareth"),
("K'Vark", "K’Vark"),
("he's escaping", "he’s escaping"),
("Drogan's", "Drogan’s"),
("'A life curse goes beyond a joke,'", "“A life curse goes beyond a joke,”"),
("'Poor judgment,'", "“Poor judgment,”"),
("I'll have my", "I’ll have my"),
("'For your final test", "“For your final test"),
("retrieving a book,'", "retrieving a book,”"),
("'The book was", "“The book was"),
("it from me.'", "it from me.”"),
("'They are no", "“They are no"),
("twice now.'", "twice now.”"),
("'Excellent. We travel", "“Excellent. We travel"),
("book inside.'", "book inside.”"),
("Mage Lord's", "Mage Lord’s"),
("mage lord's", "mage lord’s"),
("won't hold back", "won’t hold back"),
("We've got", "We’ve got"),
("you aren't leaving", "you aren’t leaving"),
("now you've given", "now you’ve given"),
("you've got", "you’ve got"),
("humankind's", "humankind’s"),
("I'm not ready to die", "I’m not ready to die"),

# Fix screw up
# conversion added in 1.9.0+svn
("‘The book was", "“The book was"),

("DID", "DiD"),
),

"wesnoth-dm" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("warn you - a party", "warn you — a party"),
("each other - and you'll", "each other — and you’ll"),
("Night is falling - that's", "Night is falling — that’s"),
("work by now - I did not", "work by now — I did not"),
("seeking you - you see", "seeking you — you see"),
("Of course - do you", "Of course — do you"),
("Knalga - the rumor", "Knalga — the rumor"),
("Worse news - the", "Worse news — the"),
("been to the west - will the", "been to the west — will the"),
("the dead - should", "the dead — should"),
("Illuven - lesser", "Illuven — lesser"),
("need protection - cost", "need protection — cost"),
("No thanks - we'll manage by ourselves...", "No thanks — we’ll manage by ourselves..."),
("Let's move on - the less", "Let’s move on — the less"),
("We should camp for the night now - we", "We should camp for the night now — we"),
("Those standing stones - they summon", "Those standing stones — they summon"),
("possible - I want to get us", "possible — I want to get us"),
("they are woses - tree guardians", "they are woses — tree guardians"),
("no alternative - we must get", "no alternative — we must get"),
("things in Wesnoth - we must fight", "things in Wesnoth — we must fight"),
("stirred on the island -", "stirred on the island —"),
("see my greatest achievement - an", "see my greatest achievement — an"),
("must be informed immediately -", "must be informed immediately —"),
("This forest looks quiet - too quiet.", "This forest looks quiet — too quiet."),
("No - you can't be dead!", "No — you can’t be dead!"),
("of our help too - this", "of our help too — this"),

# Fix screw up
# conversion added in 1.9.0-svn
("each other — and you'll", "each other — and you’ll"),
("Night is falling — that's", "Night is falling — that’s"),
("No thanks — we'll manage by ourselves...", "No thanks — we’ll manage by ourselves..."),
("Let's move on — the less", "Let’s move on — the less"),
("No — you can't be dead!", "No — you can’t be dead!"),

# Correct capitalization
# conversion added in 1.9.0-svn
("Clash at the manor", "Clash at the Manor"),
("Shadows in the dark", "Shadows in the Dark"),
("Face of the enemy", "Face of the Enemy"),

# Straight apostrophes and quotes to curly ones
# conversion added in 1.9.0-svn
("Delfador's Memoirs", "Delfador’s Memoirs"),
("'The Great'", "“The Great”"),
("Don't die!", "Don’t die!"),
("Methor's", "Methor’s"),
("I don't like", "I don’t like"),
("I've told you", "I’ve told you"),
("father's", "father’s"),
("After a night's rest", "After a night’s rest"),
("And if it's archers you need", "And if it’s archers you need"),
("Leollyn's", "Leollyn’s"),
("king's", "king’s"),
("Lionel's", "Lionel’s"),
("I'm honored that", "I’m honored that"),
("Here's", "Here’s"),
("It's been a pleasure", "It’s been a pleasure"),
("You'll", "You’ll"),
("I think that's all", "I think that’s all"),

("a reward for Delfador's bravery, I am", "a reward for Delfador’s bravery, I am"),
("a trace of Iliah-Malal's way between", "a trace of Iliah-Malal’s way between"),
("A wizard's staff of power.", "A wizard’s staff of power."),
("am Ulrek, chieftain o' the clan of", "am Ulrek, chieftain o’ the clan of"),
("aware of the King's", "aware of the King’s"),
("barbarian for you. We're", "barbarian for you. We’re"),
("be left in Iliah-Malal's hands.", "be left in Iliah-Malal’s hands."),
("been sent at Asheviere's", "been sent at Asheviere’s"),
("book's help, it will", "book’s help, it will"),
("But as Delfador's meditation deepened, and", "But as Delfador’s meditation deepened, and"),
("But the elder mage's health was taxed", "But the elder mage’s health was taxed"),
("called 'Heir to the Throne'. In it is", "called ‘Heir to the Throne’. In it is"),
("comrades' screams as they", "comrades’ screams as they"),
("Delfador became Garard II's", "Delfador became Garard II’s"),
("Delfador's head. Had that", "Delfador’s head. Had that"),
("Delfador's last great quest,", "Delfador’s last great quest,"),
("Delfador's troop, hurrying west", "Delfador’s troop, hurrying west"),
("Delfador's wandering time with", "Delfador’s wandering time with"),
("enemy. I'm sure they'll be no match", "enemy. I’m sure they’ll be no match"),
("evil in the reader's", "evil in the reader’s"),
("final blow, for Iliah-Malal's un-", "final blow, for Iliah-Malal’s un-"),
("final blow, for Iliah-Malal's un-life must", "final blow, for Iliah-Malal’s un-life must"),
("Following the alliance's victory and the", "Following the alliance’s victory and the"),
("Garard that way! 'Suffer our", "Garard that way! “Suffer our"),
("go against the King's", "go against the King’s"),
("had already in Delfador's time been a", "had already in Delfador’s time been a"),
("his portal before it's too late, and", "his portal before it’s too late, and"),
("hospitality? I'm lost, and freezing", "hospitality? I’m lost, and freezing"),
("how to close Iliah-Malal's portal to the", "how to close Iliah-Malal’s portal to the"),
("Human, ye ha' fought well. I", "Human, ye ha’ fought well. I"),
("Iliah-Malal's forces. But the", "Iliah-Malal’s forces. But the"),
("in having the King's ear; there was", "in having the King’s ear; there was"),
("It's a long story...", "It’s a long story..."),
("It's hopeless, I've lost all track", "It’s hopeless, I’ve lost all track"),
("It's the only way!", "It’s the only way!"),
("Lionel, the King's most trusted general,", "Lionel, the King’s most trusted general,"),
("Malal's army and Weldyn....", "Malal’s army and Weldyn..."),
("Malal's portal! All is", "Malal’s portal! All is"),
("necromancer's footsteps in a", "necromancer’s footsteps in a"),
("night like that! Let's leave this evil", "night like that! Let’s leave this evil"),
("not obey the King's order.", "not obey the King’s order."),
("one of the garrison's message-riders to him,", "one of the garrison’s message-riders to him,"),
("portal. But with Iliah-Malal's army at large", "portal. But with Iliah-Malal’s army at large"),
("rooted in the land's", "rooted in the land’s"),
("say my kin ha' been ungrateful for", "say my kin ha’ been ungrateful for"),
("some of the Book's least dangerous secrets", "some of the Book’s least dangerous secrets"),
("speak with the King's voice!", "speak with the King’s voice!"),
("still have the son's ear, though being", "still have the son’s ear, though being"),
("Sythan's village.", "Sythan’s village."),
("tale called the 'Legend of", "tale called the ‘Legend of"),
("than I thought. Iliah-Malal's offensive has begun.", "than I thought. Iliah-Malal’s offensive has begun."),
("thank you. Now let's", "thank you. Now let’s"),
("The blow interrupted Delfador's meditation. He began", "The blow interrupted Delfador’s meditation. He began"),
("The book's curse is already", "The book’s curse is already"),
("the road...especially where you'll be", "the road... especially where you’ll be"),
("think so. But...Asheviere, Garard's queen and Eldred's mother,", "think so. But... Asheviere, Garard’s queen and Eldred’s mother,"),
("to the north ye'd need to have ta'en, along the", "to the north ye’d need to have ta’en, along the"),
("to the northeast; I'll lead you there.", "to the northeast; I’ll lead you there."),
("to visit her family's demesne and her", "to visit her family’s demesne and her"),
("unseen they reached Asheviere's family demesne.", "unseen they reached Asheviere’s family demesne."),
("wanna cross it, you'll", "wanna cross it, you’ll"),
("We ha' seen those undead", "We ha’ seen those undead"),
("Wesmere'. Prolonged in life", "Wesmere’. Prolonged in life"),
("with orcs now. We'll take", "with orcs now. We’ll take"),
("wrath' indeed... I'll show him wrath!", "wrath” indeed... I’ll show him wrath!"),
("you, but I don't feel very fated. In fact, I've", "you, but I don’t feel very fated. In fact, I’ve"),
),

"wesnoth-dw" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("hearten the folk. And -", "hearten the folk. And —"),
("if you will permit - you", "if you will permit — you"),
("a week ago - wanted", "a week ago — wanted"),
("disturbing that a Kai - and", "disturbing that a Kai — and"),
("- would run here", "— would run here"),
("to be survive.", "to survive."),

# Straight quotes to curly quotes
# conversion added in 1.9.0+svn
("owner's", "owner’s"),
("I'll try on the ring.", "I’ll try on the ring."),
("I'll take this ring", "I’ll take this ring"),
("were many more on the way. I'm", "were many more on the way. I’m"),
("kai's", "kai’s"),
("Laudiss's", "Laudiss’s"),
("'Lord Ravanal'.", "“Lord Ravanal”."),
("Krellis'", "Krellis’"),
("Mal-Ravanal's Forces", "Mal-Ravanal’s Forces"),
("I've never even met", "I’ve never even met"),
("it's licking my hand", "it’s licking my hand"),
("It's kind of cute", "It’s kind of cute"),
("Aren't you going to kill", "Aren’t you going to kill"),
("Tyegëa's", "Tyegëa’s"),
("Let's hope those", "Let’s hope those"),
("It's a storm", "It’s a storm"),
("I'll take it", "I’ll take it"),
("I'll just leave", "I’ll just leave"),
("Don't let him go too", "Don’t let him go too"),
("I wouldn't do it", "I wouldn’t do it"),
("you're helping me", "you’re helping me"),
("He's one", "He’s one"),
("We don't want", "We don’t want"),
("we'll cut", "we’ll cut"),
("You're still squatting", "You’re still squatting"),
("Now you're in", "Now you’re in"),
("here's a yummy", "here’s a yummy"),
("Don't kill me", "Don’t kill me"),
("I've been so lonely", "I’ve been so lonely"),
("But he's tame", "But he’s tame"),
("He'll do what you", "He’ll do what you"),
("He's a really nice", "He’s a really nice"),
("however, and the residents didn't", "however, and the residents didn’t"),
("Cylanna's", "Cylanna’s"),
("Maudin's", "Maudin’s"),
("Imirna's", "Imirna’s"),
("Tyegëa's", "Tyegëa’s"),
("I think something's", "I think something’s"),
("aren't any more", "aren’t any more"),
("to FIND it isn't the", "to FIND it isn’t the"),
("he's dead of course", "he’s dead of course"),
("Caladon's", "Caladon’s"),
("I'VE COME FOR YOUR", "I’VE COME FOR YOUR"),
("if it doesn't want", "if it doesn’t want"),
("I don't know if that", "I don’t know if that"),
("It's MINE!", "It’s MINE!"),
("it's true that you helped", "it’s true that you helped"),
("I won't kill you.", "I won’t kill you."),
("I don't trust you,", "I don’t trust you,"),
("You haven't left yet? My patience", "You haven’t left yet? My patience"),
("I'll carry", "I’ll carry"),
("This doesn't look too bad.", "This doesn’t look too bad."),
("Bring 'em on!", "Bring ’em on!"),
("pendant on it. I'm sure the shape", "pendant on it. I’m sure the shape"),
("be useful. I'll take it.", "be useful. I’ll take it."),
("destroying Mal-Ravanal's henchmen", "destroying Mal-Ravanal’s henchmen"),
("I can't swim, so", "I can’t swim, so"),
("don't like fish", "don’t like fish"),
("And don't touch the ships", "And don’t touch the ships"),
("I'll give you my", "I’ll give you my"),
("She has a reason. Don't", "She has a reason. Don’t"),
("but he hasn't really cared", "but he hasn’t really cared"),
("I'm sure many townsfolk would", "I’m sure many townsfolk would"),
("It doesn't seem to have helped its previous owner. I don't want it.", "It doesn’t seem to have helped its previous owner. I don’t want it."),
("Let's hope those wild wolves", "Let’s hope those wild wolves"),
("don't get to the feast", "don’t get to the feast"),

# Spelling fix in 1.9.0+svn @ 46436
("truely formidible", "truly formidable"),

# Convert fake em dash
("A king who is skilled at combat commands great respect--especially", "A king who is skilled at combat commands great respect — especially"),

# Convert straight apostrophe
("I'll join you right now!", "I’ll join you right now!"),

# Add missing diaeresis
("Teeloa returned to his home", "Teelöa returned to his home"),

# Fix spelling
("I beleive it is undead", "I believe it is undead"),

# Convert straight apostrophe
("But it isn't attacking <i>me</i>.", "But it isn’t attacking <i>me</i>."),
),

"wesnoth-ei" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("Wait, before we go anywhere - who", "Wait, before we go anywhere — who"),
("This adept is weak - we may", "This adept is weak — we may"),
("onward - we must defeat", "onward — we must defeat"),
("playing a sick game - whenever", "playing a sick game — whenever"),
("to take care of - we must", "to take care of — we must"),
("to help us - but the orcs", "to help us — but the orcs"),
("It looks repairable - we", "It looks repairable — we"),
("Run for your - what the", "Run for your — what the"),
# ... and aint -> ain't
# conversion added in 1.9.0-svn
("I aint charging gold -", "I ain’t charging gold —"),
("'T'aint safe", "’T’ain’t safe"),
# Make it unspaced...
# conversion added in 1.9.0-svn
("may be able to help us in - ", "may be able to help us in—"),
("Wait - what just happened?", "Wait — what just happened?"),

# Fix screw up
# conversion added in 1.9.0-svn
("I ain't charging gold —", "I ain’t charging gold —"),
("'T'ain't safe", "’T’ain’t safe"),

("author of Wesnoth's troubles.", "author of Wesnoth’s troubles."),
("Aye, we'll help ye, for", "Aye, we’ll help ye, for"),
("by the orcs fa' sure!", "by the orcs fa’ sure!"),
("c'n stop runnin'...", "c’n stop runnin’..."),
("can get to Weld'n an' all so I", "can get to Weld’n an’ all so I"),
("can't get across. It'll help you to", "can’t get across. It’ll help you to"),
("caves for centuries. We'll no' be scattered now", "caves for centuries. We’ll no’ be scattered now"),
("Commander of the King's Horse. And take", "Commander of the King’s Horse. And take"),
("Deal. I c'n blow'er up once I", "Deal. I c’n blow’er up once I"),
("go north-east. I'll go north-west.", "go north-east. I’ll go north-west."),
("Gweddry's men retreated swiftly", "Gweddry’s men retreated swiftly"),
("Gweddry, I don't think you should", "Gweddry, I don’t think you should"),
("have crossed. Now let's see if we", "have crossed. Now let’s see if we"),
("He says: 'Well done; you have", "He says: “Well done; you have"),
("He will reach Mal-Ravanal's capital, and I", "He will reach Mal-Ravanal’s capital, and I"),
("his terms? I won't accept anything obviously", "his terms? I won’t accept anything obviously"),
("I'll be followin' yah from now", "I’ll be followin’ yah from now"),
("I'm an enginea'. I s'pect you'll have a need", "I’m an enginea’. I s’pect you’ll have a need"),
("It doesn't matter anyway. The", "It doesn’t matter anyway. The"),
("It doesn't matter. They are", "It doesn’t matter. They are"),
("look impassable. I don't think we will", "look impassable. I don’t think we will"),
("lord's hordes at dawn", "lord’s hordes at dawn"),
("may know the undead's weakness.", "may know the undead’s weakness."),
("My clan ha' lived in these", "My clan ha’ lived in these"),
("my eq'pment is. But ev'ryone not ov'r theah on that", "my eq’pment is. But ev’ryone not ov’r theah on that"),
("no refuge for Gweddry's weary men, and", "no refuge for Gweddry’s weary men, and"),
("No, I don't think so. I", "No, I don’t think so. I"),
("not defeat my Lord's minions, it was", "not defeat my Lord’s minions, it was"),
("point north of Glyn's Forest. If we", "point north of Glyn’s Forest. If we"),
("seems, well merited. Mal-Ravanal's", "seems, well merited. Mal-Ravanal’s"),
("serve in my Master's undead hordes!", "serve in my Master’s undead hordes!"),
("services. I bet you're", "services. I bet you’re"),
("several weeks, and Gweddry's men started to", "several weeks, and Gweddry’s men started to"),
("The Bitter Swamp's ill reputation is,", "The Bitter Swamp’s ill reputation is,"),
("the fate of Gweddry's band, but all", "the fate of Gweddry’s band, but all"),
("the men of Gweddry's troop that they", "the men of Gweddry’s troop that they"),
("These people won't listen to reason,", "These people won’t listen to reason,"),
("they can't escape until our", "they can’t escape until our"),
("this river. Now let's go!", "this river. Now let’s go!"),
("those orcs are chasin' you, and if", "those orcs are chasin’ you, and if"),
("to keep the Crown's", "to keep the Crown’s"),
("to know what you're talking about... but,", "to know what you’re talking about... but,"),
("to that signpost ov'r dere. That’s where", "to that signpost ov’r dere. That’s where"),
("undead. But we can't help you until", "undead. But we can’t help you until"),
("undead. ’T’ain’t safe 'round 'ere!", "undead. ’T’ain’t safe ’round ’ere!"),
("up that bridge ov'r theah.", "up that bridge ov’r theah."),
("Very well, we'll protect you from", "Very well, we’ll protect you from"),
("We didn't capture enough of", "We didn’t capture enough of"),
("We're in the land", "We’re in the land"),
("well once the King's forces arrived. There", "well once the King’s forces arrived. There"),
("Well, we didn't have to blow", "Well, we didn’t have to blow"),
("when it blows w'll", "when it blows w’ll"),
("Why don't we see if", "Why don’t we see if"),
("won't be able to", "won’t be able to"),
("work yet complete. Mal-Ravanal's main host has", "work yet complete. Mal-Ravanal’s main host has"),
("year of Konrad II's reign, and there", "year of Konrad II’s reign, and there"),
("your name.'", "your name.”"),
),


"wesnoth-httt" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("and the support of my men - from", "and the support of my men — from"),
("NE - Dan'Tonk", "NE — Dan’Tonk"),
("SE - Fort Tahn", "SE — Fort Tahn"),
("The Valley of Death - The Princess's Revenge", "The Valley of Death — The Princess’s Revenge"),
("the hills - there are undead about!", "the hills — there are undead about!"),
("those gryphon eggs - they", "those gryphon eggs — they"),
("- Delfador's insistence", "— Delfador’s insistence"),
("Look - orcs are", "Look — orcs are"),
("A frail human - or worse, an elf -", "A frail human — or worse, an elf —"),
("out to the heir - I", "out to the heir — I"),
("gruesome sight - a fleet", "gruesome sight — a fleet"),
("introduce myself - I", "introduce myself — I"),
("my warning - prepare", "my warning — prepare"),
("princess - the heiress", "princess — the heiress"),
("don't try to fight us - you", "don’t try to fight us — you"),
("Princess Li'sar - here?", "Princess Li’sar — here?"),
("Look - you can", "Look — you can"),
("century - a generation", "century — a generation"),
("vast human army - his", "vast human army — his"),
#("mother's", "mother’s"),
#("'Fight no one great", "“Fight no one great"),
#("from his shoulders.'", "from his shoulders.”"),
#("You'd", "You’d"),

# Fix screw up
# conversion added in 1.9.0-svn
("NE — Dan'Tonk", "NE — Dan’Tonk"),
("The Valley of Death — The Princess's Revenge", "The Valley of Death — The Princess’s Revenge"),
("— Delfador's insistence", "— Delfador’s insistence"),
("don't try to fight us — you", "don’t try to fight us — you"),
("Princess Li'sar — here?", "Princess Li’sar — here?"),

# Convert straight apostrophes and quotation marks
# conversion added in 1.9.0+svn
("Don't die, Delfador!", "Don’t die, Delfador!"),
("city's finest defenders", "city’s finest defenders"),
("Usadar Q'kai", "Usadar Q’kai"),
("Can't we strike", "Can’t we strike"),
("Managa'Gwin", "Managa’Gwin"),
("up, worm! I'm the boss!", "up, worm! I’m the boss!"),
("'em!", "’em!"),
("I hope he's safe!", "I hope he’s safe!"),
("you, Delfador? You're coming with", "you, Delfador? You’re coming with"),
("Delfador! I can't do it", "Delfador! I can’t do it"),
("days' travel if", "days’ travel if"),
("with me, aren't you, Delfador?", "with me, aren’t you, Delfador?"),
("only six days' march up", "only six days’ march up"),
("Muff Malal's Peninsula", "Muff Malal’s Peninsula"),
("Let's hope these", "Let’s hope these"),
("It's a shame", "It’s a shame"),
("the Elensefar Thieves' Guild. We", "the Elensefar Thieves’ Guild. We"),
("if you didn't trust us,", "if you didn’t trust us,"),
("across from Elensefar's docks.", "across from Elensefar’s docks."),
("from the city's northern gate.", "from the city’s northern gate."),
("flag if it's dark?", "flag if it’s dark?"),
("Let's expel these", "Let’s expel these"),
("banks of Elensefar's port district,", "banks of Elensefar’s port district,"),
("To Konrad's surprise, they", "To Konrad’s surprise, they"),
("across the river's mouth.", "across the river’s mouth."),
("with the Ka'lian, the Great", "with the Ka’lian, the Great"),
("but surely Asheviere's men will", "but surely Asheviere’s men will"),
("of Fire? What's that?", "of Fire? What’s that?"),
("Wesmere, for Asheviere's orcs have", "Wesmere, for Asheviere’s orcs have"),
("The queen's agents were", "The queen’s agents were"),
("Elensefar's rescue had", "Elensefar’s rescue had"),
("it was, Konrad's army avoided", "it was, Konrad’s army avoided"),
("important to Asheviere's", "important to Asheviere’s"),
("Dan'Tonk, we are", "Dan’Tonk, we are"),
("Force Li'sar's surrender (reduce", "Force Li’sar’s surrender (reduce"),
("Elmar's Crossing", "Elmar’s Crossing"),
("\"Li'sar\"", "\"Li’sar\""),
("the princess, Li'sar. The daughter", "the princess, Li’sar. The daughter"),
("I'm no impostor.", "I’m no impostor."),
("I surrender! Don't hurt me,", "I surrender! Don’t hurt me,"),
("said before I'm not an", "said before I’m not an"),
("near, impostor! I'll not be", "near, impostor! I’ll not be"),
("I'll seize their", "I’ll seize their"),
("You've discovered my", "You’ve discovered my"),
("ploy! It doesn't matter, I'll still rout", "ploy! It doesn’t matter, I’ll still rout"),
("foiled, but I'll still flush", "foiled, but I’ll still flush"),
("Well, let's start climbing", "Well, let’s start climbing"),
("the mother Gryphon's eggs!", "the mother Gryphon’s eggs!"),
("circling the mountain's peak. Several", "circling the mountain’s peak. Several"),
("glared in Konrad's direction, their", "glared in Konrad’s direction, their"),
("by the river's edge!", "by the river’s edge!"),
("Isn't there somewhere", "Isn’t there somewhere"),
("Wesnoth. We can't let them", "Wesnoth. We can’t let them"),
("tore at Konrad's army, signaling", "tore at Konrad’s army, signaling"),
("Halgar Du'nar", "Halgar Du’nar"),
("Eventually Delfador's insistence that", "Eventually Delfador’s insistence that"),
("Who... who's here?", "Who... who’s here?"),
("been? I haven't seen you", "been? I haven’t seen you"),
("We can't get in!", "We can’t get in!"),
("But Konrad's party was", "But Konrad’s party was"),
("It's so dark", "It’s so dark"),
("'Never done us any harm?' Why, I", "‘Never done us any harm?’ Why, I"),
("haven't had such", "haven’t had such"),
("What's this? Gryphons", "What’s this? Gryphons"),
("your anger. Relgorn's manner may", "your anger. Relgorn’s manner may"),
("'Guest quarters'.", "‘Guest quarters’."),
("My uncle's family drowned", "My uncle’s family drowned"),
("What's this, more", "What’s this, more"),
("Some sort o' monster's appendage set", "Some sort o’ monster’s appendage set"),
("I was brawlin' with these", "I was brawlin’ with these"),
("stink, and I'm a", "stink, and I’m a"),
("my drift. Canna' swim past", "my drift. Canna’ swim past"),
("no' for lack o' tryin'.", "no’ for lack o’ tryin’."),
("around here. Let's go.", "around here. Let’s go."),
("last outpost o' civilization in", "last outpost o’ civilization in"),
("sway. You’ll no'", "sway. You’ll no’"),
("Northeast o' my keep,", "Northeast o’ my keep,"),
("Princess. Now let's continue", "Princess. Now let’s continue"),
("I can't believe it", "I can’t believe it"),
("Konrad or Li'sar", "Konrad or Li’sar"),
("An earthquake! We'll be trapped!", "An earthquake! We’ll be trapped!"),
("from the ruby's inner fire.", "from the ruby’s inner fire."),
("old wizard's normally inscrutable", "old wizard’s normally inscrutable"),
("Unan-Ka'tall", "Unan-Ka’tall"),
("thinking... I don't really want", "thinking... I don’t really want"),
("in my people's ancient tongue.", "in my people’s ancient tongue."),
("'The River of Bones'. Great and", "‘The River of Bones’. Great and"),
("come under Konrad's leadership, and", "come under Konrad’s leadership, and"),
("on Queen Asheviere's ill-", "on Queen Asheviere’s ill-"),
("belonged to Ila'alion, a mighty", "belonged to Ila’alion, a mighty"),
("Ila'alion's aid as", "Ila’alion’s aid as"),
("do to one's soul? What", "do to one’s soul? What"),
("became my mother's most trusted", "became my mother’s most trusted"),
("mother's rule, but", "mother’s rule, but"),
("my brothers. Asheviere's masterwork of", "my brothers. Asheviere’s masterwork of"),
("of the Lich's once-mortal", "of the Lich’s once-mortal"),
("of the Lich's", "of the Lich’s"),
("Aimucasur's lordship over", "Aimucasur’s lordship over"),
("does to one's soul? What", "does to one’s soul? What"),
("weeks past. We've found no", "weeks past. We’ve found no"),
("Haven't you found", "Haven’t you found"),
("I don't recommend it.", "I don’t recommend it."),
("am Princess Li'sar, the only", "am Princess Li’sar, the only"),
("Konrad and Li'sar)", "Konrad and Li’sar)"),
("Now when we've", "Now when we’ve"),
("I don't know. Maybe", "I don’t know. Maybe"),
("We drakes don't", "We drakes don’t"),
("go further we'll be forced", "go further we’ll be forced"),
("no choice. We'll fight our", "no choice. We’ll fight our"),
("We've found $random", "We’ve found $random"),
("Hurrah! We've killed their", "Hurrah! We’ve killed their"),
("the drakes haven't finished him", "the drakes haven’t finished him"),
("Princess Li'sar, do I", "Princess Li’sar, do I"),
("help him, I'll do my", "help him, I’ll do my"),
("am Princess Li'sar, and we", "am Princess Li’sar, and we"),
("Truly, I don't recommend entering", "Truly, I don’t recommend entering"),
("I've reached the", "I’ve reached the"),
("El'rien", "El’rien"),
("again! We don't know what", "again! We don’t know what"),
("was afraid that's what you", "was afraid that’s what you"),
("you, Lord El'rien. The hospitality", "you, Lord El’rien. The hospitality"),
("El'rien, we have", "El’rien, we have"),
("Elf, but I'm afraid I", "Elf, but I’m afraid I"),
("is rightfully Konrad's. You have", "is rightfully Konrad’s. You have"),
("Delfador! Li'sar has become", "Delfador! Li’sar has become"),
("friend. I don't want to", "friend. I don’t want to"),
("are mistaken. Li'sar is the", "are mistaken. Li’sar is the"),
("Li'sar should be", "Li’sar should be"),
("go with Li'sar.", "go with Li’sar."),
("Tell me, Li'sar, are you", "Tell me, Li’sar, are you"),
("been with Li'sar in the", "been with Li’sar in the"),
("dangerous! I didn't even think", "dangerous! I didn’t even think"),
("Kalenz's champion courtiers", "Kalenz’s champion courtiers"),
("Konrad's small band", "Konrad’s small band"),
("Konrad's heart leapt", "Konrad’s heart leapt"),
("the traitor Li'sar, with the", "the traitor Li’sar, with the"),
("Delfador, don't you think", "Delfador, don’t you think"),
("but it isn't to be.", "but it isn’t to be."),
("Konrad and Li'sar, to", "Konrad and Li’sar, to"),
("their forces! Let's see if", "their forces! Let’s see if"),
("knew that Asheviere's wrath would", "knew that Asheviere’s wrath would"),
("little prince's broken body,", "little prince’s broken body,"),
("if it weren't for her", "if it weren’t for her"),
("all of Asheviere's evil,", "all of Asheviere’s evil,"),
("this Li'sar is fit", "this Li’sar is fit"),
("mother. The land's blood is", "mother. The land’s blood is"),
("Li'sar strikes a", "Li’sar strikes a"),
("at last! Li'sar! You will", "at last! Li’sar! You will"),
("but I won't ever forget", "but I won’t ever forget"),
("was ended. Li'sar, daughter of", "was ended. Li’sar, daughter of"),
("Delfador became Li'sar's High Counselor,", "Delfador became Li’sar’s High Counselor,"),
("noble in Li'sar's court. He married Li'sar, and together", "noble in Li’sar’s court. He married Li’sar, and together"),
("is sleeping! You'd better not", "is sleeping! You’d better not"),
("reach it, Li'sar. I hope", "reach it, Li’sar. I hope"),
("know what I'm doing. Come,", "know what I’m doing. Come,"),
("of battle, Eldred's men turned", "of battle, Eldred’s men turned"),
("Garard's queen, Asheviere,", "Garard’s queen, Asheviere,"),
("than her husband's had.", "than her husband’s had."),
("But Garard's arch-mage, Delfador,", "But Garard’s arch-mage, Delfador,"),
("with his mother's advice", "with his mother’s advice"),
("'Fight no one", "“Fight no one"),
("from his shoulders.'", "from his shoulders.”"),
("the end Delfador's men were", "the end Delfador’s men were"),
("son's lifeless body,", "son’s lifeless body,"),
("the great mage's staff.", "the great mage’s staff."),
("news of Asheviere's orders reached", "news of Asheviere’s orders reached"),
("youngest of Garard's nephews, thereby", "youngest of Garard’s nephews, thereby"),
("sadly as Asheviere's reign of", "sadly as Asheviere’s reign of"),

# fix added in 1.10.0+svn
("Second, who you most", "Second, whom you most"),
),

"wesnoth-l" : (
# Typo fix at r44124
("devestating", "devastating"),
# Convert makeshift dashes and other stuff:
# conversion added in 1.9.0-svn
("are on the hunt - and", "are on the hunt — and"),
("and ruthlessness - and their", "and ruthlessness — and their"),
("of death - would only", "of death — would only"),
("my father - your grandfather - brought", "my father — your grandfather — brought"),
("catacombs - cover me.", "catacombs — cover me."),
("Liberty - Epilogue", "Liberty — Epilogue"),
("'If you wish to find us, head southwest. When you reach the land's end, just keep going. See you in the Three Sisters, old friend. - Relnan'", "“If you wish to find us, head southwest. When you reach the land’s end, just keep going. See you in the Three Sisters, old friend. ― Relnan”"),
# Fix screw up
# conversion added in 1.9.0-svn
("'If you wish to find us, head southwest. When you reach the land's end, just keep going. See you in the Three Sisters, old friend. — Relnan'", "“If you wish to find us, head southwest. When you reach the land’s end, just keep going. See you in the Three Sisters, old friend. ― Relnan”"),
# Straight apostrophes and quotes to curly ones
# conversion added in 1.9.0-svn
("By the sound of the wolves, the scent trail of Fal Khag's killers ends here.", "By the sound of the wolves, the scent trail of Fal Khag’s killers ends here."),
("But when the effects fall on mere peasants, the wrong of it may not seem so obvious from a noble's chair.", "But when the effects fall on mere peasants, the wrong of it may not seem so obvious from a noble’s chair."),
("Then they'll be back in force.", "Then they’ll be back in force."),
("Well, it won't be long until they report back to the local garrison with the details of your encounter.", "Well, it won’t be long until they report back to the local garrison with the details of your encounter."),
("In your many years as Delwyn's magistrate I have never known your judgment to err.", "In your many years as Delwyn’s magistrate I have never known your judgment to err."),
("there would be no help for the villagers, ground beneath the wheels of Asheviere's wrath.", "there would be no help for the villagers, ground beneath the wheels of Asheviere’s wrath."),
("After more than a week of fierce fighting, the main body of Asheviere's", "After more than a week of fierce fighting, the main body of Asheviere’s"),
("I think it's working!", "I think it’s working!"),
("It's done.", "It’s done."),
("I'm inside the tower! I'm going", "I’m inside the tower! I’m going"),
("I think... I think they're trying to storm Halstead itself... the fools!", "I think... I think they’re trying to storm Halstead itself... the fools!"),
("Baldras, Gwydion is Lord Maddock's son. This battle isn't just about Annuvin anymore.", "Baldras, Gwydion is Lord Maddock’s son. This battle isn’t just about Annuvin anymore."),
("They're in for a surprise.", "They’re in for a surprise."),
("They must think Wesnoth's army is spread out right now.", "They must think Wesnoth’s army is spread out right now."),
("I hope you're right.", "I hope you’re right."),
("I don't see how we can bring it down.", "I don’t see how we can bring it down."),
("the base of Halstead's frozen waves of stone.", "the base of Halstead’s frozen waves of stone."),
("from the earth's living rock.", "from the earth’s living rock."),
("338 years before Asheviere's betrayal,", "338 years before Asheviere’s betrayal,"),
("I'm the crazy one. Let's finish this folly.", "I’m the crazy one. Let’s finish this folly."),
("I haven't run this much in years.", "I haven’t run this much in years."),
("They've seen us, RUN!", "They’ve seen us, RUN!"),
("we're all in big trouble...", "we’re all in big trouble..."),
("so if we're careful enough we", "so if we’re careful enough we"),
("They're looking for us.", "They’re looking for us."),
("I'm not sure where we're supposed to go now.", "I’m not sure where we’re supposed to go now."),
("from sleep... we're not exactly sure.", "from sleep... we’re not exactly sure."),
("We don't. You have sought us out.", "We don’t. You have sought us out."),
("The former King's magic ministry", "The former King’s magic ministry"),
("It's not noble work, but our pursuits require plenty of gold.", "It’s not noble work, but our pursuits require plenty of gold."),
("Rest well tonight, because tomorrow's battle", "Rest well tonight, because tomorrow’s battle"),
("It's an ambush!", "It’s an ambush!"),
("I am seen so far from my Lord's borders.", "I am seen so far from my Lord’s borders."),
("It's this or nothing.", "It’s this or nothing."),
("You just said we can't beat their entire army!", "You just said we can’t beat their entire army!"),
("They mustn't take one step without", "They mustn’t take one step without"),
("What's worse is that she appears to", "What’s worse is that she appears to"),
("'We fled like criminals in the night, but we made sure that no one would harass us anymore.'", "“We fled like criminals in the night, but we made sure that no one would harass us anymore.”"),
("'Baldras, You would have been proud. We gave 'em hell. But in the end, it wasn't enough.'", "“Baldras, You would have been proud. We gave ’em hell. But in the end, it wasn’t enough.”"),
("with bitter humor he realized that Lord Maddock's men", "with bitter humor he realized that Lord Maddock’s men"),
("convinced Asheviere's second in command", "convinced Asheviere’s second in command"),
("As the main body of Asheviere's army", "As the main body of Asheviere’s army"),
("The spectacle of Halstead's destruction stunned them into", "The spectacle of Halstead’s destruction stunned them into"),
("Indeed, Asheviere's armies", "Indeed, Asheviere’s armies"),
("a second time, we're going to", "a second time, we’re going to"),
("I don't understand.", "I don’t understand."),
("Unless you want me to round up the city's", "Unless you want me to round up the city’s"),
("the King's son and betrayer.", "the King’s son and betrayer."),
("Queen's", "Queen’s"),
("the King's rule", "the King’s rule"),
("But we'll need to chase them all down if we're going to stop them.", "But we’ll need to chase them all down if we’re going to stop them."),
("And Harper... don't get yourself killed. I'm responsible for you now that your father is gone, and I won't dishonor his memory by breaking my promise to keep you safe.", "And Harper... don’t get yourself killed. I’m responsible for you now that your father is gone, and I won’t dishonor his memory by breaking my promise to keep you safe."),
("All right now, let's", "All right now, let’s"),
("They're getting closer", "They’re getting closer"),
("They'll come out when they see those riders, or us, approaching, but there aren't many of them... I wouldn't count on them being able to protect the village alone for long.", "They’ll come out when they see those riders, or us, approaching, but there aren’t many of them... I wouldn’t count on them being able to protect the village alone for long."),
("What's happening here?", "What’s happening here?"),
("Some mages are thrown out of the mage's", "Some mages are thrown out of the mage’s"),
("In the process they pick up a good deal of the Knight's", "In the process they pick up a good deal of the Knight’s"),
("take responsibility for the community's", "take responsibility for the community’s"),
("the village's wisest and most", "the village’s wisest and most"),
("but as Asheviere's grip", "but as Asheviere’s grip"),
("I don't want to touch it.", "I don’t want to touch it."),
("With bitter humor he realized that Lord Maddock's", "With bitter humor he realized that Lord Maddock’s"),

# Insert a diaeresis
("I am not so naive", "I am not so naïve"),
),

"wesnoth-low" : (
# Spelling fixes required at r44124
("engaged wit the defenders", "engaged with the defenders"),
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("follow you, Kalenz - but", "follow you, Kalenz — but"),
("Kalenz - lead us", "Kalenz — lead us"),
("them aid - it's clear", "them aid — it’s clear"),

# Fix screw up
# conversion added in 1.9.0-svn
("them aid — it's clear", "them aid — it’s clear"),

# Correct capitalization
# conversion added in 1.9.0-svn
("Breaking the siege", "Breaking the Siege"),
("Council ruling", "Council Ruling"),
("The Chief must die", "The Chief Must Die"),
("News from the front", "News from the Front"),
("Battle of the book", "Battle of the Book"),
("Council of hard choices", "Council of Hard Choices"),
("Ka'lian under attack", "Ka’lian under Attack"), # Also convert apostrophe to its curly form
("Hostile mountains", "Hostile Mountains"),

("Let's storm one of the orcs", "Let’s storm one of the orcs"),
("Let's see if ever the payment", "Let’s see if ever the payment"),
("Hasn't enough blood been", "Hasn’t enough blood been"),
("Don't dare to hurt her!", "Don’t dare to hurt her!"),

# Experimental conversion - apostrophes and em dashes
# Legend of Wesmere only
# Simons Mith, 23/08/2010, build 45968
("\"T'baran\"", "\"T’baran\""),
("Absolutely. Let's do it!", "Absolutely. Let’s do it!"),
("After the council's decision, Kalenz", "After the council’s decision, Kalenz"),
("and trolls dinna' go there", "and trolls dinna’ go there"),
("are but children's tales...still, be", "are but children’s tales...still, be"),
("are fresher; I'll go.", "are fresher; I’ll go."),
("at the Ka'lian took an", "at the Ka’lian took an"),
("behind me; I'll guard you", "behind me; I’ll guard you"),
("Believing that Wesmere's seeming safety", "Believing that Wesmere’s seeming safety"),
("both their mother's healing gifts", "both their mother’s healing gifts"),
("But Kalenz's story was", "But Kalenz’s story was"),
("But the Ka'lian's gold is the Ka'lian's. Would you", "But the Ka’lian’s gold is the Ka’lian’s. Would you"),
("by the council's passivity. But", "by the council’s passivity. But"),
("carry out Crelanu's dangerous but", "carry out Crelanu’s dangerous but"),
("Chieftain? But he's guarded better", "Chieftain? But he’s guarded better"),
("Cleodil's doubt is", "Cleodil’s doubt is"),
("council in Ka'lian and enlist", "council in Ka’lian and enlist"),
("Crelanu's book ...", "Crelanu’s book ..."),
("Crelanu's philter affecting", "Crelanu’s philter affecting"),
("Crelanu's place should", "Crelanu’s place should"),
("Dancer's Green", "Dancer’s Green"),
("different matter; I'm sure I", "different matter; I’m sure I"),
("do. I ha' heard tale", "do. I ha’ heard tale"),
("Don't flee. We", "Don’t flee. We"),
("Don't worry overmuch", "Don’t worry overmuch"),
("dose of Crelanu's invisibility philter. I'll go back", "dose of Crelanu’s invisibility philter. I’ll go back"),
("elves and I'll get da", "elves and I’ll get da"),
("Elves' Last Stand", "Elves’ Last Stand"),
("eventually, Landar's increasingly harsh", "eventually, Landar’s increasingly harsh"),
("find some o' my people", "find some o’ my people"),
("for Landar's revolt. But", "for Landar’s revolt. But"),
("from the Ka'lian.", "from the Ka’lian."),
("from the wolves-", "from the wolves —"),
("Gentlemen, don't squabble. We", "Gentlemen, don’t squabble. We"),
("get it. I'll catch up", "get it. I’ll catch up"),
("go, Kalenz, I'll guard the Ka'lian till our", "go, Kalenz, I’ll guard the Ka’lian till our"),
("gold and we'll get", "gold and we’ll get"),
("hate in Landar's heart. Something", "hate in Landar’s heart. Something"),
("heads! They ha' been too", "heads! They ha’ been too"),
("horse-pokers!! Run, let's get out", "horse-pokers!! Run, let’s get out"),
("How? Don't tell me", "How? Don’t tell me"),
("humans fall, we'll be", "humans fall, we’ll be"),
("I agree. What's done is", "I agree. What’s done is"),
("I dinna' think it", "I dinna’ think it"),
("If it weren't for Kalenz", "If it weren’t for Kalenz"),
("If the Ka'lian is too", "If the Ka’lian is too"),
("is the council's decision.", "is the council’s decision."),
("It willna' be easy.", "It willna’ be easy."),
("It's an ancient", "It’s an ancient"),
("It's not a dastone, it's the stone:", "It’s not a dastone, it’s the stone:"),
("It's not much", "It’s not much"),
("It's Olurf! He", "It’s Olurf! He"),
("It's the source", "It’s the source"),
("It's...it's monstrous!", "It’s...it’s monstrous!"),
("I'm sorry I", "I’m sorry I"),
("Kalenz. The Ka'lian council should", "Kalenz. The Ka’lian council should"),
("Kalenz's long childhood", "Kalenz’s long childhood"),
("Kalenz's veterans joined", "Kalenz’s veterans joined"),
("Ka'lian do...How do", "Ka’lian do...How do"),
("Landar! You're back! All", "Landar! You’re back! All"),
("Landar's remaining followers' lives were", "Landar’s remaining followers’ lives were"),
("learn of Kalenz's last and", "learn of Kalenz’s last and"),
("left the Ka'lian he had", "left the Ka’lian he had"),
("life by Crelanu's philter, he", "life by Crelanu’s philter, he"),
("make a dwarf's hackles rise;", "make a dwarf’s hackles rise;"),
("Maybe they don't", "Maybe they don’t"),
("minus expenses that's about even.", "minus expenses that’s about even."),
("my kin willna' be happy", "my kin willna’ be happy"),
("my men ha' all kept", "my men ha’ all kept"),
("No, don't leave your", "No, don’t leave your"),
("Now that we've retrieved the", "Now that we’ve retrieved the"),
("of the Kalenz's elves!", "of the Kalenz’s elves!"),
("of the Ka'lian, for it", "of the Ka’lian, for it"),
("on the Ka'lian! But if", "on the Ka’lian! But if"),
("one of Aquagar's creatures. Who", "one of Aquagar’s creatures. Who"),
("orcs alone! It's all over!", "orcs alone! It’s all over!"),
("orcs are everybody's enemy. When", "orcs are everybody’s enemy. When"),
("Our wolves' teeth will", "Our wolves’ teeth will"),
("o' this river", "o’ this river"),
("Perhaps the dwarves' intransigence can", "Perhaps the dwarves’ intransigence can"),
("poorly. The Ka'lian will need", "poorly. The Ka’lian will need"),
("raid on Kalenz's home, it", "raid on Kalenz’s home, it"),
("remnant evil Crelanu's philter had", "remnant evil Crelanu’s philter had"),
("responded to Cleodil's", "responded to Cleodil’s"),
("stinking trolls ha' stepped on", "stinking trolls ha’ stepped on"),
("take back what's ours!", "take back what’s ours!"),
("take the Ka'lian, all is", "take the Ka’lian, all is"),
("the bright moon's face again!", "the bright moon’s face again!"),
("the fun wi' the orcs", "the fun wi’ the orcs"),
("the fun. I'd ha' brought a", "the fun. I’d ha’ brought a"),
("The orcs ha' been defeated.", "The orcs ha’ been defeated."),
("the rest o'", "the rest o’"),
("their Great Chief's place. You", "their Great Chief’s place. You"),
("there! Now, let's", "there! Now, let’s"),
("They are no' fighting for gold, they're here for", "They are no’ fighting for gold, they’re here for"),
("they seem. Landar's", "they seem. Landar’s"),
("to the Ka'lian was already", "to the Ka’lian was already"),
("to the Ka'lian!", "to the Ka’lian!"),
("to the Ka'lian. Thanks to", "to the Ka’lian. Thanks to"),
("to the Ka'lian. There is", "to the Ka’lian. There is"),
("to the Ka'lian...", "to the Ka’lian..."),
("to the saurians' treasury was", "to the saurians’ treasury was"),
("Too late, we've taken all", "Too late, we’ve taken all"),
("treasury's garrison before", "treasury’s garrison before"),
("us. We ha' been forced", "us. We ha’ been forced"),
("Very soon, Landar's army appeared...", "Very soon, Landar’s army appeared..."),
("we have what's left of", "we have what’s left of"),
("We'll crush those", "We’ll crush those"),
("We'll need to", "We’ll need to"),
("What 'stone', foul and", "What ‘stone’, foul and"),
("What's more, King", "What’s more, King"),
("What's that, and", "What’s that, and"),
("What's this? It", "What’s this? It"),
("Where there's a contract,", "Where there’s a contract,"),
("WHo murdered El'Isomithir!", "WHo murdered El’Isomithir!"),
("With Crelanu's book in", "With Crelanu’s book in"),
("with Kalenz's army.", "with Kalenz’s army."),
("won! The Ka'lian is safe!", "won! The Ka’lian is safe!"),
("ye think we'd let you", "ye think we’d let you"),
("Yesss! It's the elves", "Yesss! It’s the elves"),
("yet over. Aquagar's curse was", "yet over. Aquagar’s curse was"),
("you may. They'll be back.", "you may. They’ll be back."),
("you mean, 'diverted their attention'?", "you mean, ‘diverted their attention’?"),
("you our army's vanguard? Hurry,", "you our army’s vanguard? Hurry,"),
("You won't get very", "You won’t get very"),
("'dastone'. I am", "‘dastone’. I am"),

# Fix capitalization
("\"LOW\"", "\"LoW\""),

# Fix apostrophes...
("while the foes'", "while the foes’"),
("Don't let them lure", "Don’t let them lure"),

# Spelling fix added in 1.9.9
("brake the treaty!", "break the treaty!"),
),

"wesnoth-multiplayer" : (
# Spelling fix required at r44124.
("helps alot", "helps a lot"),
# Convert makeshift dashes
# conversion added in 1.9.0-svn
("2p -", "2p —"),
("3p -", "3p —"),
("4p -", "4p —"),
("5p -", "5p —"),
("6p -", "6p —"),
("8p -", "8p —"),
("9p -", "9p —"),
("- - Seven Stones and Eleven", "— Seven Stones and Eleven"),
("- - Seven Stones - and the Elven", "— Seven Stones — and the Elven"),
("Bramwythl was left behind - in their haste, no one had remembered to find", "Bramwythl was left behind — in their haste, no one had remembered to find"),
("treasure that had been lost in these watery caves- a spear whose head was", "treasure that had been lost in these watery caves— a spear whose head was"),
("Single player mode - uses the reduced strength spawns", "Single player mode — uses the reduced strength spawns"),
("Two player mode - uses the full strength spawns.", "Two player mode — uses the full strength spawns."),
("Dark Forecast - a random survival scenario", "Dark Forecast — a random survival scenario"),
("They appear along the north, south and west map edges - though they are most", "They appear along the north, south and west map edges — though they are most"),
("- map design, spawn groups and scenario concept", "— map design, spawn groups and scenario concept"),
("- WML implementation", "— WML implementation"),
("- So this is the accursed valley - a land ours for the taking.", "So this is the accursed valley — a land ours for the taking."),
("watery caves- a spear whose", "watery caves— a spear whose"),
("-- map design, spawn groups and scenario concept", "— map design, spawn groups and scenario concept"),
("- WML implementation", "— WML implementation"),

("lair of Chak'kso Ney'yks, a legendary", "lair of Chak’kso Ney’yks, a legendary"),
("steal the Basilisk's gold.", "steal the Basilisk’s gold."),
("the Basilisk's cave,", "the Basilisk’s cave,"),
("(translation: ''The Hunts-Foes-In-", "(translation: “The Hunts-Foes-In-"),
("Boiling-Rivers Clan'').", "Boiling-Rivers Clan”)."),
("text: ''After he was", "text: “After he was"),
("Upper World, Chak'kso's enraged", "Upper World, Chak’kso’s enraged"),
("through ''pathways of stone''", "through ‘pathways of stone’"),
("of Chak'kso's hatred.'' Confused,", "of Chak’kso’s hatred.” Confused,"),
("was entitled ''Ney'yks of", "was entitled “Ney’yks of"),
("Gaze,'' and told", "Gaze,” and told"),
("as the ''Chak'kso Ney'yks,'' a name", "as the “Chak’kso Ney’yks,” a name"),
("tongue, meant ''Eldest Child of Chak'kso.''", "tongue, meant “Eldest Child of Chak’kso.”"),
("Chak'kso was believed", "Chak’kso was believed"),
("as a ''great demon of the earth'' in the", "as a ‘great demon of the earth’ in the"),
("histories, Chak'kso had once", "histories, Chak’kso had once"),
("''The Ney'yks,'' or ''The Eldest Child,'' in the", "“The Ney’yks,” or “The Eldest Child,” in the"),
("'mongst the Statues here was slain", "’mongst the Statues here was slain"),
("kept the area's goblins alive", "kept the area’s goblins alive"),
("wouldn't move. As the", "wouldn’t move. As the"),
("as ''The Dragonbane.'' It was", "as ‘The Dragonbane’. It was"),
("initiated a ''counterattack''", "initiated a ‘counterattack’"),
("lake, no other's, and though", "lake, no other’s, and though"),
("the Serpent's eyes. And the", "the Serpent’s eyes. And the"),
("hero of Wesnoth's Golden Age,", "hero of Wesnoth’s Golden Age,"),
("in the sun's glory.", "in the sun’s glory."),
("warmed by the sun's rays,", "warmed by the sun’s rays,"),
("Aethec Corryn: ''Talael Ryndoc!", "Aethec Corryn: “Talael Ryndoc!"),
("the death I'd have given you!''", "the death I’d have given you!”"),
("tooth of Chak'kso Ney'yks, an Elder", "tooth of Chak’kso Ney’yks, an Elder"),
("very-much-alive Chak'kso Ney'yks, who was, oddly", "very-much-alive Chak’kso Ney’yks, who was, oddly"),
("she hasn't moved her eyes", "she hasn’t moved her eyes"),
("Lose all your team's leaders", "Lose all your team’s leaders"),
("2p — Sulla's Ruins", "2p — Sulla’s Ruins"),
("many of Wesnoth's greatest", "many of Wesnoth’s greatest"),
("4p — Isar's Cross", "4p — Isar’s Cross"),
("6p — Crusaders' Field", "6p — Crusaders’ Field"),
("Frankly, I haven't got a", "Frankly, I haven’t got a"),

# Fix spelling
("enemy spawnpoints and types", "enemy spawn points and types"),

("dog, are yer legs made of stone?!?", "dog, are yer legs made of stone?!"),
),

"wesnoth-nr" : (
# Correct some English usage at revision 44124
("fortifications and siege them", "fortifications and besiege them"),
("Form, up men!", "Form up, men!"),
("the sunlit word", "the sunlit world"),
("bows and a cudgels", "bows and cudgels"),
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("afar -- hence they kept their weapons sharp, and reckoned themselves well ", "afar — hence they kept their weapons sharp, and reckoned themselves well "),
("forgetting that their ancestors had been free - and might have completely ", "forgetting that their ancestors had been free — and might have completely "),
("lives of the people of Dwarven Doors - forever.", "lives of the people of Dwarven Doors — forever."),
("It was an early spring day like any other; the humans - joyless beneath the ", "It was an early spring day like any other; the humans — joyless beneath the "),
("orcish whip - were dispiritedly planting the yearly crop. All at once, the ", "orcish whip — were dispiritedly planting the yearly crop. All at once, the "),
("The orcs have gotten careless - look how easily I stole these weapons and ", "The orcs have gotten careless — look how easily I stole these weapons and "),
("help, unlooked for, arrived in the form of a small band of woodsrunners -- ", "help, unlooked for, arrived in the form of a small band of woodsrunners — "),
("of them to hell! Oh, and just between me and you, it is actually good fun - ", "of them to hell! Oh, and just between me and you, it is actually good fun — "),
("Tallin, this situation is hopeless - there are endless monsters swarming ", "Tallin, this situation is hopeless — there are endless monsters swarming "),
("We are in a dire situation indeed - but just see - the trolls and the ", "We are in a dire situation indeed — but just see — the trolls and the "),
("replacement - whereas for every monster we kill, it seems that two more come ", "replacement — whereas for every monster we kill, it seems that two more come "),
("creatures and they know their caves backwards and forwards - so I am sure at ", "creatures and they know their caves backwards and forwards — so I am sure at "),
("Och, the dwarves of Knalga are themselves in desperate straits - but we ", "Och, the dwarves of Knalga are themselves in desperate straits — but we "),
("this sorry state - where are they now? From what I have seen, it is mostly ", "this sorry state — where are they now? From what I have seen, it is mostly "),
("not before we surprised and slew their leader - the cursed Khazg Black-Tusk. ", "not before we surprised and slew their leader — the cursed Khazg Black-Tusk. "),
("after we were trapped here - by ones and twos in the beginning, and now by ", "after we were trapped here — by ones and twos in the beginning, and now by "),
("our axes. So, it is not their existence I fear - for we have faced far worse ", "our axes. So, it is not their existence I fear — for we have faced far worse "),
("trials - but their rising numbers gives cause for worry...", "trials — but their rising numbers gives cause for worry..."),
("the mines! Let the guardsmen stay behind along with the noncombatants - for ", "the mines! Let the guardsmen stay behind along with the noncombatants — for "),
("But my my, what do we have here - Tallin.", "But my my, what do we have here — Tallin."),
("minions - gorge on the flesh of these scurrying little rats!", "minions — gorge on the flesh of these scurrying little rats!"),
("Tallin. He's lucky, he is - or he makes his own luck.", "Tallin. He's lucky, he is — or he makes his own luck."),
("Black-Tusk, but we survived the orcs and trolls -- only to be captured by ", "Black-Tusk, but we survived the orcs and trolls — only to be captured by "),
("What was that? Oh, woe - two big slabs of rock cutting off our retreat!", "What was that? Oh, woe — two big slabs of rock cutting off our retreat!"),
("Well, if you don't mind me saying - that certainly isn't the state of ", "Well, if you don’t mind me saying — that certainly isn’t the state of "),
("Interesting. I wonder who - or what - could have created such a powerful ", "Interesting. I wonder who — or what — could have created such a powerful "),
("Thus, compelled by some strange and irresistible force, Tallin - eyes glazed ", "Thus, compelled by some strange and irresistible force, Tallin — eyes glazed "),
("and mind unthinking - did the will of his new master.", "and mind unthinking — did the will of his new master."),
("dwarves - it's because of you that we ha' made a start rebuilding Knalga in ", "dwarves — it’s because of you that we ha’ made a start rebuilding Knalga in "),
("I am here simply finishing the job Khazg Black-Tusk started years ago - the ", "I am here simply finishing the job Khazg Black-Tusk started years ago — the "),
("And I am grateful - but what should we do from here? The number of humans ", "And I am grateful — but what should we do from here? The number of humans "),
("Orcs are fickle; if we negotiated a treaty with them - assuming they would ", "Orcs are fickle; if we negotiated a treaty with them — assuming they would "),
("accept - it would just be broken as soon as the next warlord arose among ", "accept — it would just be broken as soon as the next warlord arose among "),
("I have heard all about him from my good friend Stalrag. He is - or was ", "I have heard all about him from my good friend Stalrag. He is — or was "),
("anyway, haven't heard from him in years - the chief o' the villages that lay ", "anyway, haven’t heard from him in years — the chief o’ the villages that lay "),
("As Tallin sat staring blankly at the body of his former friend - loathing ", "As Tallin sat staring blankly at the body of his former friend — loathing "),
("and hating what he had become - he received an urgent summons from his ", "and hating what he had become — he received an urgent summons from his "),
("Pew! Nick of time too - I almost bought it there!", "Pew! Nick of time too — I almost bought it there!"),
("elves - for the sorceress was, in fact, a princess of the highest rank.", "elves — for the sorceress was, in fact, a princess of the highest rank."),
("our help in rescuing their Princess? You heard their message - those elves ", "our help in rescuing their Princess? You heard their message — those elves "),
("Well, suppose we do join up with the elves - assuming they will let us - and ", "Well, suppose we do join up with the elves — assuming they will let us — and "),
("Just our luck to be here when they're mustering a field force - probably to ", "Just our luck to be here when they’re mustering a field force — probably to "),
("Later, princess, first let's get outta -- uh, let us take our leave of this ", "Later, princess, first let’s get outta — uh, let us take our leave of this "),
("him, and so he fled. As he did so, visions of his dying friends - friends ", "him, and so he fled. As he did so, visions of his dying friends — friends "),
("that he had slain - flashed before his eyes, while their voices chided him ", "that he had slain — flashed before his eyes, while their voices chided him "),
("under his control, I have done unthinkable deeds - I have studied the most ", "under his control, I have done unthinkable deeds — I have studied the most "),
("never would have risked their lives - nay, if I were not there they wouldn't ", "never would have risked their lives — nay, if I were not there they wouldn’t "),
("Ruler of Men to ensure peace, harmony and above all - justice. For hundreds ", "Ruler of Men to ensure peace, harmony and above all — justice. For hundreds "),
("Tallin, accompanied by the ghost of the king - whose name he soon discovered ", "Tallin, accompanied by the ghost of the king — whose name he soon discovered "),
("to be Abhai - set off down the tunnels in search of the Rod of Justice. ", "to be Abhai — set off down the tunnels in search of the Rod of Justice. "),
("The Amulet of Strength, how fascinating. However - though it may be a useful ", "The Amulet of Strength, how fascinating. However — though it may be a useful "),
("place - nothing there but a bunch of bats.", "place — nothing there but a bunch of bats."),
("I found a way to the surface - that is if you feel like climbing straight up ", "I found a way to the surface — that is if you feel like climbing straight up "),
("We do not wish to fight you, Great One, we simply seek the Rod of Justice - ", "We do not wish to fight you, Great One, we simply seek the Rod of Justice — "),
("countless foes and raised countless generations of drakes - and now I am ", "countless foes and raised countless generations of drakes — and now I am "),
("The only problem - or the major one, anyway - is these blasted orcs. They ", "The only problem — or the major one, anyway — is these blasted orcs. They "),
("determined to make Knalga into an orcish stronghold - enslaving or killing ", "determined to make Knalga into an orcish stronghold — enslaving or killing "),
("There is a large elvish force not far from here who - we believe anyway - ", "There is a large elvish force not far from here who — we believe anyway — "),
("hesitated and found himself unable to proceed. Abhai came forth - his ", "hesitated and found himself unable to proceed. Abhai came forth — his "),
("defenders - and fearlessly lifted the Rod in his ghostly hands. He paused ", "defenders — and fearlessly lifted the Rod in his ghostly hands. He paused "),
("Tallin raised his eyes to look into Abhai's and Abhai met his gaze - ", "Tallin raised his eyes to look into Abhai’s and Abhai met his gaze — "),
("your doubts at rest.' Tallin held the gaze for one more long moment - and ", "your doubts at rest.” Tallin held the gaze for one more long moment — and "),
("Yes, Tallin - it is I.", "Yes, Tallin — it is I."),
("How did you get here - I thought you you were dead... and for that matter, ", "How did you get here — I thought you you were dead... and for that matter, "),
("of unimaginable splendor, glory and transcendence - the very realm of the ", "of unimaginable splendor, glory and transcendence — the very realm of the "),
("They are all here, princess! Hand picked by your father - the finest and ", "They are all here, princess! Hand picked by your father — the finest and "),
("They are all here, princess! Hand picked by your father - the finest and ", "They are all here, princess! Hand picked by your father — the finest and "),
("and leave them a small guard. Then they pressed onward to rescue Hidel - and ", "and leave them a small guard. Then they pressed onward to rescue Hidel — and "),
("Cheer up -- you won't have to live with your failure for long...*snicker*", "Cheer up — you won’t have to live with your failure for long...*snicker*"),
("have a few scores to settle with you. Take them, troops - I want no orc left ", "have a few scores to settle with you. Take them, troops — I want no orc left "),
("been crushed. This relieves a blight on our land - but if we do not address ", "been crushed. This relieves a blight on our land — but if we do not address "),
("what race they come from - even orcs.", "what race they come from — even orcs."),
("Tallin, as you know, I have been separated from my race and kinsmen - well, ", "Tallin, as you know, I have been separated from my race and kinsmen — well, "),
("except Eryssa - for many years now. I wish to go back to the elvish forests ", "except Eryssa — for many years now. I wish to go back to the elvish forests "),
("Lords of Light - or Darkness - guide you on your path. For those of you who ", "Lords of Light — or Darkness — guide you on your path. For those of you who "),
("are staying - come, we have much to do.", "are staying — come, we have much to do."),
("- and unknown - world to trade with the Dwarves of Knalga.", "— and unknown — world to trade with the Dwarves of Knalga."),
("Thus, from a small, enslaved community, the people of Dwarven Doors - by ", "Thus, from a small, enslaved community, the people of Dwarven Doors — by "),
("their fortitude, valor, and wisdom - brought the Northlands out of the ", "their fortitude, valor, and wisdom — brought the Northlands out of the "),

# Fix screw up
# conversion added in 1.9.0-svn
("Cheer up — you won't have to live with your failure for long...*snicker*", "Cheer up — you won’t have to live with your failure for long...*snicker*"),
("your doubts at rest.' Tallin held the gaze for one more long moment — and ", "your doubts at rest.” Tallin held the gaze for one more long moment — and "),
("Tallin raised his eyes to look into Abhai's and Abhai met his gaze — ", "Tallin raised his eyes to look into Abhai’s and Abhai met his gaze — "),
("never would have risked their lives — nay, if I were not there they wouldn't ", "never would have risked their lives — nay, if I were not there they wouldn’t "),
("Later, princess, first let's get outta — uh, let us take our leave of this ", "Later, princess, first let’s get outta — uh, let us take our leave of this "),
("Just our luck to be here when they're mustering a field force — probably to ", "Just our luck to be here when they’re mustering a field force — probably to "),
("anyway, haven't heard from him in years — the chief o' the villages that lay ", "anyway, haven’t heard from him in years — the chief o’ the villages that lay "),
("dwarves — it's because of you that we ha' made a start rebuilding Knalga in ", "dwarves — it’s because of you that we ha’ made a start rebuilding Knalga in "),
("Well, if you don't mind me saying — that certainly isn't the state of ", "Well, if you don’t mind me saying — that certainly isn’t the state of "),

("...Don't tax yourself... princess....", "... Don’t tax yourself... princess..."),
("Abhai: I don't think death will", "Abhai: I don’t think death will"),
("Abhai: You haven't? It was an", "Abhai: You haven’t? It was an"),
("about Tallin's will grew stronger.", "about Tallin’s will grew stronger."),
("accomplished, men! Now let's crush the rest", "accomplished, men! Now let’s crush the rest"),
("again took up Rakshas's trail. They soon", "again took up Rakshas’s trail. They soon"),
("again, Morvin? Well, I'll", "again, Morvin? Well, I’ll"),
("all of a necromancer's creations, I must", "all of a necromancer’s creations, I must"),
("all, most people can't hold the Rod", "all, most people can’t hold the Rod"),
("and for all. Let's give", "and for all. Let’s give"),
("and he moved awa' up here to", "and he moved awa’ up here to"),
("And if you don't keep your mouth", "And if you don’t keep your mouth"),
("and looked to Hidel's ward as their", "and looked to Hidel’s ward as their"),
("and return here, we'll ha' made a", "and return here, we’ll ha’ made a"),
("and started down Rakshas's", "and started down Rakshas’s"),
("and the mage Ro'Arthian and his brother,", "and the mage Ro’Arthian and his brother,"),
("apprehension. 'I have faith in you, my boy,' Abhai said, 'Take this, and put", "apprehension. “I have faith in you, my boy,” Abhai said, “Take this, and put"),
("are bonny fighters wi' a sword as", "are bonny fighters wi’ a sword as"),
("are heavy-footed creatures; Rakshas's trail was readily", "are heavy-footed creatures; Rakshas’s trail was readily"),
("aren't ready to give", "aren’t ready to give"),
("as other parts o' Knalga.", "as other parts o’ Knalga."),
("At Hidel's call Tallin approached", "At Hidel’s call Tallin approached"),
("attacking her won't make the situation", "attacking her won’t make the situation"),
("away, Tallin's former friends had attacked Malifor's study with overwhelming", "away, Tallin’s former friends had attacked Malifor’s study with overwhelming"),
("Awww, aren't they the most", "Awww, aren’t they the most"),
("Aye! We'll have to lay", "Aye! We’ll have to lay"),
("Aye, it's true we do", "Aye, it’s true we do"),
("Aye. And we dinna' just want to", "Aye. And we dinna’ just want to"),
("Aye. Word ha' spread, and dwarves", "Aye. Word ha’ spread, and dwarves"),
("back to back, don't let them penetrate", "back to back, don’t let them penetrate"),
("bargain. If you don't agree to", "bargain. If you don’t agree to"),
("become legends! 'The two terrible mages,", "become legends! “The two terrible mages,"),
("body and I don't know how to", "body and I don’t know how to"),
("boys, that weakling Al'Tar can't even keep his", "boys, that weakling Al’Tar can’t even keep his"),
("brain has rotted awa' entire. Think on", "brain has rotted awa’ entire. Think on"),
("buddy, and now it's", "buddy, and now it’s"),
("But I'm sore vexed. We", "But I’m sore vexed. We"),
("but if you don't hurry up and", "but if you don’t hurry up and"),
("by himself. We don't know", "by himself. We don’t know"),
("can hear me, Ro'Arthian. Please, the fate", "can hear me, Ro’Arthian. Please, the fate"),
("certainty, those skeletons won't like us getting", "certainty, those skeletons won’t like us getting"),
("Come on boys, let's chop them to", "Come on boys, let’s chop them to"),
("Come on men, let's go", "Come on men, let’s go"),
("Come on Thera, let's destroy that old", "Come on Thera, let’s destroy that old"),
("Come on, Ro'Sothian, we really don't have to be", "Come on, Ro’Sothian, we really don’t have to be"),
("Come what may, we'll handle them. Come on now, let's find those", "Come what may, we’ll handle them. Come on now, let’s find those"),
("couldna' take it and", "couldna’ take it and"),
("Dang it! They're gone, and the", "Dang it! They’re gone, and the"),
("desecrated my life's work, now I", "desecrated my life’s work, now I"),
("destroy its arms we'll be relatively safe", "destroy its arms we’ll be relatively safe"),
("DISGUSTING LITTLE VERMIN! DON'T YOU KNOW HOW", "DISGUSTING LITTLE VERMIN! DON’T YOU KNOW HOW"),
("do business here: 'I do this for", "do business here: “I do this for"),
("do I want? I'll tell you what", "do I want? I’ll tell you what"),
("do this for me.' No, we seek", "do this for me.” No, we seek"),
("do to survive. We'll get more food", "do to survive. We’ll get more food"),
("Don't bother heading down", "Don’t bother heading down"),
("don't get it treated", "don’t get it treated"),
("Don't give in to", "Don’t give in to"),
("don't say anything to", "don’t say anything to"),
("Don't worry, Thera, you", "Don’t worry, Thera, you"),
("Don't! Please, my friend,", "Don’t! Please, my friend,"),
("down here, but ye'll not likely see", "down here, but ye’ll not likely see"),
("dragon. If you don't hand it over", "dragon. If you don’t hand it over"),
("enough. They taste unco' foul but that's how", "enough. They taste unco’ foul but that’s how"),
("Even if we don't catch them then", "Even if we don’t catch them then"),
("Even if we don't catch them, we", "Even if we don’t catch them, we"),
("everywhere, we simply didna'", "everywhere, we simply didna’"),
("extremely rich nation. Malifor's treasury is hardly", "extremely rich nation. Malifor’s treasury is hardly"),
("fact I think it's best that we", "fact I think it’s best that we"),
("flank, and now it's your", "flank, and now it’s your"),
("Fools! Don't think it's so easy to", "Fools! Don’t think it’s so easy to"),
("from Malifor's treasury and we", "from Malifor’s treasury and we"),
("from them. Now they're killing each other.", "from them. Now they’re killing each other."),
("Gee, what's with all these", "Gee, what’s with all these"),
("Geez, what a fool's quest. Everyone in", "Geez, what a fool’s quest. Everyone in"),
("go to his Master's aid. When he", "go to his Master’s aid. When he"),
("gold cannot. Now let's run down Rakshas", "gold cannot. Now let’s run down Rakshas"),
("got from my guards' boastings and foul", "got from my guards’ boastings and foul"),
("Great Chamber eh? Doesn't look like there", "Great Chamber eh? Doesn’t look like there"),
("Great. Now let's get back to", "Great. Now let’s get back to"),
("Grrr, Ha'Tuil has never failed", "Grrr, Ha’Tuil has never failed"),
("ha' been as much", "ha’ been as much"),
("ha' the manpower.", "ha’ the manpower."),
("Hamel's held open their", "Hamel’s held open their"),
("have died.... a warrior's death....", "have died... a warrior’s death..."),
("havena' forgotten the old", "havena’ forgotten the old"),
("He didn't keep us alive", "He didn’t keep us alive"),
("he disappeared again. Let's find him and", "he disappeared again. Let’s find him and"),
("help, I guess we'll nae have any", "help, I guess we’ll nae have any"),
("her was a fool's quest to begin", "her was a fool’s quest to begin"),
("Hey look, it's an ancient door.", "Hey look, it’s an ancient door."),
("Hey, that's not right. The good guys aren't supposed to die. Oh well, I'll", "Hey, that’s not right. The good guys aren’t supposed to die. Oh well, I’ll"),
("Hidel's death was a", "Hidel’s death was a"),
("Hiera'Shirsha?", "Hiera’Shirsha?"),
("his prisoners? He didn't seem to be", "his prisoners? He didn’t seem to be"),
("his prisoners? He doesn't really", "his prisoners? He doesn’t really"),
("Hmmm, you aren't as bad as", "Hmmm, you aren’t as bad as"),
("Hmmm...I think you've the right of", "Hmmm... I think you’ve the right of"),
("Hold! We aren't friends of Malifor", "Hold! We aren’t friends of Malifor"),
("host. But we can't", "host. But we can’t"),
("I am sorry, Ro'Arthian. I should have", "I am sorry, Ro’Arthian. I should have"),
("I didn't know that you", "I didn’t know that you"),
("I don't know what it", "I don’t know what it"),
("I don't think so, my", "I don’t think so, my"),
("I don't think so, you", "I don’t think so, you"),
("I don't think there is", "I don’t think there is"),
("I know you're impressive wi' a pitchfork, and", "I know you’re impressive wi’ a pitchfork, and"),
("I sense the Rod's power coming from", "I sense the Rod’s power coming from"),
("if the elves dinna' fight and pay", "if the elves dinna’ fight and pay"),
("If you don't mind me asking,", "If you don’t mind me asking,"),
("if you know what's good for you.", "if you know what’s good for you."),
("in disrepair. It doesn't look like", "in disrepair. It doesn’t look like"),
("in line. I don't know why you", "in line. I don’t know why you"),
("in mission, but Al'tar dead by peasant", "in mission, but Al’tar dead by peasant"),
("in the meantime, I'm leaving!", "in the meantime, I’m leaving!"),
("in the way. Let's give them a", "in the way. Let’s give them a"),
("In time, Knalga's caverns were refurbished,", "In time, Knalga’s caverns were refurbished,"),
("In truth, we ha' no idea. They", "In truth, we ha’ no idea. They"),
("is a risk we'll have to take.", "is a risk we’ll have to take."),
("is great...but we dinna' ha' much of weapons", "is great... but we dinna’ ha’ much of weapons"),
("is so, then it's time to end", "is so, then it’s time to end"),
("is up, men, let's hurry up and", "is up, men, let’s hurry up and"),
("it shall be. You've my leave and", "it shall be. You’ve my leave and"),
("it to 'em!", "it to ’em!"),
("It's an escape tunnel. The hole wasn't big enough for", "It’s an escape tunnel. The hole wasn’t big enough for"),
("It's too late, I", "It’s too late, I"),
("It's very easy, Tallin,", "It’s very easy, Tallin,"),
("it, Father! We don't help no-one. Not", "it, Father! We don’t help no-one. Not"),
("Knalga. And d'ye ken those two", "Knalga. And d’ye ken those two"),
("lay amidst the Dragon's hoarded", "lay amidst the Dragon’s hoarded"),
("let's get back at", "let’s get back at"),
("like someone really doesn't want us going", "like someone really doesn’t want us going"),
("little vermin and I'll be right over.", "little vermin and I’ll be right over."),
("looks of things, they've busked themselves for", "looks of things, they’ve busked themselves for"),
("Lord Hamel, some o' us would like", "Lord Hamel, some o’ us would like"),
("lost and don't come back, unless you're tired of life!", "lost and don’t come back, unless you’re tired of life!"),
("made easier by Malifor's ending.", "made easier by Malifor’s ending."),
("Malifor's evil summons, but", "Malifor’s evil summons, but"),
("me, lich, and you'll find yourself dangling", "me, lich, and you’ll find yourself dangling"),
("money they don't have.", "money they don’t have."),
("more then two days' march from here.", "more then two days’ march from here."),
("must do both. I'm thinking it might", "must do both. I’m thinking it might"),
("my friend. You won't believe what a", "my friend. You won’t believe what a"),
("nae! Ye'll need yon gold", "nae! Ye’ll need yon gold"),
("Never! If you ha' been sent by", "Never! If you ha’ been sent by"),
("none of any dwarf's or human’s business,", "none of any dwarf’s or human’s business,"),
("not so sure it's that simple Tallin.", "not so sure it’s that simple Tallin."),
("not suppress the people's joy in their", "not suppress the people’s joy in their"),
("now, anyway. Now let's get to the", "now, anyway. Now let’s get to the"),
("Now, let's get back to", "Now, let’s get back to"),
("Of course we won't do that. As", "Of course we won’t do that. As"),
("of Darkness, I haven't seen you in", "of Darkness, I haven’t seen you in"),
("of Light! Elenia! It's you! It has", "of Light! Elenia! It’s you! It has"),
("of orcs, belike they'd kill the princess,", "of orcs, belike they’d kill the princess,"),
("off many of it's", "off many of it’s"),
("Oh, dinna' mind him, that'd be Camerin. He", "Oh, dinna’ mind him, that’d be Camerin. He"),
("On Father Morvin's advice, the Council", "On Father Morvin’s advice, the Council"),
("only make the Master's punishment for you", "only make the Master’s punishment for you"),
("or at least, what's left o'", "or at least, what’s left o’"),
("orcish stronghold. We can't let that happen!", "orcish stronghold. We can’t let that happen!"),
("orcs then, princess? Don't you wish to", "orcs then, princess? Don’t you wish to"),
("our gold. Now let's run", "our gold. Now let’s run"),
("our side. Now let's plan our way", "our side. Now let’s plan our way"),
("ourselves, Tallin. We ha' failed the trust", "ourselves, Tallin. We ha’ failed the trust"),
("ourselves. We're but a remnant", "ourselves. We’re but a remnant"),
("out of his mother's arms, when Konrad", "out of his mother’s arms, when Konrad"),
("passed, but eventually Malifor's", "passed, but eventually Malifor’s"),
("ploy. Nae sooner ha' we thinned the", "ploy. Nae sooner ha’ we thinned the"),
("powerful magician called Ro'Arthian and his brother Ro'Sothian.", "powerful magician called Ro’Arthian and his brother Ro’Sothian."),
("Price?!? Don't insult us, Tallin.", "Price?! Don’t insult us, Tallin."),
("problem lad, we ha' been stranded in", "problem lad, we ha’ been stranded in"),
("quickly cut off it's arms we may", "quickly cut off it’s arms we may"),
("regenerate over time; it's doubtful we can", "regenerate over time; it’s doubtful we can"),
("Remember, Ro'Arthian, we need willing", "Remember, Ro’Arthian, we need willing"),
("Ro'Arthian and Ro'Sothian, you two have", "Ro’Arthian and Ro’Sothian, you two have"),
("Ro'Arthian's brother, Ro'Sothian went south with", "Ro’Arthian’s brother, Ro’Sothian went south with"),
("Ro'Arthian, I am not", "Ro’Arthian, I am not"),
("Ro'Arthian, listen to me. We don't want to fight", "Ro’Arthian, listen to me. We don’t want to fight"),
("Ro'Sothian! How are you", "Ro’Sothian! How are you"),
("said that I wasn't going to repay", "said that I wasn’t going to repay"),
("saying something about 'The Master'.", "saying something about ‘The Master’."),
("seems the rescue ha' any chance of", "seems the rescue ha’ any chance of"),
("Seriously Hamel, I don't know if we’re", "Seriously Hamel, I don’t know if we’re"),
("shall be, Tallin, I'll hold the caves", "shall be, Tallin, I’ll hold the caves"),
("simply ha' not the manpower.", "simply ha’ not the manpower."),
("small amount of Al'Tar's gold. He fled", "small amount of Al’Tar’s gold. He fled"),
("snot! If it wasn't for you we wouldn't have lost our", "snot! If it wasn’t for you we wouldn’t have lost our"),
("Sobbing against Tallin's chest, Eryssa told", "Sobbing against Tallin’s chest, Eryssa told"),
("Some monster that Malifor's minions greatly fear", "Some monster that Malifor’s minions greatly fear"),
("some room and you'll", "some room and you’ll"),
("south and we haven't found Malifor yet...", "south and we haven’t found Malifor yet..."),
("swords. We couldna' claim to be expert weaponsmiths wi'out kenning how to", "swords. We couldna’ claim to be expert weaponsmiths wi’out kenning how to"),
("Tallin's proposal, the endless", "Tallin’s proposal, the endless"),
("Tallin. He's lucky, he is", "Tallin. He’s lucky, he is"),
("thanks, Stalrag and Ro'Arthian. However, I have", "thanks, Stalrag and Ro’Arthian. However, I have"),
("that Hidel's elves had been", "that Hidel’s elves had been"),
("that way... We don't like it, but if that's what", "that way... We don’t like it, but if that’s what"),
("the air they canna' see over them.", "the air they canna’ see over them."),
("the dead. I don't", "the dead. I don’t"),
("the door to Malifor's study. Are we", "the door to Malifor’s study. Are we"),
("the elf! We can't let that happen!", "the elf! We can’t let that happen!"),
("the first place. Dinna' ye speak of", "the first place. Dinna’ ye speak of"),
("the orcish hordes!'", "the orcish hordes!”"),
("The orcs still haven't given up their", "The orcs still haven’t given up their"),
("the orcs. It's time to give", "the orcs. It’s time to give"),
("the rest here. We'll store it in", "the rest here. We’ll store it in"),
("the time you've laid him to", "the time you’ve laid him to"),
("the tunnels into dawn's early light.", "the tunnels into dawn’s early light."),
("The war council's thoughts quickly turned", "The war council’s thoughts quickly turned"),
("them if you can't.", "them if you can’t."),
("them: 'Hold fast your hope,", "them: “Hold fast your hope,"),
("Then one day Al'Tar, the current 'master' of Dwarven Doors,", "Then one day Al’Tar, the current ‘master’ of Dwarven Doors,"),
("then that Princess Li'sar was in hot", "then that Princess Li’sar was in hot"),
("then we thought, let's go", "then we thought, let’s go"),
("then ye could ha' done in", "then ye could ha’ done in"),
("Thera, don't you think that this isn't really a good", "Thera, don’t you think that this isn’t really a good"),
("there are, we dinna' yet ken of", "there are, we dinna’ yet ken of"),
("there skills beyond swingin' a weapon", "there skills beyond swingin’ a weapon"),
("these orcs! IN HIDEL'S", "these orcs! IN HIDEL’S"),
("these waters we can't destroy the creature.", "these waters we can’t destroy the creature."),
("Think of it, Ro'Arthian, it is a", "Think of it, Ro’Arthian, it is a"),
("This isn't looking good. Not", "This isn’t looking good. Not"),
("this stuff. But that's probably why they", "this stuff. But that’s probably why they"),
("tight. I wonder what's behind it.", "tight. I wonder what’s behind it."),
("to cut off it's arms fast enough.", "to cut off it’s arms fast enough."),
("to help us. What's more, if Stalrag", "to help us. What’s more, if Stalrag"),
("to him. Nope, we'll never hurt you.", "to him. Nope, we’ll never hurt you."),
("Together we'll rip them to", "Together we’ll rip them to"),
("training of you hasna' been for naught.", "training of you hasna’ been for naught."),
("trolls. But it ha' been", "trolls. But it ha’ been"),
("Tusk's troops, but we", "Tusk’s troops, but we"),
("uncoordinated and erratic, wi' the orcs as", "uncoordinated and erratic, wi’ the orcs as"),
("underwater creature. They'll try to pummel", "underwater creature. They’ll try to pummel"),
("up positions outside Malifor's study door.", "up positions outside Malifor’s study door."),
("up, father, he isn't answering. I am", "up, father, he isn’t answering. I am"),
("us you WON'T come back.", "us you <i>won’t</i> come back."), # New <i> tags
("wants to take 'em and fight with", "wants to take ’em and fight with"),
("was a piece o' good work, lad,", "was a piece o’ good work, lad,"),
("was decided that Tallin's men should sally", "was decided that Tallin’s men should sally"),
("was led by Ro'Arthian and was composed", "was led by Ro’Arthian and was composed"),
("water. And there doesn't seem to be", "water. And there doesn’t seem to be"),
("we are to ha' anything", "we are to ha’ anything"),
("we ha' survived all these", "we ha’ survived all these"),
("we rose against Al'Tar and defeated his", "we rose against Al’Tar and defeated his"),
("we would gain o' great tactical", "we would gain o’ great tactical"),
("We'll have to kill", "We’ll have to kill"),
("Well, that certainly isn't the state of", "Well, that certainly isn’t the state of"),
("well. Why, it can't have", "well. Why, it can’t have"),
("What's more, some of", "What’s more, some of"),
("What's with this whole 'Master' business? It's starting to make", "What’s with this whole ‘Master’ business? It’s starting to make"),
("When Tallin's party had shaken", "When Tallin’s party had shaken"),
("where we couldna' reach while the", "where we couldna’ reach while the"),
("while you and Stalrag's", "while you and Stalrag’s"),
("Whoa! Maybe he isn't so friendly after", "Whoa! Maybe he isn’t so friendly after"),
("with Hidel's head in her", "with Hidel’s head in her"),
("With Malifor's death, the magical controls over Tallin's will vanished.", "With Malifor’s death, the magical controls over Tallin’s will vanished."),
("with the orcs I'll send you a", "with the orcs I’ll send you a"),
("with you. I wouldn't even think", "with you. I wouldn’t even think"),
("wood-knot, I'd lay odds on", "wood-knot, I’d lay odds on"),
("would emerge and we'd find oursel' back in the", "would emerge and we’d find oursel’ back in the"),
("would we could ha' done more.", "would we could ha’ done more."),
("Wouldn't miss it. Maybe", "Wouldn’t miss it. Maybe"),
("wouldna' ally themselves with the likes o' us.", "wouldna’ ally themselves with the likes o’ us."),
("Wow, that's a long way", "Wow, that’s a long way"),
("wrong with pitchforks, I'll give these orcish", "wrong with pitchforks, I’ll give these orcish"),
("ye'll need and leave", "ye’ll need and leave"),
("Ye'll need to travel", "Ye’ll need to travel"),
("years, however, I ha' noticed that there", "years, however, I ha’ noticed that there"),
("you shall be free!' Then he was", "you shall be free!” Then he was"),
("your kind that won't leave US in", "your kind that won’t leave <i>us</i> in"),

# Minor usage fixes added in 1.9.9
("one you little guys you might fit", "one of you little guys might fit"),
("yours and your brothers", "your and your brother’s"),
("What in the world gave you that idea.", "What in the world gave you that idea?"),

# Converted a period to a question marke
("into disarray once more.", "into disarray once more?"),
),

"wesnoth-thot" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("then living -- all", "then living — all"),
("the Hammer -- dropped", "the Hammer — dropped"),
("to my eyes - an", "to my eyes — an"),
("captain -- but", "captain — but"),
("prisoners - which", "prisoners — which"),
("seen again - I began", "seen again — I began"),
("tightly locked - and, I", "tightly locked — and, I"),

("'Our' valley, says Master", "‘Our’ valley, says Master"),
("<i>is</i> Karrag? We can't have been more", "<i>is</i> Karrag? We can’t have been more"),
("A book. Thursagan's book, in a", "A book. Thursagan’s book, in a"),
("a teleportation device. What's more, the", "a teleportation device. What’s more, the"),
("Aiglondur, what ye ha' found is disturbing,", "Aiglondur, what ye ha’ found is disturbing,"),
("am Ratheln. If you're traveling down the", "am Ratheln. If you’re traveling down the"),
("an instructor, and they're expecting me.", "an instructor, and they’re expecting me."),
("and your precious 'Alliance'.", "and your precious ‘Alliance’."),
("Aye. And I'd not leave anyone", "Aye. And I’d not leave anyone"),
("Aye. But I've a grim feeling", "Aye. But I’ve a grim feeling"),
("Aye. Ye'll recall that in", "Aye. Ye’ll recall that in"),
("Aye? Well, you'd know such things", "Aye? Well, you’d know such things"),
("be of the Alliance's making, right enough,", "be of the Alliance’s making, right enough,"),
("be the Northern Alliance's arm today, and", "be the Northern Alliance’s arm today, and"),
("cliffs near the valley's end. They were", "cliffs near the valley’s end. They were"),
("close to the Alliance's border? That", "close to the Alliance’s border? That"),
("Don't you wonder what", "Don’t you wonder what"),
("east. I believe we've found", "east. I believe we’ve found"),
("eyes of the dwarves' history, a deep", "eyes of the dwarves’ history, a deep"),
("Fellow dwarves, hail! We're looking for a", "Fellow dwarves, hail! We’re looking for a"),
("hear it with one's own ears? It", "hear it with one’s own ears? It"),
("hills, from whom we've heard nothing since", "hills, from whom we’ve heard nothing since"),
("his victims. The 'dirtgrubbers' he seems to", "his victims. The ‘dirtgrubbers’ he seems to"),
("I thought Karrag's fight was against", "I thought Karrag’s fight was against"),
("I was told I'd find Aiglondur of", "I was told I’d find Aiglondur of"),
("I'll grant that was", "I’ll grant that was"),
("I'm surprised myself that", "I’m surprised myself that"),
("In case you haven't noticed, there is", "In case you haven’t noticed, there is"),
("It's been...it's been years since I've been down here.", "It’s been... it’s been years since I’ve been down here."),
("Karrag's will, and his", "Karrag’s will, and his"),
("Let's take him with", "Let’s take him with"),
("Nae sign o' bandits I can", "Nae sign o’ bandits I can"),
("none such since Thursagan's day. And for", "none such since Thursagan’s day. And for"),
("now and we wouldn't know it", "now and we wouldn’t know it"),
("of the Northern Alliance's lands.", "of the Northern Alliance’s lands."),
("of the pass. They've an opening", "of the pass. They’ve an opening"),
("of this shack. I've had nothing to", "of this shack. I’ve had nothing to"),
("on the dais, they're all", "on the dais, they’re all"),
("presence of Master Perrin's journeymen gives you", "presence of Master Perrin’s journeymen gives you"),
("see to my troop's care. Food will", "see to my troop’s care. Food will"),
("something doesn't feel right here.", "something doesn’t feel right here."),
("The Alliance's road ended at", "The Alliance’s road ended at"),
("the Alliance's tribes if we", "the Alliance’s tribes if we"),
("the Kal Karthans we'll have to", "the Kal Karthans we’ll have to"),
("the rank, and ye've shown the wits", "the rank, and ye’ve shown the wits"),
("the so called 'dirtgrubbers'. Why would", "the so called ‘dirtgrubbers’. Why would"),
("their journey. I'll return with your", "their journey. I’ll return with your"),
("to it. It won't do to have", "to it. It won’t do to have"),
("to the east; we've had few reports", "to the east; we’ve had few reports"),
("to the power o' the Hammer —", "to the power o’ the Hammer —"),
("us that the Hammer's power will smash", "us that the Hammer’s power will smash"),
("users from Master Perrin's academy in your", "users from Master Perrin’s academy in your"),
("was in my grandsire's time.", "was in my grandsire’s time."),
("we will regret Perrin's passing. There is", "we will regret Perrin’s passing. There is"),
("We're grateful for your assistance. And I'd like to send", "We’re grateful for your assistance. And I’d like to send"),
("We're snowed in. Our", "We’re snowed in. Our"),
("What's this? Large orcish", "What’s this? Large orcish"),
("Ye're young and not", "Ye’re young and not"),

# Fixed capitalization
("At The East Gate", "At the East Gate"),
("Reclaiming The Past", "Reclaiming the Past"),
),

"wesnoth-trow" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("Nay! Off with your hea- - -", "Nay! Off with your hea—"),
("is time - but I'll", "is time — but I’ll"),
("SE - The River Road.", "SE — The River Road."),
("SW - The", "SW — The"),
("SW - Southbay.", "SW — Southbay."),
("Hold - I see", "Hold — I see"),
("The River Road -", "The River Road —"),
("I'm a tinkin- -", "I’m a tinkin—"),
("NW - Southbay", "NW — Southbay"),
# Also, "Like" -> "like"
("More Like NW - Every orc", "More like NW — Every orc"),
("Rarlg - argh", "Rarlg — argh"),
("Sewer - Danger", "Sewer — Danger"),
("Legend has it- -", "Legend has it—"),
("your fate you- -", "your fate you—"),
("Compan- - What?", "Compan— What?"),
("your services again- -", "your services again—"),
("people - to ally", "people — to ally"),
("meet your son -", "meet your son —"),
("- As we agreed.", "— As we agreed."),
("and your people -", "and your people —"),
("their aid at all -", "their aid at all —"),
("me - I'll be dead", "me — I’ll be dead"),
("to say it - but", "to say it — but"),
("is ridiculous! I -", "is ridiculous! I —"),
("all I need - prepare", "all I need — prepare"),
("much -- from both", "much — from both"),
("a Lord -- to join", "a Lord — to join"),
("best of times - so", "best of times — so"),

# Convert another makeshift dash
# conversion added in 1.9.0+svn
("I- I die?", "I— I die?"),
("We come in peace. May our peoples-", "We come in peace. May our peoples—"),
("By the right of treaty, ancient but still true-", "By the right of treaty, ancient but still true—"),
("That tree, it speaks! No- No- We are tree-friends, really!", "That tree, it speaks! No— No— We are tree-friends, really!"),

# Fix screw up
# conversion added in 1.9.0-svn
("is time — but I'll", "is time — but I’ll"),
("I'm a tinkin—", "I’m a tinkin—"),
("me — I'll be dead", "me — I’ll be dead"),

("'Bring these Wesfolk too?", "“Bring these Wesfolk too?"),
("a century ago. I've never found my", "a century ago. I’ve never found my"),
("a fight, sir. They'll stand to the", "a fight, sir. They’ll stand to the"),
("a terrible price. Let's put Commander Aethyr", "a terrible price. Let’s put Commander Aethyr"),
("After a week's steady sailing eastward,", "After a week’s steady sailing eastward,"),
("against my people! Shek'kahan,", "against my people! Shek’kahan,"),
("Ahh, it's great to be home! I'm not much for", "Ahh, it’s great to be home! I’m not much for"),
("always think they're so clever!", "always think they’re so clever!"),
("amusement. Oh, and what's this?", "amusement. Oh, and what’s this?"),
("And now, I'm as much a", "And now, I’m as much a"),
("And why east? We've", "And why east? We’ve"),
("are, but you can't continue down this", "are, but you can’t continue down this"),
("aren't you setting us", "aren’t you setting us"),
("arrives on the elves' cursed Isle of", "arrives on the elves’ cursed Isle of"),
("arriving, and we haven't secured the frontier! We're dead!", "arriving, and we haven’t secured the frontier! We’re dead!"),
("at least we don't have to contend", "at least we don’t have to contend"),
("at the moment. I'm", "at the moment. I’m"),
("axe. Heh, heh. It's all I need", "axe. Heh, heh. It’s all I need"),
("Bah! I'm not dying today!", "Bah! I’m not dying today!"),
("be rich, and I'm not the first", "be rich, and I’m not the first"),
("bearing news. Some 'old friends’ have decided", "bearing news. Some ‘old friends’ have decided"),
("being invaded? Okay, we'll sort out our", "being invaded? Okay, we’ll sort out our"),
("Besides, I'd rather enjoy meeting", "Besides, I’d rather enjoy meeting"),
("best if you didn't mention our little", "best if you didn’t mention our little"),
("big bosses said we'd be in da", "big bosses said we’d be in da"),
("blocked. I guess we'll have to", "blocked. I guess we’ll have to"),
("blocked... I guess we'll have", "blocked... I guess we’ll have"),
("building a true 'Kingdom of Wesnoth'. I..I think a", "building a true ‘Kingdom of Wesnoth’. I..I think a"),
("but the elves aren't accustomed to war.", "but the elves aren’t accustomed to war."),
("but who knows what's going on", "but who knows what’s going on"),
("can not pass. You've already stumbled on", "can not pass. You’ve already stumbled on"),
("careful, these islands aren't safe. We're in Naga territory.", "careful, these islands aren’t safe. We’re in Naga territory."),
("Caror's Book of Fire", "Caror’s Book of Fire"),
("chatty! Get them, it's a long boat", "chatty! Get them, it’s a long boat"),
("comes behind us. You'd be slaves, or you'd be dead.", "comes behind us. You’d be slaves, or you’d be dead."),
("compromise with the Dwarves.'", "compromise with the Dwarves.”"),
("conflict is over I'll have to suffer", "conflict is over I’ll have to suffer"),
("crossing humans! Ahh, I've been with you", "crossing humans! Ahh, I’ve been with you"),
("days. I guess that's it for", "days. I guess that’s it for"),
("death'?", "death’?"),
("Don't forget about some", "Don’t forget about some"),
("down here. This isn't an old maid's drain pipe, this", "down here. This isn’t an old maid’s drain pipe, this"),
("earned both. We can't make this work", "earned both. We can’t make this work"),
("elves. But that won't", "elves. But that won’t"),
("emerged from the Lich-Lord's tomb, with the", "emerged from the Lich-Lord’s tomb, with the"),
("Enough of this, let's go get some", "Enough of this, let’s go get some"),
("far south. I don't know what's going on here. I'll stick", "far south. I don’t know what’s going on here. I’ll stick"),
("far. Who'd have thought, a", "far. Who’d have thought, a"),
("fighting. It's not stopping. We're trapped.", "fighting. It’s not stopping. We’re trapped."),
("Fire so we wouldn't have to bear", "Fire so we wouldn’t have to bear"),
("fleet is starving, we'll never pacify this", "fleet is starving, we’ll never pacify this"),
("fleet' has departed for", "fleet’ has departed for"),
("Fool of a boy's just gonna burn", "Fool of a boy’s just gonna burn"),
("for a while. It'll make a good", "for a while. It’ll make a good"),
("Free! I'm free at last!", "Free! I’m free at last!"),
("from me there'll be nothing but", "from me there’ll be nothing but"),
("go on ahead, I'll catch", "go on ahead, I’ll catch"),
("Haldric! I won't do it... I won't help. What is", "Haldric! I won’t do it... I won’t help. What is"),
("Haldric, it's bad. The elves", "Haldric, it’s bad. The elves"),
("Haldric, that's devious. I kind", "Haldric, that’s devious. I kind"),
("has translated Lich-Lord Caror's Book of Fire", "has translated Lich-Lord Caror’s Book of Fire"),
("hasn't been done in", "hasn’t been done in"),
("have been the Isle's largest city save", "have been the Isle’s largest city save"),
("have no idea what's going on outside", "have no idea what’s going on outside"),
("He's a large part", "He’s a large part"),
("here on purpose. You'd all still be", "here on purpose. You’d all still be"),
("hills, but I can't promise that my", "hills, but I can’t promise that my"),
("his merry men! We'll teach you a", "his merry men! We’ll teach you a"),
("Hmm. 'Wesnoth'. I like it. Yes, 'Wesnoth' would be fine.", "Hmm. ‘Wesnoth’. I like it. Yes, ‘Wesnoth’ would be fine."),
("Hmmmm- my kind were", "Hmmmm— my kind were"),
("homes! Heh...Tan-Vragar, I 'm", "homes! Heh... Tan-Vragar, I’m"),
("huts if it wasn't for my people.", "huts if it wasn’t for my people."),
("I can smell 'em!", "I can smell ’em!"),
("I decree a 'Pact of Mutual Aid' between our peoples.", "I decree a ‘Pact of Mutual Aid’ between our peoples."),
("I don't know who you", "I don’t know who you"),
("I go! I won't be denied!", "I go! I won’t be denied!"),
("I suspect you won't have to worry", "I suspect you won’t have to worry"),
("I were paranoid I'd say", "I were paranoid I’d say"),
("I wonder what they'll do when it", "I wonder what they’ll do when it"),
("I'd drop the sentiment", "I’d drop the sentiment"),
("I'll be dead. I mean we can't just say that", "I’ll be dead. I mean we can’t just say that"),
("I'll be glad when", "I’ll be glad when"),
("I'll begin my translation.", "I’ll begin my translation."),
("I'll return to the", "I’ll return to the"),
("I'm a dwarf, you", "I’m a dwarf, you"),
("I'm back, things went", "I’m back, things went"),
("I'm glad that's over! We have", "I’m glad that’s over! We have"),
("I'm glad that's over. The elves certainly aren't taking it easy on us. It's", "I’m glad that’s over. The elves certainly aren’t taking it easy on us. It’s"),
("I'm going to face", "I’m going to face"),
("I'm Haldric, my father", "I’m Haldric, my father"),
("I'm Haldric; my father", "I’m Haldric; my father"),
("I'm Prince Haldric, we", "I’m Prince Haldric, we"),
("I'm sure they'll eventually evacuate. Just", "I’m sure they’ll eventually evacuate. Just"),
("if Prince Haldric isn't the one to", "if Prince Haldric isn’t the one to"),
("in my pack. Let's work out what", "in my pack. Let’s work out what"),
("in other lands. It's a big world", "in other lands. It’s a big world"),
("In truth, I don't know if the", "In truth, I don’t know if the"),
("is no way I'm letting you go", "is no way I’m letting you go"),
("Is that snow?... We're doomed!", "Is that snow?... We’re doomed!"),
("is the Elder Prince's island all right.", "is the Elder Prince’s island all right."),
("is the last I'll see of my", "is the last I’ll see of my"),
("It isn't long before Haldric", "It isn’t long before Haldric"),
("it looks like they've rearmed!", "it looks like they’ve rearmed!"),
("it to Southbay. I'll", "it to Southbay. I’ll"),
("It wasn't long after he", "It wasn’t long after he"),
("It would be...highly irregular.' Bah! I'll show", "It would be... highly irregular.” Bah! I’ll show"),
("it — but that's not the", "it — but that’s not the"),
("It's better than the swamp. I'm under 5 feet", "It’s better than the swamp. I’m under 5 feet"),
("It's Commander Aethyr, of", "It’s Commander Aethyr, of"),
("It's funny that the lich-lord didn't have this on", "It’s funny that the lich-lord didn’t have this on"),
("It's not very Heroic", "It’s not very Heroic"),
("It's the Haldric! Jevyan,", "It’s the Haldric! Jevyan,"),
("Jessene, if you won't accept crown or", "Jessene, if you won’t accept crown or"),
("Jevyan himself. We can't have an illiterate", "Jevyan himself. We can’t have an illiterate"),
("Jevyan- I spit upon", "Jevyan— I spit upon"),
("just outside of Southbay-", "just outside of Southbay—"),
("kingdom be called 'Wesnoth',", "kingdom be called ‘Wesnoth’,"),
("leads to the Isle's second largest city,", "leads to the Isle’s second largest city,"),
("left for me. I'll stay. By the", "left for me. I’ll stay. By the"),
("lich is free! Let's bash him and", "lich is free! Let’s bash him and"),
("Lord Aryad said, 'If", "Lord Aryad said, “If"),
("Lords will follow. It's because of that", "Lords will follow. It’s because of that"),
("love, my family, I'll be there", "love, my family, I’ll be there"),
("mangled corpse. Then, you're going to destroy", "mangled corpse. Then, you’re going to destroy"),
("matters little now. What's this? You bring", "matters little now. What’s this? You bring"),
("maybe I won't even come to", "maybe I won’t even come to"),
("me. I'm no wizard. All", "me. I’m no wizard. All"),
("meet again, I won't be so", "meet again, I won’t be so"),
("more isolated villages, we'll need the gold!", "more isolated villages, we’ll need the gold!"),
("More of the trolls' kin are arriving", "More of the trolls’ kin are arriving"),
("my power. What! What's this? A treaty.", "my power. What! What’s this? A treaty."),
("my wrath, Jevyan, I'm coming for you!", "my wrath, Jevyan, I’m coming for you!"),
("my young prince. You've done", "my young prince. You’ve done"),
("nice thing is, they'll go back to", "nice thing is, they’ll go back to"),
("No! I don't believe it. Only", "No! I don’t believe it. Only"),
("No, Sir! I'm the sole survivor", "No, Sir! I’m the sole survivor"),
("of orcs. Apparently they're bogged down in", "of orcs. Apparently they’re bogged down in"),
("of the dragon I'm sure, who are", "of the dragon I’m sure, who are"),
("of these tree-folk we'll be trapped in", "of these tree-folk we’ll be trapped in"),
("Okay. Let's un-petrify that lich", "Okay. Let’s un-petrify that lich"),
("on convincing Jevyan, it's the only way", "on convincing Jevyan, it’s the only way"),
("on the Green Isle- we will return", "on the Green Isle— we will return"),
("orc thing, it wasn't such a bad", "orc thing, it wasn’t such a bad"),
("orcish scum, I can't bring myself to", "orcish scum, I can’t bring myself to"),
("Orcs! It's been a long", "Orcs! It’s been a long"),
("orcs. The ships aren't ready yet. We", "orcs. The ships aren’t ready yet. We"),
("our dear old 'tree-friend' might have known", "our dear old ‘tree-friend’ might have known"),
("our Naga friendsss- You’ll pay! We", "our Naga friendsss— You’ll pay! We"),
("out of time... We'll be trapped in", "out of time... We’ll be trapped in"),
("outside of Southbay, we'll be trapped. Of", "outside of Southbay, we’ll be trapped. Of"),
("person. Since I don't", "person. Since I don’t"),
("pile of dust, let's get out of", "pile of dust, let’s get out of"),
("point. Jessene, you're going to forge", "point. Jessene, you’re going to forge"),
("points and fishhooks. It's difficult to be", "points and fishhooks. It’s difficult to be"),
("power are true he's going to", "power are true he’s going to"),
("Prince Haldric, I'll concentrate on holding", "Prince Haldric, I’ll concentrate on holding"),
("Push 'em into the sea! They're in our winter", "Push ’em into the sea! They’re in our winter"),
("quickly-", "quickly—"),
("refugees, you'll be slaughtered.", "refugees, you’ll be slaughtered."),
("road follows the Isle's greatest river to", "road follows the Isle’s greatest river to"),
("rushing to a fool's death? Who'll lead these people?", "rushing to a fool’s death? Who’ll lead these people?"),
("said they would 'deal with the survivors'.", "said they would ‘deal with the survivors’."),
("seem rather intelligent. They're certainly belligerent and", "seem rather intelligent. They’re certainly belligerent and"),
("She's clever! Maybe I", "She’s clever! Maybe I"),
("ships rescued me. I'll not be", "ships rescued me. I’ll not be"),
("should 'deal' with the survivors,", "should ‘deal’ with the survivors,"),
("so are you. Aren't you the king", "so are you. Aren’t you the king"),
("so long I don't imagine he's too happy.", "so long I don’t imagine he’s too happy."),
("soldier there. The city's sewer", "soldier there. The city’s sewer"),
("Soldier, aren't you on the", "Soldier, aren’t you on the"),
("something! I think I'll save it for", "something! I think I’ll save it for"),
("soo-", "soo—"),
("Southbay won't fall without one", "Southbay won’t fall without one"),
("Southbay. We clearly can't go that way.", "Southbay. We clearly can’t go that way."),
("Southbay. We've been busy evacuating", "Southbay. We’ve been busy evacuating"),
("Southbay. You clearly can't go that way.", "Southbay. You clearly can’t go that way."),
("SWAMP! I'm under 5 feet", "<i>Swamp</i>! I’m under 5 feet"), # New <i> tags
("tall, and I don't float! Argh, have", "tall, and I don’t float! Argh, have"),
("tall, and I don't float!", "tall, and I don’t float!"),
("than any army I've ever seen just", "than any army I’ve ever seen just"),
("that bat. He won't let such a", "that bat. He won’t let such a"),
("that the elves 'have' the Ruby of", "that the elves ‘have’ the Ruby of"),
("that the elves wouldn't have sent us", "that the elves wouldn’t have sent us"),
("that the your 'third of a", "that the your ‘third of a"),
("that we've failed. We're trapped.", "that we’ve failed. We’re trapped."),
("The elves don't have the Ruby", "The elves don’t have the Ruby"),
("the Lich-Lords' dark pact with", "the Lich-Lords’ dark pact with"),
("the orcs haven't caught up with", "the orcs haven’t caught up with"),
("The orcs that didn't escape in the", "The orcs that didn’t escape in the"),
("the politics, but it's great to", "the politics, but it’s great to"),
("themselves before the Ka'lian, or the", "themselves before the Ka’lian, or the"),
("then we should 'deal' with the survivors,", "then we should ‘deal’ with the survivors,"),
("there. Failing that, there's always", "there. Failing that, there’s always"),
("they haven't met the main", "they haven’t met the main"),
("They look like they've been damaged by", "They look like they’ve been damaged by"),
("they won't be screaming at", "they won’t be screaming at"),
("they'll go looking for", "they’ll go looking for"),
("They're already holding the", "They’re already holding the"),
("They're Wesfolk, reavers and", "They’re Wesfolk, reavers and"),
("They've undoubtedly spotted the", "They’ve undoubtedly spotted the"),
("This isn't so bad. Haldric’s", "This isn’t so bad. Haldric’s"),
("this thing does, I'll just put it", "this thing does, I’ll just put it"),
("to dust. If you're", "to dust. If you’re"),
("to their aid. We'll have to weigh", "to their aid. We’ll have to weigh"),
("tongue. It means 'People of the West'. The", "tongue. It means ‘People of the West’. The"),
("trolls? I think it's", "trolls? I think it’s"),
("Trust me, you can't continue down this", "Trust me, you can’t continue down this"),
("Umm, we'll talk about it later. I'm just a little", "Umm, we’ll talk about it later. I’m just a little"),
("until the end. You've been a good", "until the end. You’ve been a good"),
("up sees that there's still", "up sees that there’s still"),
("war, but you're the real monster!", "war, but you’re the real monster!"),
("We can't all stay here", "We can’t all stay here"),
("We don't need some magic", "We don’t need some magic"),
("we fail here they'll doubtless be slaughtered.", "we fail here they’ll doubtless be slaughtered."),
("we groom the forest.'", "we groom the forest.”"),
("we prevailed. Now, I'm having a problem.", "we prevailed. Now, I’m having a problem."),
("We retrieved the Lich-Lord's book, and I", "We retrieved the Lich-Lord’s book, and I"),
("We'll have to be", "We’ll have to be"),
("We'll have to go", "We’ll have to go"),
("We're going to convince", "We’re going to convince"),
("We're only holding the", "We’re only holding the"),
("We're trapped in the", "We’re trapped in the"),
("We've defeated Jevyan's forces. The ships", "We’ve defeated Jevyan’s forces. The ships"),
("We've defeated their vanguard.", "We’ve defeated their vanguard."),
("We've escaped from the", "We’ve escaped from the"),
("We've escaped the pass.", "We’ve escaped the pass."),
("Well, I certainly don't mind the thought", "Well, I certainly don’t mind the thought"),
("well, if you can't be deterred. Remember,", "well, if you can’t be deterred. Remember,"),
("Well, we can't go back, and", "Well, we can’t go back, and"),
("What is your kind's infatuation with rushing to an 'honorable", "What is your kind’s infatuation with rushing to an ‘honorable"),
("What's that! No! The", "What’s that! No! The"),
("where it is! We've paid our friends", "where it is! We’ve paid our friends"),
("Who knows, maybe they'll be of some", "Who knows, maybe they’ll be of some"),
("winter stores. They've gone to Southbay,", "winter stores. They’ve gone to Southbay,"),
("with the elves. It's not like you", "with the elves. It’s not like you"),
("Without the Commander's noble sacrifice we", "Without the Commander’s noble sacrifice we"),
("would be 'Wes Noth'. So I suggest", "would be ‘Wes Noth’. So I suggest"),
("would sing the 'The Breaking of the Pact', an epic", "would sing the ‘The Breaking of the Pact’, an epic"),
("You can’t defeat me- Hmm, what's this? You've learned to", "You can’t defeat me— Hmm, what’s this? You’ve learned to"),
("you know what you're doing.", "you know what you’re doing."),
("You monster! You've betrayed our people", "You monster! You’ve betrayed our people"),
("You namby-pamby elves: 'We don't cut the trees,", "You namby-pamby elves: “We don’t cut the trees,"),
("You said we wouldn't see any action", "You said we wouldn’t see any action"),
("You should go 'greet' them.", "You should go ‘greet’ them."),
("you'll be joining him", "you’ll be joining him"),
("You're a fool, Jevyan.", "You’re a fool, Jevyan."),
("You're showing initiative, son! I'm proud of you!", "You’re showing initiative, son! I’m proud of you!"),
("You've been on this", "You’ve been on this"),
("You've denied me honorable", "You’ve denied me honorable"),
("your information? I don't even know your", "your information? I don’t even know your"),
("your people...and I don't think", "your people... and I don’t think"),
("your son's navigator and companion.", "your son’s navigator and companion."),

# Fix another screw up
("monolith was erected by me, ——", "monolith was erected by me, ―"),

# Eliminated a Britishism
("a fight in their favour", "a fight in their favor")
),

"wesnoth-tsg" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("My lord! The dawn is breaking - now is the time for us to attack and drive ", "My lord! The dawn is breaking — now is the time for us to attack and drive "),
("reinforcements - Did Sir Loris send you from Westin?", "reinforcements — Did Sir Loris send you from Westin?"),
("We will do our best to hold the city - you lead your men across the river ", "We will do our best to hold the city — you lead your men across the river "),
("Westin has fallen! This is a problem too great for me to handle - I must ", "Westin has fallen! This is a problem too great for me to handle — I must "),
("I have lost two brothers to Deoran - I shall not allow him to reach the ", "I have lost two brothers to Deoran — I shall not allow him to reach the "),
("My mount will not help me in these rocky paths - I will leave him here at ", "My mount will not help me in these rocky paths — I will leave him here at "),
("Also, soldiers with the -quick- trait will be useful in the dark.", "Also, soldiers with the — quick — trait will be useful in the dark."),
("We have escaped the great forest, but look! The undead are chasing us - we ", "We have escaped the great forest, but look! The undead are chasing us — we "),
("I too will make my stand here - I owe you my life and must atone for my ", "I too will make my stand here — I owe you my life and must atone for my "),
("Minister Hylas, ride with me to Westin - you must summon the Council of ", "Minister Hylas, ride with me to Westin — you must summon the Council of "),
("Far from home I fall - but not in vain! Guard our people, Deoran!", "Far from home I fall — but not in vain! Guard our people, Deoran!"),
("when they have an ally directly behind the unit they are attacking - they'll ", "when they have an ally directly behind the unit they are attacking — they’ll "),
("magic. They are slow and weak - any of your units is more than a match for ", "magic. They are slow and weak — any of your units is more than a match for "),
("Finally Ethiliel told the men to halt and pitch camp - they had reached the ", "Finally Ethiliel told the men to halt and pitch camp — they had reached the "),

# Fix screw up
# conversion added in 1.9.0-svn
("when they have an ally directly behind the unit they are attacking — they'll ", "when they have an ally directly behind the unit they are attacking — they’ll "),

# Convert apostrophes and quotation marks
# conversion added in 1.9.0-svn
("I've seen your", "I’ve seen your"),
("We'll surely all", "We’ll surely all"),
("Ithelden's", "Ithelden’s"),
("We won't be able", "We won’t be able"),
("Mal M'Brin", "Mal M’Brin"),
("Brin's", "Brin’s"),
("Gerrick's", "Gerrick’s"),
("men's", "men’s"),
("I prepare the city's defenses", "I prepare the city’s defenses"),
("works if it's the last", "works if it’s the last"),
("I've broken through", "I’ve broken through"),
("I'm lighting it", "I’m lighting it"),
("lich's", "lich’s"),
("we won't kill you", "we won’t kill you"),
("shouldn't", "shouldn’t"),
("I wouldn't have", "I wouldn’t have"),
("I've seen many", "I’ve seen many"),
("you've earned", "you’ve earned"),
("We're almost there", "We’re almost there"),
("I don't think they'll", "I don’t think they’ll"),
("Mebrin's", "Mebrin’s"),
("I'll kill you", "I’ll kill you"),
("they don't look very friendly", "they don’t look very friendly"),
("She won't listen", "She won’t listen"),
("I didn't have a choice", "I didn’t have a choice"),
("if we hadn't fallen", "if we hadn’t fallen"),
("I'm afraid it is not", "I’m afraid it is not"),
("it'd", "it’d"),
("undead aren't so tough", "undead aren’t so tough"),
("You're alive", "You’re alive"),
("You're too late", "You’re too late"),
("Mal A'kai", "Mal A’kai"),
("I hope that's the last", "I hope that’s the last"),
("Hylas's", "Hylas’s"),
("We're too late", "We’re too late"),
("They'll feel the deadly", "They’ll feel the deadly"),
("It's me", "It’s me"),
("they're in trouble", "they’re in trouble"),
("Mathin's", "Mathin’s"),
("While you're in your camp", "While you’re in your camp"),
("citadel's", "citadel’s"),
("'my kind'?", "“my kind”?"),
("I've seen your", "I’ve seen your"),
("Deoran's", "Deoran’s"),
("the capital of Wesnoth's", "the capital of Wesnoth’s"),
("Haldiel's", "Haldiel’s"),
("'Stop! Enough blood has", "“Stop! Enough blood has"),
("your service also was worthy.'", "your service also was worthy.”"),

("'chaotic', which means that", "<i>chaotic</i>, which means that"), # new <i> tags
("'Civilian' difficulty level is", "‘Civilian’ difficulty level is"),
("and I'll defend them to", "and I’ll defend them to"),
("Named 'Eyestalks' for obvious reasons,", "Named ‘Eyestalks’ for obvious reasons,"),
("your units are 'lawful'. This means that", "your units are <i>lawful</i>. This means that"), # new <i> tags

# Fix capitalization
("Return To Kerlath", "Return to Kerlath"),
("Into The Depths", "Into the Depths"),
("Proven By The Sword", "Proven by the Sword"),
),

"wesnoth-sof" : (
# Typo fixes required at r44124.
("going to back to the mines", "going back to the mines"),
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("So now I tell from whence it came -", "So now I tell from whence it came —"),
("The Fire-sceptre great -", "The Fire-sceptre great —"),
("Uh, no, wait! Lets talk business - how much will you pay us to do this for ", "Uh, no, wait! Lets talk business — how much will you pay us to do this for "),
("Fine then - ten thousand... now, what exactly do you want us to make the ", "Fine then — ten thousand... now, what exactly do you want us to make the "),
("Hey! You can't do that - this road is an elvish road. We're not letting you ", "Hey! You can’t do that — this road is an elvish road. We’re not letting you "),
("Ah, I see - you are one of the rebels. Our treaty with the elves does not ", "Ah, I see — you are one of the rebels. Our treaty with the elves does not "),
("We will see about that - if you ever get underground, which I doubt. Ha!", "We will see about that — if you ever get underground, which I doubt. Ha!"),
("Here - I'll go warn the council. You stay here and fight.", "Here — I’ll go warn the council. You stay here and fight."),
("dwarf-made stonecraft. We refuse - now let's close these gates!", "dwarf-made stonecraft. We refuse — now let’s close these gates!"),
("Just watch. The gates wi' close very soon. Then the elves outside - and, ", "Just watch. The gates wi’ close very soon. Then the elves outside — and, "),
("unfortunately, our dwarves who are still out there - wi' become irrelevant.", "unfortunately, our dwarves who are still out there — wi’ become irrelevant."),
("for the runesmith named Thursagan - the sage of fire.", "for the runesmith named Thursagan — the sage of fire."),
("members of their party - one expected, and one not.", "members of their party — one expected, and one not."),
("In the treasury. And leave it there until you're ready to work with it - I ", "In the treasury. And leave it there until you’re ready to work with it — I "),
("the finest coal, and many of the finest jewels in the land - we'll need ", "the finest coal, and many of the finest jewels in the land — we’ll need "),
("to fight them; also be prepared to spend quite some time here - mining can ", "to fight them; also be prepared to spend quite some time here — mining can "),
("Yes, although we will have to hire the miners - they don't work for free. ", "Yes, although we will have to hire the miners — they don’t work for free. "),
("were others - many others. Thus I present Theganli, the jeweler. His role is ", "were others — many others. Thus I present Theganli, the jeweler. His role is "),
("were no exception. They were crafters of crafters - they made tools. The ", "were no exception. They were crafters of crafters — they made tools. The "),
("How about this - I help you defeat them, and you let me keep the ruby when ", "How about this — I help you defeat them, and you let me keep the ruby when "),
("Well, back to the battle - we are all in the caves, but there are still ", "Well, back to the battle — we are all in the caves, but there are still "),
("Well, back to the battle - we need everyone to get into the Shorbear caves. ", "Well, back to the battle — we need everyone to get into the Shorbear caves. "),
("If ye'll permit me to say so, sir, ye're wrong. We could - ", "If ye’ll permit me to say so, sir, ye’re wrong. We could — "),
("If you'll permit me to say so, sir, you're wrong. We could - ", "If you’ll permit me to say so, sir, you’re wrong. We could — "),
("went south - back to the Wesnoth border.", "went south — back to the Wesnoth border."),
("And thus Rugnur died - a glorious death, in the eyes of the dwarven sages. ", "And thus Rugnur died — a glorious death, in the eyes of the dwarven sages. "),
("And Krawg - well, Krawg followed him. I know not how. But Krawg made his way ", "And Krawg — well, Krawg followed him. I know not how. But Krawg made his way "),
("But before it was found, legends grew up around it, and around its makers - ", "But before it was found, legends grew up around it, and around its makers — "),
# Straight apostrophes and quotes to curly ones
# conversion added in 1.9.0-svn
# (NOT YET)
#("The land of Wesnoth's banner bold", "The land of Wesnoth’s banner bold"),
#("Made by a runesmith's hand.", "Made by a runesmith’s hand."),

# Fix screw up:
# conversion added in 1.9.0-svn
("Yes, although we will have to hire the miners v they don't work for free. ", "Yes, although we will have to hire the miners — they don’t work for free. "),
("Hey! You can't do that — this road is an elvish road. We're not letting you ", "Hey! You can’t do that — this road is an elvish road. We’re not letting you "),
("If you'll permit me to say so, sir, you're wrong. We could — ", "If you’ll permit me to say so, sir, you’re wrong. We could — "),
("If ye'll permit me to say so, sir, ye're wrong. We could — ", "If ye’ll permit me to say so, sir, ye’re wrong. We could — "),
("the finest coal, and many of the finest jewels in the land — we'll need ", "the finest coal, and many of the finest jewels in the land — we’ll need "),
("In the treasury. And leave it there until you're ready to work with it — I ", "In the treasury. And leave it there until you’re ready to work with it — I "),
("unfortunately, our dwarves who are still out there — wi' become irrelevant.", "unfortunately, our dwarves who are still out there — wi’ become irrelevant."),
("Just watch. The gates wi' close very soon. Then the elves outside — and, ", "Just watch. The gates wi’ close very soon. Then the elves outside — and, "),
("dwarf-made stonecraft. We refuse — now let's close these gates!", "dwarf-made stonecraft. We refuse — now let’s close these gates!"),
("Here — I'll go warn the council. You stay here and fight.", "Here — I’ll go warn the council. You stay here and fight."),

("'Ruby of Fire', and told us", "‘Ruby of Fire’, and told us"),
("a conclusion. He can't make the sceptre", "a conclusion. He can’t make the sceptre"),
("a deal, but I'm not sure. I'll have to ask", "a deal, but I’m not sure. I’ll have to ask"),
("Aha! I've spent years looking", "Aha! I’ve spent years looking"),
("And I'll be doing the", "And I’ll be doing the"),
("And if we don't like what you", "And if we don’t like what you"),
("apparently. I think we'll have to", "apparently. I think we’ll have to"),
("Argh! I'm a retired warrior...", "Argh! I’m a retired warrior..."),
("artifact, not a child's playtoy!", "artifact, not a child’s playtoy!"),
("away, eh? I dinna' like that, but it seems it's our only", "away, eh? I dinna’ like that, but it seems it’s our only"),
("Baglur's. Krawg's done more for", "Baglur’s. Krawg’s done more for"),
("can tell I don't need the money", "can tell I don’t need the money"),
("can't run as fast", "can’t run as fast"),
("Can't you tell a", "Can’t you tell a"),
("captured! If I can't", "captured! If I can’t"),
("castle here... I don't much like fighting.", "castle here... I don’t much like fighting."),
("caves to learn Rugnur's fate.", "caves to learn Rugnur’s fate."),
("caves, but I can't! What am I", "caves, but I can’t! What am I"),
("caves, so they can't escape when it", "caves, so they can’t escape when it"),
("city and report. I'm", "city and report. I’m"),
("constitutes treason, wouldn't you?", "constitutes treason, wouldn’t you?"),
("craftsmen, and he wouldn't have gone to", "craftsmen, and he wouldn’t have gone to"),
("Don't be a fool! We can't take the elves,", "Don’t be a fool! We can’t take the elves,"),
("don't stop me if", "don’t stop me if"),
("down, or we'll force you.", "down, or we’ll force you."),
("Durstorn's rash action led", "Durstorn’s rash action led"),
("Durstorn. And I wouldn't give the Sceptre", "Durstorn. And I wouldn’t give the Sceptre"),
("dwarves! Now, if you'll excuse me, I think I'll be", "dwarves! Now, if you’ll excuse me, I think I’ll be"),
("dwarves, but now I've found you!", "dwarves, but now I’ve found you!"),
("eastern mines. Me, I'll stay here. No", "eastern mines. Me, I’ll stay here. No"),
("enough! Or, if they're not 'good enough'", "enough! Or, if they’re not ‘good enough’"),
("even if you hadn't, you've seen enough combat.", "even if you hadn’t, you’ve seen enough combat."),
("Even the elves won't challenge us here.", "Even the elves won’t challenge us here."),
("evil things. We can't let it", "evil things. We can’t let it"),
("for those, though. I'll also need to", "for those, though. I’ll also need to"),
("for us, for they'll go as fast", "for us, for they’ll go as fast"),
("Getting this cut isn't worth that much; what with Rugnur's mistake", "Getting this cut isn’t worth that much; what with Rugnur’s mistake"),
("gold he needs. Shouldn't we leave now?", "gold he needs. Shouldn’t we leave now?"),
("Good idea, Baglur! He'll need some help,", "Good idea, Baglur! He’ll need some help,"),
("Good. Now, let's get out of", "Good. Now, let’s get out of"),
("Ha! There hasn't been a border", "Ha! There hasn’t been a border"),
("He doesn't need to come here, I'll take the jewel", "He doesn’t need to come here, I’ll take the jewel"),
("He doesn't want to talk", "He doesn’t want to talk"),
("his sceptre intact he'd best send troops", "his sceptre intact he’d best send troops"),
("I don't know if we", "I don’t know if we"),
("I don't know what is", "I don’t know what is"),
("I think you'd better let me", "I think you’d better let me"),
("I'd rather face one", "I’d rather face one"),
("I'll see what I", "I’ll see what I"),
("I'm a dragoon, with Haldric II's personal bodyguard. I", "I’m a dragoon, with Haldric II’s personal bodyguard. I"),
("I'm Lord Durstorn, king", "I’m Lord Durstorn, king"),
("I'm not going to", "I’m not going to"),
("If he fails, it's on his head.", "If he fails, it’s on his head."),
("If that's what you're thinking, you're not fit to", "If that’s what you’re thinking, you’re not fit to"),
("if you did. It's not yours to", "if you did. It’s not yours to"),
("if you won't give the Sceptre", "if you won’t give the Sceptre"),
("in advance, and he'll give us five", "in advance, and he’ll give us five"),
("in any case, I've been with you", "in any case, I’ve been with you"),
("it looks like I'll ha' to come out", "it looks like I’ll ha’ to come out"),
("it seems... it can't be", "it seems... it can’t be"),
("it's not like we", "it’s not like we"),
("it, but they didn't offer", "it, but they didn’t offer"),
("it, eh? Well I'm not going", "it, eh? Well I’m not going"),
("its mine, and Rugnur's, and", "its mine, and Rugnur’s, and"),
("killed, but it isn't", "killed, but it isn’t"),
("Look, I've found something here.", "Look, I’ve found something here."),
("Made by a runesmith's hand.", "Made by a runesmith’s hand."),
("me, but I won't die with you!", "me, but I won’t die with you!"),
("miners — they don't work for free.", "miners — they don’t work for free."),
("occupied, the gates wi' close. Then the", "occupied, the gates wi’ close. Then the"),
("Och, well that's the difficulty. It's a magic gate.", "Och, well that’s the difficulty. It’s a magic gate."),
("of the trolls' territory.", "of the trolls’ territory."),
("offer. If you won't", "offer. If you won’t"),
("out, we'll make our way", "out, we’ll make our way"),
("payment, but it can't be helped. Our", "payment, but it can’t be helped. Our"),
("period of time. We'll only be", "period of time. We’ll only be"),
("pieces of silver. I'm going with you", "pieces of silver. I’m going with you"),
("property secure, the deal's off.", "property secure, the deal’s off."),
("ruby, then what? They'll probably kill us", "ruby, then what? They’ll probably kill us"),
("runes infuse the runesmith's blows with power", "runes infuse the runesmith’s blows with power"),
("so the lava doesn't kill us.", "so the lava doesn’t kill us."),
("something or I can't make the sceptre!", "something or I can’t make the sceptre!"),
("still don't trust you.", "still don’t trust you."),
("sure... well, it doesn't matter why.", "sure... well, it doesn’t matter why."),
("take their property. We'll have to buy,", "take their property. We’ll have to buy,"),
("than a thousand we'll", "than a thousand we’ll"),
("than life! So, I'm ordering", "than life! So, I’m ordering"),
("that was not Rugnur's.", "that was not Rugnur’s."),
("the Arkan-thoria. Trolls ha' overrun them, but", "the Arkan-thoria. Trolls ha’ overrun them, but"),
("the elves will no' be able to", "the elves will no’ be able to"),
("the far northlands. They're", "the far northlands. They’re"),
("The land of Wesnoth's banner bold", "The land of Wesnoth’s banner bold"),
("the last chance we'll get to do", "the last chance we’ll get to do"),
("the sceptre. Why don't we send him", "the sceptre. Why don’t we send him"),
("the stone's power, but contain", "the stone’s power, but contain"),
("the surface world. He's late!", "the surface world. He’s late!"),
("the Wesnothian army; I'd prefer to", "the Wesnothian army; I’d prefer to"),
("then go north. He's", "then go north. He’s"),
("Then maybe we'll walk out of", "Then maybe we’ll walk out of"),
("Then no deal. I'll gather my things", "Then no deal. I’ll gather my things"),
("these dwarves too? They're attacking my clan", "these dwarves too? They’re attacking my clan"),
("This plan, it isn't worse than staying", "This plan, it isn’t worse than staying"),
("those Surghan mercenaries. I'll bet those elves", "those Surghan mercenaries. I’ll bet those elves"),
("though, and we can't stay here forever.", "though, and we can’t stay here forever."),
("time! Haldric killed Shek'kahan my", "time! Haldric killed Shek’kahan my"),
("to do? I don't see how this", "to do? I don’t see how this"),
("to go. Alanin, you've served in the", "to go. Alanin, you’ve served in the"),
("to his shop. He'll need", "to his shop. He’ll need"),
("to plan the sceptre's design... what in", "to plan the sceptre’s design... what in"),
("to the elves, I'll take it from", "to the elves, I’ll take it from"),
("Uh, technically it's Haldric’s, and I", "Uh, technically it’s Haldric’s, and I"),
("up, Baglur! You don't know what you're talking about. I've analyzed the", "up, Baglur! You don’t know what you’re talking about. I’ve analyzed the"),
("up, Rugnur! You don't know what you're talking about. I've analyzed the", "up, Rugnur! You don’t know what you’re talking about. I’ve analyzed the"),
("Wait! Rugnur, dinna' they tell you", "Wait! Rugnur, dinna’ they tell you"),
("Wait, what's going on? The", "Wait, what’s going on? The"),
("We can't just go in", "We can’t just go in"),
("we wanted to. We're surrounded!", "we wanted to. We’re surrounded!"),
("Well, I think we've gotten a good", "Well, I think we’ve gotten a good"),
("Well, Thursagan, we've reached the eastern", "Well, Thursagan, we’ve reached the eastern"),
("Well, we've succeeded in stopping", "Well, we’ve succeeded in stopping"),
("Wesnothian army. Why don't you rejoin it?", "Wesnothian army. Why don’t you rejoin it?"),
("we’re here, we'll be able to", "we’re here, we’ll be able to"),
("whatever it is can't be worse than", "whatever it is can’t be worse than"),
("where I can't help!", "where I can’t help!"),
("Where's that fool Rugnur gotten to? It's time for him", "Where’s that fool Rugnur gotten to? It’s time for him"),
("with us. He's somewhere up here.", "with us. He’s somewhere up here."),
("would stop the stone's", "would stop the stone’s"),
("year. Oh well, what's done is done. We'll have to work", "year. Oh well, what’s done is done. We’ll have to work"),
("yes, but I don't know where it", "yes, but I don’t know where it"),
("Yes, of course, let's just all be", "Yes, of course, let’s just all be"),
("You don't dictate what I", "You don’t dictate what I"),
("You don't have authority over", "You don’t have authority over"),
("You don't want to make", "You don’t want to make"),
("You know yours aren't acceptable, but you", "You know yours aren’t acceptable, but you"),
("you missed. Now, I'd say that attacking", "you missed. Now, I’d say that attacking"),
("You're a king?! Must", "You’re a king?! Must"),
("You're not my ally. You're more against me", "You’re not my ally. You’re more against me"),
("you, else we'll be overrun by", "you, else we’ll be overrun by"),

# Convert apostrophe
("Don't forget to recruit the miners.", "Don’t forget to recruit the miners."),

# Usage fixes in 1.9.9
("Och, its some o’ them cave-dwarves.", "Och, it’s some o’ them cave-dwarves."),
("I guess you would to plan", "I guess you would want to plan"),
),

"wesnoth-sotbe" :(
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("the offer. However, on the way to the city of Dwarven Doors - the ", "the offer. However, on the way to the city of Dwarven Doors — the "),
("headquarters of the Northern Alliance - Karun and his small group of ", "headquarters of the Northern Alliance — Karun and his small group of "),
("The peace treaty still held however, partially because the treaty - although ", "The peace treaty still held however, partially because the treaty — although "),
("arranged by the Northern Alliance - was between the local earls and the ", "arranged by the Northern Alliance — was between the local earls and the "),
("Alliance. The other - and perhaps the main - reason why the treaty held was ", "Alliance. The other — and perhaps the main — reason why the treaty held was "),
("followed were relatively peaceful and prosperous - at least, for the humans.", "followed were relatively peaceful and prosperous — at least, for the humans."),
("In the thirteenth year of the Lord Protectorship of Howgarth III - the ", "In the thirteenth year of the Lord Protectorship of Howgarth III — the "),
("successor of Rahul I - tension began to rise between orcish tribes and human ", "successor of Rahul I — tension began to rise between orcish tribes and human "),
("on their lands. Then, Earl Lanbec'h - the most powerful human warlord of the ", "on their lands. Then, Earl Lanbec’h — the most powerful human warlord of the "),
("North - determined to abolish the orcish menace raised an army and conferred ", "North — determined to abolish the orcish menace raised an army and conferred "),
("Who is this unlicked whelp? Grunts - kill him and bring me his head!", "Who is this unlicked whelp? Grunts — kill him and bring me his head!"),
("Wise decision, Kapou'e. By the size of that army, this is no mere raid - it ", "Wise decision, Kapou’e. By the size of that army, this is no mere raid — it "),
("And this is that stupid human who dares march against Kapou'e - Son of the ", "And this is that stupid human who dares march against Kapou’e — Son of the "),
("Hey, Chief, I was thinking - dwarves are pretty slow; why don't we just ", "Hey, Chief, I was thinking — dwarves are pretty slow; why don’t we just "),
("As soon as the sandstorm died down, Kapou'e and his men - eager to escape ", "As soon as the sandstorm died down, Kapou’e and his men — eager to escape "),
("the searing heat - left the oasis and continued on their trek.", "the searing heat — left the oasis and continued on their trek."),
("and all the natural sounds of the forest died away - leaving everything ", "and all the natural sounds of the forest died away — leaving everything "),
("Leave your people here Kapou'e - they are safe for the moment - and go ", "Leave your people here Kapou’e — they are safe for the moment — and go "),
("friends - get them!", "friends — get them!"),
("After a fair amount of squabbling - for some of the older warlords were ", "After a fair amount of squabbling — for some of the older warlords were "),
("reluctant to let this young upstart lead them - and a few consequent ", "reluctant to let this young upstart lead them — and a few consequent "),
("Kapou'e placed one force under the leadership of Shan Taum the Smug, who - ", "Kapou’e placed one force under the leadership of Shan Taum the Smug, who — "),
("despite his obnoxious nature - was a fierce and capable leader. He was sent ", "despite his obnoxious nature — was a fierce and capable leader. He was sent "),
("He sent the shamans, Pirk, Gork and Vraurk back to Borstep - a city just ", "He sent the shamans, Pirk, Gork and Vraurk back to Borstep — a city just "),
("north of the Mourned Hills - to organize any remaining orcish forces as well ", "north of the Mourned Hills — to organize any remaining orcish forces as well "),
("Kapou'e himself - desiring to settle this business once and for all - led ", "Kapou’e himself — desiring to settle this business once and for all — led "),
("The orcs are making headway. The city must not fall - call the reserves!", "The orcs are making headway. The city must not fall — call the reserves!"),
("fall. A few days after the first snowfall a goblin rider - half dead from ", "fall. A few days after the first snowfall a goblin rider — half dead from "),
("exhaustion - raced into Dorset with the news that Borstep was under siege by ", "exhaustion — raced into Dorset with the news that Borstep was under siege by "),
("give him a good chance to clobber the orcs together again - the old ", "give him a good chance to clobber the orcs together again — the old "),
("Kapou'e's exploits had been widely circulated - from his rescue of the ", "Kapou’e’s exploits had been widely circulated — from his rescue of the "),
("horde was dispersed he appointed three more shamans - with the consent of ", "horde was dispersed he appointed three more shamans — with the consent of "),
("all - to make the Great Council complete again.", "all — to make the Great Council complete again."),

# Fix screw up
# conversion added in 1.9.0-svn
("Kapou'e's exploits had been widely circulated — from his rescue of the ", "Kapou’e’s exploits had been widely circulated — from his rescue of the "),
("Kapou'e himself — desiring to settle this business once and for all — led ", "Kapou’e himself — desiring to settle this business once and for all — led "),
("Kapou'e placed one force under the leadership of Shan Taum the Smug, who — ", "Kapou’e placed one force under the leadership of Shan Taum the Smug, who — "),
("Leave your people here Kapou'e — they are safe for the moment — and go ", "Leave your people here Kapou’e — they are safe for the moment — and go "),
("As soon as the sandstorm died down, Kapou'e and his men — eager to escape ", "As soon as the sandstorm died down, Kapou’e and his men — eager to escape "),
("Hey, Chief, I was thinking — dwarves are pretty slow; why don't we just ", "Hey, Chief, I was thinking — dwarves are pretty slow; why don’t we just "),
("And this is that stupid human who dares march against Kapou'e — Son of the ", "And this is that stupid human who dares march against Kapou’e — Son of the "),
("Wise decision, Kapou'e. By the size of that army, this is no mere raid — it ", "Wise decision, Kapou’e. By the size of that army, this is no mere raid — it "),
("on their lands. Then, Earl Lanbec'h — the most powerful human warlord of the ", "on their lands. Then, Earl Lanbec’h — the most powerful human warlord of the "),

("A human-worm's dog. Animal meat", "A human-worm’s dog. Animal meat"),
("a shock to Kapou'e to", "a shock to Kapou’e to"),
("after Kapou'e's arrival, riders caught", "after Kapou’e’s arrival, riders caught"),
("after this desert, we'll have to cross", "after this desert, we’ll have to cross"),
("Ahhh, it's 'the Prince' from the Silent", "Ahhh, it’s ‘the Prince’ from the Silent"),
("all the shamans 'suck up' to him just", "all the shamans ‘suck up’ to him just"),
("an invitation to Kapou'e to join", "an invitation to Kapou’e to join"),
("And if that isn't", "And if that isn’t"),
("and if we aren't", "and if we aren’t"),
("and occupying Dorset, Kapou'e and his men", "and occupying Dorset, Kapou’e and his men"),
("and your people won't be safe. You", "and your people won’t be safe. You"),
("apprehension and rage, Kapou'e sent his riders", "apprehension and rage, Kapou’e sent his riders"),
("as his son, it's up to you", "as his son, it’s up to you"),
("As Kapou'e expected, by the", "As Kapou’e expected, by the"),
("As Kapou'e saw the hardened", "As Kapou’e saw the hardened"),
("at Barag Gór, Kapou'e, accompanied by the", "at Barag Gór, Kapou’e, accompanied by the"),
("at last. We don't have much time,", "at last. We don’t have much time,"),
("automatically assume that Kapou'e tricked and killed", "automatically assume that Kapou’e tricked and killed"),
("back in the tribe's homeland.", "back in the tribe’s homeland."),
("battle, my grunts! We'll show these humans", "battle, my grunts! We’ll show these humans"),
("been led by Kapou'e this far had", "been led by Kapou’e this far had"),
("before. Flar'Tar was sent directly", "before. Flar’Tar was sent directly"),
("Black-Eye will never 'flee'. But making a 'strategic", "Black-Eye will never ‘flee’. But making a ‘strategic"),
("Black-Eye. If you hadn't arrived when you did, I don't", "Black-Eye. If you hadn’t arrived when you did, I don’t"),
("But, Father, we don't know what they", "But, Father, we don’t know what they"),
("came from or you'll find your head", "came from or you’ll find your head"),
("Catch up when you're done.", "Catch up when you’re done."),
("cave or something. We'll be through here", "cave or something. We’ll be through here"),
("Cease your lies, Lanbec'h. We all know", "Cease your lies, Lanbec’h. We all know"),
("Chief, Inarix hasn't yet made it", "Chief, Inarix hasn’t yet made it"),
("count on the assassins' guild to help", "count on the assassins’ guild to help"),
("crossed the mountains. We're close to the", "crossed the mountains. We’re close to the"),
("Don't be afraid. If", "Don’t be afraid. If"),
("Don't be so cowardly.", "Don’t be so cowardly."),
("Earl Lanbec'h, you slimy coward,", "Earl Lanbec’h, you slimy coward,"),
("Earl's forces and withdrew", "Earl’s forces and withdrew"),
("eastern Mourned Hills. Kapou'e himself made a", "eastern Mourned Hills. Kapou’e himself made a"),
("eighth year of Kapou'e's reign, and then", "eighth year of Kapou’e’s reign, and then"),
("Exasperated, Kapou'e launched an attack", "Exasperated, Kapou’e launched an attack"),
("Finally we've done it! Now let's wait out this", "Finally we’ve done it! Now let’s wait out this"),
("For his part Kapou'e", "For his part Kapou’e"),
("force if necessary. Al'Brock was sent west,", "force if necessary. Al’Brock was sent west,"),
("Forest. Unfortunately, I don't think", "Forest. Unfortunately, I don’t think"),
("formally bestowed upon Kapou'e.", "formally bestowed upon Kapou’e."),
("Fortunately, they didn't encounter any giant", "Fortunately, they didn’t encounter any giant"),
("found his generals, Al'Brock and Flar'Tar had already arrived", "found his generals, Al’Brock and Flar’Tar had already arrived"),
("friends! Those orcs don't imagine they are", "friends! Those orcs don’t imagine they are"),
("giant scorpion yet. I'm sure it would", "giant scorpion yet. I’m sure it would"),
("Good work, men, we've done it! Now we don't have to worry", "Good work, men, we’ve done it! Now we don’t have to worry"),
("green hills! We've reached the lands", "green hills! We’ve reached the lands"),
("had dealings with Kapou'e's", "had dealings with Kapou’e’s"),
("having been settled, Kapou'e began to inquire", "having been settled, Kapou’e began to inquire"),
("hell. Your beloved Kapou'e will be", "hell. Your beloved Kapou’e will be"),
("Hey Chief, it's pretty odd. Shan Taum doesn't seem to have", "Hey Chief, it’s pretty odd. Shan Taum doesn’t seem to have"),
("his word, once Lanbec'h had been slain,", "his word, once Lanbec’h had been slain,"),
("Hold, Earl Lanbec'h. I have reached", "Hold, Earl Lanbec’h. I have reached"),
("how to dissemble. I'll bet he sent", "how to dissemble. I’ll bet he sent"),
("Hurry-hurry, friends. Lanbec'h and his kind", "Hurry-hurry, friends. Lanbec’h and his kind"),
("I am Kapou'e, son of the", "I am Kapou’e, son of the"),
("I can't leave these renegades", "I can’t leave these renegades"),
("I don't know if you", "I don’t know if you"),
("I don't know. Orcs have", "I don’t know. Orcs have"),
("I don't really know, one", "I don’t really know, one"),
("I hope we won't find one! They", "I hope we won’t find one! They"),
("I wouldn't worry about the", "I wouldn’t worry about the"),
("I'm hungry and we", "I’m hungry and we"),
("I'm not that sure.", "I’m not that sure."),
("I've been told the", "I’ve been told the"),
("I've never pushed so", "I’ve never pushed so"),
("III demanded that Kapou'e give Dorset", "III demanded that Kapou’e give Dorset"),
("implications of the shamans' deaths begin to", "implications of the shamans’ deaths begin to"),
("in this fight. I'm sorry for the", "in this fight. I’m sorry for the"),
("is done then Kapou'e will give you", "is done then Kapou’e will give you"),
("is going well, let's surround them on", "is going well, let’s surround them on"),
("It's been a chilly", "It’s been a chilly"),
("It's dawn!", "It’s dawn!"),
("it's time to show", "it’s time to show"),
("its own weakness. Prestim's walls are", "its own weakness. Prestim’s walls are"),
("Kapou'e agreed to meet", "Kapou’e agreed to meet"),
("Kapou'e and his followers", "Kapou’e and his followers"),
("Kapou'e and the shamans", "Kapou’e and the shamans"),
("Kapou'e had cowed into", "Kapou’e had cowed into"),
("Kapou'e refused to do", "Kapou’e refused to do"),
("Kapou'e ruled for many", "Kapou’e ruled for many"),
("Kapou'e suspected some sort", "Kapou’e suspected some sort"),
("Kapou'e vowed that this", "Kapou’e vowed that this"),
("Kapou'e, both attacks were", "Kapou’e, both attacks were"),
("Kapou'e, but he decided", "Kapou’e, but he decided"),
("Kapou'e, if Prestim falls,", "Kapou’e, if Prestim falls,"),
("Kapou'e, it is essential", "Kapou’e, it is essential"),
("Kapou'e, rumors tell of", "Kapou’e, rumors tell of"),
("kid for me. I'll be in my", "kid for me. I’ll be in my"),
("killing the shamans, Kapou'e was requested to", "killing the shamans, Kapou’e was requested to"),
("Lanbec'h flatly refused to", "Lanbec’h flatly refused to"),
("Lanbec'h led the army", "Lanbec’h led the army"),
("Lanbec'h was no doubt", "Lanbec’h was no doubt"),
("Let's slay them and", "Let’s slay them and"),
("made his plans, Kapou'e set out for", "made his plans, Kapou’e set out for"),
("messenger brought Rahul Karun's reply.", "messenger brought Rahul Karun’s reply."),
("miles away. They won't be", "miles away. They won’t be"),
("mountain pass. Kapou'e began to recognize", "mountain pass. Kapou’e began to recognize"),
("my good graces. I'll", "my good graces. I’ll"),
("My son, you're old enough to", "My son, you’re old enough to"),
("no choice ... I've never tasted scorpions", "no choice ... I’ve never tasted scorpions"),
("No really, it's true. He is", "No really, it’s true. He is"),
("of the Black-Eye. I'm Jetto, master of", "of the Black-Eye. I’m Jetto, master of"),
("of turmoil, we don't", "of turmoil, we don’t"),
("Oh, so it's another group o' stinkin' orcs, eh. Begone,", "Oh, so it’s another group o’ stinkin’ orcs, eh. Begone,"),
("on us! They can't be more then", "on us! They can’t be more then"),
("on your victory, Kapou'e. You do honor", "on your victory, Kapou’e. You do honor"),
("orcish leader Kapou'e, the son of", "orcish leader Kapou’e, the son of"),
("orcish tribes and didn't have any direct", "orcish tribes and didn’t have any direct"),
("Orcs, bring me Kapou'e's head. I must", "Orcs, bring me Kapou’e’s head. I must"),
("out of the traitor's skull, Kapou'e gathered his men", "out of the traitor’s skull, Kapou’e gathered his men"),
("over three months, Kapou'e and his exhausted", "over three months, Kapou’e and his exhausted"),
("pathetic humans come, we'll deal with", "pathetic humans come, we’ll deal with"),
("pint-sized idiot, we aren't out to kill", "pint-sized idiot, we aren’t out to kill"),
("sight of Earl Lanbec'h's army.", "sight of Earl Lanbec’h’s army."),
("snows melted and Kapou'e readied his defenses.", "snows melted and Kapou’e readied his defenses."),
("So the 'great and mighty' elves are now", "So the ‘great and mighty’ elves are now"),
("that Grüü, you don't look so good.", "that Grüü, you don’t look so good."),
("that, if Earl Lanbec'h continued in", "that, if Earl Lanbec’h continued in"),
("that? I've not forgotten what", "that? I’ve not forgotten what"),
("That’s right! Now let's drive those mincing", "That’s right! Now let’s drive those mincing"),
("The hell I don't, Earl Lanbec'h, the Northern Alliance", "The hell I don’t, Earl Lanbec’h, the Northern Alliance"),
("the hour. If you're not ready by", "the hour. If you’re not ready by"),
("the human however, Kapou'e", "the human however, Kapou’e"),
("the humans, pfff. You're one to talk.", "the humans, pfff. You’re one to talk."),
("the land well. Flar'Tar and Al'Brock, both of you", "the land well. Flar’Tar and Al’Brock, both of you"),
("the leadership of Kapou'e, the", "the leadership of Kapou’e, the"),
("the Northern Alliance. Kapou'e declined. Instead he", "the Northern Alliance. Kapou’e declined. Instead he"),
("the route that Kapou'e had used to", "the route that Kapou’e had used to"),
("the shamans. It won't work", "the shamans. It won’t work"),
("the Silent Forest, Kapou'e and his party", "the Silent Forest, Kapou’e and his party"),
("then never. Now it's time to kill!", "then never. Now it’s time to kill!"),
("Then, Kapou'e launched his desperate", "Then, Kapou’e launched his desperate"),
("there, old chap. You're deemed too insignificant", "there, old chap. You’re deemed too insignificant"),
("Throughout this nightmare Kapou'e drove himself and", "Throughout this nightmare Kapou’e drove himself and"),
("to a minimum, Kapou'e also", "to a minimum, Kapou’e also"),
("to counterattack. We can't defeat them by", "to counterattack. We can’t defeat them by"),
("to deal with. He's left that pleasure", "to deal with. He’s left that pleasure"),
("to him. Delighted, Kapou'e sent detachments off", "to him. Delighted, Kapou’e sent detachments off"),
("to know why you're here,", "to know why you’re here,"),
("to our one. We'll just have", "to our one. We’ll just have"),
("to steal from Kapou'e, Son of the", "to steal from Kapou’e, Son of the"),
("told us that that's where the shamans", "told us that that’s where the shamans"),
("troll, so you don't know. These lands", "troll, so you don’t know. These lands"),
("us if we don't have to worry", "us if we don’t have to worry"),
("wasn't joking about making", "wasn’t joking about making"),
("we destroy Earl Lanbec'h. Smite, stab, slay!", "we destroy Earl Lanbec’h. Smite, stab, slay!"),
("we meet any, we'll cut them to", "we meet any, we’ll cut them to"),
("we'll discuss it.", "we’ll discuss it."),
("We'll head for Barag", "We’ll head for Barag"),
("We've received messengers from", "We’ve received messengers from"),
("We've succeeded! We've passed through the", "We’ve succeeded! We’ve passed through the"),
("well known than Kapou'e.", "well known than Kapou’e."),
("Well let's see, first he", "Well let’s see, first he"),
("which Earl Lanbec'h had retreated.", "which Earl Lanbec’h had retreated."),
("Who is that? Kapou'e! What are you", "Who is that? Kapou’e! What are you"),
("who weren't killed in combat", "who weren’t killed in combat"),
("withdrawal' is another matter.", "withdrawal’ is another matter."),
("won't meet them.", "won’t meet them."),
("yield to fear, Kapou'e rallied his warriors", "yield to fear, Kapou’e rallied his warriors"),
("You don't understand. Orcs have", "You don’t understand. Orcs have"),
("you miserable coward?! I'll make", "you miserable coward?! I’ll make"),
("you so why don't you just", "you so why don’t you just"),
("You were right, Kapou'e, the", "You were right, Kapou’e, the"),
("You're right. I can", "You’re right. I can"),
("young upstart called Kapou'e. He", "young upstart called Kapou’e. He"),

("eaten that Grüü,", "eaten that, Grüü"),

# Fix capitalization
("\"Shan Taum The Smug\"", "\"Shan Taum the Smug\""),
("The Siege Of Barag Gór", "The Siege of Barag Gór"),
("End Of Peace", "End of Peace"),

# Convert apostrophes
("must know they can't break through", "must know they can’t break through"),
("Great Horde. We can't", "Great Horde. We can’t"),
("Defeat Earl Lanbec'h", "Defeat Earl Lanbec’h"),
("the other tribes won't think", "the other tribes won’t think"),
("What's this? This weakling", "What’s this? This weakling"),
("Bah! I'm the only", "Bah! I’m the only"),
("amongst themselves! Now's our", "amongst themselves! Now’s our"),
),

"wesnoth-tb" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("A Tale of Two Brothers - Epilogue", "A Tale of Two Brothers — Epilogue"),

# Convert straight apostrophes and quotation marks
## conversion added in 1.9.0-svn
("The village's mage Bjarn", "The village’s mage Bjarn"),
("'Fear and obey Mordak the Mage!'", "“Fear and obey Mordak the Mage!”"),
("Mordak's", "Mordak’s"),
("more than two days' ride", "more than two days’ ride"),
("brother's", "brother’s"),
("We're chasing after", "We’re chasing after"),
("master's", "master’s"),
("let's catch those", "let’s catch those"),
("We think there's", "We think there’s"),
("They've captured my brother", "They’ve captured my brother"),
("Arne's", "Arne’s"),
("let's kill some orcs", "let’s kill some orcs"),
("you're supposed", "you’re supposed"),
("isn't it", "isn’t it"),
("aren't our relief", "aren’t our relief"),
("they're holding Bjarn", "they’re holding Bjarn"),
("day's", "day’s"),
("I don't think we can rescue", "I don’t think we can rescue"),

# Fix capitalization
# conversion added in 1.9.0+svn
("Rooting Out A Mage", "Rooting Out a Mage"),

# Punctuation error.
# fixed in 1.9.1+svn
("brother still And", "brother still. And"),
),

"wesnoth-units" : (
# Spelling fixes required at r44124
("diminuitive", "diminutive"),
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("allowed - grudgingly - to", "allowed—grudgingly—to"),
("horseback - in fact", "horseback — in fact"),
("is quite justified -", "is quite justified —"),
("their own race - this power", "their own race — this power"),
("archer - though the heft", "archer — though the heft"),
("dangerous enough - the deadliness", "dangerous enough — the deadliness"),
("in battle - if only", "in battle — if only"),
("accompany it - it is a sin", "accompany it — it is a sin"),
("rarely seen - standing", "rarely seen — standing"),

# Fix spacing of old results
# conversion added in 1.9.0+svn
("allowed — grudgingly — to", "allowed—grudgingly—to"),

# Convert straight quotes to curly quotes
# conversion added in 1.9.0+svn
("particularly good at ""resolving"" territorial", "particularly good at “resolving” territorial"),
("the power they've learned to focus", "the power they’ve learned to focus"),
("challenge his tribe's leader for", "challenge his tribe’s leader for"),
("although it's most commonly", "although it’s most commonly"),
("one's own goals", "one’s own goals"),
("commands of it's master", "commands of it’s master"),
("This doesn't work", "This doesn’t work"),
("target's", "target’s"),
("enemies'", "enemies’"),

("known as 'Armageddon Drakes' are", "known as ‘Armageddon Drakes’ are"),
("protect their fellows' honor, and", "protect their fellows’ honor, and"),
("to the drake's traditional metal", "to the drake’s traditional metal"),
("good at \"resolving\" territorial disputes", "good at “resolving” territorial disputes"),
("challenge his tribe's", "challenge his tribe’s"),
("enemy where it's least", "enemy where it’s least"),
("supplement their bodies' momentum, adding", "supplement their bodies’ momentum, adding"),
("tribe's army. No", "tribe’s army. No"),
("of these 'thundersticks' is terrifying", "of these ‘thundersticks’ is terrifying"),
("from these 'thundersticks' can take", "from these ‘thundersticks’ can take"),
("any of humanity's fresh", "any of humanity’s fresh"),
("the elvish 'Avengers' comes from", "the elvish ‘Avengers’ comes from"),
("titled as 'sorceresses' by other", "titled as ‘sorceresses’ by other"),
("'Dire wolves' differ from", "‘Dire wolves’ differ from"),
("from these beasts' paws into", "from these beasts’ paws into"),
("disrupt the enemy's battle", "disrupt the enemy’s battle"),
("The Goblin 'Knights' have little", "The Goblin ‘Knights’ have little"),
("Nonetheless, a rouser's frenzied goading", "Nonetheless, a rouser’s frenzied goading"),
("are called 'Goblins' and are", "are called ‘Goblins’ and are"),
("for the paladins' loyalty is", "for the paladins’ loyalty is"),
("liege's apparent virtue.", "liege’s apparent virtue."),
("as the 'Grand Knights'", "as the ‘Grand Knights’"),
("custom of Wesnoth's nobility that", "custom of Wesnoth’s nobility that"),
("Magi as 'Red Magi'. Likewise, the", "Magi as ‘Red Magi’. Likewise, the"),
("signify the mage's ability to", "signify the mage’s ability to"),
("warriors, the 'Red Magi' have a", "warriors, the ‘Red Magi’ have a"),
("service, former 'footpads' rise up", "service, former ‘footpads’ rise up"),
("victim's belongings. Masters", "victim’s belongings. Masters"),
("of outlaw, they're not comfortable", "of outlaw, they’re not comfortable"),
("derisively called 'footpads' by their", "derisively called ‘footpads’ by their"),
("against their employer's enemies, often", "against their employer’s enemies, often"),
("known as 'Ruffians'. With enough", "known as ‘Ruffians’. With enough"),
("the various 'guilds' of", "the various ‘guilds’ of"),
("in a 'fair fight'.", "in a ‘fair fight’."),
("However, if you're hurling peasants", "However, if you’re hurling peasants"),
("your foes, you're clearly out", "your foes, you’re clearly out"),
("nets, mermen can't give any", "nets, mermen can’t give any"),
("are called 'Slayers' by their", "are called ‘Slayers’ by their"),
("arbalest or 'slurbow'; a", "arbalest or ‘slurbow’; a"),
("whatever 'societal' function they", "whatever ‘societal’ function they"),
("as large 'sticks and stones' serve them", "as large ‘sticks and stones’ serve them"),
("necrophage, or 'devourer of the dead', is a", "necrophage, or ‘devourer of the dead’, is a"),
("- a process", "— a process"),
("sight of one's", "sight of one’s"),
("immortality from nature's grasp, the", "immortality from nature’s grasp, the"),
("of black magic's first goal:", "of black magic’s first goal:"),
("is considered 'black magic' is the", "is considered ‘black magic’ is the"),
("alone caused humanity's condemnation of", "alone caused humanity’s condemnation of"),
("a dark sorcerer's creations, some", "a dark sorcerer’s creations, some"),
("dubbed simply 'Bone-Shooters' by their", "dubbed simply ‘Bone-Shooters’ by their"),
("known as 'Deathblades' were masters", "known as ‘Deathblades’ were masters"),
("from the 'skeletons' which often", "from the ‘skeletons’ which often"),
("In their masters' hands, these", "In their masters’ hands, these"),
("masters' powers. They", "masters’ powers. They"),

("taking a foe's head clean off.", "taking a foe’s head clean off."),
("poor horses' stamina", "poor horses’ stamina"),

# "naptha" -> "naphtha"
("naptha attack", "naphtha attack"),
),

"wesnoth-utbs" : (
# Convert makeshift dashes:
# conversion added in 1.9.0-svn
("Strike hard and fast and also be careful--right, this is going to be fun.", "Strike hard and fast and also be careful—right, this is going to be fun."),
("There might be, but I don't--", "There might be, but I don’t—"),
("You idiot--", "You idiot—"),
("unfortunately we found your men dead--", "unfortunately we found your men dead—"),
("Well, actually they were fleeing from--", "Well, actually they were fleeing from—"),
("Look, if you'll just let me explain--", "Look, if you’ll just let me explain—"),
("This is--", "This is—"),
("This entire journey has been based on a lie--", "This entire journey has been based on a lie—"),
("the last few generations--if anything the land had grown even more", "the last few generations—if anything the land had grown even more"),
("But if if this battle", "But if this battle"),

# Fix screw up
# conversion added in 1.9.0-svn
("Look, if you'll just let me explain—", "Look, if you’ll just let me explain—"),
("There might be, but I don't—", "There might be, but I don’t—"),

# Convert straight apostrophes and quotation marks
# conversion added in 1.9.0-svn
("What's happened? Oh Eloh, the craters are everywhere, everything is gone, ruined. I can hardly recognize our village. I didn't think it could be this bad.", "What’s happened? Oh Eloh, the craters are everywhere, everything is gone, ruined. I can hardly recognize our village. I didn’t think it could be this bad."),
("C'mon", "Come on"),
("Tanuil's", "Tanuil’s"),
("That's", "That’s"),
("Uria's", "Uria’s"),
("If we don't stop them", "If we don’t stop them"),
("Then let's join the battle!", "Then let’s join the battle!"),
("let's get rid of them", "let’s get rid of them"),
("If we don't", "If we don’t"),
("things won't be", "things won’t be"),
("we can't dwell on the dead", "we can’t dwell on the dead"),
("Let's keep exploring the wreckage.", "Let’s keep exploring the wreckage."),
("They're destroyed at last.", "They’re destroyed at last."),
("But there's no time", "But there’s no time"),
("so we'll have", "so we’ll have"),
("I'm fine. I'm afraid only", "I’m fine. I’m afraid only"),
("Maybe they're hiding in the stables. Let's go check.", "Maybe they’re hiding in the stables. Let’s go check."),
("We'll need your help", "We’ll need your help"),
("They've agreed to", "They’ve agreed to"),
("couldn't", "couldn’t"),
("Eloh's", "Eloh’s"),
("I've fought", "I’ve fought"),
("We'll just have to find out.", "We’ll just have to find out."),
("I suppose we wouldn't", "I suppose we wouldn’t"),
("there'll be one heck", "there’ll be one heck"),
("You've been working on", "You’ve been working on"),
("Hey Kaleh, how's", "Hey Kaleh, how’s"),
("I'm not quite", "I’m not quite"),
("Yechnagoth's", "Yechnagoth’s"),
("Go'hag", "Go’hag"),
("she's", "she’s"),
("Looks like that's the", "Looks like that’s the"),
("it's a long story", "it’s a long story"),
("you're a mage", "you’re a mage"),
("We'd", "We’d"),
("Let's cleanse", "Let’s cleanse"),
("if it's a fight they want, it's a fight they'll", "if it’s a fight they want, it’s a fight they’ll"),
("That's the last", "That’s the last"),
("there's still dried blood on the stones. It's", "there’s still dried blood on the stones. It’s"),
("they're going to be sorry", "they’re going to be sorry"),
("It's a good", "It’s a good"),
("don't kill me", "don’t kill me"),
("We're just", "We’re just"),
("It's Holy Water.", "It’s Holy Water."),
("we won't be able", "we won’t be able"),
("we've made it", "we’ve made it"),
("I've been searching", "I’ve been searching"),
("I'm searching for the", "I’m searching for the"),
("No, I haven't.", "No, I haven’t."),
("I'm not sure. I've read various references to it, but nothing specific. I've been searching for it for a long time. All I know is that it was a very powerful magical wand and that it was some sort of symbol of royalty in the old empire, but I have no idea where it might be. So I scour the land, learning all I can about the olden days. I'm sure it must be somewhere.", "I’m not sure. I’ve read various references to it, but nothing specific. I’ve been searching for it for a long time. All I know is that it was a very powerful magical wand and that it was some sort of symbol of royalty in the old empire, but I have no idea where it might be. So I scour the land, learning all I can about the olden days. I’m sure it must be somewhere."),
("exhausted. We've taken", "exhausted. We’ve taken"),
("I don't...", "I don’t..."),
("or what's left", "or what’s left"),
("I'm impressed.", "I’m impressed."),
("You've been working", "You’ve been working"),
("couldn't be worse", "couldn’t be worse"),
("they haven't had the", "they haven’t had the"),
("I don't think we've explored", "I don’t think we’ve explored"),
("We've explored the village and I think we've", "We’ve explored the village and I think we’ve"),
("glad you're here", "glad you’re here"),
("'You must be strong, young elf", "“You must be strong, young elf"),
("you and protect you.'", "you and protect you.”"),
("We can't leave them", "We can’t leave them"),
("I don't want to tarry", "I don’t want to tarry"),
("What's that to", "What’s that to"),
("won't be forgotten", "won’t be forgotten"),
("Zhul's", "Zhul’s"),
("friends'", "friends’"),
("you don't recruit", "you don’t recruit"),
("I don't see a thing", "I don’t see a thing"),
("poor person's body", "poor person’s body"),
("There doesn't seem", "There doesn’t seem"),
("wait...what's this", "wait...what’s this"),
("I've heard tales", "I’ve heard tales"),
("Traveler's Ring", "Traveler’s Ring"),

# Convert more apostrophes/quotes and makeshift dashes
# conversion added in 1.9.0+svn
("I'll be back soon!", "I’ll be back soon!"),
("Let's aid the dwarves.", "Let’s aid the dwarves."),
("Let's aid the trolls.", "Let’s aid the trolls."),
("Kaleh, a quick question-", "Kaleh, a quick question—"),

("'traders' used to come", "‘traders’ used to come"),
("'Yechnagoth' and 'Zhangor'. She said that", "‘Yechnagoth’ and ‘Zhangor’. She said that"),
("a land of 'kill or be killed'. Outlaws, ogres", "a land of ‘kill or be killed’. Outlaws, ogres"),
("a lesson they won't soon forget. Attack!", "a lesson they won’t soon forget. Attack!"),
("a minute, I don't see any targets", "a minute, I don’t see any targets"),
("after your enemies fall' Grave robbing is", "after your enemies fall” Grave robbing is"),
("again, Kaleh, this isn't the desert. The", "again, Kaleh, this isn’t the desert. The"),
("again. You numbskulls aren't getting the easy", "again. You numbskulls aren’t getting the easy"),
("Ah, it's a hard knock", "Ah, it’s a hard knock"),
("all contact. I don't know if any", "all contact. I don’t know if any"),
("all this way, I'm not going to", "all this way, I’m not going to"),
("also declared that Zhangor's name and his", "also declared that Zhangor’s name and his"),
("And besides, we can't afford to lose", "And besides, we can’t afford to lose"),
("and better. We haven't seen one of", "and better. We haven’t seen one of"),
("and joining Tanstafaal. I'm afraid that while", "and joining Tanstafaal. I’m afraid that while"),
("and protect you.” I'm", "and protect you.” I’m"),
("and so I wasn't told the location.", "and so I wasn’t told the location."),
("Anyway you've really gotten us", "Anyway you’ve really gotten us"),
("anyway? I'm going to have", "anyway? I’m going to have"),
("are close. Let's be far away", "are close. Let’s be far away"),
("are hard pressed. We're going to have", "are hard pressed. We’re going to have"),
("are pitch black! It's as bad as", "are pitch black! It’s as bad as"),
("are the words: 'May you have the", "are the words: “May you have the"),
("are, and we don't much care. Tremble", "are, and we don’t much care. Tremble"),
("aren't safe. Can you", "aren’t safe. Can you"),
("as a hermit crab's shell. It must", "as a hermit crab’s shell. It must"),
("as a warrior. Let's respect his wishes", "as a warrior. Let’s respect his wishes"),
("as you think. Let's explore the village", "as you think. Let’s explore the village"),
("attacking it. We can't stop now!", "attacking it. We can’t stop now!"),
("Aye, there's still life in", "Aye, there’s still life in"),
("be left that I'm not leading my", "be left that I’m not leading my"),
("be magical. Maybe that's why he was", "be magical. Maybe that’s why he was"),
("be too sure. I'll", "be too sure. I’ll"),
("beast you aren't familiar with. The", "beast you aren’t familiar with. The"),
("become useful eventually. I'll keep it just", "become useful eventually. I’ll keep it just"),
("been a mirage. I've been out", "been a mirage. I’ve been out"),
("Before we leave, there's one thing I", "Before we leave, there’s one thing I"),
("being burnt alive! It's terrible, I", "being burnt alive! It’s terrible, I"),
("better. But we don't know", "better. But we don’t know"),
("blue stuff. I don't know", "blue stuff. I don’t know"),
("blue. The pool isn't very deep, I", "blue. The pool isn’t very deep, I"),
("broken. But I don't think we have", "broken. But I don’t think we have"),
("but frankly we didn't have enough dwarves", "but frankly we didn’t have enough dwarves"),
("But I still don't understand how you", "But I still don’t understand how you"),
("But no, they wouldn't kill me. They", "But no, they wouldn’t kill me. They"),
("But still she doesn't stand a", "But still she doesn’t stand a"),
("but to forgive divine'?", "but to forgive divine”?"),
("but, well, we don't like going that", "but, well, we don’t like going that"),
("by orcs. I wouldn't be surprised if", "by orcs. I wouldn’t be surprised if"),
("by the Sea God's hand I call", "by the Sea God’s hand I call"),
("can damage it. It's almost as if", "can damage it. It’s almost as if"),
("can't bear to watch.", "can’t bear to watch."),
("can't last forever, and", "can’t last forever, and"),
("Can't the dead ever", "Can’t the dead ever"),
("caves, and we won't have the advantages", "caves, and we won’t have the advantages"),
("charges! If we can't stop them then", "charges! If we can’t stop them then"),
("charges! If we can't stop them", "charges! If we can’t stop them"),
("close to the trolls' main lair; you", "close to the trolls’ main lair; you"),
("Come on! I ain't going anywhere for", "Come on! I ain’t going anywhere for"),
("consequences would be. I'm not", "consequences would be. I’m not"),
("creature was, she doesn't smell any better", "creature was, she doesn’t smell any better"),
("cross our blades... It's time", "cross our blades... It’s time"),
("Curse that girl! She'll be the death", "Curse that girl! She’ll be the death"),
("dangers we don't. We really are", "dangers we don’t. We really are"),
("darkness, and you won't be lacking a", "darkness, and you won’t be lacking a"),
("darkness? And I'm afraid that while", "darkness? And I’m afraid that while"),
("Dead?! You 'found' them you say?", "Dead?! You ‘found’ them you say?"),
("defenses'. It would have", "defenses’. It would have"),
("didn't have vile humans", "didn’t have vile humans"),
("didn't talk much, but", "didn’t talk much, but"),
("died fighting gloriously. I'm so ashamed. I", "died fighting gloriously. I’m so ashamed. I"),
("disappears at a moment's notice. No one", "disappears at a moment’s notice. No one"),
("do you mean 'lead you to me'? You are a god, don't you exist", "do you mean ‘lead you to me’? You are a god, don’t you exist"),
("Do you think I'm stupid enough to", "Do you think I’m stupid enough to"),
("doesn't know where to", "doesn’t know where to"),
("doesn't look good.", "doesn’t look good."),
("doesn't make sense.", "doesn’t make sense."),
("Don't ask where the", "Don’t ask where the"),
("Don't blame yourself. You didn't know. If we", "Don’t blame yourself. You didn’t know. If we"),
("don't do anything dangerous.", "don’t do anything dangerous."),
("don't have to spend.", "don’t have to spend."),
("don't know how long.", "don’t know how long."),
("don't know what happened.", "don’t know what happened."),
("Don't listen to them,", "Don’t listen to them,"),
("don't seem as threatening", "don’t seem as threatening"),
("don't suppose they're going to mind", "don’t suppose they’re going to mind"),
("Don't think you know", "Don’t think you know"),
("don't want the rest", "don’t want the rest"),
("Don't worry. We're not the monsters", "Don’t worry. We’re not the monsters"),
("Don't you worry about us, Kaleh. We'll be careful. I'll organize five bands", "Don’t you worry about us, Kaleh. We’ll be careful. I’ll organize five bands"),
("Don't you worry, we won't rest until we", "Don’t you worry, we won’t rest until we"),
("down this way. It's odd, I guess", "down this way. It’s odd, I guess"),
("dry ground, and you've been through a", "dry ground, and you’ve been through a"),
("due north. From Esanoo's description it sounds", "due north. From Esanoo’s description it sounds"),
("duplicate their ancestors' previous success, the", "duplicate their ancestors’ previous success, the"),
("dwarves. And you don't really need me,", "dwarves. And you don’t really need me,"),
("effect! I don't think the trolls", "effect! I don’t think the trolls"),
("elf... But you won't be so lucky", "elf... But you won’t be so lucky"),
("Eloh and the merfolk's god, and delighting", "Eloh and the merfolk’s god, and delighting"),
("escapes the valley, we'll be in trouble.", "escapes the valley, we’ll be in trouble."),
("explored no further. I'm", "explored no further. I’m"),
("faster if you don't interrupt me. Now,", "faster if you don’t interrupt me. Now,"),
("fear not the dark.' And again, just", "fear not the dark.” And again, just"),
("fear will surely appear' It looks", "fear will surely appear’ It looks"),
("fighting, but she doesn't smell any better", "fighting, but she doesn’t smell any better"),
("find another infestation. They've", "find another infestation. They’ve"),
("first. I don't think I want", "first. I don’t think I want"),
("for rescuing me. We'll show those humans", "for rescuing me. We’ll show those humans"),
("former owner certainly won't miss it.", "former owner certainly won’t miss it."),
("forward to that. I'm afraid my scales", "forward to that. I’m afraid my scales"),
("free us. I can't believe the humans", "free us. I can’t believe the humans"),
("from them, I can't leave now.", "from them, I can’t leave now."),
("from, but she wouldn't answer, she just", "from, but she wouldn’t answer, she just"),
("from. You really don't want to know.", "from. You really don’t want to know."),
("fruitful lives. Don't let our sacrifices", "fruitful lives. Don’t let our sacrifices"),
("further. I'm freezing cold, wet, and I can't see a thing. I'm not exactly", "further. I’m freezing cold, wet, and I can’t see a thing. I’m not exactly"),
("Garak were here, he'd know more about", "Garak were here, he’d know more about"),
("go through those 'light", "go through those ‘light"),
("Good, then let's get out of", "Good, then let’s get out of"),
("Good, until then we'll settle around that", "Good, until then we’ll settle around that"),
("Good, we'll move our people", "Good, we’ll move our people"),
("Good. We're safe for now.", "Good. We’re safe for now."),
("Greetings. I'm afraid I am", "Greetings. I’m afraid I am"),
("group of enemies. Let's hope", "group of enemies. Let’s hope"),
("guard, if we don't come out in", "guard, if we don’t come out in"),
("Ha, you're trapped. I've got you right", "Ha, you’re trapped. I’ve got you right"),
("hadn't come Grog would", "hadn’t come Grog would"),
("half-manned. They obviously didn't expect any serious", "half-manned. They obviously didn’t expect any serious"),
("happened! Who in Moradin's name rigged the", "happened! Who in Moradin’s name rigged the"),
("have defeated me, don't you? Foolish boy,", "have defeated me, don’t you? Foolish boy,"),
("he died a warrior's", "he died a warrior’s"),
("He'll be slaughtered! He", "He’ll be slaughtered! He"),
("He's dead. Rest in", "He’s dead. Rest in"),
("He's probably dead, then,", "He’s probably dead, then,"),
("heh, no you didn't find me. I", "heh, no you didn’t find me. I"),
("here are insects, I'll be very", "here are insects, I’ll be very"),
("here too long we'll roast alive. I don't even want to", "here too long we’ll roast alive. I don’t even want to"),
("here, and yet there's so", "here, and yet there’s so"),
("here, but I don't know what made", "here, but I don’t know what made"),
("Hey, c'mon, maybe we can", "Hey, c’mon, maybe we can"),
("Hey, what's this? There seems", "Hey, what’s this? There seems"),
("him through. Well here's your chance. Come", "him through. Well here’s your chance. Come"),
("his scouts haven't reported back yet. I'm starting to get", "his scouts haven’t reported back yet. I’m starting to get"),
("horizon, but I don't know what it", "horizon, but I don’t know what it"),
("Humans aren't just the bandits and outlaws you're familiar with from", "Humans aren’t just the bandits and outlaws you’re familiar with from"),
("humans doing here? I've never seen so", "humans doing here? I’ve never seen so"),
("humans much anyway. I'll be glad to", "humans much anyway. I’ll be glad to"),
("hurts and I, I'm stuck!", "hurts and I, I’m stuck!"),
("I am, I'm a fire mage. I've been traveling for", "I am, I’m a fire mage. I’ve been traveling for"),
("I can't even begin to", "I can’t even begin to"),
("I do. But it's just a theory. I don't want to say", "I do. But it’s just a theory. I don’t want to say"),
("I don't care! Do you think I'm going to let", "I don’t care! Do you think I’m going to let"),
("I don't get it. All", "I don’t get it. All"),
("I don't know what 'they' are, but we can't go back. Prepare", "I don’t know what ‘they’ are, but we can’t go back. Prepare"),
("I don't know. When I", "I don’t know. When I"),
("I don't see any spiders,", "I don’t see any spiders,"),
("I don't want to be", "I don’t want to be"),
("I fear our people's fate", "I fear our people’s fate"),
("I for one can't wait to feel", "I for one can’t wait to feel"),
("I lived long, I'm not afraid of", "I lived long, I’m not afraid of"),
("I thought that's what all the", "I thought that’s what all the"),
("I'd better head back", "I’d better head back"),
("I'll blow the charges", "I’ll blow the charges"),
("I'll just be happy", "I’ll just be happy"),
("I'll muster the rest", "I’ll muster the rest"),
("I'm exhausted after walking", "I’m exhausted after walking"),
("I'm not sure. I've read various references", "I’m not sure. I’ve read various references"),
("I'm sorry Kaleh, I", "I’m sorry Kaleh, I"),
("I'm sorry, Kaleh. My", "I’m sorry, Kaleh. My"),
("I'm still not sure", "I’m still not sure"),
("I've followed that boy this far, I'm not going to", "I’ve followed that boy this far, I’m not going to"),
("I've heard of dwarves,", "I’ve heard of dwarves,"),
("I've heard of your", "I’ve heard of your"),
("I've saved his sorry", "I’ve saved his sorry"),
("I've seen the ancient", "I’ve seen the ancient"),
("in the end, I'm afraid you just", "in the end, I’m afraid you just"),
("in the wall. 'If you dare", "in the wall. “If you dare"),
("in this chamber. You're surprised to see", "in this chamber. You’re surprised to see"),
("Incoming! Ugh, it's big, hairy, and", "Incoming! Ugh, it’s big, hairy, and"),
("instead of a coward's. For that I", "instead of a coward’s. For that I"),
("interesting. In truth, I'd never heard of", "interesting. In truth, I’d never heard of"),
("into dust. All that's", "into dust. All that’s"),
("into the caves you'll", "into the caves you’ll"),
("is one they don't have to. Still,", "is one they don’t have to. Still,"),
("is so hot it's stifling; I can", "is so hot it’s stifling; I can"),
("island. I bet it's", "island. I bet it’s"),
("it can't kill it.", "it can’t kill it."),
("It wouldn't have been my", "It wouldn’t have been my"),
("it's a hard knock", "it’s a hard knock"),
("It's a huge underground", "It’s a huge underground"),
("It's a long story,", "It’s a long story,"),
("It's another one of", "It’s another one of"),
("It's complicated, and I don't know how much", "It’s complicated, and I don’t know how much"),
("It's hard to imagine.", "It’s hard to imagine."),
("It's not important. What's important is that", "It’s not important. What’s important is that"),
("It's okay, you're not used to", "It’s okay, you’re not used to"),
("It's some sort of", "It’s some sort of"),
("its power. We don't normally tolerate using", "its power. We don’t normally tolerate using"),
("jungle, but I can't tell what", "jungle, but I can’t tell what"),
("just fine. I don't know", "just fine. I don’t know"),
("Kaleh, I wouldn't be surprised if", "Kaleh, I wouldn’t be surprised if"),
("Kaleh, we don't have time for", "Kaleh, we don’t have time for"),
("Kali's just a hardass because he's bitter that he", "Kali’s just a hardass because he’s bitter that he"),
("keep practicing until I'm satisfied.", "keep practicing until I’m satisfied."),
("killing trolls. And you're not going anywhere until I'm done with you.", "killing trolls. And you’re not going anywhere until I’m done with you."),
("Lady's name are you", "Lady’s name are you"),
("lands when I don't know what's in front of us. We've lost too many", "lands when I don’t know what’s in front of us. We’ve lost too many"),
("Leader wouldn't want any unpleasant", "Leader wouldn’t want any unpleasant"),
("leaderless. I guess it's every ant for", "leaderless. I guess it’s every ant for"),
("leadership so far; I'll wait", "leadership so far; I’ll wait"),
("let her die? I've never seen fire", "let her die? I’ve never seen fire"),
("Let's blast those monsters", "Let’s blast those monsters"),
("lightly defended; he's sent most of", "lightly defended; he’s sent most of"),
("Like I haven't killed enough undead recently. Why can't these guys just", "Like I haven’t killed enough undead recently. Why can’t these guys just"),
("like. I suppose we'll just try", "like. I suppose we’ll just try"),
("listen to the mages' protests. So, seeking", "listen to the mages’ protests. So, seeking"),
("lived underground, Elyssa. I've", "lived underground, Elyssa. I’ve"),
("location. So she didn't tell us where", "location. So she didn’t tell us where"),
("long as you didn't try to cheat", "long as you didn’t try to cheat"),
("Looks like he's been squirreling away", "Looks like he’s been squirreling away"),
("Looks like we don't have any choice", "Looks like we don’t have any choice"),
("made it outside. I'd forgotten how big", "made it outside. I’d forgotten how big"),
("many of our people's faith is wavering.", "many of our people’s faith is wavering."),
("master know about 'Yechnagoth' and 'Zhangor'? If Eloh", "master know about ‘Yechnagoth’ and ‘Zhangor’? If Eloh"),
("May the Sea God's bounty bless you", "May the Sea God’s bounty bless you"),
("Maybe it's just fish, but", "Maybe it’s just fish, but"),
("me if I don't take you at", "me if I don’t take you at"),
("meantime, we'll go ahead and", "meantime, we’ll go ahead and"),
("merfolk's help we would", "merfolk’s help we would"),
("merfolk, then we don't have any time", "merfolk, then we don’t have any time"),
("mess. I think you're right", "mess. I think you’re right"),
("more fire guardians. I'm going to be", "more fire guardians. I’m going to be"),
("my hand. It doesn't seem to have", "my hand. It doesn’t seem to have"),
("my life. I can't believe I was", "my life. I can’t believe I was"),
("named it 'Quenoth Isle' in honor of", "named it ‘Quenoth Isle’ in honor of"),
("nasty orcses. And 'specially stinking elves.", "nasty orcses. And ’specially stinking elves."),
("need to explain, it's pretty obvious what you're up to. Here", "need to explain, it’s pretty obvious what you’re up to. Here"),
("No I haven't, I don't often explore underground", "No I haven’t, I don’t often explore underground"),
("no idea what you're babbling about, elf, but you're just lucky you", "no idea what you’re babbling about, elf, but you’re just lucky you"),
("No we can't. But for now", "No we can’t. But for now"),
("No, I don't think you shall", "No, I don’t think you shall"),
("No, you don't have to introduce", "No, you don’t have to introduce"),
("northeast and northwest. Don't go too far,", "northeast and northwest. Don’t go too far,"),
("not enough. Look, you're a wonderful girl", "not enough. Look, you’re a wonderful girl"),
("not sure why I'm the one she", "not sure why I’m the one she"),
("Not worth... Whoa! What's happening?!", "Not worth... Whoa! What’s happening?!"),
("Now that we've set up a", "Now that we’ve set up a"),
("Now that you've found the way", "Now that you’ve found the way"),
("Nym is Kaleh's childhood friend who", "Nym is Kaleh’s childhood friend who"),
("Nym, and Kaleh, you've", "Nym, and Kaleh, you’ve"),
("oases aren't easy to find", "oases aren’t easy to find"),
("obsidian wall. I can't find any entrances", "obsidian wall. I can’t find any entrances"),
("obsidian wall. I can't find any way", "obsidian wall. I can’t find any way"),
("of here, but it's already flooding.", "of here, but it’s already flooding."),
("of me boys didn't think you could", "of me boys didn’t think you could"),
("of the desert? It's a rather barren", "of the desert? It’s a rather barren"),
("of the orcs. I'd seen sandstorms", "of the orcs. I’d seen sandstorms"),
("of water: it's even creating a", "of water: it’s even creating a"),
("of Wesnoth. And it's isn't", "of Wesnoth. And it’s isn’t"),
("Oh they're no need to explain, it's pretty obvious what you're up to. Here", "Oh, there’s no need to explain, it’s pretty obvious what you’re up to. Here"),
("Oh, Eloh. They're all dead. Butchered.", "Oh, Eloh. They’re all dead. Butchered."),
("on the horizon. We'll surely be", "on the horizon. We’ll surely be"),
("on Trolls. But you'll see,", "on Trolls. But you’ll see,"),
("Once we've escaped and are", "Once we’ve escaped and are"),
("orcs and goblins. We'll never make it", "orcs and goblins. We’ll never make it"),
("our destination. It wasn't even an island,", "our destination. It wasn’t even an island,"),
("our god, I don't remember Zhul mentioning", "our god, I don’t remember Zhul mentioning"),
("our people dwindle, it's going to become", "our people dwindle, it’s going to become"),
("our people dwindle, it's going to", "our people dwindle, it’s going to"),
("outpost isn't guarded as heavily", "outpost isn’t guarded as heavily"),
("Part of me didn't think we would", "Part of me didn’t think we would"),
("past the orcs. It's odd, I guess", "past the orcs. It’s odd, I guess"),
("peacefully or I'll make you sorry you didn't.", "peacefully or I’ll make you sorry you didn’t."),
("people already. I don't want to lead", "people already. I don’t want to lead"),
("people, I don't want to kill", "people, I don’t want to kill"),
("Pinnacle Rock. It's the highest land", "Pinnacle Rock. It’s the highest land"),
("place and they can't reach him there.", "place and they can’t reach him there."),
("places you can't.", "places you can’t."),
("planned or not, what's important is what", "planned or not, what’s important is what"),
("possible, but we won't last long in", "possible, but we won’t last long in"),
("pray, but I'm going in there!", "pray, but I’m going in there!"),
("protecting our encampments isn't enough, darkness and", "protecting our encampments isn’t enough, darkness and"),
("put here, but it's obviously powerful and", "put here, but it’s obviously powerful and"),
("rally more reinforcements. We're hurtin', Kaleh, I'll", "rally more reinforcements. We’re hurtin’, Kaleh, I’ll"),
("range. When they don't have a Queen", "range. When they don’t have a Queen"),
("recent losses we don't have the", "recent losses we don’t have the"),
("reconnaissance. We'll send out small", "reconnaissance. We’ll send out small"),
("red glowing runes. I'm not", "red glowing runes. I’m not"),
("refer to as 'The Golden Age'.", "refer to as ‘The Golden Age’."),
("reinforcements have arrived. We'll surely", "reinforcements have arrived. We’ll surely"),
("Remember, Kaleh, it's nasty fighting underground.", "Remember, Kaleh, it’s nasty fighting underground."),
("right you runts, let's try this again.", "right you runts, let’s try this again."),
("right, I'm going in. The", "right, I’m going in. The"),
("river and lake. I'm glad we weren't downstream of that", "river and lake. I’m glad we weren’t downstream of that"),
("river. I sure wouldn't want to be", "river. I sure wouldn’t want to be"),
("sacrificed them on Zhangor's bloody altars. Zhangor", "sacrificed them on Zhangor’s bloody altars. Zhangor"),
("safety. We certainly wouldn't want", "safety. We certainly wouldn’t want"),
("said $intl_ally_name, that you'd been here before?", "said $intl_ally_name, that you’d been here before?"),
("seem particularly inhospitable! I've been", "seem particularly inhospitable! I’ve been"),
("seen. It's almost unnatural.", "seen. It’s almost unnatural."),
("serve the merfolk's foul god.", "serve the merfolk’s foul god."),
("set, then I won't try to convince", "set, then I won’t try to convince"),
("snowy peaks. So that's how several thousand", "snowy peaks. So that’s how several thousand"),
("so big. And what's", "so big. And what’s"),
("so far. I don't think I", "so far. I don’t think I"),
("Some of trolls didn't think you strong", "Some of trolls didn’t think you strong"),
("Sometimes called the 'hollow men', spectres form the", "Sometimes called the ‘hollow men’, spectres form the"),
("south. And we can't hang around here", "south. And we can’t hang around here"),
("spider. Aren't we lucky!", "spider. Aren’t we lucky!"),
("spilled already. Knock 'em out, wound them,", "spilled already. Knock ’em out, wound them,"),
("stay dead? And aren't they trapped by", "stay dead? And aren’t they trapped by"),
("stayed in the captain's quarters and", "stayed in the captain’s quarters and"),
("still haunted by Nym's death in that", "still haunted by Nym’s death in that"),
("Still, the trolls don't seem to be", "Still, the trolls don’t seem to be"),
("stone wall. This can't be a coincidence.", "stone wall. This can’t be a coincidence."),
("strange vision. I didn't have long to", "strange vision. I didn’t have long to"),
("such as the elves' stats and the", "such as the elves’ stats and the"),
("summoned back I don't think we", "summoned back I don’t think we"),
("sunlit lands. When you're ready I'll show you", "sunlit lands. When you’re ready I’ll show you"),
("suns. She said 'Have", "suns. She said “Have"),
("support, so we don't want to run", "support, so we don’t want to run"),
("sure what I'm doing up here.", "sure what I’m doing up here."),
("surface, and besides there's still lots of", "surface, and besides there’s still lots of"),
("swear those castles weren't there at sunset.", "swear those castles weren’t there at sunset."),
("tastes terrible, I can't drink this! What", "tastes terrible, I can’t drink this! What"),
("tell me to 'kill the unbelievers'? If we had", "tell me to ‘kill the unbelievers’? If we had"),
("Tenacious savages, aren't they? But these", "Tenacious savages, aren’t they? But these"),
("than I do. I've", "than I do. I’ve"),
("Thank you, I'm sure you will", "Thank you, I’m sure you will"),
("Thank you. I'm sorry, I don't know what came", "Thank you. I’m sorry, I don’t know what came"),
("that appears by Tanstafaal's", "that appears by Tanstafaal’s"),
("that I don't know where she", "that I don’t know where she"),
("that Yechnagoth wasn't the one who", "that Yechnagoth wasn’t the one who"),
("that? It sure doesn't look good. The", "that? It sure doesn’t look good. The"),
("the best, so I'll be your", "the best, so I’ll be your"),
("the cavern nicely. I'm just thankful the", "the cavern nicely. I’m just thankful the"),
("the dangers we don't. We really are", "the dangers we don’t. We really are"),
("the essence, so let's move out as", "the essence, so let’s move out as"),
("the hills, you can't even see all that water, it's", "the hills, you can’t even see all that water, it’s"),
("the island's foul inhabitants, and", "the island’s foul inhabitants, and"),
("the left. And it's big enough", "the left. And it’s big enough"),
("the olden days. I'm sure it must", "the olden days. I’m sure it must"),
("the past, and there's nothing we can", "the past, and there’s nothing we can"),
("The point is, it's not her fight, it's our fight too.", "The point is, it’s not her fight, it’s our fight too."),
("the rubble. It doesn't look", "the rubble. It doesn’t look"),
("the shamans. The troll's military leader was", "the shamans. The troll’s military leader was"),
("The stinkin' elves have freed", "The stinkin’ elves have freed"),
("the stone. But there's no", "the stone. But there’s no"),
("the suns. But I've", "the suns. But I’ve"),
("the troll menace. They've been", "the troll menace. They’ve been"),
("the unbelievers proves your-", "the unbelievers proves your—"),
("the way. It's not far.", "the way. It’s not far."),
("the world that hadn't been plagued with", "the world that hadn’t been plagued with"),
("their fear of Zhangor's wrath, the", "their fear of Zhangor’s wrath, the"),
("them. I think that's where they are", "them. I think that’s where they are"),
("them. Whatever 'them' was.", "them. Whatever ‘them’ was."),
("Then we'll just have to", "Then we’ll just have to"),
("There isn't much left of", "There isn’t much left of"),
("these branching tunnels we'll have no idea", "these branching tunnels we’ll have no idea"),
("these secret tunnels, you'd think they could", "these secret tunnels, you’d think they could"),
("These smaller hills aren't as hard to", "These smaller hills aren’t as hard to"),
("these tunnels deserted. I'd be", "these tunnels deserted. I’d be"),
("these tunnels deserted. I'll be", "these tunnels deserted. I’ll be"),
("they are, it can't be good.", "they are, it can’t be good."),
("they look. And they're huge! Well, the", "they look. And they’re huge! Well, the"),
("They obviously weren't enough. You go", "They obviously weren’t enough. You go"),
("they shan't. We shall kill", "they shan’t. We shall kill"),
("They're a proud people,", "They’re a proud people,"),
("They've come for my precious. It's mine, yes it is. They shan't have it, no", "They’ve come for my precious. It’s mine, yes it is. They shan’t have it, no"),
("They've crossed the chasm!", "They’ve crossed the chasm!"),
("thing to a god's actual", "thing to a god’s actual"),
("this time, we won't always be so", "this time, we won’t always be so"),
("those caves aren't safe. Can you", "those caves aren’t safe. Can you"),
("through this inferno I'm", "through this inferno I’m"),
("to a head. I'm worried about Tanstafaal", "to a head. I’m worried about Tanstafaal"),
("to be judged. They'll deal with you", "to be judged. They’ll deal with you"),
("to challenge me, I'll devour your soul", "to challenge me, I’ll devour your soul"),
("to give you. I'm afraid that where", "to give you. I’m afraid that where"),
("to ignore enemy units' zones of", "to ignore enemy units’ zones of"),
("to me? This isn't a negotiation. You", "to me? This isn’t a negotiation. You"),
("to meet the merfolk's leader!", "to meet the merfolk’s leader!"),
("to open it. Let's go, I'd really like to", "to open it. Let’s go, I’d really like to"),
("to respect your leaders' decisions. Kaleh has", "to respect your leaders’ decisions. Kaleh has"),
("to the base. They're going to sacrifice", "to the base. They’re going to sacrifice"),
("tomorrow night, then I'll tell you", "tomorrow night, then I’ll tell you"),
("Too late. And it's just sand inside.", "Too late. And it’s just sand inside."),
("trolls' secret weapon.", "trolls’ secret weapon."),
("trolls. And you don't really need me,", "trolls. And you don’t really need me,"),
("trolls. But there's no point dwelling", "trolls. But there’s no point dwelling"),
("truly came from. I'd been to some", "truly came from. I’d been to some"),
("tunnels towards the Dwarves'", "tunnels towards the Dwarves’"),
("Ugh. I'm covered in blood", "Ugh. I’m covered in blood"),
("Undead don't have to breathe and I don't think a little", "Undead don’t have to breathe and I don’t think a little"),
("Until I know what's out there, I'm not taking any", "Until I know what’s out there, I’m not taking any"),
("upon the elves' fears and demanded", "upon the elves’ fears and demanded"),
("us back. Well, we'll show them. It", "us back. Well, we’ll show them. It"),
("us if we can't end this battle", "us if we can’t end this battle"),
("us, we'll stick to the", "us, we’ll stick to the"),
("used to. So we'd best be extra", "used to. So we’d best be extra"),
("very close to dwarves' main lair. You", "very close to dwarves’ main lair. You"),
("Very well. I've trusted your decisions", "Very well. I’ve trusted your decisions"),
("vile, that I can't imagine them worshiping", "vile, that I can’t imagine them worshiping"),
("wait a minute, you're elves!?!", "wait a minute, you’re elves!?!"),
("wall of stone. I've had me boys", "wall of stone. I’ve had me boys"),
("wand. I don't have much experience", "wand. I don’t have much experience"),
("want to flee, don't you? But you", "want to flee, don’t you? But you"),
("want to know what's out there.", "want to know what’s out there."),
("warriors. I'm sure they have", "warriors. I’m sure they have"),
("wastes if we can't find", "wastes if we can’t find"),
("water-skins and rations. We'll have enough if", "water-skins and rations. We’ll have enough if"),
("way. But they aren't even elves,", "way. But they aren’t even elves,"),
("We don't know who you", "We don’t know who you"),
("we know you aren't servants of Yechnagoth,", "we know you aren’t servants of Yechnagoth,"),
("We will remember Esanoo's sacrifice.", "We will remember Esanoo’s sacrifice."),
("We won't. If we keep", "We won’t. If we keep"),
("We'll meet you along", "We’ll meet you along"),
("We're in luck, a", "We’re in luck, a"),
("We're still trying to", "We’re still trying to"),
("We've come far and", "We’ve come far and"),
("We've come out on", "We’ve come out on"),
("We've found both keys.", "We’ve found both keys."),
("We've reached what looks", "We’ve reached what looks"),
("We've run out of", "We’ve run out of"),
("We've taken too long", "We’ve taken too long"),
("Well, Nym's right, we don't know what's out there. So", "Well, Nym’s right, we don’t know what’s out there. So"),
("Well, we can't spend all day", "Well, we can’t spend all day"),
("west of here. Don't worry about losing", "west of here. Don’t worry about losing"),
("what's inside...", "what’s inside..."),
("What's that, boy? Are", "What’s that, boy? Are"),
("What's this on our borders? Stinkin' elves and more undead? We'll teach them", "What’s this on our borders? Stinkin’ elves and more undead? We’ll teach them"),
("What's this? Hidden underneath", "What’s this? Hidden underneath"),
("What's this? His clothes", "What’s this? His clothes"),
("What's this? Someone has", "What’s this? Someone has"),
("Who are you? What's behind that black", "Who are you? What’s behind that black"),
("who knows what we'll encounter deep under", "who knows what we’ll encounter deep under"),
("with their tactics; we've lost several of", "with their tactics; we’ve lost several of"),
("with us, but I'm afraid we don't", "with us, but I’m afraid we don’t"),
("won't help me, then", "won’t help me, then"),
("won't let a couple", "won’t let a couple"),
("won't lose all your", "won’t lose all your"),
("years. Her commands don't make any sense; she doesn't seem", "years. Her commands don’t make any sense; she doesn’t seem"),
("Yes, but I didn't explore very far.", "Yes, but I didn’t explore very far."),
("you can't drink it?", "you can’t drink it?"),
("You didn't expect the other", "You didn’t expect the other"),
("you first arrived, we've had a few", "you first arrived, we’ve had a few"),
("you know what you're doing, Kaleh.", "you know what you’re doing, Kaleh."),
("you not say: 'to err is elven,", "you not say: “to err is elven,"),
("You rescued Nog's brother, and he", "You rescued Nog’s brother, and he"),
("you sacrifice for Garak's sake?", "you sacrifice for Garak’s sake?"),
("You saved Grog's life. Grog still", "You saved Grog’s life. Grog still"),
("you see him, he's big, green and", "you see him, he’s big, green and"),
("you with. We haven't sent anyone to", "you with. We haven’t sent anyone to"),
("You're being very cryptic,", "You’re being very cryptic,"),
("you, I don't think that the", "you, I don’t think that the"),
("your word. We haven't seen elves for", "your word. We haven’t seen elves for"),
("Zur... Come mortal, let's cross our blades... It's time for", "Zur... Come mortal, let’s cross our blades... It’s time for"),

# Fix screw up
# conversion added in 1.9.0+svn
("in the wall. ‘If you dare", "in the wall. “If you dare"),

# Fixed capitalization
("you ever heard of The", "you ever heard of the"),

# Kill the hyphens
("Mid-day", "Midday"),
("During the daytime (Dawn, Morning, Mid-day", "During the daytime (Dawn, Morning, Midday"),

# Get rid of duplicate word
("but we we are not broken yet", "but we are not broken yet"),
),

"1.8-announcement" : (
# conversion added shortly before 1.8.0, might be relevant for the 1.10.0 announcement
("WML events an AI components", "WML events and AI components"),
("1.7.3", "1.7.13"),
("/tags/1.8/", "/tags/1.8.0/"),
),

"1.10-announcement" : (
("roleplaying", "role-playing"),
),

# Bulk apostrophe and dash conversion, pass 1, all campaigns EXCEPT
# LOW (already sent as another patch)
# HTTT and DW (being done by Espreon and ancestral)
# AOI, DID, L (already fully converted)
# Simons Mith, 23/08/2010, based on build 45968
# Also about twelve new pairs of <i> tags, marked with comments
# From here downwards to next comment block...

"wesnoth-tutorial" : (
("<big>You have selected Li'sar.</big>", "<big>You have selected Li’sar.</big>"),
("a ranged attack (3-3; or 3 damage,", "a ranged attack (3–3; or 3 damage,"),
("are in its 'Zone of Control'", "are in its <i>Zone of Control</i>"), # new <i> tags
("attack! I'd better grab more", "attack! I’d better grab more"),
("be? The sword (5-4) or the bow (3-3)? I suppose you'll", "be? The sword (5–4) or the bow (3–3)? I suppose you’ll"),
("but I think you're", "but I think you’re"),
("can make sure you've used all your", "can make sure you’ve used all your"),
("captured our village! You'd better get him", "captured our village! You’d better get him"),
("dark blue water? It's too deep for", "dark blue water? It’s too deep for"),
("designed as a beginner's campaign.", "designed as a beginner’s campaign."),
("Don't forget about your", "Don’t forget about your"),
("dummy's attacks are magical,", "dummy’s attacks are magical,"),
("enough experience and you'll become more powerful.", "enough experience and you’ll become more powerful."),
("every time, you'll drop from $student_hp", "every time, you’ll drop from $student_hp"),
("experience so it's more efficient to", "experience so it’s more efficient to"),
("female^Unfortunately, you've used up your", "female^Unfortunately, you’ve used up your"),
("female^You're about to be", "female^You’re about to be"),
("from the 'Actions' menu.", "from the ‘Actions’ menu."),
("gold per turn. You're only losing 1", "gold per turn. You’re only losing 1"),
("healed 2 hitpoints! I'd better attack it", "healed 2 hitpoints! I’d better attack it"),
("However, both units don't have much", "However, both units don’t have much"),
("If a unit doesn't do anything for", "If a unit doesn’t do anything for"),
("if your leader (Li'sar) is killed.", "if your leader (Li’sar) is killed."),
("into that village! He's not as stupid", "into that village! He’s not as stupid"),
("into the forest they'll", "into the forest they’ll"),
("It's very dangerous to", "It’s very dangerous to"),
("its melee attack (3-5).", "its melee attack (3–5)."),
("Konrad, Li'sar and Delfador are", "Konrad, Li’sar and Delfador are"),
("Li'sar will attack.", "Li’sar will attack."),
("miss $deadguy.name|, but it's better than one", "miss $deadguy.name|, but it’s better than one"),
("Now, Li'sar, I will leave", "Now, Li’sar, I will leave"),
("once with the 'Show Enemy Moves' command", "once with the <b>Show Enemy Moves</b> command"), # new <b> tags
("out of the enemy's reach!", "out of the enemy’s reach!"),
("right-clicking and selecting 'Recruit'. This time you", "right-clicking and selecting <b>Recruit</b>. This time you"), # new <b> tags
("select a unit, you'll see percentages for", "select a unit, you’ll see percentages for"),
("select the attacker (Li'sar) then the target", "select the attacker (Li’sar) then the target"),
("troops before clicking 'End Scenario' to continue to", "troops before clicking <b>End Scenario</b> to continue to"), # new <b> tags
("turn by pressing 'n' to", "turn by pressing <b>n</b> to"), # new <b> tags
("Unfortunately, you've used up your", "Unfortunately, you’ve used up your"),
("used a sword (5-4; or 5 damage,", "used a sword (5–4; or 5 damage,"),
("water, you'll have to kill", "water, you’ll have to kill"),
("When 'n' no longer selects", "When <b>n</b> no longer selects"), # new <b> tags
("Whenever you're on a keep,", "Whenever you’re on a keep,"),
("Yes. It's a magical quintain.", "Yes. It’s a magical quintain."),
("you are playing Li'sar.", "you are playing Li’sar."),
("You can press 'u' to undo most", "You can press <b>u</b> to undo most"), # new <b> tags
("you can press 'u' to undo,", "you can press <b>u</b> to undo,"), # new <b> tags
("you recruited first; they'll be a lot", "you recruited first; they’ll be a lot"),
("You wouldn't do anything stupid", "You wouldn’t do anything stupid"),
("You're about to be", "You’re about to be"),
("you're going to need", "you’re going to need"),
("You've captured all the", "You’ve captured all the"),
("your keep for Li'sar to capture, as", "your keep for Li’sar to capture, as"),
),

# Pass 2: Bulk apostrophe conversion by Simons Mith
# ~ 600 apostrophes changed to sexed single and double quotes
# No other changes made.

"unspecified" : (
# Single-word changes; currently commented out as they give duplicate msgids
# Although they're only names, so they should be safe to apply. Except
# possibly Li'sar

("\"Al'Brock\"", "\"Al’Brock\""),
("\"Al'Mar\"", "\"Al’Mar\""),
("\"Al'Tar\"", "\"Al’Tar\""),
("\"Ar'Muff\"", "\"Ar’Muff\""),
("\"Bak'man\"", "\"Bak’man\""),
("\"Flar'Tar\"", "\"Flar’Tar\""),
("\"Ha'Tang\"", "\"Ha’Tang\""),
("\"Ha'Tuil\"", "\"Ha’Tuil\""),
("\"Hida'tul\"", "\"Hida’tul\""),
("\"J'rem\"", "\"J’rem\""),
("\"Kapou'e\"", "\"Kapou’e\""),
("\"Lanbec'h\"", "\"Lanbec’h\""),
("\"Ro'Arthian\"", "\"Ro’Arthian\""),
("\"Ro'Sothian\"", "\"Ro’Sothian\""),
("\"Shek'kahan\"", "\"Shek’kahan\""),
("\"T'bhan\"", "\"T’bhan\""),
("\"Thu'lok\"", "\"Thu’lok\""),
("\"Ug'lok\"", "\"Ug’lok\""),
("\"Ut'Tan-Grilg\"", "\"Ut’Tan-Grilg\""),
("\"Ut'Tan-Grorag\"", "\"Ut’Tan-Grorag\""),
("\"Ut'Tan-Vrork\"", "\"Ut’Tan-Vrork\""),

# Generates long po line: 86 bytes
# ("kind that won't leave <i>us</i>", "kind that won’t leave <i>us</i>"),

# May generate long po line: 85 bytes
("It's the Prince", "It’s the Prince"),

# The rest all appear safe to apply

("'Lord'? Oh no!", "‘Lord’? Oh no!"),
("'Prince Haldric the Dragonbane' sounds rather", "‘Prince Haldric the Dragonbane’ sounds rather"),
("'The Great Chamber'? Hmmm, wonder", "‘The Great Chamber’? Hmmm, wonder"),
("'The master’s study' eh? I bet we'll find Malifor", "‘The master’s study’ eh? I bet we’ll find Malifor"),
("(<i>Grunt... strain...</i>) It's not moving.", "(<i>Grunt... strain...</i>) It’s not moving."),
("(<i>Reads</i>) <i>Kannin a'kana du'masi a'forigln de'amp.</i>", "(<i>Reads</i>) <i>Kannin a’kana du’masi a’forigln de’amp.</i>"),
("(<i>Shrugs</i>) I don't think we", "(<i>Shrugs</i>) I don’t think we"),
("... Can't... The... heat...", "... Can’t... The... heat..."),
("... Don't tax yourself...", "... Don’t tax yourself..."),
("800 gold! We're rich!!", "800 gold! We’re rich!!"),
("<i>I</i> want? I'll tell you", "<i>I</i> want? I’ll tell you"),
("<i>Swamp</i>?! I'm under five", "<i>Swamp</i>?! I’m under five"),
("a hundred. Don't lose it.", "a hundred. Don’t lose it."),
("a minute, you're elves?!", "a minute, you’re elves?!"),
("a sec, Ro'Sothian, you don't want to", "a sec, Ro’Sothian, you don’t want to"),
("a sword. I'm fairly sure you'll win.", "a sword. I’m fairly sure you’ll win."),
("a true 'Kingdom of Wesnoth'. I... I", "a true ‘Kingdom of Wesnoth’. I... I"),
("aan ogras, us'll elp yoo!", "aan ogras, us’ll elp yoo!"),
("about trolls? I'm not sure I'd want to", "about trolls? I’m not sure I’d want to"),
("Actually... we won't be coming.", "Actually... we won’t be coming."),
("Afraid you'll find out you're not of", "Afraid you’ll find out you’re not of"),
("Ah, they don't scare me!", "Ah, they don’t scare me!"),
("Ahh yes, that's better.", "Ahh yes, that’s better."),
("Ahhh, I'm so tired.", "Ahhh, I’m so tired."),
("All I'm saying is", "All I’m saying is"),
("All right! Blast'n time!", "All right! Blast’n time!"),
("All right, let's move out", "All right, let’s move out"),
("All right, let's move out,", "All right, let’s move out,"),
("all... but I'm sure none", "all... but I’m sure none"),
("alliance, Earl Lanbec'h.", "alliance, Earl Lanbec’h."),
("Amazing, I haven't seen a", "Amazing, I haven’t seen a"),
("and arrows won't work.", "and arrows won’t work."),
("and Earl Lanbec'h", "and Earl Lanbec’h"),
("and fight... What's this?", "and fight... What’s this?"),
("and hammers o' every dwarf", "and hammers o’ every dwarf"),
("and I don't think I", "and I don’t think I"),
("and I— I'm stuck!", "and I— I’m stuck!"),
("And look, he's gone. Now", "And look, he’s gone. Now"),
("And now we'll never know", "And now we’ll never know"),
("and undisturbed, Ro'Arthian.", "and undisturbed, Ro’Arthian."),
("And what's the bad", "And what’s the bad"),
("any better, we'll probably die", "any better, we’ll probably die"),
("Anything you can't handle?", "Anything you can’t handle?"),
("are arriving. They've surrounded us! We're doomed.", "are arriving. They’ve surrounded us! We’re doomed."),
("are but children's tales... Still,", "are but children’s tales... Still,"),
("Argh! I'll just come", "Argh! I’ll just come"),
("Argh! I'm dead! Well,", "Argh! I’m dead! Well,"),
("Argh! I'm done. My", "Argh! I’m done. My"),
("at least he's finally at", "at least he’s finally at"),
("at least you're improving.", "at least you’re improving."),
("attacks more. You're becoming too", "attacks more. You’re becoming too"),
("Awesome, let's go!", "Awesome, let’s go!"),
("Aye! Down wi' the orcs!", "Aye! Down wi’ the orcs!"),
("Aye, it's still bandit", "Aye, it’s still bandit"),
("A’right then, we'll wait fa' later, eh?", "A’right then, we’ll wait fa’ later, eh?"),
("be called 'Dragonbane'.", "be called ‘Dragonbane’."),
("be mad! I'll offer one", "be mad! I’ll offer one"),
("be quiet, I'll explain it", "be quiet, I’ll explain it"),
("be useful, I'll take it.", "be useful, I’ll take it."),
("be... highly irregular.' Bah! I'll show you", "be... highly irregular.” Bah! I’ll show you"),
("bet this wasn't the welcome", "bet this wasn’t the welcome"),
("Black-Eye! We won't submit to", "Black-Eye! We won’t submit to"),
("break free! Don't let that", "break free! Don’t let that"),
("bridge yet, Cap'n?", "bridge yet, Cap’n?"),
("But don't you need", "But don’t you need"),
("But hasn't Eloh told", "But hasn’t Eloh told"),
("but it doesn't sound friendly.", "but it doesn’t sound friendly."),
("but we dinna' ha' much of", "but we dinna’ ha’ much of"),
("But we don't want to", "But we don’t want to"),
("but you haven't beat us", "but you haven’t beat us"),
("But... Asheviere, Garard's queen and Eldred's mother, seemed", "But... Asheviere, Garard’s queen and Eldred’s mother, seemed"),
("calling you 'the Dragonbane'.", "calling you ‘the Dragonbane’."),
("can just 'pass through' a Drake", "can just ‘pass through’ a Drake"),
("Can't our units", "Can’t our units"),
("Capture Ro'Arthian.", "Capture Ro’Arthian."),
("Careful, you don't know what's lurking in", "Careful, you don’t know what’s lurking in"),
("Chief! It's true, the", "Chief! It’s true, the"),
("Chief, we can't be blowing", "Chief, we can’t be blowing"),
("Chop 'em down. It's cool in", "Chop ’em down. It’s cool in"),
("click on Li'sar", "click on Li’sar"),
("coal that wi' melt this", "coal that wi’ melt this"),
("Come mortal, let's cross our", "Come mortal, let’s cross our"),
("coming! Smash 'em good!", "coming! Smash ’em good!"),
("course you didn't. That’s why it's called sneaking.", "course you didn’t. That’s why it’s called sneaking."),
("dares disturb Shek'kahan the Terrible?", "dares disturb Shek’kahan the Terrible?"),
("Dead-Man's Ford", "Dead-Man’s Ford"),
("Death of Al'Brock", "Death of Al’Brock"),
("Death of Flar'Tar", "Death of Flar’Tar"),
("Death of Kapou'e", "Death of Kapou’e"),
("Death of Li'sar", "Death of Li’sar"),
("Death of Ro'Arthian", "Death of Ro’Arthian"),
("Death of Ro'Sothian", "Death of Ro’Sothian"),
("Defeat Lanbec'h", "Defeat Lanbec’h"),
("Delfador to Ur-Thorodor's castle", "Delfador to Ur-Thorodor’s castle"),
("died a hero's death, one", "died a hero’s death, one"),
("died... a warrior's death... You", "died... a warrior’s death... You"),
("do. I don't know why,", "do. I don’t know why,"),
("do? We can't go much", "do? We can’t go much"),
("doing? I haven't let you", "doing? I haven’t let you"),
("Don't kill me,", "Don’t kill me,"),
("Don't make me", "Don’t make me"),
("Don't try to", "Don’t try to"),
("Don't we have", "Don’t we have"),
("Don't worry about", "Don’t worry about"),
("Don't worry Kaleh,", "Don’t worry Kaleh,"),
("Don't you intend", "Don’t you intend"),
("drama, but that's no excuse", "drama, but that’s no excuse"),
("dwarf. But he's been beaten", "dwarf. But he’s been beaten"),
("dwarves and stinkin' elves, we", "dwarves and stinkin’ elves, we"),
("dwarves, it don't matter!", "dwarves, it don’t matter!"),
("eh? You won't find many", "eh? You won’t find many"),
("elves, we can't trust them!", "elves, we can’t trust them!"),
("enough. Well, let's get them!", "enough. Well, let’s get them!"),
("envy your kind's prowess when", "envy your kind’s prowess when"),
("especially where you'll be going,", "especially where you’ll be going,"),
("expect, Chief? They're elves after", "expect, Chief? They’re elves after"),
("explain, Esanoo. We'll have to", "explain, Esanoo. We’ll have to"),
("Father, I'd like to", "Father, I’d like to"),
("feel like I'm forgetting something.", "feel like I’m forgetting something."),
("few here don't stand a", "few here don’t stand a"),
("filthy elves! I'll...", "filthy elves! I’ll..."),
("Finally! We'll fight at", "Finally! We’ll fight at"),
("Finally! You don't know how", "Finally! You don’t know how"),
("First, why don't you tell", "First, why don’t you tell"),
("for my family's memory, you", "for my family’s memory, you"),
("for now we've got to", "for now we’ve got to"),
("forbidden lore you've been taught.", "forbidden lore you’ve been taught."),
("Free, I'm free, and", "Free, I’m free, and"),
("Freedom! Don't worry about", "Freedom! Don’t worry about"),
("from? I can't see a", "from? I can’t see a"),
("frontier or it's all over!", "frontier or it’s all over!"),
("gave us, they're glowing!", "gave us, they’re glowing!"),
("GO HOME! AIN'T NUTTIN TO", "GO HOME! AIN’T NUTTIN TO"),
("go. Now let's see where", "go. Now let’s see where"),
("gold! We don't need more!", "gold! We don’t need more!"),
("grab him! Don't let him", "grab him! Don’t let him"),
("Greetings from Kapou'e, Son of", "Greetings from Kapou’e, Son of"),
("Grunt, and I'm done for!", "Grunt, and I’m done for!"),
("guy. We aren't gonna hurt", "guy. We aren’t gonna hurt"),
("Ha! <i>You</i> don't trust <i>me</i>?", "Ha! <i>You</i> don’t trust <i>me</i>?"),
("Haldric enters Southbay's sewer", "Haldric enters Southbay’s sewer"),
("Haldric, what's going on? What's the plan?", "Haldric, what’s going on? What’s the plan?"),
("Haldric, you're too paranoid.", "Haldric, you’re too paranoid."),
("hands of Kalenz's elves!", "hands of Kalenz’s elves!"),
("have allied wi' the humans...", "have allied wi’ the humans..."),
("he disappeared. Let's find him", "he disappeared. Let’s find him"),
("he go? Let's find him", "he go? Let’s find him"),
("He's in his", "He’s in his"),
("He's raising our", "He’s raising our"),
("He's still breathing.", "He’s still breathing."),
("He's stirring.", "He’s stirring."),
("He's... he's still breathing!", "He’s... he’s still breathing!"),
("heed the King's voice. My", "heed the King’s voice. My"),
("help you? You're not a", "help you? You’re not a"),
("Help! They're everywhere!", "Help! They’re everywhere!"),
("Help, I'm drowning!", "Help, I’m drowning!"),
("Hey look! It's a dwarf!", "Hey look! It’s a dwarf!"),
("Hey look! It's a troll!", "Hey look! It’s a troll!"),
("Hey look, it's a troll!", "Hey look, it’s a troll!"),
("Hey! Look who's back!", "Hey! Look who’s back!"),
("Hey, don't light it up, I'm not safe!", "Hey, don’t light it up, I’m not safe!"),
("Hey, don't you always", "Hey, don’t you always"),
("Hey, there's somebody hidden", "Hey, there’s somebody hidden"),
("Hey, what's going on", "Hey, what’s going on"),
("him — he's big, green", "him — he’s big, green"),
("him. We don't want him", "him. We don’t want him"),
("His brother Ro'Arthian is the", "His brother Ro’Arthian is the"),
("Hm... Me le' no'w!", "Hm... Me le’ no’w!"),
("Hmph! You're just happy", "Hmph! You’re just happy"),
("Hold it! What's going on", "Hold it! What’s going on"),
("hold Southbay. We're done for!", "hold Southbay. We’re done for!"),
("home, now we'll take yer", "home, now we’ll take yer"),
("How'd they get", "How’d they get"),
("Huh? Didn't a messenger", "Huh? Didn’t a messenger"),
("Huh? Who's there, who", "Huh? Who’s there, who"),
("Huh?! Can't be, the", "Huh?! Can’t be, the"),
("hundred, but don't go up,", "hundred, but don’t go up,"),
("Hurry, friends, let's set up", "Hurry, friends, let’s set up"),
("I almost can't believe it.", "I almost can’t believe it."),
("I can't be finished", "I can’t be finished"),
("I don't care, get", "I don’t care, get"),
("I don't find that", "I don’t find that"),
("I don't know, but", "I don’t know, but"),
("I don't know, see", "I don’t know, see"),
("I don't see how", "I don’t see how"),
("I don't think diplomacy", "I don’t think diplomacy"),
("I don't think I", "I don’t think I"),
("I expect we'll be facing", "I expect we’ll be facing"),
("I guess we'll just have", "I guess we’ll just have"),
("I just don't know who", "I just don’t know who"),
("I s'ppose we'll hav' to wade", "I s’ppose we’ll hav’ to wade"),
("I think I'll fight the", "I think I’ll fight the"),
("I think I'll say that", "I think I’ll say that"),
("I think I'll take the", "I think I’ll take the"),
("I think I'll wait a", "I think I’ll wait a"),
("I think I've figured something", "I think I’ve figured something"),
("I think I've found it.", "I think I’ve found it."),
("I think I've got it.", "I think I’ve got it."),
("I think you've the right", "I think you’ve the right"),
("I wasn't talking to", "I wasn’t talking to"),
("I'd like to", "I’d like to"),
("I'll do it, sir. I'll sacrifice myself", "I’ll do it, sir. I’ll sacrifice myself"),
("I'll not go", "I’ll not go"),
("I'll recruit some", "I’ll recruit some"),
("I'll take twenty-five", "I’ll take twenty-five"),
("I'll tell them", "I’ll tell them"),
("I'll try, sir.", "I’ll try, sir."),
("I'm afraid, Kaleh,", "I’m afraid, Kaleh,"),
("I'm always careful. I'll be back", "I’m always careful. I’ll be back"),
("I'm back, Kaleh.", "I’m back, Kaleh."),
("I'm coming... coming...", "I’m coming... coming..."),
("I'm fine. But", "I’m fine. But"),
("I'm gonna make", "I’m gonna make"),
("I'm here, chief!", "I’m here, chief!"),
("I'm still not", "I’m still not"),
("I'm thirsty... <i>Gulp", "I’m thirsty... <i>Gulp"),
("I'm too young", "I’m too young"),
("I've asked and", "I’ve asked and"),
("I've found the", "I’ve found the"),
("I've seen some", "I’ve seen some"),
("if anything, it's our road!", "if anything, it’s our road!"),
("if you don't mind, I'll go with", "if you don’t mind, I’ll go with"),
("Iliah-Malal's body lay", "Iliah-Malal’s body lay"),
("Impossible! I canna' believe any", "Impossible! I canna’ believe any"),
("is departing. We're trapped!", "is departing. We’re trapped!"),
("is I, Kapou'e, son of", "is I, Kapou’e, son of"),
("is this 'Landar' you speak", "is this ‘Landar’ you speak"),
("is this thing's daughter?", "is this thing’s daughter?"),
("It couldna' been more", "It couldna’ been more"),
("It couldna' ha' been more", "It couldna’ ha’ been more"),
("it sure isn't gonna be", "it sure isn’t gonna be"),
("It won't be said", "It won’t be said"),
("It won't be that", "It won’t be that"),
("it you wouldn't mind if", "it you wouldn’t mind if"),
("it! We weren't fast enough.", "it! We weren’t fast enough."),
("It's a 'he', and yes, they're actually very", "It’s a ‘he’, and yes, they’re actually very"),
("It's a dust", "It’s a dust"),
("It's a heck", "It’s a heck"),
("It's been... it's been years since I've been down", "It’s been... it’s been years since I’ve been down"),
("It's better than", "It’s better than"),
("It's called 'The End', foul orc.", "It’s called ‘The End’, foul orc."),
("It's cooler here,", "It’s cooler here,"),
("It's just ‘Tallin’,", "It’s just ‘Tallin’,"),
("It's locked, but", "It’s locked, but"),
("It's no good,", "It’s no good,"),
("It's no use", "It’s no use"),
("It's not very", "It’s not very"),
("It's Rakshas!", "It’s Rakshas!"),
("It's talking to", "It’s talking to"),
("It's the only", "It’s the only"),
("It's time to", "It’s time to"),
("It's too late!", "It’s too late!"),
("It's too quiet...", "It’s too quiet..."),
("It's very easy,", "It’s very easy,"),
("It's... it's monstrous!", "It’s... it’s monstrous!"),
("its some o' them cave-dwarves.", "its some o’ them cave-dwarves."),
("Kaleh, you can't just go", "Kaleh, you can’t just go"),
("Keep of El'Ithsomir", "Keep of El’Ithsomir"),
("kill Mal-Ravanal. He's here somewhere...", "kill Mal-Ravanal. He’s here somewhere..."),
("kill me, I'm not telling", "kill me, I’m not telling"),
("kin would ha' helped a", "kin would ha’ helped a"),
("Krawg sme' o'cz in iz 'ave!", "Krawg sme’ o’cz in iz ’ave!"),
("Let's clear out", "Let’s clear out"),
("Let's finish off", "Let’s finish off"),
("Let's make them", "Let’s make them"),
("Let's put those", "Let’s put those"),
("Let's send these", "Let’s send these"),
("Living... Soon, you'll be more", "Living... Soon, you’ll be more"),
("long as they're in charge you'll have to", "long as they’re in charge you’ll have to"),
("long enough. Let's go!", "long enough. Let’s go!"),
("Look, there's a dragon", "Look, there’s a dragon"),
("looks like he's been beaten", "looks like he’s been beaten"),
("looks like he's in bad", "looks like he’s in bad"),
("Lord El'Isomithir", "Lord El’Isomithir"),
("M' 'u!", "M’ ’u!"),
("makes your Master's punishment from", "makes your Master’s punishment from"),
("Mal-Ravanal's Capital", "Mal-Ravanal’s Capital"),
("Malifor the Great's Study", "Malifor the Great’s Study"),
("Many ships. We're being invaded!", "Many ships. We’re being invaded!"),
("mask. You... you're not with", "mask. You... you’re not with"),
("Maybe they won't be hostile.", "Maybe they won’t be hostile."),
("Maybe they'll clear it", "Maybe they’ll clear it"),
("me, Kaleh, I'm dying...", "me, Kaleh, I’m dying..."),
("mentioned I can't float.", "mentioned I can’t float."),
("mine, and don't touch it!", "mine, and don’t touch it!"),
("Monsters! We're being invaded", "Monsters! We’re being invaded"),
("Move Kapou'e to the", "Move Kapou’e to the"),
("Move Li'sar next to", "Move Li’sar next to"),
("Move Li'sar to another", "Move Li’sar to another"),
("Move Li'sar to capture", "Move Li’sar to capture"),
("Move Li'sar to the", "Move Li’sar to the"),
("much as I'd like to, we don't have time", "much as I’d like to, we don’t have time"),
("Must obey... Can't resist... I...", "Must obey... Can’t resist... I..."),
("Must... Can't... Must... Help", "Must... Can’t... Must... Help"),
("My love, I'll be there", "My love, I’ll be there"),
("my my, what's in that", "my my, what’s in that"),
("never mind, I'll get you", "never mind, I’ll get you"),
("no choice... I've never tasted", "no choice... I’ve never tasted"),
("no game, Li'sar!", "no game, Li’sar!"),
("no ghosts, I'll take it.", "no ghosts, I’ll take it."),
("No you won't, you soldier", "No you won’t, you soldier"),
("No! I'll not go", "No! I’ll not go"),
("No! It's all over!", "No! It’s all over!"),
("no! Please don't die!", "no! Please don’t die!"),
("No!! They'll kill me!", "No!! They’ll kill me!"),
("No, $unit.name can't die now!", "No, $unit.name can’t die now!"),
("No, don't...", "No, don’t..."),
("No, it's an elf!", "No, it’s an elf!"),
("No, we can't. But for", "No, we can’t. But for"),
("No. I can't ask you", "No. I can’t ask you"),
("No. It can't be!", "No. It can’t be!"),
("no. You don't understand! We", "no. You don’t understand! We"),
("Noo!! I can't be promoted", "Noo!! I can’t be promoted"),
("Nooo! Don't kill me,", "Nooo! Don’t kill me,"),
("not say, 'To err is", "not say, “To err is"),
("now $intl_ally_name, I'm busy.", "now $intl_ally_name, I’m busy."),
("now between Iliah-Malal's army and", "now between Iliah-Malal’s army and"),
("Now let's get going,", "Now let’s get going,"),
("Now let's make mush", "Now let’s make mush"),
("now that you've triggered it.", "now that you’ve triggered it."),
("now, and don't look back!", "now, and don’t look back!"),
("now, Father. I'm a little", "now, Father. I’m a little"),
("Nym! No! Don't open—", "Nym! No! Don’t open—"),
("Nym. I didn't hear you", "Nym. I didn’t hear you"),
("of here. Don't make me", "of here. Don’t make me"),
("of here. We're trapped!", "of here. We’re trapped!"),
("of surrender! Don't kill me...", "of surrender! Don’t kill me..."),
("of the Ka'lian do... How", "of the Ka’lian do... How"),
("of this. I'm feeling ill!", "of this. I’m feeling ill!"),
("of this. I'm out of", "of this. I’m out of"),
("of time... We'll never beat", "of time... We’ll never beat"),
("off the 'or the orcs", "off the ‘or the orcs"),
("Oh no, I'm defeated.", "Oh no, I’m defeated."),
("on boys, let's give it", "on boys, let’s give it"),
("on brother, let's get outta", "on brother, let’s get outta"),
("one escape. Let's hope they don't all!", "one escape. Let’s hope they don’t all!"),
("One of Iliah-Malal's creatures. So", "One of Iliah-Malal’s creatures. So"),
("one of Malifor's experiments.", "one of Malifor’s experiments."),
("one of Tallin's men...", "one of Tallin’s men..."),
("Oooooh, cool, it's a drake!", "Oooooh, cool, it’s a drake!"),
("Orcs sighted! They've seized the", "Orcs sighted! They’ve seized the"),
("Orcs, kill'em all! Humans,", "Orcs, kill’em all! Humans,"),
("orcs’ chest! It's filled with", "orcs’ chest! It’s filled with"),
("our hammers can't manage.", "our hammers can’t manage."),
("Ow, I'm stuck!", "Ow, I’m stuck!"),
("Perhaps he didn't want to", "Perhaps he didn’t want to"),
("Phew, it's hot down", "Phew, it’s hot down"),
("Pillage! Let's plunder these", "Pillage! Let’s plunder these"),
("powerful, why can't you destroy", "powerful, why can’t you destroy"),
("Prestim's walls are rock-hard, they'll waste themselves", "Prestim’s walls are rock-hard, they’ll waste themselves"),
("pretty small, let's see if", "pretty small, let’s see if"),
("Price?! Don't insult us,", "Price?! Don’t insult us,"),
("protect us, they're elves!", "protect us, they’re elves!"),
("Put Ro'Sothian into that", "Put Ro’Sothian into that"),
("Quick, honey, we've got to", "Quick, honey, we’ve got to"),
("quiet Zhul, I'll explain it", "quiet Zhul, I’ll explain it"),
("ran away, didn't you?", "ran away, didn’t you?"),
("reinforcements coming! We're trapped! All", "reinforcements coming! We’re trapped! All"),
("Relax, he's a friend.", "Relax, he’s a friend."),
("remain here we'll die.", "remain here we’ll die."),
("Reserves! We can't let them", "Reserves! We can’t let them"),
("Resist until Dacyn's return in", "Resist until Dacyn’s return in"),
("Resist until Dacyn's return tomorrow", "Resist until Dacyn’s return tomorrow"),
("right boys, let's go!", "right boys, let’s go!"),
("right people, let's move out!", "right people, let’s move out!"),
("Right. We'll probably have", "Right. We’ll probably have"),
("Ro'Arthian, are you", "Ro’Arthian, are you"),
("Ro'Arthian, bah! Don't utter that", "Ro’Arthian, bah! Don’t utter that"),
("Ro'Arthian, send a", "Ro’Arthian, send a"),
("Ro'Sothian, is that", "Ro’Sothian, is that"),
("rumbling. And what's that roaring", "rumbling. And what’s that roaring"),
("running, or we'll be fish-bait", "running, or we’ll be fish-bait"),
("said that you'd been here", "said that you’d been here"),
("save us, it's... it's an elf.", "save us, it’s... it’s an elf."),
("second thought, it's better to", "second thought, it’s better to"),
("secret door. Let's see where", "secret door. Let’s see where"),
("secrets of Crelanu's book...", "secrets of Crelanu’s book..."),
("shall spoil, we'll starve!", "shall spoil, we’ll starve!"),
("She's just full", "She’s just full"),
("She's... She's beautiful.", "She’s... She’s beautiful."),
("shut up. Let's go kill", "shut up. Let’s go kill"),
("Sigh, we'll talk later.", "Sigh, we’ll talk later."),
("Sister! Don't you recognize", "Sister! Don’t you recognize"),
("skeletons! Where?! Let's go burn 'em all!", "skeletons! Where?! Let’s go burn ’em all!"),
("So far it's been danger", "So far it’s been danger"),
("so sure it's that simple,", "so sure it’s that simple,"),
("so tough don't you? Well", "so tough don’t you? Well"),
("So you don't have the", "So you don’t have the"),
("So, d'ya want me", "So, d’ya want me"),
("so. I can't see anything.", "so. I can’t see anything."),
("someday maybe we'll meet again.", "someday maybe we’ll meet again."),
("Someone's coming. Quick,", "Someone’s coming. Quick,"),
("somewhere. We can't be sure.", "somewhere. We can’t be sure."),
("Sorry, won't do it", "Sorry, won’t do it"),
("sounds good. I'll stay here", "sounds good. I’ll stay here"),
("Spiders aren't insects.", "Spiders aren’t insects."),
("Stalrag from Ro'Sothian.", "Stalrag from Ro’Sothian."),
("stand still you're a dead", "stand still you’re a dead"),
("stay and 're-educate' these un-orcs.", "stay and ‘re-educate’ these un-orcs."),
("stinking trees, we'll show you!", "stinking trees, we’ll show you!"),
("stop him. Let's go!", "stop him. Let’s go!"),
("sure there aren't any spiders?", "sure there aren’t any spiders?"),
("Sure, we'll help you", "Sure, we’ll help you"),
("tale, for it's worth it.", "tale, for it’s worth it."),
("Tallin, we can't do it", "Tallin, we can’t do it"),
("Tallin, we ha' failed to", "Tallin, we ha’ failed to"),
("Tell him it's time.", "Tell him it’s time."),
("tell me what's going on", "tell me what’s going on"),
("tell you, Kapou'e?", "tell you, Kapou’e?"),
("than goblins. We'll break them!", "than goblins. We’ll break them!"),
("Thank you. I'm sure the", "Thank you. I’m sure the"),
("Thanks, father. Don't worry, I'll take care", "Thanks, father. Don’t worry, I’ll take care"),
("That hurts. Let's try not", "That hurts. Let’s try not"),
("That isn't him...", "That isn’t him..."),
("that snow...? We're doomed!", "that snow...? We’re doomed!"),
("that this isn't going as", "that this isn’t going as"),
("Thats right. I'll be coming", "That’s right. I’ll be coming"),
("That’s... that's the Rod", "That’s... that’s the Rod"),
("The 'true people' speak through", "The ‘true people’ speak through"),
("The dragon's cave has", "The dragon’s cave has"),
("the happier I'll be.", "the happier I’ll be."),
("the horizon. Jevyan's fleet is here! It's all over.", "the horizon. Jevyan’s fleet is here! It’s all over."),
("The Ka'lian", "The Ka’lian"),
("The maker's mark is", "The maker’s mark is"),
("the sewer. We're doomed!", "the sewer. We’re doomed!"),
("the ship isn't here yet.", "the ship isn’t here yet."),
("the swamp. I'm under five", "the swamp. I’m under five"),
("The world won't miss him", "The world won’t miss him"),
("them sacks o' bones.", "them sacks o’ bones."),
("Then it's time to", "Then it’s time to"),
("Then let's kill them!", "Then let’s kill them!"),
("then return Li'sar to the", "then return Li’sar to the"),
("Then, into Southbay's sewer.", "Then, into Southbay’s sewer."),
("there alone. She'll kill you!", "there alone. She’ll kill you!"),
("these tunnels aren't as bad", "these tunnels aren’t as bad"),
("They're coming this", "They’re coming this"),
("They're definitely of the 'attack first, ask questions later' variety.", "They’re definitely of the ‘attack first, ask questions later’ variety."),
("They're here!", "They’re here!"),
("They're raising the", "They’re raising the"),
("They're... the humans", "They’re... the humans"),
("thing just won't stay dead!", "thing just won’t stay dead!"),
("This can't be the", "This can’t be the"),
("This rabble won't be a", "This rabble won’t be a"),
("this real? I'm coming, I'm coming.", "this real? I’m coming, I’m coming."),
("this thing? It's huge!", "this thing? It’s huge!"),
("This trash's gone.", "This trash’s gone."),
("This'll splinter your", "This’ll splinter your"),
("those orcs can't get to", "those orcs can’t get to"),
("thousand, and I'll go no", "thousand, and I’ll go no"),
("Tirigaz. But what's going on?", "Tirigaz. But what’s going on?"),
("to back! Don't let them", "to back! Don’t let them"),
("to his master's aid. When", "to his master’s aid. When"),
("to move Li'sar", "to move Li’sar"),
("to say, 'I told you so'?", "to say, “I told you so”?"),
("to think. It's all so", "to think. It’s all so"),
("to you, Kapou'e, our imbecile", "to you, Kapou’e, our imbecile"),
("too late! They're burning the", "too late! They’re burning the"),
("too late. We've taken what", "too late. We’ve taken what"),
("too long! We'll never be", "too long! We’ll never be"),
("too! Chief doesn't let me", "too! Chief doesn’t let me"),
("Treasury! Cool, let's go loot", "Treasury! Cool, let’s go loot"),
("tribe anymore, Kapou'e.", "tribe anymore, Kapou’e."),
("troll hole? Didn't you?", "troll hole? Didn’t you?"),
("trolls. But you'll see, Griknagh", "trolls. But you’ll see, Griknagh"),
("Uh... I don't think so.", "Uh... I don’t think so."),
("Undead. Bah! We're pros at", "Undead. Bah! We’re pros at"),
("Ungrateful minx! I'm sorely tempted", "Ungrateful minx! I’m sorely tempted"),
("us <i>you</i> won't come back.", "us <i>you</i> won’t come back."),
("us but I'm afraid we can't help you.", "us but I’m afraid we can’t help you."),
("Very well. I'll do my", "Very well. I’ll do my"),
("Very well. We'll go south", "Very well. We’ll go south"),
("victory against Al'Tar.", "victory against Al’Tar."),
("was fun, wasn't it, Grüü?", "was fun, wasn’t it, Grüü?"),
("way already! We're running out", "way already! We’re running out"),
("we die, it's our only", "we die, it’s our only"),
("We don't have any", "We don’t have any"),
("We don't. Not everything", "We don’t. Not everything"),
("we go, let's give this", "we go, let’s give this"),
("We haven't a moment", "We haven’t a moment"),
("we should ha' allied wi' the humans...", "we should ha’ allied wi’ the humans..."),
("We'll ask later,", "We’ll ask later,"),
("We'll be back,", "We’ll be back,"),
("We'll go through", "We’ll go through"),
("We'll make sure.", "We’ll make sure."),
("We're in. Now:", "We’re in. Now:"),
("We're surrounded! I", "We’re surrounded! I"),
("We're surrounded! The", "We’re surrounded! The"),
("We're through the", "We’re through the"),
("We've captured all", "We’ve captured all"),
("We've cleared the beach. Let's return to", "We’ve cleared the beach. Let’s return to"),
("We've defeated the orcs. Let's rig the", "We’ve defeated the orcs. Let’s rig the"),
("We've slain the", "We’ve slain the"),
("we? I can't see where", "we? I can’t see where"),
("Well, fine, I'll hear your", "Well, fine, I’ll hear your"),
("Well, it's pointless to", "Well, it’s pointless to"),
("Well, let's see what's behind it.", "Well, let’s see what’s behind it."),
("Well, there's only one", "Well, there’s only one"),
("Well, what's behind the", "Well, what’s behind the"),
("Well, where's their leader? I don't see him.", "Well, where’s their leader? I don’t see him."),
("well. It isn't as if we weren't expecting this.", "well. It isn’t as if we weren’t expecting this."),
("Wha'? Yah fail'd to give", "Wha’? Yah fail’d to give"),
("what cause ha' I heard naught o' this?", "what cause ha’ I heard naught o’ this?"),
("What in Moradin's name are", "What in Moradin’s name are"),
("What'd he want? And why'd you have", "What’d he want? And why’d you have"),
("What's going on", "What’s going on"),
("What's that strange", "What’s that strange"),
("What's this? A", "What’s this? A"),
("What's this? You", "What’s this? You"),
("What's up big", "What’s up big"),
("What's with this 'Master' business? It's starting to", "What’s with this ‘Master’ business? It’s starting to"),
("What's wrong, Kaleh? Don't you trust", "What’s wrong, Kaleh? Don’t you trust"),
("What... Kapou'e! What in", "What... Kapou’e! What in"),
("What? What's going on", "What? What’s going on"),
("Whenever you're on a", "Whenever you’re on a"),
("Who is 'they'?", "Who is ‘they’?"),
("who murdered El'Isomithir!", "who murdered El’Isomithir!"),
("Why can't you wield", "Why can’t you wield"),
("Why d'ye say that?", "Why d’ye say that?"),
("Why don't you try 'knocking'?", "Why don’t you try ‘knocking’?"),
("Why? What's wrong?", "Why? What’s wrong?"),
("will eat you' part.", "will eat you’ part."),
("will surely appear.' It looks", "will surely appear.” It looks"),
("with them, Kapou'e, they aren't going to", "with them, Kapou’e, they aren’t going to"),
("Yeah, I can't wait to", "Yeah, I can’t wait to"),
("Yeah, we'll be heroes!", "Yeah, we’ll be heroes!"),
("yer mouth! Let's just get 'em.", "yer mouth! Let’s just get ’em."),
("Yes master, I'll make him", "Yes master, I’ll make him"),
("Yes, I'll take it.", "Yes, I’ll take it."),
("Yes, I'm still figuring", "Yes, I’m still figuring"),
("Yess! It's the elves", "Yess! It’s the elves"),
("you are, I'll come up", "you are, I’ll come up"),
("You aren't actually going", "You aren’t actually going"),
("You don't know where", "You don’t know where"),
("You don't turn your", "You don’t turn your"),
("you idiot, I'm still over", "you idiot, I’m still over"),
("you if you'd 'remove' just our", "you if you’d ‘remove’ just our"),
("you sure that's wise?", "you sure that’s wise?"),
("you sure you're okay?", "you sure you’re okay?"),
("you the undead's prisoners?", "you the undead’s prisoners?"),
("You won't get much", "You won’t get much"),
("You're going to", "You’re going to"),
("You're in no", "You’re in no"),
("You're making a", "You’re making a"),
("You're really beat", "You’re really beat"),
("You're right, Elenia,", "You’re right, Elenia,"),
("You're right, Hamel,", "You’re right, Hamel,"),
("You're telling me.", "You’re telling me."),
("You've been this", "You’ve been this"),
("You've fought orcs", "You’ve fought orcs"),
("You've learned well, Li'sar!", "You’ve learned well, Li’sar!"),
("You've learned well,", "You’ve learned well,"),
("You've put my", "You’ve put my"),
("you. Now let's get going,", "you. Now let’s get going,"),
("young prince, you're not as", "young prince, you’re not as"),
("your enemies fall.' Grave robbing", "your enemies fall.’ Grave robbing"),
("your face? We've never seen", "your face? We’ve never seen"),
("Your word can't be trusted.", "Your word can’t be trusted."),

# Added missing apostrophe
("Thats right", "That‘s right"),
),
# End of pass 2, bulk apostrophe conversion

# Apostrophe conversion, core data
# by Simons Mith, based on build 46240 or so (2010-09-03)

"wesnoth-editor" : (
("Can't Undo", "Can’t Undo"),
("Can't Redo", "Can’t Redo"),

# Fix capitalization
("Create new map", "Create New Map"),
("Choose a mask to apply", "Choose a Mask to Apply"),
("Choose target map", "Choose Target Map"),
),

"wesnoth-lib" : (
# Convert makeshift dashes
# conversion added in 1.9.0-svn
("Player Info - ", "Player Info — "),

("Don't ask me again!", "Don’t ask me again!"),
("dedicated server 'wesnothd'", "dedicated server ‘wesnothd’"),
("the game's settings", "the game’s settings"),

# Fix capitalization
("Lava chasm", "Lava Chasm"),

#this rules seems to be f***ed up and not in sync with src/addon/manager.cpp:999
#uncomment it for the moment, please only reactivate after capitalization of
#'add-ons' is/was fixed everywhere!
#("Update add-ons", "Update Add-ons"),

# Use "Wooden"
("Wood Bridge", "Wooden Bridge"),

# Use "Gray"
("Grey Deep Water", "Gray Deep Water"),
("Clean Grey Cobbles", "Clean Gray Cobbles"),

# Fix apostrophes
("Hides allies'", "Hides allies’"),

# Fix apostrophes and capitalization
("Hide Allies' Plans by Default", "Hide allies’ plans by default"),

# Fix capitalization
("Maximum Auto-Saves", "Maximum auto-saves"),
),

"wesnoth-manual" : (
("Mage Delfador - pay attention", "Mage Delfador — pay attention"),
("about 10-20 scenarios.", "about 10–20 scenarios."),
("Life and Death - Experience", "Life and Death — Experience"),
("- i.e. the leader", "— i.e. the leader"),
("voluntarily - usually", "voluntarily — usually"),
("2 + villages - maximum(0, upkeep - villages)", "2 + villages − maximum(0, upkeep − villages)"),
("-25%", "−25%"),
("number keys 2-7 will identify", "number keys 2–7 will identify"),
("want to attack -", "want to attack —"),
("written as 5-4,", "written as 5–4,"),
("level units - to give", "level units — to give"),
("Plan ahead - think about", "Plan ahead — think about"),
),

"wesnoth-help" : (
("Immune to drain, poison and plague", "Immune to drain, poison, and plague"),

("Receive only 50% defense in land-based", "Receives only 50% defense in land-based"),

# Kill useless initial single quote
("'The drakish tradition", "The drakish tradition"),

# 'whose' -> 'who'
("Drakish scrollkeeper whose has", "Drakish scrollkeeper who has"),
),

"wesnoth-manpages" : (
("of on-line players", "of online players"),
),

# conversion added in 1.10.0+svn
"wesnoth-tutorial" : (
("$unit.type", "$unit.language_name"),
),

}

# Speak, if all argument files are newer than this timestamp
# Try to use UTC here
# date --utc "+%s  # %c"
timecheck = 1283156523  # Mo 30 Aug 2010 08:22:03 UTC

import os, sys, time, stat, re
try:
    from multiprocessing import Pool, cpu_count
    def parallel_map(*args, **kw):
        pool = Pool(cpu_count())
        return pool.map(*args, **kw)
except ImportError:
    print "Failed to import 'multiprocessing' module. Multiple cpu cores won't be utilized"
    parallel_map = map

def process_file(path):
    before = open(path, "r").read()
    decommented = re.sub("#.*", "", before)
    lines = before.split('\n')
    for (domain, fixes) in stringfixes.items():
        # In case of screwed-up pairs that are hard to find, uncomment the following:
        #for fix in fixes:
        #    if len(fix) != 2:
        #        print fix
        for (old, new) in fixes:
            if old is new:
                #complain loudly
                print "pofix: old string\n\t\"%s\"\n equals new string\n\t\"%s\"\nexiting." % (old, new)
                sys.exit(1)
            #this check is problematic and the last clause is added to prevent false
            #positives in case that new is a substring of old, though this can also
            #lead to "real" probs not found, the real check would be "does replacing
            #old with new lead to duplicate msgids? (including old ones marked with #~)"
            #which is not easily done in the current design...
            elif new in decommented and old in decommented and not new in old:
                print "pofix: %s already includes the new string\n\t\"%s\"\nbut also the old\n\t\"%s\"\nthis needs handfixing for now since it likely creates duplicate msgids." % (path, new, old)
            else:
                for (i, line) in enumerate(lines):
                    if line and line[0] != '#':
                        lines[i] = lines[i].replace(old, new)
    after = '\n'.join(lines)
    if after != before:
        print "pofix: %s modified" % path
        # Save a backup
        os.rename(path, path + "-bak")
        # Write out transformed version
        ofp = open(path, "w")
        ofp.write(after)
        ofp.close()
        return 1
    else:
        return 0

if __name__ == '__main__':
    newer = 0
    modified = 0
    pocount = 0
    files = []
    for path in sys.argv[1:]:
        if not path.endswith(".po") and not path.endswith(".pot") and not path.endswith(".cfg"):
            continue
        pocount += 1
        # Notice how many files are newer than the time check
        statinfo = os.stat(path)
        if statinfo.st_mtime > timecheck:
            newer += 1
	files.append(path)
    modified = sum(parallel_map(process_file, files))
    print "pofix: %d files processed, %d files modified, %d files newer" \
          % (pocount, modified, newer)
    if pocount > 1 and newer == pocount:
        print "pofix: script may be obsolete"
