#include "libMems/IntervalList.h"
#include "libMems/Aligner.h"
#include "libGenome/gnFASSource.h"
#include <fstream>
#include <string>
#include <vector>
#include <iomanip>

using namespace std;
using namespace genome;
using namespace mems;


int main( int argc, char* argv[] )
{
	// 
	if( argc != 3 )
	{
		cerr << "Usage: sortContigs <Mauve Alignment> <reference sequence #>\n";
		cerr << "Where <Mauve Alignment> is the .mln file generated by Mauve, NOT the .alignment\n";
		cerr << "Sequences are numbered from 0 in the order they were input to Mauve\n";
		cerr << "This program will write out a new reordered FastA file for each of the non-reference sequences\n";
		return -1;
	}
	ifstream aln_file( argv[1] );
	if( !aln_file.is_open() )
	{
		cerr << "Error opening \"" << argv[1] << "\"\n";
		return -1;
	}

	uint ref_seqI = atoi( argv[2] ); 

try{
	IntervalList iv_list;
	iv_list.ReadList( aln_file );
	cerr << "Read " << argv[1] << endl;
	LoadSequences(iv_list, &cout );

	// remove all but the n-way intervals
	IntervalList nway_iv_list;
	for( uint ivI = 0; ivI < iv_list.size(); ivI++ )
	{
		uint def_seqI = 0;
		for( ; def_seqI < iv_list.seq_table.size(); def_seqI++ )
			if( iv_list[ivI].Start( def_seqI ) == 0 )
				break;
		if( def_seqI == iv_list.seq_table.size() )
			nway_iv_list.push_back( iv_list[ivI] );
	}
	iv_list.erase(iv_list.begin(), iv_list.end() );
	iv_list.insert( iv_list.end(), nway_iv_list.begin(), nway_iv_list.end() );

	// compute LCB adjacencies
	vector< int64 > weights = vector< int64 >( iv_list.size(), 1 );
	vector< LCB > adjacencies;
	cerr << "computeLCBAdjacencies\n";
	computeLCBAdjacencies_v2( iv_list, weights, adjacencies );
	uint seq_count = iv_list.seq_filename.size();
	vector< gnSequence* > new_seq_table = vector< gnSequence* >( seq_count );
	for( uint seqI = 0; seqI < seq_count; seqI++ )
		new_seq_table[seqI] = new gnSequence();
	delete new_seq_table[ref_seqI];
	new_seq_table[ref_seqI] = iv_list.seq_table[ref_seqI];

	uint leftmost_lcb = 0;
	for( ; leftmost_lcb < adjacencies.size(); leftmost_lcb++ )
		if( adjacencies[ leftmost_lcb ].left_adjacency[ref_seqI] == -1 )
			break;
	uint adjI = leftmost_lcb;
	vector< set< uint > > placed_contigs = vector< set< uint > >( iv_list.seq_table.size() );
	cerr << "placing contigs\n";

	while( adjI != -1 && adjI != -2 && adjI < adjacencies.size() )
	{
		for( uint seqI = 0; seqI < seq_count; seqI++ )
		{
			if( seqI == ref_seqI )
				continue;
			int64 lend = absolut(adjacencies[ adjI ].left_end[seqI] );
			int64 rend = absolut(adjacencies[ adjI ].right_end[seqI] ) - 1;
			bool cur_forward = (adjacencies[ adjI ].left_end[seqI] > 0);
			bool ref_forward = (adjacencies[ adjI ].left_end[ref_seqI] > 0);
			bool forward = cur_forward == ref_forward;
			uint r_contig, l_contig;
			try{
				l_contig = iv_list.seq_table[seqI]->contigIndexByBase( absolut(lend) );
			}catch( gnException& gne )
			{
				cerr << gne << endl;
				cerr << "Thrown while getting contig for base lend: " << absolut(lend) << endl;
			}
			try{
				r_contig = iv_list.seq_table[seqI]->contigIndexByBase( absolut(rend) );
			}catch( gnException& gne )
			{
				cerr << gne << endl;
				cerr << "Thrown while getting contig for base rend: " << absolut(rend) << endl;
			}

			uint first_contig = forward? l_contig : r_contig;
			uint last_contig = forward? r_contig : l_contig;
			first_contig++;
			last_contig++;
			try{
			for( uint contigI = first_contig; forward? (contigI <= last_contig) : (contigI >= last_contig); (forward? contigI++ : contigI--) )
			{
				// place these if they haven't already been placed
				set< uint >::iterator p_iter = placed_contigs[seqI].find(contigI-1);
				if( p_iter != placed_contigs[seqI].end() )
					continue;	// already placed this contig
				try{
				(*new_seq_table[ seqI ]) += iv_list.seq_table[seqI]->contig(contigI-1);
				}catch( gnException& gne ){
					cerr << gne << endl;
					cerr << "Thrown while accessing seq " << seqI << " contig " << contigI-1<< endl;
				}
				if(!forward)
					new_seq_table[seqI]->setReverseComplement( true, new_seq_table[seqI]->contigListLength()-1 );
				placed_contigs[seqI].insert( contigI-1 );
			}
			}catch( gnException& gne ){
				cerr << gne << endl;
				cerr << "Thrown while adding contigs in the range: " << first_contig << " to " << last_contig << endl;
			}
		}
		adjI = adjacencies[ adjI ].right_adjacency[ref_seqI];
	}
	cerr << "adding unplaced contigs\n";

	// add any remaining contigs that the alignment didn't place
	for( uint seqI = 0; seqI < seq_count; seqI++ )
	{
		if( seqI == ref_seqI )
			continue;
		for( uint contigI = 0; contigI < iv_list.seq_table[seqI]->contigListLength(); contigI++ )
		{
			// place this contig if it hasn't already been placed
			set< uint >::iterator p_iter = placed_contigs[seqI].find(contigI);
			if( p_iter != placed_contigs[seqI].end() )
				continue;	// already placed this contig
			(*new_seq_table[ seqI ]) += iv_list.seq_table[seqI]->contig(contigI);
			placed_contigs[seqI].insert( contigI );
		}
	}

	cerr << "writing reordered sequence\n";
	for( uint seqI = 0; seqI < seq_count; seqI++ )
	{
		for( uint contigI = 0; contigI < new_seq_table[seqI]->contigListSize(); contigI++ )
		{
			string name = new_seq_table[seqI]->contigName( contigI );
			stringstream ss( name );
			string new_name;
			ss >> new_name;
			stringstream new_ss;
			int fillsize = ceil(log((double)new_seq_table[seqI]->contigListSize())/log(10.0));
			new_ss << setfill('0') << setw(fillsize);
			new_ss << contigI << "_" << new_name;
			if( new_seq_table[seqI]->isReverseComplement(contigI) )
				new_ss << "-";
			else
				new_ss << "+";
			new_seq_table[seqI]->setContigName( contigI, new_ss.str() );
		}
		if( seqI == ref_seqI )
			continue;	// reference sequence didn't change
		string o_filename = iv_list.seq_filename[ seqI ] + ".reordered";
		ofstream out_file( o_filename.c_str() );
		if( !out_file.is_open() )
		{
			cerr << "Error opening \"" << o_filename << "\"\n";
			return -1;
		}
		gnFASSource::Write( *new_seq_table[seqI], out_file, false, false );
	}

}catch(gnException& gne){
	cerr << gne << endl;
}
}
