/***************************************************************************
 *   Copyright (C) 2009 by Kai Dombrowe <just89@gmx.de>                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

#include "listwidgetitem.h"
#include "listwidget.h"

#include <QPainter>
#include <QStyleOptionGraphicsItem>
#include <QWidget>
#include <QGraphicsSceneHoverEvent>
#include <QGraphicsSceneMouseEvent>

#include <Plasma/FrameSvg>
#include <Nepomuk/KRatingPainter>
#include <KDebug>
#include <Plasma/Theme>
#include <Plasma/PopupApplet>
#include <QGraphicsView>
#include <QApplication>

ListWidgetItem::ListWidgetItem(QGraphicsWidget *parent)
    : QGraphicsWidget(parent),
    m_backgroundSvg(new Plasma::FrameSvg(this)),
    m_hoverSvg(new Plasma::FrameSvg(this)),
    m_selectionSvg(new Plasma::FrameSvg(this)),
    m_title(""),
    m_rating(0),
    m_x(false),
    m_command(""),
    m_state(Normal),
    m_isSelected(false),
    m_hoverRating(-1),
    m_dragStarted(false),
    m_lastPos(-1, -1)
{

    m_backgroundSvg->setImagePath("widgets/frame");
    m_backgroundSvg->setCacheAllRenderedFrames(true);
    m_backgroundSvg->setElementPrefix("plain");
    
    m_hoverSvg->setImagePath("widgets/viewitem");
    m_hoverSvg->setCacheAllRenderedFrames(true);
    m_hoverSvg->setElementPrefix("hover");
    
    m_selectionSvg->setImagePath("widgets/viewitem");
    m_selectionSvg->setCacheAllRenderedFrames(true);
    m_selectionSvg->setElementPrefix("selected");

    m_runPixmap = KIcon("system-run").pixmap(32,32);
    m_editPixmap = KIcon("edit-rename").pixmap(32, 32);
    m_removePixmap = KIcon("list-remove").pixmap(32, 32);

    setAcceptHoverEvents(true);
    setAcceptDrops(true);

    setFont(Plasma::Theme::defaultTheme()->font(Plasma::Theme::DesktopFont));

}


ListWidgetItem::~ListWidgetItem()
{

    delete m_backgroundSvg;
    delete m_hoverSvg;
    delete m_selectionSvg;

}


void ListWidgetItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{

    Q_UNUSED(option);
    Q_UNUSED(widget);
    
    painter->setPen(Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor));
    
    m_backgroundSvg->resizeFrame(size());
    m_backgroundSvg->paintFrame(painter, contentsRect());

    QFontMetricsF fm(font());
        
    QRectF textRect = contentsRect();
    textRect.setHeight(fm.height());
    textRect.setLeft((textRect.left()+(fm.height()*2)));
    
    painter->drawText(textRect, m_title);
    
    QRectF ratingRect = textRect;
    ratingRect.moveTop(textRect.bottom());
    ratingRect.setWidth(fm.height()*5);
    m_ratingRect = ratingRect;
    
    KRatingPainter::paintRating(painter, ratingRect.toRect(), Qt::AlignLeft, m_rating, m_hoverRating);
    
    QRectF iconRect = contentsRect();
    iconRect.setHeight(fm.height()*2);
    iconRect.setRight(textRect.left());
    
    painter->drawPixmap(iconRect.toRect(), KIcon(m_icon).pixmap(32,32));
    
    if (m_state != Normal || m_isSelected) {
        m_hoverSvg->resizeFrame(contentsRect().size());
        m_hoverSvg->paintFrame(painter, contentsRect());

        QRectF runRect = contentsRect();
        runRect.setLeft(textRect.left()+fm.width(m_title)+5);
        runRect.setHeight(fm.height());
        runRect.moveCenter(QPointF(runRect.center().x(), runRect.height()/2));
        runRect.setWidth(fm.height());
        m_runRect = runRect;
    
        painter->drawPixmap(runRect.toRect(), m_runPixmap);

        m_editRect = m_runRect;
        m_editRect.moveLeft(m_editRect.right()+5);
        painter->drawPixmap(m_editRect.toRect(), m_editPixmap);

        m_removeRect = m_editRect;
        m_removeRect.moveLeft(m_removeRect.right()+5);
        painter->drawPixmap(m_removeRect.toRect(), m_removePixmap);
    }
    
    if (m_isSelected) {
        m_selectionSvg->resizeFrame(contentsRect().size());
        m_selectionSvg->paintFrame(painter, contentsRect());
    }
    
}


QString ListWidgetItem::title() const
{

    return m_title;

}


int ListWidgetItem::rating() const
{

    return m_rating;

}


QString ListWidgetItem::icon() const
{

    return m_icon;

}


bool ListWidgetItem::X() const
{

    return m_x;

}


QString ListWidgetItem::command() const
{

    return m_command;

}


bool ListWidgetItem::isSelected() const
{

    return m_isSelected;

}


void ListWidgetItem::setTitle(const QString &newTitle)
{

    if (m_title == newTitle)
        return;
        
    m_title = newTitle;
    calcSize();
    update();

}


void ListWidgetItem::setRating(const int &newRating)
{

    if (m_rating == newRating)
        return;
    
    m_rating = newRating;
    update();
    
}


void ListWidgetItem::setIcon(const QString &newIcon)
{

    if (m_icon == newIcon)
        return;
        
    m_icon = newIcon;
    update();

}


void ListWidgetItem::setX(const bool &newX)
{

    if (m_x == newX) {
        return;
    }

    m_x = newX;
    
}


void ListWidgetItem::setCommand(const QString &newCommand)
{

    m_command = newCommand;

}


void ListWidgetItem::calcSize()
{

    QFontMetrics fm(font());

    qreal height = (fm.height()*2);
    setMinimumHeight(height);
    setPreferredHeight(height);
    
    qreal width = fm.width(m_title);
    width += (fm.height()*2); // game icon
    width += ((fm.height()*3)+15); // run/edit/remove icon
    setMinimumWidth(width);
    setPreferredWidth(width);
    setMaximumHeight(preferredHeight());
    
}


void ListWidgetItem::hoverEnterEvent(QGraphicsSceneHoverEvent *event)
{

    Q_UNUSED(event);
    if (m_state != Hover) {
        m_state = Hover;
        update();
    }

}


void ListWidgetItem::hoverLeaveEvent(QGraphicsSceneHoverEvent *event)
{

    Q_UNUSED(event);
    if (m_state != Normal) {
        m_state = Normal;
        update();
    }
    
    if (m_hoverRating != -1) {
        m_hoverRating = -1;
        update();
    }

}


void ListWidgetItem::hoverMoveEvent(QGraphicsSceneHoverEvent *event)
{

    bool change = false;
    if (m_runRect.contains(event->pos()) && m_state != Run) {
        m_state = Run;
        m_runPixmap = KIcon("system-run").pixmap(32, 32, KIcon::Active);
        change = true;
    } else if (!m_runRect.contains(event->pos()) && m_state == Run) {
        m_runPixmap = KIcon("system-run").pixmap(32, 32);
        m_state = Normal;
        change = true;
    }

    if (m_editRect.contains(event->pos()) && m_state != Edit) {
        m_state = Edit;
        m_editPixmap = KIcon("edit-rename").pixmap(32, 32, KIcon::Active);
        change = true;
    } else if (!m_editRect.contains(event->pos()) && m_state == Edit) {
        m_editPixmap = KIcon("edit-rename").pixmap(32, 32);
        m_state = Normal;
        change = true;
    }
    
    
    if (m_removeRect.contains(event->pos()) && m_state != Remove) {
        m_state = Remove;
        m_removePixmap = KIcon("list-remove").pixmap(32, 32, KIcon::Active);
        change = true;
    } else if(!m_removeRect.contains(event->pos()) && m_state == Remove) {
        m_removePixmap = KIcon("list-remove").pixmap(32, 32);
        m_state = Normal;
        change = true;
    }

    if (m_ratingRect.contains(event->pos())) {
        int old = m_hoverRating;
        m_hoverRating = KRatingPainter::getRatingFromPosition(m_ratingRect.toRect(), Qt::AlignLeft, Qt::LeftToRight, event->pos().toPoint());
        if (m_hoverRating != old)
            change = true;
    }
    
    if (m_state == Normal && contentsRect().contains(event->pos())) {
        m_state = Hover;
        update();
    } else if (change) {
        update();
    }

}


void ListWidgetItem::mousePressEvent(QGraphicsSceneMouseEvent *event)
{

    m_lastPos = event->pos();

}


void ListWidgetItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{

    m_lastPos = QPointF(-1, -1);
    if (m_editRect.contains(event->pos())) {
        emit edit();
        return;
    } else if (m_runRect.contains(event->pos())) {
        emit run();
        return;
    } else if (m_removeRect.contains(event->pos())) {
        emit remove();
        return;
    } else if (m_ratingRect.contains(event->pos())) {
        m_rating = m_hoverRating;
    } else {
        setSelected(!m_isSelected);
    }
    update();
    
}


void ListWidgetItem::setSelected(const bool &select)
{

    if (m_isSelected == select)
        return;
        
    m_isSelected = select;
    if (m_isSelected)
        emit selected();
    update();

}


void ListWidgetItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{

    QGraphicsWidget::mouseMoveEvent(event);
    if (m_lastPos != QPointF(-1, -1) &&
        (m_lastPos.toPoint() - event->pos().toPoint()).manhattanLength() >= QApplication::startDragDistance() &&
            !m_dragStarted) {
        kDebug() << "starting drag";
        m_dragStarted = true;
        QMimeData *mime = new QMimeData;
        mime->setData("INTERNAL", "");
        mime->setData("Index", QString::number(static_cast<ListWidget*>(parentItem())->indexOf(this)).toLatin1());
        QDrag *drag = new QDrag(static_cast<Plasma::PopupApplet*>(parentItem()->parentItem())->view());
        drag->setPixmap(KIcon(icon()).pixmap(32, 32));
        drag->setMimeData(mime);
        kDebug() << "drag started";
        drag->exec();        
        m_dragStarted = false;
    }

}

        
        
#include "listwidgetitem.moc"