/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile$

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkMaterialInterfaceProcessLoading
// .SECTION Description
// Data type to represent a node in a multiprocess job
// and its current loading state.

#ifndef vtkMaterialInterfaceProcessLoading_h
#define vtkMaterialInterfaceProcessLoading_h

#include "vtkPVVTKExtensionsDefaultModule.h" //needed for exports

#include <cassert>
#include "vtkType.h"
#include <iostream>
#include <vector>

class VTKPVVTKEXTENSIONSDEFAULT_EXPORT vtkMaterialInterfaceProcessLoading
{
  public:
    enum {ID=0, LOADING=1, SIZE=2};
    //
    vtkMaterialInterfaceProcessLoading(){ this->Initialize(-1,0); }
    //
    ~vtkMaterialInterfaceProcessLoading(){ this->Initialize(-1,0); }
    // Description:
    // Set the id and load factor.
    void Initialize(int id, vtkIdType loadFactor)
    {
      this->Data[ID]=id;
      this->Data[LOADING]=loadFactor;
    }
    // Description:
    // Comparision of two objects loading.
    bool operator<(const vtkMaterialInterfaceProcessLoading &rhs) const
    {
      return this->Data[LOADING]<rhs.Data[LOADING];
    }
    // Description:
    // Comparision of two objects loading.
    bool operator<=(const vtkMaterialInterfaceProcessLoading &rhs) const
    {
      return this->Data[LOADING]<=rhs.Data[LOADING];
    }
    // Description:
    // Comparision of two objects loading.
    bool operator>(const vtkMaterialInterfaceProcessLoading &rhs) const
    {
      return this->Data[LOADING]>rhs.Data[LOADING];
    }
    // Description:
    // Comparision of two objects loading.
    bool operator>=(const vtkMaterialInterfaceProcessLoading &rhs) const
    {
      return this->Data[LOADING]>=rhs.Data[LOADING];
    }
    // Description:
    // Comparision of two objects loading.
    bool operator==(const vtkMaterialInterfaceProcessLoading &rhs) const
    {
      return this->Data[LOADING]==rhs.Data[LOADING];
    }
    // Description:
    // Return the process id.
    vtkIdType GetId() const{ return this->Data[ID]; }
    // Description:
    // Return the load factor.
    vtkIdType GetLoadFactor() const{ return this->Data[LOADING]; }
    // Description:
    // Add to the load factor.
    vtkIdType UpdateLoadFactor(vtkIdType loadFactor)
    {
      assert("Update would make loading negative."
             && (this->Data[LOADING]+loadFactor)>=0);
      return this->Data[LOADING]+=loadFactor;
    }
  private:
    vtkIdType Data[SIZE];
};
VTKPVVTKEXTENSIONSDEFAULT_EXPORT
std::ostream &operator<<(std::ostream &sout, const vtkMaterialInterfaceProcessLoading &fp);
VTKPVVTKEXTENSIONSDEFAULT_EXPORT
std::ostream &operator<<(std::ostream &sout, const std::vector<vtkMaterialInterfaceProcessLoading> &vfp);
#endif


// VTK-HeaderTest-Exclude: vtkMaterialInterfaceProcessLoading.h
