///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file SimpleGeometryObject.h
 * \brief Contains the definition of the Core::SimpleGeometryObject class.
 */

#ifndef __OVITO_SIMPLE_GEOM_OBJECT_H
#define __OVITO_SIMPLE_GEOM_OBJECT_H

#include <core/Core.h>
#include <mesh/tri/TriMesh.h>
#include "../SceneObject.h"

namespace Core {

/**
 * \brief Base class for all simple geometry objects that are based on a Mesh.
 *
 * Subclasses must implements the buildMesh() method.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT SimpleGeometryObject : public SceneObject
{
protected:

	/// \brief Default constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	SimpleGeometryObject(bool isLoading) : SceneObject(isLoading), meshValidity(TimeNever) {}

public:

	/// \brief Makes the object render itself into a viewport.
	/// \param time The animation time at which to render the object
	/// \param contextNode The node context used to render the object.
	/// \param vp The viewport to render in.
	///
	/// This implementation just renders the Mesh produced by buildMesh().
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp);

	/// \brief Asks the object whether it should show in a rendered image.
	virtual bool isRenderable() { return true; }

	/// \brief Renders the object in preview rendering mode using OpenGL.
	/// \param time The animation time at which to render the object
	/// \param view Describes the camera projection.
	/// \param contextNode The node context used to render the object.
	/// \param imageWidth The width of the rendering buffer in pixels.
	/// \param imageHeight The height of the rendering buffer in pixels.
	/// \param glcontext The window that provides the OpenGL rendering context.
	/// \return \c false if the rendering has been aborted by the user; \c true otherwise.
	virtual bool renderPreview(TimeTicks time, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

	/// \brief Computes the bounding box of the object.
	/// \param time The animation time for which the bounding box should be computed.
	/// \param contextNode The scene node to which this scene object belongs to.
	/// \return The bounding box of the object in local object coordinates.
	///
	/// This implementation returns the bounding box of the Mesh produced by buildMesh().
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode) {
		if(!meshValidity.contains(time))
			buildMesh(time, mesh, meshValidity);
		return mesh.boundingBox();
	}

	/// \brief This asks the object whether it supports the conversion to another object type.
	/// \param objectClass The destination type. This must be a SceneObject derived class.
	/// \return \c true if this object can be converted to the requested type given by \a objectClass or any sub-class thereof.
	///         \c false if the conversion is not possible.
	///
	/// This implementation allows the conversion to a MeshObject.
	virtual bool canConvertTo(PluginClassDescriptor* objectClass);

	/// \brief Lets the object convert itself to another object type.
	/// \param objectClass The destination type. This must be a SceneObject derived class.
	/// \param time The time at which to convert the object.
	/// \return The newly created object or \c NULL if no conversion is possible.
	///
	/// This implementation handles the conversion to a MeshObject.
	virtual SceneObject::SmartPtr convertTo(PluginClassDescriptor* objectClass, TimeTicks time);

protected:

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg) {
		if(msg->type() == REFTARGET_CHANGED) {
			// Invalidate mesh cache when a parameter has changed.
			meshValidity.setEmpty();
		}
		return SceneObject::onRefTargetMessage(source, msg);
	}

	/// Is called when the value of a reference field of this RefMaker changes.
	virtual void onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget) {
		// Invalidate mesh cache when a parameter controller has been replaced.
		meshValidity.setEmpty();
        SceneObject::onRefTargetReplaced(field, oldTarget, newTarget);
	}

	/// Is called when a reference target has been added to a list reference field of this RefMaker.
	virtual void onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex) {
        SceneObject::onRefTargetInserted(field, newTarget, listIndex);
		// Invalidate mesh cache when a parameter controller has been added.
		meshValidity.setEmpty();
	}

	/// Is called when a reference target has been removed from a list reference field of this RefMaker.
	virtual void onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex) {
        SceneObject::onRefTargetRemoved(field, oldTarget, listIndex);
		// Invalidate mesh cache when a parameter controller has been removed.
		meshValidity.setEmpty();
	}

	/// Is called when the value of a non-animatable property field of this RefMaker has changed.
	virtual void onPropertyFieldValueChanged(const PropertyFieldDescriptor& field) {
        SceneObject::onPropertyFieldValueChanged(field);
		// Invalidate mesh cache when a parameter has changed.
		meshValidity.setEmpty();
	}

protected:

	/// \brief Builds the mesh representation of this geometry object.
	/// \param[in] time The animation time at which to generate the mesh.
	/// \param[out] mesh The mesh object that should be filled by the method. Please note
	///                  that the mesh passed to this method can contain the old geometry from the last
	///                  call to buildMesh(), so the method has to clear it first.
	/// \param[out] meshValidity The method implementation stores the validty interval of the generated mesh in this variable.
	///
	/// The SimpleGeometryObject class calls this virtual method to let it generate the mesh based on the internal
	/// parameters of the object.
	virtual void buildMesh(TimeTicks time, TriMesh& mesh, TimeInterval& meshValidity) = 0;

private:

	/// The validity interval of the cached mesh.
	TimeInterval meshValidity;

	/// The cached mesh generated by this object.
	TriMesh mesh;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(SimpleGeometryObject)
};


};

#endif // __OVITO_SIMPLE_GEOM_OBJECT_H
