/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CompoundExpression.h"

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Instructions.h>

#include "../LLVMBackend/CodeGenerator_p.h"
#include "../Debug.h"
#include "../Macros_p.h"
#include "../Type.h"
#include "../Type_p.h"
#include "../LLVMBackend/ExpressionResult_p.h"
#include "../LLVMBackend/ExpressionGenerationContext_p.h"
#include "GenerationVisitor.h"
#include "../LLVMBackend/Visitor_p.h"
#include "GarbageCollectionStatement.h"

using namespace GTLCore::AST;

CompoundExpression::CompoundExpression( const GTLCore::Type* _type, const std::vector<Expression*>& _expressions) : m_expressions(_expressions), m_type(_type)
{
  GTL_ASSERT(_type);
}

CompoundExpression::~CompoundExpression( )
{
  for( std::vector<Expression*>::iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
  {
    delete *it;
  }
}

const GTLCore::Type* CompoundExpression::type() const
{
  return m_type;
}

bool CompoundExpression::isConstant() const
{
  return m_constant;
}

LLVMBackend::ExpressionResult CompoundExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  GTL_ASSERT( _egc.currentBasicBlock() );
  GTL_DEBUG( *m_type << " " << m_type->dataType() << " " << Type::ARRAY << " " << Type::STRUCTURE << " " << Type::VECTOR );
  
  switch( m_type->dataType() )
  {
    case Type::VECTOR:
    {
      GTL_DEBUG( m_expressions.size() << " " << m_type->vectorSize() );
      GTL_ASSERT( m_expressions.size() == m_type->vectorSize() );
      llvm::Value* resultLoad = LLVMBackend::CodeGenerator::createVector( _gc.llvmContext(), m_type );
      // Affect the same value to each element of the vector
      for(std::size_t i = 0; i < m_expressions.size(); ++i)
      {
        llvm::Value* val = m_expressions[i]->generateValue( _gc, _egc ).value();
        GTL_DEBUG( i << " " << *val );
        resultLoad = llvm::InsertElementInst::Create( resultLoad, val, LLVMBackend::CodeGenerator::integerToConstant( _gc.llvmContext(), gtl_uint32(i) ), "", _egc.currentBasicBlock() );
      }
      return LLVMBackend::ExpressionResult( resultLoad, m_type );
    }
    case Type::ARRAY:
    {
      // Allocate memory for an array
      llvm::Value* ptr = LLVMBackend::CodeGenerator::allocateMemory(_gc, type()->d->type(_gc.llvmContext()), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), INT32_C(1)), _egc.currentBasicBlock());
      
      // The initial size is equal to the number of elements in the compound
      std::list<llvm::Value*> sizes;
      sizes.push_back(LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), gtl_uint32(m_expressions.size())));
      
      // Initialize the array's memory
      const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor(type());
      _egc.setCurrentBasicBlock(visitor->initialise( _gc, _egc.currentBasicBlock(), ptr, type(), sizes ));
      
      // Set the values
      const Type* mtype = type()->embeddedType();
      const LLVMBackend::Visitor* mvisitor = LLVMBackend::Visitor::getVisitorFor( mtype );
      for(std::size_t i = 0; i < m_expressions.size(); ++i)
      {
        llvm::Value* nptrToOwnMember = visitor->pointerToIndex(_gc, _egc, ptr, type(),  LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), gtl_uint32(i)));
        
        llvm::Value* val = m_expressions[i]->generateValue( _gc, _egc ).value();
        
        _egc.setCurrentBasicBlock(mvisitor->set( _gc, _egc.currentBasicBlock(), nptrToOwnMember, mtype, val, mtype));
      }
      _egc.setCurrentBasicBlock(visitor->mark(_gc, _egc.currentBasicBlock(), ptr, type(), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), INT32_C(-1))));
      
      return LLVMBackend::ExpressionResult(ptr, m_type, true);
    }
    case Type::STRUCTURE:
    {
      GTL_ASSERT(m_expressions.size() == type()->countStructDataMembers());
      llvm::Value* ptr = LLVMBackend::CodeGenerator::allocateMemory(_gc, type()->d->type(_gc.llvmContext()), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), INT32_C(1)), _egc.currentBasicBlock());
      
      const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor(type());
      _egc.setCurrentBasicBlock(visitor->initialise( _gc, _egc.currentBasicBlock(), ptr, type(), std::list<llvm::Value*>()));

      for(std::size_t i = 0; i < type()->countStructDataMembers(); ++i)
      {
        const Type* mtype = type()->structDataMember(i).type();
        llvm::Value* nptrToOwnMember = LLVMBackend::StructureVisitor::pointerToValue(_gc, _egc.currentBasicBlock(), ptr, i);
        llvm::Value* val = m_expressions[i]->generateValue( _gc, _egc ).value();
        
        const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor( mtype );
        _egc.setCurrentBasicBlock(visitor->set( _gc, _egc.currentBasicBlock(), nptrToOwnMember, mtype, val, mtype));
      }
      _egc.setCurrentBasicBlock(visitor->mark(_gc, _egc.currentBasicBlock(), ptr, type(), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), INT32_C(-1))));
      return LLVMBackend::ExpressionResult(ptr, m_type, true);
    }
    default:
      GTL_ABORT("Unsupported");
  }
}

ExpressionResultSP CompoundExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  std::vector<ExpressionResultSP> results;
  results.reserve(m_expressions.size());
  foreach( Expression* expr, m_expressions)
  {
    results.push_back( expr->generateValue(_generationVisitor) );
  }
  return _generationVisitor->createCompoundValue( results, m_type, annotation());
}

void CompoundExpression::markAsReturnExpression()
{
  for(std::size_t i = 0; i < m_expressions.size(); ++i)
  {
    m_expressions[i]->markAsReturnExpression();
  }
}
