/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var bench = require( '@stdlib/bench' );
var randu = require( '@stdlib/random/base/randu' );
var floor = require( '@stdlib/math/base/special/floor' );
var isNonNegativeIntegerArray = require( '@stdlib/assert/is-nonnegative-integer-array' ).primitives;
var numel = require( './../../base/numel' );
var pkg = require( './../package.json' ).name;
var ind2sub = require( './../lib' );


// MAIN //

bench( pkg, function benchmark( b ) {
	var shape;
	var len;
	var out;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	len = numel( shape );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, idx );
		if ( out.length !== shape.length ) {
			b.fail( 'should return an array with expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array containing nonnegative integers' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::memory_reuse:assign', function benchmark( b ) {
	var shape;
	var len;
	var out;
	var idx;
	var s;
	var i;

	shape = [ 10, 10, 10 ];
	len = numel( shape );
	out = [ 0, 0, 0 ];

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		s = ind2sub.assign( shape, idx, out );
		if ( s !== out ) {
			b.fail( 'should return output array' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( s ) ) {
		b.fail( 'should return an array containing nonnegative integers' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=wrap', function benchmark( b ) {
	var shape;
	var opts;
	var len;
	var out;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	len = numel( shape );
	opts = {
		'mode': 'wrap'
	};

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, idx, opts );
		if ( out.length !== shape.length ) {
			b.fail( 'should return an array with expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array containing nonnegative integers' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=clamp', function benchmark( b ) {
	var shape;
	var opts;
	var len;
	var out;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	len = numel( shape );
	opts = {
		'mode': 'clamp'
	};

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, idx, opts );
		if ( out.length !== shape.length ) {
			b.fail( 'should return an array with expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array containing nonnegative integers' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=throw', function benchmark( b ) {
	var shape;
	var opts;
	var len;
	var out;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	len = numel( shape );
	opts = {
		'mode': 'throw'
	};

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, idx, opts );
		if ( out.length !== shape.length ) {
			b.fail( 'should return an array with expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array containing nonnegative integers' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':order=row-major', function benchmark( b ) {
	var shape;
	var opts;
	var len;
	var out;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	len = numel( shape );
	opts = {
		'order': 'row-major'
	};

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, idx, opts );
		if ( out.length !== shape.length ) {
			b.fail( 'should return an array with expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array containing nonnegative integers' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':order=column-major', function benchmark( b ) {
	var shape;
	var opts;
	var len;
	var out;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	len = numel( shape );
	opts = {
		'order': 'column-major'
	};

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, idx, opts );
		if ( out.length !== shape.length ) {
			b.fail( 'should return an array with expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array containing nonnegative integers' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});
