/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.options.editor.onsave;

import java.awt.BorderLayout;
import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.prefs.Preferences;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.text.JTextComponent;
import org.netbeans.api.editor.EditorRegistry;
import org.netbeans.api.options.OptionsDisplayer;
import org.netbeans.modules.options.editor.spi.PreferencesCustomizer;
import org.netbeans.spi.options.OptionsPanelController;
import org.openide.util.WeakListeners;

/**
 * Panel for "On Save" tab in options.
 *
 * @author Miloslav Metelka
 */
@OptionsPanelController.Keywords(keywords = {"#KW_OnSave"}, location = OptionsDisplayer.EDITOR, tabTitle="#CTL_OnSave_DisplayName")
public class OnSaveTabPanel extends JPanel implements PropertyChangeListener {
    
    private OnSaveCommonPanel commonPanel;

    private OnSaveTabSelector selector;

    private PropertyChangeListener weakListener;

    public OnSaveTabPanel() {
        initComponents();
        // Languages combobox renderer
        cboLanguage.setRenderer(new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                if ((value instanceof String) && selector != null) {
                    value = selector.getLanguageName((String) value);
                }
                return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            }
        });
        
        commonPanel = new OnSaveCommonPanel();
        commonPanelContainer.setLayout(new BorderLayout());
        commonPanelContainer.add(commonPanel, BorderLayout.WEST);
        
        customPanelContainer.setLayout(new BorderLayout());
    }

    private String storedMimeType = null;

    public void setSelector(OnSaveTabSelector selector) {
        if (selector == null) {
            storedMimeType = (String)cboLanguage.getSelectedItem();
        }

        if (this.selector != null) {
            this.selector.removePropertyChangeListener(weakListener);
        }

        this.selector = selector;

        if (this.selector != null) {
            this.weakListener = WeakListeners.propertyChange(this, this.selector);
            this.selector.addPropertyChangeListener(weakListener);
            DefaultComboBoxModel model = new DefaultComboBoxModel();
            String preSelectMimeType = null;
            for (String mimeType : this.selector.getMimeTypes()) {
                model.addElement(mimeType);
                if (mimeType.equals(storedMimeType)) {
                    preSelectMimeType = mimeType;
                }
            }
            cboLanguage.setModel(model);

            // Pre-select a language
            if (preSelectMimeType == null) {
                JTextComponent pane = EditorRegistry.lastFocusedComponent();
                preSelectMimeType = pane != null ? (String)pane.getDocument().getProperty("mimeType") : ""; // NOI18N
            }
            cboLanguage.setSelectedItem(preSelectMimeType);
            if (!preSelectMimeType.equals(cboLanguage.getSelectedItem())) {
                cboLanguage.setSelectedIndex(0);
            }
        } else {
            cboLanguage.setModel(new DefaultComboBoxModel());
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        lLanguage = new javax.swing.JLabel();
        cboLanguage = new javax.swing.JComboBox();
        commonPanelContainer = new javax.swing.JPanel();
        customPanelContainer = new javax.swing.JPanel();

        lLanguage.setLabelFor(cboLanguage);
        org.openide.awt.Mnemonics.setLocalizedText(lLanguage, org.openide.util.NbBundle.getMessage(OnSaveTabPanel.class, "OnSaveTabPanel.lLanguage.text")); // NOI18N

        cboLanguage.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                languageChanged(evt);
            }
        });

        javax.swing.GroupLayout commonPanelContainerLayout = new javax.swing.GroupLayout(commonPanelContainer);
        commonPanelContainer.setLayout(commonPanelContainerLayout);
        commonPanelContainerLayout.setHorizontalGroup(
            commonPanelContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 241, Short.MAX_VALUE)
        );
        commonPanelContainerLayout.setVerticalGroup(
            commonPanelContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 90, Short.MAX_VALUE)
        );

        javax.swing.GroupLayout customPanelContainerLayout = new javax.swing.GroupLayout(customPanelContainer);
        customPanelContainer.setLayout(customPanelContainerLayout);
        customPanelContainerLayout.setHorizontalGroup(
            customPanelContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 0, Short.MAX_VALUE)
        );
        customPanelContainerLayout.setVerticalGroup(
            customPanelContainerLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 137, Short.MAX_VALUE)
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(customPanelContainer, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(commonPanelContainer, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                        .addComponent(lLanguage)
                        .addGap(3, 3, 3)
                        .addComponent(cboLanguage, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(0, 0, Short.MAX_VALUE)))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(lLanguage)
                    .addComponent(cboLanguage, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(commonPanelContainer, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(customPanelContainer, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addContainerGap())
        );
    }// </editor-fold>//GEN-END:initComponents

    private void languageChanged(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_languageChanged
        selector.setSelectedMimeType((String)cboLanguage.getSelectedItem());
    }//GEN-LAST:event_languageChanged

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JComboBox cboLanguage;
    private javax.swing.JPanel commonPanelContainer;
    private javax.swing.JPanel customPanelContainer;
    private javax.swing.JLabel lLanguage;
    // End of variables declaration//GEN-END:variables

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        String mimeType = selector.getSelectedMimeType();
        Preferences prefs = selector.getPreferences(mimeType);
        Preferences globalPrefs = selector.getPreferences("");
        commonPanel.update(prefs, globalPrefs);

        customPanelContainer.setVisible(false);
        customPanelContainer.removeAll();
        PreferencesCustomizer c = selector.getSelectedCustomizer();
        if (c != null) {
            customPanelContainer.add(c.getComponent(), BorderLayout.WEST);
        }
        customPanelContainer.setVisible(true);
    }

}
