//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Base/Pixel/SphericalPixel.cpp
//! @brief     Implements class SphericalPixel.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Base/Pixel/SphericalPixel.h"
#include "Base/Axis/Bin.h"
#include "Base/Vector/GisasDirection.h"

SphericalPixel::SphericalPixel(const Bin1D& alpha_bin, const Bin1D& phi_bin)
    : m_alpha(alpha_bin.lowerBound())
    , m_phi(phi_bin.lowerBound())
    , m_dalpha(alpha_bin.binSize())
    , m_dphi(phi_bin.binSize())
{
    auto solid_angle_value = std::abs(m_dphi * (std::sin(m_alpha + m_dalpha) - std::sin(m_alpha)));
    m_solid_angle = solid_angle_value <= 0.0 ? 1.0 : solid_angle_value;
}

SphericalPixel* SphericalPixel::clone() const
{
    return new SphericalPixel(*this);
}

SphericalPixel* SphericalPixel::createZeroSizePixel(double x, double y) const
{
    const double phi = m_phi + x * m_dphi;
    const double alpha = m_alpha + y * m_dalpha;
    return new SphericalPixel(Bin1D::At(alpha), Bin1D::At(phi));
}

R3 SphericalPixel::getK(double x, double y, double wavelength) const
{
    const double phi = m_phi + x * m_dphi;
    const double alpha = m_alpha + y * m_dalpha;
    return vecOfLambdaAlphaPhi(wavelength, alpha, phi);
}

double SphericalPixel::integrationFactor(double /* x */, double y) const
{
    if (m_dalpha == 0.0)
        return 1.0;
    const double alpha = m_alpha + y * m_dalpha;
    return std::cos(alpha) * m_dalpha / (std::sin(m_alpha + m_dalpha) - std::sin(m_alpha));
}

double SphericalPixel::solidAngle() const
{
    return m_solid_angle;
}
