/*
 * Copyright (c) 2019-2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sve_hybrid_fp16_mla_6x4VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<__fp16> A_arg,
    size_t M, size_t N, const __fp16 *B_ptr, IndirectOutputArg<__fp16> output_arg,
    const __fp16 *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
        __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const __fp16 *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p5.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 66f\n"
      "cmp %x[M], #0x4\n"
      "bgt 53f\n"
      "beq 40f\n"
      "cmp %x[M], #0x2\n"
      "bgt 27f\n"
      "beq 14f\n"
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.h, x20, x11\n"
      "inch x20\n"
      "whilelt p3.h, x20, x11\n"
      "inch x20\n"
      "whilelt p2.h, x20, x11\n"
      "inch x20\n"
      "whilelt p1.h, x20, x11\n"
      "cbz x12, 3f\n"
      "ld1h { z8.h }, p5/Z, [x12]\n"
      "ld1h { z9.h }, p5/Z, [x12, #1, MUL VL]\n"
      "ld1h { z10.h }, p5/Z, [x12, #2, MUL VL]\n"
      "ld1h { z11.h }, p5/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "b 5f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 4f\n"
      "ld1h { z8.h }, p4/Z, [x9]\n"
      "ld1h { z9.h }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1h { z10.h }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z11.h }, p1/Z, [x9, #3, MUL VL]\n"
      "b 5f\n"
      "4:"  // Height 1: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "5:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "6:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 7f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 8f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "b 8f\n"
      "7:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "8:"  // Height 1: input setup done
      "cmp x27, #0x8\n"
      "ble 10f\n"
      "9:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z0.h }, p0/Z, [x26]\n"
      "ld1h { z16.h }, p5/Z, [x10]\n"
      "fmla z8.h, z16.h, z0.h[0]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z9.h, z16.h, z0.h[0]\n"
      "ld1h { z16.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z10.h, z16.h, z0.h[0]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z11.h, z16.h, z0.h[0]\n"
      "ld1h { z16.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z8.h, z16.h, z0.h[1]\n"
      "ld1h { z16.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z9.h, z16.h, z0.h[1]\n"
      "ld1h { z16.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z10.h, z16.h, z0.h[1]\n"
      "ld1h { z16.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z11.h, z16.h, z0.h[1]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[2]\n"
      "fmla z9.h, z16.h, z0.h[2]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[2]\n"
      "fmla z11.h, z16.h, z0.h[2]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[3]\n"
      "fmla z9.h, z16.h, z0.h[3]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[3]\n"
      "fmla z11.h, z16.h, z0.h[3]\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[4]\n"
      "fmla z9.h, z16.h, z0.h[4]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[4]\n"
      "fmla z11.h, z16.h, z0.h[4]\n"
      "ld1h { z17.h }, p5/Z, [x10, #4, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[5]\n"
      "fmla z9.h, z16.h, z0.h[5]\n"
      "ld1h { z17.h }, p5/Z, [x10, #6, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z17.h, z0.h[5]\n"
      "fmla z11.h, z16.h, z0.h[5]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[6]\n"
      "fmla z9.h, z16.h, z0.h[6]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[6]\n"
      "fmla z11.h, z16.h, z0.h[6]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[7]\n"
      "fmla z9.h, z16.h, z0.h[7]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "sub x27, x27, #0x8\n"
      "cmp x27, #0x8\n"
      "fmla z10.h, z17.h, z0.h[7]\n"
      "fmla z11.h, z16.h, z0.h[7]\n"
      "add x26, x26, #0x10\n"
      "bgt 9b\n"
      "10:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z0.h }, p0/Z, [x26]\n"
      "ld1h { z16.h }, p5/Z, [x10]\n"
      "fmla z8.h, z16.h, z0.h[0]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z9.h, z16.h, z0.h[0]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[0]\n"
      "fmla z11.h, z16.h, z0.h[0]\n"
      "addvl x10, x10, #4\n"
      "ble 11f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[1]\n"
      "fmla z9.h, z16.h, z0.h[1]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[1]\n"
      "fmla z11.h, z16.h, z0.h[1]\n"
      "addvl x10, x10, #4\n"
      "ble 11f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[2]\n"
      "fmla z9.h, z16.h, z0.h[2]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[2]\n"
      "fmla z11.h, z16.h, z0.h[2]\n"
      "addvl x10, x10, #4\n"
      "ble 11f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[3]\n"
      "fmla z9.h, z16.h, z0.h[3]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[3]\n"
      "fmla z11.h, z16.h, z0.h[3]\n"
      "addvl x10, x10, #4\n"
      "ble 11f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[4]\n"
      "fmla z9.h, z16.h, z0.h[4]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[4]\n"
      "fmla z11.h, z16.h, z0.h[4]\n"
      "addvl x10, x10, #4\n"
      "ble 11f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[5]\n"
      "fmla z9.h, z16.h, z0.h[5]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[5]\n"
      "fmla z11.h, z16.h, z0.h[5]\n"
      "addvl x10, x10, #4\n"
      "ble 11f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[6]\n"
      "fmla z9.h, z16.h, z0.h[6]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[6]\n"
      "fmla z11.h, z16.h, z0.h[6]\n"
      "addvl x10, x10, #4\n"
      "ble 11f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[7]\n"
      "fmla z9.h, z16.h, z0.h[7]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[7]\n"
      "fmla z11.h, z16.h, z0.h[7]\n"
      "addvl x10, x10, #4\n"
      "11:"  // Height 1: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 6b\n"
      "tbz %x[flags], #1, 12f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rh { z17.h }, p5/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z16.h }, p5/Z, [x20]\n"
      "fmin z8.h, p5/M, z8.h, z17.h\n"
      "fmin z9.h, p5/M, z9.h, z17.h\n"
      "fmin z10.h, p5/M, z10.h, z17.h\n"
      "fmin z11.h, p5/M, z11.h, z17.h\n"
      "fmax z8.h, p5/M, z8.h, z16.h\n"
      "fmax z9.h, p5/M, z9.h, z16.h\n"
      "fmax z10.h, p5/M, z10.h, z16.h\n"
      "fmax z11.h, p5/M, z11.h, z16.h\n"
      "12:"  // Height 1: No activation
      "st1h { z8.h }, p4, [x9]\n"
      "st1h { z9.h }, p3, [x9, #1, MUL VL]\n"
      "st1h { z10.h }, p2, [x9, #2, MUL VL]\n"
      "st1h { z11.h }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "13:"  // Height 1: Writeback done
      "dech x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 2b\n"
      "b 80f\n"
      "14:"  // Height 2
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "15:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.h, x20, x11\n"
      "inch x20\n"
      "whilelt p3.h, x20, x11\n"
      "inch x20\n"
      "whilelt p2.h, x20, x11\n"
      "inch x20\n"
      "whilelt p1.h, x20, x11\n"
      "cbz x12, 16f\n"
      "ld1h { z8.h }, p5/Z, [x12]\n"
      "ld1h { z9.h }, p5/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1h { z10.h }, p5/Z, [x12, #2, MUL VL]\n"
      "ld1h { z11.h }, p5/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "addvl x12, x12, #4\n"
      "b 18f\n"
      "16:"  // Height 2: no bias
      "tbz %x[flags], #0, 17f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x20, x9, x20, LSL #1\n"
      "ld1h { z8.h }, p4/Z, [x9]\n"
      "ld1h { z9.h }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1h { z10.h }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z11.h }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1h { z12.h }, p4/Z, [x20]\n"
      "ld1h { z13.h }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1h { z14.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z15.h }, p1/Z, [x20, #3, MUL VL]\n"
      "b 18f\n"
      "17:"  // Height 2: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "18:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "19:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 20f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 21f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "b 21f\n"
      "20:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "21:"  // Height 2: input setup done
      "cmp x27, #0x8\n"
      "ble 23f\n"
      "22:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z1.h }, p0/Z, [x26]\n"
      "ld1rqh { z0.h }, p0/Z, [x25]\n"
      "sub x27, x27, #0x8\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z1.h[0]\n"
      "fmla z12.h, z17.h, z0.h[0]\n"
      "fmla z9.h, z16.h, z1.h[0]\n"
      "fmla z13.h, z16.h, z0.h[0]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z17.h, z1.h[0]\n"
      "fmla z14.h, z17.h, z0.h[0]\n"
      "ld1h { z17.h }, p5/Z, [x10, #4, MUL VL]\n"
      "cmp x27, #0x8\n"
      "fmla z11.h, z16.h, z1.h[0]\n"
      "fmla z15.h, z16.h, z0.h[0]\n"
      "ld1h { z16.h }, p5/Z, [x10, #5, MUL VL]\n"
      "add x26, x26, #0x10\n"
      "fmla z8.h, z17.h, z1.h[1]\n"
      "fmla z12.h, z17.h, z0.h[1]\n"
      "ld1h { z17.h }, p5/Z, [x10, #6, MUL VL]\n"
      "add x25, x25, #0x10\n"
      "fmla z9.h, z16.h, z1.h[1]\n"
      "fmla z13.h, z16.h, z0.h[1]\n"
      "ld1h { z16.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z17.h, z1.h[1]\n"
      "fmla z14.h, z17.h, z0.h[1]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z11.h, z16.h, z1.h[1]\n"
      "fmla z15.h, z16.h, z0.h[1]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z17.h, z1.h[2]\n"
      "fmla z12.h, z17.h, z0.h[2]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z9.h, z16.h, z1.h[2]\n"
      "fmla z13.h, z16.h, z0.h[2]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z17.h, z1.h[2]\n"
      "fmla z14.h, z17.h, z0.h[2]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z11.h, z16.h, z1.h[2]\n"
      "fmla z15.h, z16.h, z0.h[2]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z17.h, z1.h[3]\n"
      "fmla z12.h, z17.h, z0.h[3]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z9.h, z16.h, z1.h[3]\n"
      "fmla z13.h, z16.h, z0.h[3]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z17.h, z1.h[3]\n"
      "fmla z14.h, z17.h, z0.h[3]\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "fmla z11.h, z16.h, z1.h[3]\n"
      "fmla z15.h, z16.h, z0.h[3]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z1.h[4]\n"
      "fmla z12.h, z17.h, z0.h[4]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z16.h, z1.h[4]\n"
      "fmla z13.h, z16.h, z0.h[4]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z17.h, z1.h[4]\n"
      "fmla z14.h, z17.h, z0.h[4]\n"
      "ld1h { z17.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z11.h, z16.h, z1.h[4]\n"
      "fmla z15.h, z16.h, z0.h[4]\n"
      "ld1h { z16.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z17.h, z1.h[5]\n"
      "fmla z12.h, z17.h, z0.h[5]\n"
      "ld1h { z17.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z9.h, z16.h, z1.h[5]\n"
      "fmla z13.h, z16.h, z0.h[5]\n"
      "ld1h { z16.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z17.h, z1.h[5]\n"
      "fmla z14.h, z17.h, z0.h[5]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z11.h, z16.h, z1.h[5]\n"
      "fmla z15.h, z16.h, z0.h[5]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z17.h, z1.h[6]\n"
      "fmla z12.h, z17.h, z0.h[6]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z9.h, z16.h, z1.h[6]\n"
      "fmla z13.h, z16.h, z0.h[6]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z17.h, z1.h[6]\n"
      "fmla z14.h, z17.h, z0.h[6]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z11.h, z16.h, z1.h[6]\n"
      "fmla z15.h, z16.h, z0.h[6]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z17.h, z1.h[7]\n"
      "fmla z12.h, z17.h, z0.h[7]\n"
      "ld1h { z17.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z9.h, z16.h, z1.h[7]\n"
      "fmla z13.h, z16.h, z0.h[7]\n"
      "ld1h { z16.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z17.h, z1.h[7]\n"
      "fmla z14.h, z17.h, z0.h[7]\n"
      "fmla z11.h, z16.h, z1.h[7]\n"
      "fmla z15.h, z16.h, z0.h[7]\n"
      "bgt 22b\n"
      "23:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z0.h }, p0/Z, [x26]\n"
      "ld1rqh { z1.h }, p0/Z, [x25]\n"
      "subs x27, x27, #0x1\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[0]\n"
      "fmla z12.h, z17.h, z1.h[0]\n"
      "fmla z9.h, z16.h, z0.h[0]\n"
      "fmla z13.h, z16.h, z1.h[0]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[0]\n"
      "fmla z14.h, z17.h, z1.h[0]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[0]\n"
      "fmla z15.h, z16.h, z1.h[0]\n"
      "ble 24f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[1]\n"
      "fmla z12.h, z17.h, z1.h[1]\n"
      "fmla z9.h, z16.h, z0.h[1]\n"
      "fmla z13.h, z16.h, z1.h[1]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[1]\n"
      "fmla z14.h, z17.h, z1.h[1]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[1]\n"
      "fmla z15.h, z16.h, z1.h[1]\n"
      "ble 24f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[2]\n"
      "fmla z12.h, z17.h, z1.h[2]\n"
      "fmla z9.h, z16.h, z0.h[2]\n"
      "fmla z13.h, z16.h, z1.h[2]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[2]\n"
      "fmla z14.h, z17.h, z1.h[2]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[2]\n"
      "fmla z15.h, z16.h, z1.h[2]\n"
      "ble 24f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[3]\n"
      "fmla z12.h, z17.h, z1.h[3]\n"
      "fmla z9.h, z16.h, z0.h[3]\n"
      "fmla z13.h, z16.h, z1.h[3]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[3]\n"
      "fmla z14.h, z17.h, z1.h[3]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[3]\n"
      "fmla z15.h, z16.h, z1.h[3]\n"
      "ble 24f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[4]\n"
      "fmla z12.h, z17.h, z1.h[4]\n"
      "fmla z9.h, z16.h, z0.h[4]\n"
      "fmla z13.h, z16.h, z1.h[4]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[4]\n"
      "fmla z14.h, z17.h, z1.h[4]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[4]\n"
      "fmla z15.h, z16.h, z1.h[4]\n"
      "ble 24f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[5]\n"
      "fmla z12.h, z17.h, z1.h[5]\n"
      "fmla z9.h, z16.h, z0.h[5]\n"
      "fmla z13.h, z16.h, z1.h[5]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[5]\n"
      "fmla z14.h, z17.h, z1.h[5]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[5]\n"
      "fmla z15.h, z16.h, z1.h[5]\n"
      "ble 24f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[6]\n"
      "fmla z12.h, z17.h, z1.h[6]\n"
      "fmla z9.h, z16.h, z0.h[6]\n"
      "fmla z13.h, z16.h, z1.h[6]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z10.h, z17.h, z0.h[6]\n"
      "fmla z14.h, z17.h, z1.h[6]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[6]\n"
      "fmla z15.h, z16.h, z1.h[6]\n"
      "ble 24f\n"
      "ld1h { z17.h }, p5/Z, [x10]\n"
      "ld1h { z16.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z17.h, z0.h[7]\n"
      "fmla z12.h, z17.h, z1.h[7]\n"
      "fmla z9.h, z16.h, z0.h[7]\n"
      "fmla z13.h, z16.h, z1.h[7]\n"
      "ld1h { z17.h }, p5/Z, [x10, #2, MUL VL]\n"
      "ld1h { z16.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z17.h, z0.h[7]\n"
      "fmla z14.h, z17.h, z1.h[7]\n"
      "addvl x10, x10, #4\n"
      "fmla z11.h, z16.h, z0.h[7]\n"
      "fmla z15.h, z16.h, z1.h[7]\n"
      "24:"  // Height 2: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 19b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #1\n"
      "tbz %x[flags], #1, 25f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rh { z17.h }, p5/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z16.h }, p5/Z, [x20]\n"
      "fmin z8.h, p5/M, z8.h, z17.h\n"
      "fmin z9.h, p5/M, z9.h, z17.h\n"
      "fmin z10.h, p5/M, z10.h, z17.h\n"
      "fmin z11.h, p5/M, z11.h, z17.h\n"
      "fmin z12.h, p5/M, z12.h, z17.h\n"
      "fmin z13.h, p5/M, z13.h, z17.h\n"
      "fmin z14.h, p5/M, z14.h, z17.h\n"
      "fmin z15.h, p5/M, z15.h, z17.h\n"
      "fmax z8.h, p5/M, z8.h, z16.h\n"
      "fmax z9.h, p5/M, z9.h, z16.h\n"
      "fmax z10.h, p5/M, z10.h, z16.h\n"
      "fmax z11.h, p5/M, z11.h, z16.h\n"
      "fmax z12.h, p5/M, z12.h, z16.h\n"
      "fmax z13.h, p5/M, z13.h, z16.h\n"
      "fmax z14.h, p5/M, z14.h, z16.h\n"
      "fmax z15.h, p5/M, z15.h, z16.h\n"
      "25:"  // Height 2: No activation
      "st1h { z8.h }, p4, [x9]\n"
      "st1h { z9.h }, p3, [x9, #1, MUL VL]\n"
      "st1h { z10.h }, p2, [x9, #2, MUL VL]\n"
      "st1h { z11.h }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1h { z12.h }, p4, [x25]\n"
      "st1h { z13.h }, p3, [x25, #1, MUL VL]\n"
      "st1h { z14.h }, p2, [x25, #2, MUL VL]\n"
      "st1h { z15.h }, p1, [x25, #3, MUL VL]\n"
      "26:"  // Height 2: Writeback done
      "dech x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 15b\n"
      "b 80f\n"
      "27:"  // Height 3
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "28:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.h, x20, x11\n"
      "inch x20\n"
      "whilelt p3.h, x20, x11\n"
      "inch x20\n"
      "whilelt p2.h, x20, x11\n"
      "inch x20\n"
      "whilelt p1.h, x20, x11\n"
      "cbz x12, 29f\n"
      "ld1h { z8.h }, p5/Z, [x12]\n"
      "ld1h { z9.h }, p5/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1h { z10.h }, p5/Z, [x12, #2, MUL VL]\n"
      "ld1h { z11.h }, p5/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "b 31f\n"
      "29:"  // Height 3: no bias
      "tbz %x[flags], #0, 30f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x21, x9, x20, LSL #1\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z8.h }, p4/Z, [x9]\n"
      "ld1h { z9.h }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1h { z10.h }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z11.h }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1h { z12.h }, p4/Z, [x21]\n"
      "ld1h { z13.h }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1h { z14.h }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1h { z15.h }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1h { z16.h }, p4/Z, [x20]\n"
      "ld1h { z17.h }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z19.h }, p1/Z, [x20, #3, MUL VL]\n"
      "b 31f\n"
      "30:"  // Height 3: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "31:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "32:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 33f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 34f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "b 34f\n"
      "33:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "34:"  // Height 3: input setup done
      "cmp x27, #0x8\n"
      "ble 36f\n"
      "35:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z2.h }, p0/Z, [x26]\n"
      "ld1rqh { z1.h }, p0/Z, [x25]\n"
      "sub x27, x27, #0x8\n"
      "ld1rqh { z0.h }, p0/Z, [x24]\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "fmla z8.h, z21.h, z2.h[0]\n"
      "fmla z12.h, z21.h, z1.h[0]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z16.h, z21.h, z0.h[0]\n"
      "fmla z9.h, z20.h, z2.h[0]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[0]\n"
      "fmla z17.h, z20.h, z0.h[0]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "cmp x27, #0x8\n"
      "fmla z10.h, z21.h, z2.h[0]\n"
      "fmla z14.h, z21.h, z1.h[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla z18.h, z21.h, z0.h[0]\n"
      "fmla z11.h, z20.h, z2.h[0]\n"
      "ld1h { z21.h }, p5/Z, [x10, #4, MUL VL]\n"
      "add x24, x24, #0x10\n"
      "fmla z15.h, z20.h, z1.h[0]\n"
      "fmla z19.h, z20.h, z0.h[0]\n"
      "ld1h { z20.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z21.h, z2.h[1]\n"
      "fmla z12.h, z21.h, z1.h[1]\n"
      "fmla z16.h, z21.h, z0.h[1]\n"
      "fmla z9.h, z20.h, z2.h[1]\n"
      "ld1h { z21.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[1]\n"
      "fmla z17.h, z20.h, z0.h[1]\n"
      "ld1h { z20.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z21.h, z2.h[1]\n"
      "fmla z14.h, z21.h, z1.h[1]\n"
      "fmla z18.h, z21.h, z0.h[1]\n"
      "fmla z11.h, z20.h, z2.h[1]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z15.h, z20.h, z1.h[1]\n"
      "fmla z19.h, z20.h, z0.h[1]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z21.h, z2.h[2]\n"
      "fmla z12.h, z21.h, z1.h[2]\n"
      "fmla z16.h, z21.h, z0.h[2]\n"
      "fmla z9.h, z20.h, z2.h[2]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[2]\n"
      "fmla z17.h, z20.h, z0.h[2]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z21.h, z2.h[2]\n"
      "fmla z14.h, z21.h, z1.h[2]\n"
      "fmla z18.h, z21.h, z0.h[2]\n"
      "fmla z11.h, z20.h, z2.h[2]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z15.h, z20.h, z1.h[2]\n"
      "fmla z19.h, z20.h, z0.h[2]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z21.h, z2.h[3]\n"
      "fmla z12.h, z21.h, z1.h[3]\n"
      "fmla z16.h, z21.h, z0.h[3]\n"
      "fmla z9.h, z20.h, z2.h[3]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[3]\n"
      "fmla z17.h, z20.h, z0.h[3]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z21.h, z2.h[3]\n"
      "fmla z14.h, z21.h, z1.h[3]\n"
      "fmla z18.h, z21.h, z0.h[3]\n"
      "fmla z11.h, z20.h, z2.h[3]\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "fmla z15.h, z20.h, z1.h[3]\n"
      "fmla z19.h, z20.h, z0.h[3]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z2.h[4]\n"
      "fmla z12.h, z21.h, z1.h[4]\n"
      "fmla z16.h, z21.h, z0.h[4]\n"
      "fmla z9.h, z20.h, z2.h[4]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[4]\n"
      "fmla z17.h, z20.h, z0.h[4]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z21.h, z2.h[4]\n"
      "fmla z14.h, z21.h, z1.h[4]\n"
      "fmla z18.h, z21.h, z0.h[4]\n"
      "fmla z11.h, z20.h, z2.h[4]\n"
      "ld1h { z21.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z15.h, z20.h, z1.h[4]\n"
      "fmla z19.h, z20.h, z0.h[4]\n"
      "ld1h { z20.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z21.h, z2.h[5]\n"
      "fmla z12.h, z21.h, z1.h[5]\n"
      "fmla z16.h, z21.h, z0.h[5]\n"
      "fmla z9.h, z20.h, z2.h[5]\n"
      "ld1h { z21.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[5]\n"
      "fmla z17.h, z20.h, z0.h[5]\n"
      "ld1h { z20.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z21.h, z2.h[5]\n"
      "fmla z14.h, z21.h, z1.h[5]\n"
      "fmla z18.h, z21.h, z0.h[5]\n"
      "fmla z11.h, z20.h, z2.h[5]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z15.h, z20.h, z1.h[5]\n"
      "fmla z19.h, z20.h, z0.h[5]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z21.h, z2.h[6]\n"
      "fmla z12.h, z21.h, z1.h[6]\n"
      "fmla z16.h, z21.h, z0.h[6]\n"
      "fmla z9.h, z20.h, z2.h[6]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[6]\n"
      "fmla z17.h, z20.h, z0.h[6]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z21.h, z2.h[6]\n"
      "fmla z14.h, z21.h, z1.h[6]\n"
      "fmla z18.h, z21.h, z0.h[6]\n"
      "fmla z11.h, z20.h, z2.h[6]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z15.h, z20.h, z1.h[6]\n"
      "fmla z19.h, z20.h, z0.h[6]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z21.h, z2.h[7]\n"
      "fmla z12.h, z21.h, z1.h[7]\n"
      "fmla z16.h, z21.h, z0.h[7]\n"
      "fmla z9.h, z20.h, z2.h[7]\n"
      "ld1h { z21.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[7]\n"
      "fmla z17.h, z20.h, z0.h[7]\n"
      "ld1h { z20.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z21.h, z2.h[7]\n"
      "fmla z14.h, z21.h, z1.h[7]\n"
      "fmla z18.h, z21.h, z0.h[7]\n"
      "fmla z11.h, z20.h, z2.h[7]\n"
      "fmla z15.h, z20.h, z1.h[7]\n"
      "fmla z19.h, z20.h, z0.h[7]\n"
      "bgt 35b\n"
      "36:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z0.h }, p0/Z, [x26]\n"
      "ld1rqh { z1.h }, p0/Z, [x25]\n"
      "subs x27, x27, #0x1\n"
      "ld1rqh { z2.h }, p0/Z, [x24]\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "fmla z8.h, z21.h, z0.h[0]\n"
      "fmla z12.h, z21.h, z1.h[0]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z16.h, z21.h, z2.h[0]\n"
      "fmla z9.h, z20.h, z0.h[0]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[0]\n"
      "fmla z17.h, z20.h, z2.h[0]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[0]\n"
      "fmla z14.h, z21.h, z1.h[0]\n"
      "fmla z18.h, z21.h, z2.h[0]\n"
      "fmla z11.h, z20.h, z0.h[0]\n"
      "fmla z15.h, z20.h, z1.h[0]\n"
      "fmla z19.h, z20.h, z2.h[0]\n"
      "ble 37f\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z0.h[1]\n"
      "fmla z12.h, z21.h, z1.h[1]\n"
      "fmla z16.h, z21.h, z2.h[1]\n"
      "fmla z9.h, z20.h, z0.h[1]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z13.h, z20.h, z1.h[1]\n"
      "fmla z17.h, z20.h, z2.h[1]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[1]\n"
      "fmla z14.h, z21.h, z1.h[1]\n"
      "fmla z18.h, z21.h, z2.h[1]\n"
      "fmla z11.h, z20.h, z0.h[1]\n"
      "fmla z15.h, z20.h, z1.h[1]\n"
      "fmla z19.h, z20.h, z2.h[1]\n"
      "ble 37f\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z0.h[2]\n"
      "fmla z12.h, z21.h, z1.h[2]\n"
      "fmla z16.h, z21.h, z2.h[2]\n"
      "fmla z9.h, z20.h, z0.h[2]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z13.h, z20.h, z1.h[2]\n"
      "fmla z17.h, z20.h, z2.h[2]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[2]\n"
      "fmla z14.h, z21.h, z1.h[2]\n"
      "fmla z18.h, z21.h, z2.h[2]\n"
      "fmla z11.h, z20.h, z0.h[2]\n"
      "fmla z15.h, z20.h, z1.h[2]\n"
      "fmla z19.h, z20.h, z2.h[2]\n"
      "ble 37f\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z0.h[3]\n"
      "fmla z12.h, z21.h, z1.h[3]\n"
      "fmla z16.h, z21.h, z2.h[3]\n"
      "fmla z9.h, z20.h, z0.h[3]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z13.h, z20.h, z1.h[3]\n"
      "fmla z17.h, z20.h, z2.h[3]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[3]\n"
      "fmla z14.h, z21.h, z1.h[3]\n"
      "fmla z18.h, z21.h, z2.h[3]\n"
      "fmla z11.h, z20.h, z0.h[3]\n"
      "fmla z15.h, z20.h, z1.h[3]\n"
      "fmla z19.h, z20.h, z2.h[3]\n"
      "ble 37f\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z0.h[4]\n"
      "fmla z12.h, z21.h, z1.h[4]\n"
      "fmla z16.h, z21.h, z2.h[4]\n"
      "fmla z9.h, z20.h, z0.h[4]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z13.h, z20.h, z1.h[4]\n"
      "fmla z17.h, z20.h, z2.h[4]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[4]\n"
      "fmla z14.h, z21.h, z1.h[4]\n"
      "fmla z18.h, z21.h, z2.h[4]\n"
      "fmla z11.h, z20.h, z0.h[4]\n"
      "fmla z15.h, z20.h, z1.h[4]\n"
      "fmla z19.h, z20.h, z2.h[4]\n"
      "ble 37f\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z0.h[5]\n"
      "fmla z12.h, z21.h, z1.h[5]\n"
      "fmla z16.h, z21.h, z2.h[5]\n"
      "fmla z9.h, z20.h, z0.h[5]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z13.h, z20.h, z1.h[5]\n"
      "fmla z17.h, z20.h, z2.h[5]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[5]\n"
      "fmla z14.h, z21.h, z1.h[5]\n"
      "fmla z18.h, z21.h, z2.h[5]\n"
      "fmla z11.h, z20.h, z0.h[5]\n"
      "fmla z15.h, z20.h, z1.h[5]\n"
      "fmla z19.h, z20.h, z2.h[5]\n"
      "ble 37f\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z0.h[6]\n"
      "fmla z12.h, z21.h, z1.h[6]\n"
      "fmla z16.h, z21.h, z2.h[6]\n"
      "fmla z9.h, z20.h, z0.h[6]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z13.h, z20.h, z1.h[6]\n"
      "fmla z17.h, z20.h, z2.h[6]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[6]\n"
      "fmla z14.h, z21.h, z1.h[6]\n"
      "fmla z18.h, z21.h, z2.h[6]\n"
      "fmla z11.h, z20.h, z0.h[6]\n"
      "fmla z15.h, z20.h, z1.h[6]\n"
      "fmla z19.h, z20.h, z2.h[6]\n"
      "ble 37f\n"
      "ld1h { z21.h }, p5/Z, [x10]\n"
      "ld1h { z20.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z21.h, z0.h[7]\n"
      "fmla z12.h, z21.h, z1.h[7]\n"
      "fmla z16.h, z21.h, z2.h[7]\n"
      "fmla z9.h, z20.h, z0.h[7]\n"
      "ld1h { z21.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z20.h, z1.h[7]\n"
      "fmla z17.h, z20.h, z2.h[7]\n"
      "ld1h { z20.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z21.h, z0.h[7]\n"
      "fmla z14.h, z21.h, z1.h[7]\n"
      "fmla z18.h, z21.h, z2.h[7]\n"
      "fmla z11.h, z20.h, z0.h[7]\n"
      "fmla z15.h, z20.h, z1.h[7]\n"
      "fmla z19.h, z20.h, z2.h[7]\n"
      "37:"  // Height 3: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 32b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "tbz %x[flags], #1, 38f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rh { z21.h }, p5/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z20.h }, p5/Z, [x20]\n"
      "fmin z8.h, p5/M, z8.h, z21.h\n"
      "fmin z9.h, p5/M, z9.h, z21.h\n"
      "fmin z10.h, p5/M, z10.h, z21.h\n"
      "fmin z11.h, p5/M, z11.h, z21.h\n"
      "fmin z12.h, p5/M, z12.h, z21.h\n"
      "fmin z13.h, p5/M, z13.h, z21.h\n"
      "fmin z14.h, p5/M, z14.h, z21.h\n"
      "fmin z15.h, p5/M, z15.h, z21.h\n"
      "fmin z16.h, p5/M, z16.h, z21.h\n"
      "fmin z17.h, p5/M, z17.h, z21.h\n"
      "fmin z18.h, p5/M, z18.h, z21.h\n"
      "fmin z19.h, p5/M, z19.h, z21.h\n"
      "fmax z8.h, p5/M, z8.h, z20.h\n"
      "fmax z9.h, p5/M, z9.h, z20.h\n"
      "fmax z10.h, p5/M, z10.h, z20.h\n"
      "fmax z11.h, p5/M, z11.h, z20.h\n"
      "fmax z12.h, p5/M, z12.h, z20.h\n"
      "fmax z13.h, p5/M, z13.h, z20.h\n"
      "fmax z14.h, p5/M, z14.h, z20.h\n"
      "fmax z15.h, p5/M, z15.h, z20.h\n"
      "fmax z16.h, p5/M, z16.h, z20.h\n"
      "fmax z17.h, p5/M, z17.h, z20.h\n"
      "fmax z18.h, p5/M, z18.h, z20.h\n"
      "fmax z19.h, p5/M, z19.h, z20.h\n"
      "38:"  // Height 3: No activation
      "st1h { z8.h }, p4, [x9]\n"
      "st1h { z9.h }, p3, [x9, #1, MUL VL]\n"
      "st1h { z10.h }, p2, [x9, #2, MUL VL]\n"
      "st1h { z11.h }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1h { z12.h }, p4, [x25]\n"
      "st1h { z13.h }, p3, [x25, #1, MUL VL]\n"
      "st1h { z14.h }, p2, [x25, #2, MUL VL]\n"
      "st1h { z15.h }, p1, [x25, #3, MUL VL]\n"
      "st1h { z16.h }, p4, [x24]\n"
      "st1h { z17.h }, p3, [x24, #1, MUL VL]\n"
      "st1h { z18.h }, p2, [x24, #2, MUL VL]\n"
      "st1h { z19.h }, p1, [x24, #3, MUL VL]\n"
      "39:"  // Height 3: Writeback done
      "dech x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 28b\n"
      "b 80f\n"
      "40:"  // Height 4
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "41:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.h, x20, x11\n"
      "inch x20\n"
      "whilelt p3.h, x20, x11\n"
      "inch x20\n"
      "whilelt p2.h, x20, x11\n"
      "inch x20\n"
      "whilelt p1.h, x20, x11\n"
      "cbz x12, 42f\n"
      "ld1h { z8.h }, p5/Z, [x12]\n"
      "ld1h { z9.h }, p5/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1h { z10.h }, p5/Z, [x12, #2, MUL VL]\n"
      "ld1h { z11.h }, p5/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z20.d, z8.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "b 44f\n"
      "42:"  // Height 4: no bias
      "tbz %x[flags], #0, 43f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x22, x9, x20, LSL #1\n"
      "add x21, x22, x20, LSL #1\n"
      "ld1h { z8.h }, p4/Z, [x9]\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z9.h }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1h { z10.h }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z11.h }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1h { z12.h }, p4/Z, [x22]\n"
      "ld1h { z13.h }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1h { z14.h }, p2/Z, [x22, #2, MUL VL]\n"
      "ld1h { z15.h }, p1/Z, [x22, #3, MUL VL]\n"
      "ld1h { z16.h }, p4/Z, [x21]\n"
      "ld1h { z17.h }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1h { z19.h }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1h { z20.h }, p4/Z, [x20]\n"
      "ld1h { z21.h }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1h { z22.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [x20, #3, MUL VL]\n"
      "b 44f\n"
      "43:"  // Height 4: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "44:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "45:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 46f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 47f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "b 47f\n"
      "46:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "47:"  // Height 4: input setup done
      "cmp x27, #0x8\n"
      "ble 49f\n"
      "48:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z3.h }, p0/Z, [x26]\n"
      "ld1rqh { z2.h }, p0/Z, [x25]\n"
      "sub x27, x27, #0x8\n"
      "ld1rqh { z1.h }, p0/Z, [x24]\n"
      "ld1rqh { z0.h }, p0/Z, [x23]\n"
      "cmp x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[0]\n"
      "fmla z12.h, z25.h, z2.h[0]\n"
      "fmla z16.h, z25.h, z1.h[0]\n"
      "fmla z20.h, z25.h, z0.h[0]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "add x25, x25, #0x10\n"
      "fmla z9.h, z24.h, z3.h[0]\n"
      "fmla z13.h, z24.h, z2.h[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "fmla z17.h, z24.h, z1.h[0]\n"
      "fmla z21.h, z24.h, z0.h[0]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z25.h, z3.h[0]\n"
      "fmla z14.h, z25.h, z2.h[0]\n"
      "fmla z18.h, z25.h, z1.h[0]\n"
      "fmla z22.h, z25.h, z0.h[0]\n"
      "ld1h { z25.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z11.h, z24.h, z3.h[0]\n"
      "fmla z15.h, z24.h, z2.h[0]\n"
      "fmla z19.h, z24.h, z1.h[0]\n"
      "fmla z23.h, z24.h, z0.h[0]\n"
      "ld1h { z24.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[1]\n"
      "fmla z12.h, z25.h, z2.h[1]\n"
      "fmla z16.h, z25.h, z1.h[1]\n"
      "fmla z20.h, z25.h, z0.h[1]\n"
      "ld1h { z25.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z9.h, z24.h, z3.h[1]\n"
      "fmla z13.h, z24.h, z2.h[1]\n"
      "fmla z17.h, z24.h, z1.h[1]\n"
      "fmla z21.h, z24.h, z0.h[1]\n"
      "ld1h { z24.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z25.h, z3.h[1]\n"
      "fmla z14.h, z25.h, z2.h[1]\n"
      "fmla z18.h, z25.h, z1.h[1]\n"
      "fmla z22.h, z25.h, z0.h[1]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z11.h, z24.h, z3.h[1]\n"
      "fmla z15.h, z24.h, z2.h[1]\n"
      "fmla z19.h, z24.h, z1.h[1]\n"
      "fmla z23.h, z24.h, z0.h[1]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[2]\n"
      "fmla z12.h, z25.h, z2.h[2]\n"
      "fmla z16.h, z25.h, z1.h[2]\n"
      "fmla z20.h, z25.h, z0.h[2]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z9.h, z24.h, z3.h[2]\n"
      "fmla z13.h, z24.h, z2.h[2]\n"
      "fmla z17.h, z24.h, z1.h[2]\n"
      "fmla z21.h, z24.h, z0.h[2]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z25.h, z3.h[2]\n"
      "fmla z14.h, z25.h, z2.h[2]\n"
      "fmla z18.h, z25.h, z1.h[2]\n"
      "fmla z22.h, z25.h, z0.h[2]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z11.h, z24.h, z3.h[2]\n"
      "fmla z15.h, z24.h, z2.h[2]\n"
      "fmla z19.h, z24.h, z1.h[2]\n"
      "fmla z23.h, z24.h, z0.h[2]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[3]\n"
      "fmla z12.h, z25.h, z2.h[3]\n"
      "fmla z16.h, z25.h, z1.h[3]\n"
      "fmla z20.h, z25.h, z0.h[3]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z9.h, z24.h, z3.h[3]\n"
      "fmla z13.h, z24.h, z2.h[3]\n"
      "fmla z17.h, z24.h, z1.h[3]\n"
      "fmla z21.h, z24.h, z0.h[3]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z25.h, z3.h[3]\n"
      "fmla z14.h, z25.h, z2.h[3]\n"
      "fmla z18.h, z25.h, z1.h[3]\n"
      "fmla z22.h, z25.h, z0.h[3]\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "fmla z11.h, z24.h, z3.h[3]\n"
      "fmla z15.h, z24.h, z2.h[3]\n"
      "fmla z19.h, z24.h, z1.h[3]\n"
      "fmla z23.h, z24.h, z0.h[3]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[4]\n"
      "fmla z12.h, z25.h, z2.h[4]\n"
      "fmla z16.h, z25.h, z1.h[4]\n"
      "fmla z20.h, z25.h, z0.h[4]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z24.h, z3.h[4]\n"
      "fmla z13.h, z24.h, z2.h[4]\n"
      "fmla z17.h, z24.h, z1.h[4]\n"
      "fmla z21.h, z24.h, z0.h[4]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z25.h, z3.h[4]\n"
      "fmla z14.h, z25.h, z2.h[4]\n"
      "fmla z18.h, z25.h, z1.h[4]\n"
      "fmla z22.h, z25.h, z0.h[4]\n"
      "ld1h { z25.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z11.h, z24.h, z3.h[4]\n"
      "fmla z15.h, z24.h, z2.h[4]\n"
      "fmla z19.h, z24.h, z1.h[4]\n"
      "fmla z23.h, z24.h, z0.h[4]\n"
      "ld1h { z24.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[5]\n"
      "fmla z12.h, z25.h, z2.h[5]\n"
      "fmla z16.h, z25.h, z1.h[5]\n"
      "fmla z20.h, z25.h, z0.h[5]\n"
      "ld1h { z25.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z9.h, z24.h, z3.h[5]\n"
      "fmla z13.h, z24.h, z2.h[5]\n"
      "fmla z17.h, z24.h, z1.h[5]\n"
      "fmla z21.h, z24.h, z0.h[5]\n"
      "ld1h { z24.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z25.h, z3.h[5]\n"
      "fmla z14.h, z25.h, z2.h[5]\n"
      "fmla z18.h, z25.h, z1.h[5]\n"
      "fmla z22.h, z25.h, z0.h[5]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z11.h, z24.h, z3.h[5]\n"
      "fmla z15.h, z24.h, z2.h[5]\n"
      "fmla z19.h, z24.h, z1.h[5]\n"
      "fmla z23.h, z24.h, z0.h[5]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[6]\n"
      "fmla z12.h, z25.h, z2.h[6]\n"
      "fmla z16.h, z25.h, z1.h[6]\n"
      "fmla z20.h, z25.h, z0.h[6]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z9.h, z24.h, z3.h[6]\n"
      "fmla z13.h, z24.h, z2.h[6]\n"
      "fmla z17.h, z24.h, z1.h[6]\n"
      "fmla z21.h, z24.h, z0.h[6]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z25.h, z3.h[6]\n"
      "fmla z14.h, z25.h, z2.h[6]\n"
      "fmla z18.h, z25.h, z1.h[6]\n"
      "fmla z22.h, z25.h, z0.h[6]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z11.h, z24.h, z3.h[6]\n"
      "fmla z15.h, z24.h, z2.h[6]\n"
      "fmla z19.h, z24.h, z1.h[6]\n"
      "fmla z23.h, z24.h, z0.h[6]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z25.h, z3.h[7]\n"
      "fmla z12.h, z25.h, z2.h[7]\n"
      "fmla z16.h, z25.h, z1.h[7]\n"
      "fmla z20.h, z25.h, z0.h[7]\n"
      "ld1h { z25.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z9.h, z24.h, z3.h[7]\n"
      "fmla z13.h, z24.h, z2.h[7]\n"
      "fmla z17.h, z24.h, z1.h[7]\n"
      "fmla z21.h, z24.h, z0.h[7]\n"
      "ld1h { z24.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z25.h, z3.h[7]\n"
      "fmla z14.h, z25.h, z2.h[7]\n"
      "fmla z18.h, z25.h, z1.h[7]\n"
      "fmla z22.h, z25.h, z0.h[7]\n"
      "fmla z11.h, z24.h, z3.h[7]\n"
      "fmla z15.h, z24.h, z2.h[7]\n"
      "fmla z19.h, z24.h, z1.h[7]\n"
      "fmla z23.h, z24.h, z0.h[7]\n"
      "bgt 48b\n"
      "49:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z0.h }, p0/Z, [x26]\n"
      "ld1rqh { z1.h }, p0/Z, [x25]\n"
      "subs x27, x27, #0x1\n"
      "ld1rqh { z2.h }, p0/Z, [x24]\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[0]\n"
      "fmla z12.h, z25.h, z1.h[0]\n"
      "fmla z16.h, z25.h, z2.h[0]\n"
      "fmla z20.h, z25.h, z3.h[0]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z24.h, z0.h[0]\n"
      "fmla z13.h, z24.h, z1.h[0]\n"
      "fmla z17.h, z24.h, z2.h[0]\n"
      "fmla z21.h, z24.h, z3.h[0]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[0]\n"
      "fmla z14.h, z25.h, z1.h[0]\n"
      "fmla z18.h, z25.h, z2.h[0]\n"
      "fmla z22.h, z25.h, z3.h[0]\n"
      "fmla z11.h, z24.h, z0.h[0]\n"
      "fmla z15.h, z24.h, z1.h[0]\n"
      "fmla z19.h, z24.h, z2.h[0]\n"
      "fmla z23.h, z24.h, z3.h[0]\n"
      "ble 50f\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[1]\n"
      "fmla z12.h, z25.h, z1.h[1]\n"
      "fmla z16.h, z25.h, z2.h[1]\n"
      "fmla z20.h, z25.h, z3.h[1]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z9.h, z24.h, z0.h[1]\n"
      "fmla z13.h, z24.h, z1.h[1]\n"
      "fmla z17.h, z24.h, z2.h[1]\n"
      "fmla z21.h, z24.h, z3.h[1]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[1]\n"
      "fmla z14.h, z25.h, z1.h[1]\n"
      "fmla z18.h, z25.h, z2.h[1]\n"
      "fmla z22.h, z25.h, z3.h[1]\n"
      "fmla z11.h, z24.h, z0.h[1]\n"
      "fmla z15.h, z24.h, z1.h[1]\n"
      "fmla z19.h, z24.h, z2.h[1]\n"
      "fmla z23.h, z24.h, z3.h[1]\n"
      "ble 50f\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[2]\n"
      "fmla z12.h, z25.h, z1.h[2]\n"
      "fmla z16.h, z25.h, z2.h[2]\n"
      "fmla z20.h, z25.h, z3.h[2]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z9.h, z24.h, z0.h[2]\n"
      "fmla z13.h, z24.h, z1.h[2]\n"
      "fmla z17.h, z24.h, z2.h[2]\n"
      "fmla z21.h, z24.h, z3.h[2]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[2]\n"
      "fmla z14.h, z25.h, z1.h[2]\n"
      "fmla z18.h, z25.h, z2.h[2]\n"
      "fmla z22.h, z25.h, z3.h[2]\n"
      "fmla z11.h, z24.h, z0.h[2]\n"
      "fmla z15.h, z24.h, z1.h[2]\n"
      "fmla z19.h, z24.h, z2.h[2]\n"
      "fmla z23.h, z24.h, z3.h[2]\n"
      "ble 50f\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[3]\n"
      "fmla z12.h, z25.h, z1.h[3]\n"
      "fmla z16.h, z25.h, z2.h[3]\n"
      "fmla z20.h, z25.h, z3.h[3]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z9.h, z24.h, z0.h[3]\n"
      "fmla z13.h, z24.h, z1.h[3]\n"
      "fmla z17.h, z24.h, z2.h[3]\n"
      "fmla z21.h, z24.h, z3.h[3]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[3]\n"
      "fmla z14.h, z25.h, z1.h[3]\n"
      "fmla z18.h, z25.h, z2.h[3]\n"
      "fmla z22.h, z25.h, z3.h[3]\n"
      "fmla z11.h, z24.h, z0.h[3]\n"
      "fmla z15.h, z24.h, z1.h[3]\n"
      "fmla z19.h, z24.h, z2.h[3]\n"
      "fmla z23.h, z24.h, z3.h[3]\n"
      "ble 50f\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[4]\n"
      "fmla z12.h, z25.h, z1.h[4]\n"
      "fmla z16.h, z25.h, z2.h[4]\n"
      "fmla z20.h, z25.h, z3.h[4]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z9.h, z24.h, z0.h[4]\n"
      "fmla z13.h, z24.h, z1.h[4]\n"
      "fmla z17.h, z24.h, z2.h[4]\n"
      "fmla z21.h, z24.h, z3.h[4]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[4]\n"
      "fmla z14.h, z25.h, z1.h[4]\n"
      "fmla z18.h, z25.h, z2.h[4]\n"
      "fmla z22.h, z25.h, z3.h[4]\n"
      "fmla z11.h, z24.h, z0.h[4]\n"
      "fmla z15.h, z24.h, z1.h[4]\n"
      "fmla z19.h, z24.h, z2.h[4]\n"
      "fmla z23.h, z24.h, z3.h[4]\n"
      "ble 50f\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[5]\n"
      "fmla z12.h, z25.h, z1.h[5]\n"
      "fmla z16.h, z25.h, z2.h[5]\n"
      "fmla z20.h, z25.h, z3.h[5]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z9.h, z24.h, z0.h[5]\n"
      "fmla z13.h, z24.h, z1.h[5]\n"
      "fmla z17.h, z24.h, z2.h[5]\n"
      "fmla z21.h, z24.h, z3.h[5]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[5]\n"
      "fmla z14.h, z25.h, z1.h[5]\n"
      "fmla z18.h, z25.h, z2.h[5]\n"
      "fmla z22.h, z25.h, z3.h[5]\n"
      "fmla z11.h, z24.h, z0.h[5]\n"
      "fmla z15.h, z24.h, z1.h[5]\n"
      "fmla z19.h, z24.h, z2.h[5]\n"
      "fmla z23.h, z24.h, z3.h[5]\n"
      "ble 50f\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[6]\n"
      "fmla z12.h, z25.h, z1.h[6]\n"
      "fmla z16.h, z25.h, z2.h[6]\n"
      "fmla z20.h, z25.h, z3.h[6]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "subs x27, x27, #0x1\n"
      "fmla z9.h, z24.h, z0.h[6]\n"
      "fmla z13.h, z24.h, z1.h[6]\n"
      "fmla z17.h, z24.h, z2.h[6]\n"
      "fmla z21.h, z24.h, z3.h[6]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[6]\n"
      "fmla z14.h, z25.h, z1.h[6]\n"
      "fmla z18.h, z25.h, z2.h[6]\n"
      "fmla z22.h, z25.h, z3.h[6]\n"
      "fmla z11.h, z24.h, z0.h[6]\n"
      "fmla z15.h, z24.h, z1.h[6]\n"
      "fmla z19.h, z24.h, z2.h[6]\n"
      "fmla z23.h, z24.h, z3.h[6]\n"
      "ble 50f\n"
      "ld1h { z25.h }, p5/Z, [x10]\n"
      "ld1h { z24.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z25.h, z0.h[7]\n"
      "fmla z12.h, z25.h, z1.h[7]\n"
      "fmla z16.h, z25.h, z2.h[7]\n"
      "fmla z20.h, z25.h, z3.h[7]\n"
      "ld1h { z25.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z24.h, z0.h[7]\n"
      "fmla z13.h, z24.h, z1.h[7]\n"
      "fmla z17.h, z24.h, z2.h[7]\n"
      "fmla z21.h, z24.h, z3.h[7]\n"
      "ld1h { z24.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z25.h, z0.h[7]\n"
      "fmla z14.h, z25.h, z1.h[7]\n"
      "fmla z18.h, z25.h, z2.h[7]\n"
      "fmla z22.h, z25.h, z3.h[7]\n"
      "fmla z11.h, z24.h, z0.h[7]\n"
      "fmla z15.h, z24.h, z1.h[7]\n"
      "fmla z19.h, z24.h, z2.h[7]\n"
      "fmla z23.h, z24.h, z3.h[7]\n"
      "50:"  // Height 4: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 45b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "tbz %x[flags], #1, 51f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rh { z25.h }, p5/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z24.h }, p5/Z, [x20]\n"
      "fmin z8.h, p5/M, z8.h, z25.h\n"
      "fmin z9.h, p5/M, z9.h, z25.h\n"
      "fmin z10.h, p5/M, z10.h, z25.h\n"
      "fmin z11.h, p5/M, z11.h, z25.h\n"
      "fmin z12.h, p5/M, z12.h, z25.h\n"
      "fmin z13.h, p5/M, z13.h, z25.h\n"
      "fmin z14.h, p5/M, z14.h, z25.h\n"
      "fmin z15.h, p5/M, z15.h, z25.h\n"
      "fmin z16.h, p5/M, z16.h, z25.h\n"
      "fmin z17.h, p5/M, z17.h, z25.h\n"
      "fmin z18.h, p5/M, z18.h, z25.h\n"
      "fmin z19.h, p5/M, z19.h, z25.h\n"
      "fmin z20.h, p5/M, z20.h, z25.h\n"
      "fmin z21.h, p5/M, z21.h, z25.h\n"
      "fmin z22.h, p5/M, z22.h, z25.h\n"
      "fmin z23.h, p5/M, z23.h, z25.h\n"
      "fmax z8.h, p5/M, z8.h, z24.h\n"
      "fmax z9.h, p5/M, z9.h, z24.h\n"
      "fmax z10.h, p5/M, z10.h, z24.h\n"
      "fmax z11.h, p5/M, z11.h, z24.h\n"
      "fmax z12.h, p5/M, z12.h, z24.h\n"
      "fmax z13.h, p5/M, z13.h, z24.h\n"
      "fmax z14.h, p5/M, z14.h, z24.h\n"
      "fmax z15.h, p5/M, z15.h, z24.h\n"
      "fmax z16.h, p5/M, z16.h, z24.h\n"
      "fmax z17.h, p5/M, z17.h, z24.h\n"
      "fmax z18.h, p5/M, z18.h, z24.h\n"
      "fmax z19.h, p5/M, z19.h, z24.h\n"
      "fmax z20.h, p5/M, z20.h, z24.h\n"
      "fmax z21.h, p5/M, z21.h, z24.h\n"
      "fmax z22.h, p5/M, z22.h, z24.h\n"
      "fmax z23.h, p5/M, z23.h, z24.h\n"
      "51:"  // Height 4: No activation
      "st1h { z8.h }, p4, [x9]\n"
      "st1h { z9.h }, p3, [x9, #1, MUL VL]\n"
      "st1h { z10.h }, p2, [x9, #2, MUL VL]\n"
      "st1h { z11.h }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1h { z12.h }, p4, [x25]\n"
      "st1h { z13.h }, p3, [x25, #1, MUL VL]\n"
      "st1h { z14.h }, p2, [x25, #2, MUL VL]\n"
      "st1h { z15.h }, p1, [x25, #3, MUL VL]\n"
      "st1h { z16.h }, p4, [x24]\n"
      "st1h { z17.h }, p3, [x24, #1, MUL VL]\n"
      "st1h { z18.h }, p2, [x24, #2, MUL VL]\n"
      "st1h { z19.h }, p1, [x24, #3, MUL VL]\n"
      "st1h { z20.h }, p4, [x23]\n"
      "st1h { z21.h }, p3, [x23, #1, MUL VL]\n"
      "st1h { z22.h }, p2, [x23, #2, MUL VL]\n"
      "st1h { z23.h }, p1, [x23, #3, MUL VL]\n"
      "52:"  // Height 4: Writeback done
      "dech x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 41b\n"
      "b 80f\n"
      "53:"  // Height 5
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "54:"  // Height 5: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.h, x20, x11\n"
      "inch x20\n"
      "whilelt p3.h, x20, x11\n"
      "inch x20\n"
      "whilelt p2.h, x20, x11\n"
      "inch x20\n"
      "whilelt p1.h, x20, x11\n"
      "cbz x12, 55f\n"
      "ld1h { z8.h }, p5/Z, [x12]\n"
      "ld1h { z9.h }, p5/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1h { z10.h }, p5/Z, [x12, #2, MUL VL]\n"
      "ld1h { z11.h }, p5/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z20.d, z8.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "b 57f\n"
      "55:"  // Height 5: no bias
      "tbz %x[flags], #0, 56f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x9, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "ld1h { z8.h }, p4/Z, [x9]\n"
      "add x21, x22, x20, LSL #1\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z9.h }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1h { z10.h }, p2/Z, [x9, #2, MUL VL]\n"
      "ld1h { z11.h }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1h { z12.h }, p4/Z, [x23]\n"
      "ld1h { z13.h }, p3/Z, [x23, #1, MUL VL]\n"
      "ld1h { z14.h }, p2/Z, [x23, #2, MUL VL]\n"
      "ld1h { z15.h }, p1/Z, [x23, #3, MUL VL]\n"
      "ld1h { z16.h }, p4/Z, [x22]\n"
      "ld1h { z17.h }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x22, #2, MUL VL]\n"
      "ld1h { z19.h }, p1/Z, [x22, #3, MUL VL]\n"
      "ld1h { z20.h }, p4/Z, [x21]\n"
      "ld1h { z21.h }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1h { z22.h }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1h { z24.h }, p4/Z, [x20]\n"
      "ld1h { z25.h }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z27.h }, p1/Z, [x20, #3, MUL VL]\n"
      "b 57f\n"
      "56:"  // Height 5: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "57:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "58:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 59f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 60f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "b 60f\n"
      "59:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "60:"  // Height 5: input setup done
      "cmp x27, #0x8\n"
      "ble 62f\n"
      "61:"  // Height 5: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z4.h }, p0/Z, [x26]\n"
      "ld1rqh { z3.h }, p0/Z, [x25]\n"
      "sub x27, x27, #0x8\n"
      "ld1rqh { z2.h }, p0/Z, [x24]\n"
      "ld1rqh { z1.h }, p0/Z, [x23]\n"
      "cmp x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "ld1rqh { z0.h }, p0/Z, [x22]\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "fmla z8.h, z29.h, z4.h[0]\n"
      "fmla z12.h, z29.h, z3.h[0]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z16.h, z29.h, z2.h[0]\n"
      "fmla z20.h, z29.h, z1.h[0]\n"
      "add x25, x25, #0x10\n"
      "fmla z24.h, z29.h, z0.h[0]\n"
      "fmla z9.h, z28.h, z4.h[0]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "add x24, x24, #0x10\n"
      "fmla z13.h, z28.h, z3.h[0]\n"
      "fmla z17.h, z28.h, z2.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "fmla z21.h, z28.h, z1.h[0]\n"
      "fmla z25.h, z28.h, z0.h[0]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z29.h, z4.h[0]\n"
      "fmla z14.h, z29.h, z3.h[0]\n"
      "fmla z18.h, z29.h, z2.h[0]\n"
      "fmla z22.h, z29.h, z1.h[0]\n"
      "fmla z26.h, z29.h, z0.h[0]\n"
      "fmla z11.h, z28.h, z4.h[0]\n"
      "ld1h { z29.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z15.h, z28.h, z3.h[0]\n"
      "fmla z19.h, z28.h, z2.h[0]\n"
      "fmla z23.h, z28.h, z1.h[0]\n"
      "fmla z27.h, z28.h, z0.h[0]\n"
      "ld1h { z28.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z29.h, z4.h[1]\n"
      "fmla z12.h, z29.h, z3.h[1]\n"
      "fmla z16.h, z29.h, z2.h[1]\n"
      "fmla z20.h, z29.h, z1.h[1]\n"
      "fmla z24.h, z29.h, z0.h[1]\n"
      "fmla z9.h, z28.h, z4.h[1]\n"
      "ld1h { z29.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z13.h, z28.h, z3.h[1]\n"
      "fmla z17.h, z28.h, z2.h[1]\n"
      "fmla z21.h, z28.h, z1.h[1]\n"
      "fmla z25.h, z28.h, z0.h[1]\n"
      "ld1h { z28.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z29.h, z4.h[1]\n"
      "fmla z14.h, z29.h, z3.h[1]\n"
      "fmla z18.h, z29.h, z2.h[1]\n"
      "fmla z22.h, z29.h, z1.h[1]\n"
      "fmla z26.h, z29.h, z0.h[1]\n"
      "fmla z11.h, z28.h, z4.h[1]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z15.h, z28.h, z3.h[1]\n"
      "fmla z19.h, z28.h, z2.h[1]\n"
      "fmla z23.h, z28.h, z1.h[1]\n"
      "fmla z27.h, z28.h, z0.h[1]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z29.h, z4.h[2]\n"
      "fmla z12.h, z29.h, z3.h[2]\n"
      "fmla z16.h, z29.h, z2.h[2]\n"
      "fmla z20.h, z29.h, z1.h[2]\n"
      "fmla z24.h, z29.h, z0.h[2]\n"
      "fmla z9.h, z28.h, z4.h[2]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z13.h, z28.h, z3.h[2]\n"
      "fmla z17.h, z28.h, z2.h[2]\n"
      "fmla z21.h, z28.h, z1.h[2]\n"
      "fmla z25.h, z28.h, z0.h[2]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z29.h, z4.h[2]\n"
      "fmla z14.h, z29.h, z3.h[2]\n"
      "fmla z18.h, z29.h, z2.h[2]\n"
      "fmla z22.h, z29.h, z1.h[2]\n"
      "fmla z26.h, z29.h, z0.h[2]\n"
      "fmla z11.h, z28.h, z4.h[2]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z15.h, z28.h, z3.h[2]\n"
      "fmla z19.h, z28.h, z2.h[2]\n"
      "fmla z23.h, z28.h, z1.h[2]\n"
      "fmla z27.h, z28.h, z0.h[2]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z29.h, z4.h[3]\n"
      "fmla z12.h, z29.h, z3.h[3]\n"
      "fmla z16.h, z29.h, z2.h[3]\n"
      "fmla z20.h, z29.h, z1.h[3]\n"
      "fmla z24.h, z29.h, z0.h[3]\n"
      "fmla z9.h, z28.h, z4.h[3]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z13.h, z28.h, z3.h[3]\n"
      "fmla z17.h, z28.h, z2.h[3]\n"
      "fmla z21.h, z28.h, z1.h[3]\n"
      "fmla z25.h, z28.h, z0.h[3]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z29.h, z4.h[3]\n"
      "fmla z14.h, z29.h, z3.h[3]\n"
      "fmla z18.h, z29.h, z2.h[3]\n"
      "fmla z22.h, z29.h, z1.h[3]\n"
      "fmla z26.h, z29.h, z0.h[3]\n"
      "fmla z11.h, z28.h, z4.h[3]\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "fmla z15.h, z28.h, z3.h[3]\n"
      "fmla z19.h, z28.h, z2.h[3]\n"
      "fmla z23.h, z28.h, z1.h[3]\n"
      "fmla z27.h, z28.h, z0.h[3]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z4.h[4]\n"
      "fmla z12.h, z29.h, z3.h[4]\n"
      "fmla z16.h, z29.h, z2.h[4]\n"
      "fmla z20.h, z29.h, z1.h[4]\n"
      "fmla z24.h, z29.h, z0.h[4]\n"
      "fmla z9.h, z28.h, z4.h[4]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z3.h[4]\n"
      "fmla z17.h, z28.h, z2.h[4]\n"
      "fmla z21.h, z28.h, z1.h[4]\n"
      "fmla z25.h, z28.h, z0.h[4]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z29.h, z4.h[4]\n"
      "fmla z14.h, z29.h, z3.h[4]\n"
      "fmla z18.h, z29.h, z2.h[4]\n"
      "fmla z22.h, z29.h, z1.h[4]\n"
      "fmla z26.h, z29.h, z0.h[4]\n"
      "fmla z11.h, z28.h, z4.h[4]\n"
      "ld1h { z29.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z15.h, z28.h, z3.h[4]\n"
      "fmla z19.h, z28.h, z2.h[4]\n"
      "fmla z23.h, z28.h, z1.h[4]\n"
      "fmla z27.h, z28.h, z0.h[4]\n"
      "ld1h { z28.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z29.h, z4.h[5]\n"
      "fmla z12.h, z29.h, z3.h[5]\n"
      "fmla z16.h, z29.h, z2.h[5]\n"
      "fmla z20.h, z29.h, z1.h[5]\n"
      "fmla z24.h, z29.h, z0.h[5]\n"
      "fmla z9.h, z28.h, z4.h[5]\n"
      "ld1h { z29.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z13.h, z28.h, z3.h[5]\n"
      "fmla z17.h, z28.h, z2.h[5]\n"
      "fmla z21.h, z28.h, z1.h[5]\n"
      "fmla z25.h, z28.h, z0.h[5]\n"
      "ld1h { z28.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z29.h, z4.h[5]\n"
      "fmla z14.h, z29.h, z3.h[5]\n"
      "fmla z18.h, z29.h, z2.h[5]\n"
      "fmla z22.h, z29.h, z1.h[5]\n"
      "fmla z26.h, z29.h, z0.h[5]\n"
      "fmla z11.h, z28.h, z4.h[5]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z15.h, z28.h, z3.h[5]\n"
      "fmla z19.h, z28.h, z2.h[5]\n"
      "fmla z23.h, z28.h, z1.h[5]\n"
      "fmla z27.h, z28.h, z0.h[5]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z29.h, z4.h[6]\n"
      "fmla z12.h, z29.h, z3.h[6]\n"
      "fmla z16.h, z29.h, z2.h[6]\n"
      "fmla z20.h, z29.h, z1.h[6]\n"
      "fmla z24.h, z29.h, z0.h[6]\n"
      "fmla z9.h, z28.h, z4.h[6]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z13.h, z28.h, z3.h[6]\n"
      "fmla z17.h, z28.h, z2.h[6]\n"
      "fmla z21.h, z28.h, z1.h[6]\n"
      "fmla z25.h, z28.h, z0.h[6]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z29.h, z4.h[6]\n"
      "fmla z14.h, z29.h, z3.h[6]\n"
      "fmla z18.h, z29.h, z2.h[6]\n"
      "fmla z22.h, z29.h, z1.h[6]\n"
      "fmla z26.h, z29.h, z0.h[6]\n"
      "fmla z11.h, z28.h, z4.h[6]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z15.h, z28.h, z3.h[6]\n"
      "fmla z19.h, z28.h, z2.h[6]\n"
      "fmla z23.h, z28.h, z1.h[6]\n"
      "fmla z27.h, z28.h, z0.h[6]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z29.h, z4.h[7]\n"
      "fmla z12.h, z29.h, z3.h[7]\n"
      "fmla z16.h, z29.h, z2.h[7]\n"
      "fmla z20.h, z29.h, z1.h[7]\n"
      "fmla z24.h, z29.h, z0.h[7]\n"
      "fmla z9.h, z28.h, z4.h[7]\n"
      "ld1h { z29.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z13.h, z28.h, z3.h[7]\n"
      "fmla z17.h, z28.h, z2.h[7]\n"
      "fmla z21.h, z28.h, z1.h[7]\n"
      "fmla z25.h, z28.h, z0.h[7]\n"
      "ld1h { z28.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z29.h, z4.h[7]\n"
      "fmla z14.h, z29.h, z3.h[7]\n"
      "fmla z18.h, z29.h, z2.h[7]\n"
      "fmla z22.h, z29.h, z1.h[7]\n"
      "fmla z26.h, z29.h, z0.h[7]\n"
      "fmla z11.h, z28.h, z4.h[7]\n"
      "fmla z15.h, z28.h, z3.h[7]\n"
      "fmla z19.h, z28.h, z2.h[7]\n"
      "fmla z23.h, z28.h, z1.h[7]\n"
      "fmla z27.h, z28.h, z0.h[7]\n"
      "bgt 61b\n"
      "62:"  // Height 5: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z0.h }, p0/Z, [x26]\n"
      "ld1rqh { z1.h }, p0/Z, [x25]\n"
      "subs x27, x27, #0x1\n"
      "ld1rqh { z2.h }, p0/Z, [x24]\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "ld1rqh { z4.h }, p0/Z, [x22]\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "fmla z8.h, z29.h, z0.h[0]\n"
      "fmla z12.h, z29.h, z1.h[0]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z16.h, z29.h, z2.h[0]\n"
      "fmla z20.h, z29.h, z3.h[0]\n"
      "fmla z24.h, z29.h, z4.h[0]\n"
      "fmla z9.h, z28.h, z0.h[0]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[0]\n"
      "fmla z17.h, z28.h, z2.h[0]\n"
      "fmla z21.h, z28.h, z3.h[0]\n"
      "fmla z25.h, z28.h, z4.h[0]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[0]\n"
      "fmla z14.h, z29.h, z1.h[0]\n"
      "fmla z18.h, z29.h, z2.h[0]\n"
      "fmla z22.h, z29.h, z3.h[0]\n"
      "fmla z26.h, z29.h, z4.h[0]\n"
      "fmla z11.h, z28.h, z0.h[0]\n"
      "fmla z15.h, z28.h, z1.h[0]\n"
      "fmla z19.h, z28.h, z2.h[0]\n"
      "fmla z23.h, z28.h, z3.h[0]\n"
      "fmla z27.h, z28.h, z4.h[0]\n"
      "ble 63f\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z0.h[1]\n"
      "fmla z12.h, z29.h, z1.h[1]\n"
      "fmla z16.h, z29.h, z2.h[1]\n"
      "fmla z20.h, z29.h, z3.h[1]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z29.h, z4.h[1]\n"
      "fmla z9.h, z28.h, z0.h[1]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[1]\n"
      "fmla z17.h, z28.h, z2.h[1]\n"
      "fmla z21.h, z28.h, z3.h[1]\n"
      "fmla z25.h, z28.h, z4.h[1]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[1]\n"
      "fmla z14.h, z29.h, z1.h[1]\n"
      "fmla z18.h, z29.h, z2.h[1]\n"
      "fmla z22.h, z29.h, z3.h[1]\n"
      "fmla z26.h, z29.h, z4.h[1]\n"
      "fmla z11.h, z28.h, z0.h[1]\n"
      "fmla z15.h, z28.h, z1.h[1]\n"
      "fmla z19.h, z28.h, z2.h[1]\n"
      "fmla z23.h, z28.h, z3.h[1]\n"
      "fmla z27.h, z28.h, z4.h[1]\n"
      "ble 63f\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z0.h[2]\n"
      "fmla z12.h, z29.h, z1.h[2]\n"
      "fmla z16.h, z29.h, z2.h[2]\n"
      "fmla z20.h, z29.h, z3.h[2]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z29.h, z4.h[2]\n"
      "fmla z9.h, z28.h, z0.h[2]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[2]\n"
      "fmla z17.h, z28.h, z2.h[2]\n"
      "fmla z21.h, z28.h, z3.h[2]\n"
      "fmla z25.h, z28.h, z4.h[2]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[2]\n"
      "fmla z14.h, z29.h, z1.h[2]\n"
      "fmla z18.h, z29.h, z2.h[2]\n"
      "fmla z22.h, z29.h, z3.h[2]\n"
      "fmla z26.h, z29.h, z4.h[2]\n"
      "fmla z11.h, z28.h, z0.h[2]\n"
      "fmla z15.h, z28.h, z1.h[2]\n"
      "fmla z19.h, z28.h, z2.h[2]\n"
      "fmla z23.h, z28.h, z3.h[2]\n"
      "fmla z27.h, z28.h, z4.h[2]\n"
      "ble 63f\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z0.h[3]\n"
      "fmla z12.h, z29.h, z1.h[3]\n"
      "fmla z16.h, z29.h, z2.h[3]\n"
      "fmla z20.h, z29.h, z3.h[3]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z29.h, z4.h[3]\n"
      "fmla z9.h, z28.h, z0.h[3]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[3]\n"
      "fmla z17.h, z28.h, z2.h[3]\n"
      "fmla z21.h, z28.h, z3.h[3]\n"
      "fmla z25.h, z28.h, z4.h[3]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[3]\n"
      "fmla z14.h, z29.h, z1.h[3]\n"
      "fmla z18.h, z29.h, z2.h[3]\n"
      "fmla z22.h, z29.h, z3.h[3]\n"
      "fmla z26.h, z29.h, z4.h[3]\n"
      "fmla z11.h, z28.h, z0.h[3]\n"
      "fmla z15.h, z28.h, z1.h[3]\n"
      "fmla z19.h, z28.h, z2.h[3]\n"
      "fmla z23.h, z28.h, z3.h[3]\n"
      "fmla z27.h, z28.h, z4.h[3]\n"
      "ble 63f\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z0.h[4]\n"
      "fmla z12.h, z29.h, z1.h[4]\n"
      "fmla z16.h, z29.h, z2.h[4]\n"
      "fmla z20.h, z29.h, z3.h[4]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z29.h, z4.h[4]\n"
      "fmla z9.h, z28.h, z0.h[4]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[4]\n"
      "fmla z17.h, z28.h, z2.h[4]\n"
      "fmla z21.h, z28.h, z3.h[4]\n"
      "fmla z25.h, z28.h, z4.h[4]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[4]\n"
      "fmla z14.h, z29.h, z1.h[4]\n"
      "fmla z18.h, z29.h, z2.h[4]\n"
      "fmla z22.h, z29.h, z3.h[4]\n"
      "fmla z26.h, z29.h, z4.h[4]\n"
      "fmla z11.h, z28.h, z0.h[4]\n"
      "fmla z15.h, z28.h, z1.h[4]\n"
      "fmla z19.h, z28.h, z2.h[4]\n"
      "fmla z23.h, z28.h, z3.h[4]\n"
      "fmla z27.h, z28.h, z4.h[4]\n"
      "ble 63f\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z0.h[5]\n"
      "fmla z12.h, z29.h, z1.h[5]\n"
      "fmla z16.h, z29.h, z2.h[5]\n"
      "fmla z20.h, z29.h, z3.h[5]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z29.h, z4.h[5]\n"
      "fmla z9.h, z28.h, z0.h[5]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[5]\n"
      "fmla z17.h, z28.h, z2.h[5]\n"
      "fmla z21.h, z28.h, z3.h[5]\n"
      "fmla z25.h, z28.h, z4.h[5]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[5]\n"
      "fmla z14.h, z29.h, z1.h[5]\n"
      "fmla z18.h, z29.h, z2.h[5]\n"
      "fmla z22.h, z29.h, z3.h[5]\n"
      "fmla z26.h, z29.h, z4.h[5]\n"
      "fmla z11.h, z28.h, z0.h[5]\n"
      "fmla z15.h, z28.h, z1.h[5]\n"
      "fmla z19.h, z28.h, z2.h[5]\n"
      "fmla z23.h, z28.h, z3.h[5]\n"
      "fmla z27.h, z28.h, z4.h[5]\n"
      "ble 63f\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z0.h[6]\n"
      "fmla z12.h, z29.h, z1.h[6]\n"
      "fmla z16.h, z29.h, z2.h[6]\n"
      "fmla z20.h, z29.h, z3.h[6]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z29.h, z4.h[6]\n"
      "fmla z9.h, z28.h, z0.h[6]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[6]\n"
      "fmla z17.h, z28.h, z2.h[6]\n"
      "fmla z21.h, z28.h, z3.h[6]\n"
      "fmla z25.h, z28.h, z4.h[6]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[6]\n"
      "fmla z14.h, z29.h, z1.h[6]\n"
      "fmla z18.h, z29.h, z2.h[6]\n"
      "fmla z22.h, z29.h, z3.h[6]\n"
      "fmla z26.h, z29.h, z4.h[6]\n"
      "fmla z11.h, z28.h, z0.h[6]\n"
      "fmla z15.h, z28.h, z1.h[6]\n"
      "fmla z19.h, z28.h, z2.h[6]\n"
      "fmla z23.h, z28.h, z3.h[6]\n"
      "fmla z27.h, z28.h, z4.h[6]\n"
      "ble 63f\n"
      "ld1h { z29.h }, p5/Z, [x10]\n"
      "ld1h { z28.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z29.h, z0.h[7]\n"
      "fmla z12.h, z29.h, z1.h[7]\n"
      "fmla z16.h, z29.h, z2.h[7]\n"
      "fmla z20.h, z29.h, z3.h[7]\n"
      "fmla z24.h, z29.h, z4.h[7]\n"
      "fmla z9.h, z28.h, z0.h[7]\n"
      "ld1h { z29.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z13.h, z28.h, z1.h[7]\n"
      "fmla z17.h, z28.h, z2.h[7]\n"
      "fmla z21.h, z28.h, z3.h[7]\n"
      "fmla z25.h, z28.h, z4.h[7]\n"
      "ld1h { z28.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z29.h, z0.h[7]\n"
      "fmla z14.h, z29.h, z1.h[7]\n"
      "fmla z18.h, z29.h, z2.h[7]\n"
      "fmla z22.h, z29.h, z3.h[7]\n"
      "fmla z26.h, z29.h, z4.h[7]\n"
      "fmla z11.h, z28.h, z0.h[7]\n"
      "fmla z15.h, z28.h, z1.h[7]\n"
      "fmla z19.h, z28.h, z2.h[7]\n"
      "fmla z23.h, z28.h, z3.h[7]\n"
      "fmla z27.h, z28.h, z4.h[7]\n"
      "63:"  // Height 5: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 58b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "tbz %x[flags], #1, 64f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rh { z29.h }, p5/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z28.h }, p5/Z, [x20]\n"
      "fmin z8.h, p5/M, z8.h, z29.h\n"
      "fmin z9.h, p5/M, z9.h, z29.h\n"
      "fmin z10.h, p5/M, z10.h, z29.h\n"
      "fmin z11.h, p5/M, z11.h, z29.h\n"
      "fmin z12.h, p5/M, z12.h, z29.h\n"
      "fmin z13.h, p5/M, z13.h, z29.h\n"
      "fmin z14.h, p5/M, z14.h, z29.h\n"
      "fmin z15.h, p5/M, z15.h, z29.h\n"
      "fmin z16.h, p5/M, z16.h, z29.h\n"
      "fmin z17.h, p5/M, z17.h, z29.h\n"
      "fmin z18.h, p5/M, z18.h, z29.h\n"
      "fmin z19.h, p5/M, z19.h, z29.h\n"
      "fmin z20.h, p5/M, z20.h, z29.h\n"
      "fmin z21.h, p5/M, z21.h, z29.h\n"
      "fmin z22.h, p5/M, z22.h, z29.h\n"
      "fmin z23.h, p5/M, z23.h, z29.h\n"
      "fmin z24.h, p5/M, z24.h, z29.h\n"
      "fmin z25.h, p5/M, z25.h, z29.h\n"
      "fmin z26.h, p5/M, z26.h, z29.h\n"
      "fmin z27.h, p5/M, z27.h, z29.h\n"
      "fmax z8.h, p5/M, z8.h, z28.h\n"
      "fmax z9.h, p5/M, z9.h, z28.h\n"
      "fmax z10.h, p5/M, z10.h, z28.h\n"
      "fmax z11.h, p5/M, z11.h, z28.h\n"
      "fmax z12.h, p5/M, z12.h, z28.h\n"
      "fmax z13.h, p5/M, z13.h, z28.h\n"
      "fmax z14.h, p5/M, z14.h, z28.h\n"
      "fmax z15.h, p5/M, z15.h, z28.h\n"
      "fmax z16.h, p5/M, z16.h, z28.h\n"
      "fmax z17.h, p5/M, z17.h, z28.h\n"
      "fmax z18.h, p5/M, z18.h, z28.h\n"
      "fmax z19.h, p5/M, z19.h, z28.h\n"
      "fmax z20.h, p5/M, z20.h, z28.h\n"
      "fmax z21.h, p5/M, z21.h, z28.h\n"
      "fmax z22.h, p5/M, z22.h, z28.h\n"
      "fmax z23.h, p5/M, z23.h, z28.h\n"
      "fmax z24.h, p5/M, z24.h, z28.h\n"
      "fmax z25.h, p5/M, z25.h, z28.h\n"
      "fmax z26.h, p5/M, z26.h, z28.h\n"
      "fmax z27.h, p5/M, z27.h, z28.h\n"
      "64:"  // Height 5: No activation
      "st1h { z8.h }, p4, [x9]\n"
      "st1h { z9.h }, p3, [x9, #1, MUL VL]\n"
      "st1h { z10.h }, p2, [x9, #2, MUL VL]\n"
      "st1h { z11.h }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1h { z12.h }, p4, [x25]\n"
      "st1h { z13.h }, p3, [x25, #1, MUL VL]\n"
      "st1h { z14.h }, p2, [x25, #2, MUL VL]\n"
      "st1h { z15.h }, p1, [x25, #3, MUL VL]\n"
      "st1h { z16.h }, p4, [x24]\n"
      "st1h { z17.h }, p3, [x24, #1, MUL VL]\n"
      "st1h { z18.h }, p2, [x24, #2, MUL VL]\n"
      "st1h { z19.h }, p1, [x24, #3, MUL VL]\n"
      "st1h { z20.h }, p4, [x23]\n"
      "st1h { z21.h }, p3, [x23, #1, MUL VL]\n"
      "st1h { z22.h }, p2, [x23, #2, MUL VL]\n"
      "st1h { z23.h }, p1, [x23, #3, MUL VL]\n"
      "st1h { z24.h }, p4, [x22]\n"
      "st1h { z25.h }, p3, [x22, #1, MUL VL]\n"
      "st1h { z26.h }, p2, [x22, #2, MUL VL]\n"
      "st1h { z27.h }, p1, [x22, #3, MUL VL]\n"
      "65:"  // Height 5: Writeback done
      "dech x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 54b\n"
      "b 80f\n"
      "66:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0xc\n"
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "67:"  // Height 6: Column loop
      "mov x20, #0x0\n"
      "whilelt p4.h, x20, x11\n"
      "inch x20\n"
      "whilelt p3.h, x20, x11\n"
      "inch x20\n"
      "whilelt p2.h, x20, x11\n"
      "inch x20\n"
      "whilelt p1.h, x20, x11\n"
      "cbz x12, 68f\n"
      "ld1h { z8.h }, p5/Z, [x12]\n"
      "ld1h { z9.h }, p5/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1h { z10.h }, p5/Z, [x12, #2, MUL VL]\n"
      "ld1h { z11.h }, p5/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z20.d, z8.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "mov z28.d, z8.d\n"
      "mov z29.d, z9.d\n"
      "mov z30.d, z10.d\n"
      "mov z31.d, z11.d\n"
      "b 70f\n"
      "68:"  // Height 6: no bias
      "tbz %x[flags], #0, 69f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "ld1h { z8.h }, p4/Z, [x9]\n"
      "add x22, x23, x20, LSL #1\n"
      "add x21, x22, x20, LSL #1\n"
      "ld1h { z9.h }, p3/Z, [x9, #1, MUL VL]\n"
      "ld1h { z10.h }, p2/Z, [x9, #2, MUL VL]\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z11.h }, p1/Z, [x9, #3, MUL VL]\n"
      "ld1h { z12.h }, p4/Z, [x24]\n"
      "ld1h { z13.h }, p3/Z, [x24, #1, MUL VL]\n"
      "ld1h { z14.h }, p2/Z, [x24, #2, MUL VL]\n"
      "ld1h { z15.h }, p1/Z, [x24, #3, MUL VL]\n"
      "ld1h { z16.h }, p4/Z, [x23]\n"
      "ld1h { z17.h }, p3/Z, [x23, #1, MUL VL]\n"
      "ld1h { z18.h }, p2/Z, [x23, #2, MUL VL]\n"
      "ld1h { z19.h }, p1/Z, [x23, #3, MUL VL]\n"
      "ld1h { z20.h }, p4/Z, [x22]\n"
      "ld1h { z21.h }, p3/Z, [x22, #1, MUL VL]\n"
      "ld1h { z22.h }, p2/Z, [x22, #2, MUL VL]\n"
      "ld1h { z23.h }, p1/Z, [x22, #3, MUL VL]\n"
      "ld1h { z24.h }, p4/Z, [x21]\n"
      "ld1h { z25.h }, p3/Z, [x21, #1, MUL VL]\n"
      "ld1h { z26.h }, p2/Z, [x21, #2, MUL VL]\n"
      "ld1h { z27.h }, p1/Z, [x21, #3, MUL VL]\n"
      "ld1h { z28.h }, p4/Z, [x20]\n"
      "ld1h { z29.h }, p3/Z, [x20, #1, MUL VL]\n"
      "ld1h { z30.h }, p2/Z, [x20, #2, MUL VL]\n"
      "ld1h { z31.h }, p1/Z, [x20, #3, MUL VL]\n"
      "b 70f\n"
      "69:"  // Height 6: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "mov z30.b, #0x0\n"
      "mov z31.b, #0x0\n"
      "70:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "71:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 72f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 73f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "add x21, x21, x20, LSL #1\n"
      "b 73f\n"
      "72:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "add x21, x22, x21, LSL #1\n"
      "73:"  // Height 6: input setup done
      "cmp x27, #0x8\n"
      "ble 75f\n"
      "74:"  // Height 6: Multiply loop: Main loop head
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z7.h }, p0/Z, [x26]\n"
      "ld1rqh { z6.h }, p0/Z, [x25]\n"
      "sub x27, x27, #0x8\n"
      "ld1rqh { z5.h }, p0/Z, [x24]\n"
      "ld1rqh { z4.h }, p0/Z, [x23]\n"
      "cmp x27, #0x8\n"
      "add x26, x26, #0x10\n"
      "ld1rqh { z3.h }, p0/Z, [x22]\n"
      "ld1rqh { z2.h }, p0/Z, [x21]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "ld1h { z1.h }, p5/Z, [x10]\n"
      "ld1h { z0.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[0]\n"
      "fmla z12.h, z1.h, z6.h[0]\n"
      "fmla z16.h, z1.h, z5.h[0]\n"
      "fmla z20.h, z1.h, z4.h[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "fmla z24.h, z1.h, z3.h[0]\n"
      "fmla z28.h, z1.h, z2.h[0]\n"
      "ld1h { z1.h }, p5/Z, [x10, #2, MUL VL]\n"
      "add x21, x21, #0x10\n"
      "fmla z9.h, z0.h, z7.h[0]\n"
      "fmla z13.h, z0.h, z6.h[0]\n"
      "fmla z17.h, z0.h, z5.h[0]\n"
      "fmla z21.h, z0.h, z4.h[0]\n"
      "fmla z25.h, z0.h, z3.h[0]\n"
      "fmla z29.h, z0.h, z2.h[0]\n"
      "ld1h { z0.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z1.h, z7.h[0]\n"
      "fmla z14.h, z1.h, z6.h[0]\n"
      "fmla z18.h, z1.h, z5.h[0]\n"
      "fmla z22.h, z1.h, z4.h[0]\n"
      "fmla z26.h, z1.h, z3.h[0]\n"
      "fmla z30.h, z1.h, z2.h[0]\n"
      "ld1h { z1.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z11.h, z0.h, z7.h[0]\n"
      "fmla z15.h, z0.h, z6.h[0]\n"
      "fmla z19.h, z0.h, z5.h[0]\n"
      "fmla z23.h, z0.h, z4.h[0]\n"
      "fmla z27.h, z0.h, z3.h[0]\n"
      "fmla z31.h, z0.h, z2.h[0]\n"
      "ld1h { z0.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[1]\n"
      "fmla z12.h, z1.h, z6.h[1]\n"
      "fmla z16.h, z1.h, z5.h[1]\n"
      "fmla z20.h, z1.h, z4.h[1]\n"
      "fmla z24.h, z1.h, z3.h[1]\n"
      "fmla z28.h, z1.h, z2.h[1]\n"
      "ld1h { z1.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z9.h, z0.h, z7.h[1]\n"
      "fmla z13.h, z0.h, z6.h[1]\n"
      "fmla z17.h, z0.h, z5.h[1]\n"
      "fmla z21.h, z0.h, z4.h[1]\n"
      "fmla z25.h, z0.h, z3.h[1]\n"
      "fmla z29.h, z0.h, z2.h[1]\n"
      "ld1h { z0.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z1.h, z7.h[1]\n"
      "fmla z14.h, z1.h, z6.h[1]\n"
      "fmla z18.h, z1.h, z5.h[1]\n"
      "fmla z22.h, z1.h, z4.h[1]\n"
      "fmla z26.h, z1.h, z3.h[1]\n"
      "fmla z30.h, z1.h, z2.h[1]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z11.h, z0.h, z7.h[1]\n"
      "fmla z15.h, z0.h, z6.h[1]\n"
      "fmla z19.h, z0.h, z5.h[1]\n"
      "fmla z23.h, z0.h, z4.h[1]\n"
      "fmla z27.h, z0.h, z3.h[1]\n"
      "fmla z31.h, z0.h, z2.h[1]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[2]\n"
      "fmla z12.h, z1.h, z6.h[2]\n"
      "fmla z16.h, z1.h, z5.h[2]\n"
      "fmla z20.h, z1.h, z4.h[2]\n"
      "fmla z24.h, z1.h, z3.h[2]\n"
      "fmla z28.h, z1.h, z2.h[2]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z9.h, z0.h, z7.h[2]\n"
      "fmla z13.h, z0.h, z6.h[2]\n"
      "fmla z17.h, z0.h, z5.h[2]\n"
      "fmla z21.h, z0.h, z4.h[2]\n"
      "fmla z25.h, z0.h, z3.h[2]\n"
      "fmla z29.h, z0.h, z2.h[2]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z1.h, z7.h[2]\n"
      "fmla z14.h, z1.h, z6.h[2]\n"
      "fmla z18.h, z1.h, z5.h[2]\n"
      "fmla z22.h, z1.h, z4.h[2]\n"
      "fmla z26.h, z1.h, z3.h[2]\n"
      "fmla z30.h, z1.h, z2.h[2]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z11.h, z0.h, z7.h[2]\n"
      "fmla z15.h, z0.h, z6.h[2]\n"
      "fmla z19.h, z0.h, z5.h[2]\n"
      "fmla z23.h, z0.h, z4.h[2]\n"
      "fmla z27.h, z0.h, z3.h[2]\n"
      "fmla z31.h, z0.h, z2.h[2]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[3]\n"
      "fmla z12.h, z1.h, z6.h[3]\n"
      "fmla z16.h, z1.h, z5.h[3]\n"
      "fmla z20.h, z1.h, z4.h[3]\n"
      "fmla z24.h, z1.h, z3.h[3]\n"
      "fmla z28.h, z1.h, z2.h[3]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z9.h, z0.h, z7.h[3]\n"
      "fmla z13.h, z0.h, z6.h[3]\n"
      "fmla z17.h, z0.h, z5.h[3]\n"
      "fmla z21.h, z0.h, z4.h[3]\n"
      "fmla z25.h, z0.h, z3.h[3]\n"
      "fmla z29.h, z0.h, z2.h[3]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z1.h, z7.h[3]\n"
      "fmla z14.h, z1.h, z6.h[3]\n"
      "fmla z18.h, z1.h, z5.h[3]\n"
      "fmla z22.h, z1.h, z4.h[3]\n"
      "fmla z26.h, z1.h, z3.h[3]\n"
      "fmla z30.h, z1.h, z2.h[3]\n"
      "ld1h { z1.h }, p5/Z, [x10]\n"
      "fmla z11.h, z0.h, z7.h[3]\n"
      "fmla z15.h, z0.h, z6.h[3]\n"
      "fmla z19.h, z0.h, z5.h[3]\n"
      "fmla z23.h, z0.h, z4.h[3]\n"
      "fmla z27.h, z0.h, z3.h[3]\n"
      "fmla z31.h, z0.h, z2.h[3]\n"
      "ld1h { z0.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[4]\n"
      "fmla z12.h, z1.h, z6.h[4]\n"
      "fmla z16.h, z1.h, z5.h[4]\n"
      "fmla z20.h, z1.h, z4.h[4]\n"
      "fmla z24.h, z1.h, z3.h[4]\n"
      "fmla z28.h, z1.h, z2.h[4]\n"
      "ld1h { z1.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z0.h, z7.h[4]\n"
      "fmla z13.h, z0.h, z6.h[4]\n"
      "fmla z17.h, z0.h, z5.h[4]\n"
      "fmla z21.h, z0.h, z4.h[4]\n"
      "fmla z25.h, z0.h, z3.h[4]\n"
      "fmla z29.h, z0.h, z2.h[4]\n"
      "ld1h { z0.h }, p5/Z, [x10, #3, MUL VL]\n"
      "fmla z10.h, z1.h, z7.h[4]\n"
      "fmla z14.h, z1.h, z6.h[4]\n"
      "fmla z18.h, z1.h, z5.h[4]\n"
      "fmla z22.h, z1.h, z4.h[4]\n"
      "fmla z26.h, z1.h, z3.h[4]\n"
      "fmla z30.h, z1.h, z2.h[4]\n"
      "ld1h { z1.h }, p5/Z, [x10, #4, MUL VL]\n"
      "fmla z11.h, z0.h, z7.h[4]\n"
      "fmla z15.h, z0.h, z6.h[4]\n"
      "fmla z19.h, z0.h, z5.h[4]\n"
      "fmla z23.h, z0.h, z4.h[4]\n"
      "fmla z27.h, z0.h, z3.h[4]\n"
      "fmla z31.h, z0.h, z2.h[4]\n"
      "ld1h { z0.h }, p5/Z, [x10, #5, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[5]\n"
      "fmla z12.h, z1.h, z6.h[5]\n"
      "fmla z16.h, z1.h, z5.h[5]\n"
      "fmla z20.h, z1.h, z4.h[5]\n"
      "fmla z24.h, z1.h, z3.h[5]\n"
      "fmla z28.h, z1.h, z2.h[5]\n"
      "ld1h { z1.h }, p5/Z, [x10, #6, MUL VL]\n"
      "fmla z9.h, z0.h, z7.h[5]\n"
      "fmla z13.h, z0.h, z6.h[5]\n"
      "fmla z17.h, z0.h, z5.h[5]\n"
      "fmla z21.h, z0.h, z4.h[5]\n"
      "fmla z25.h, z0.h, z3.h[5]\n"
      "fmla z29.h, z0.h, z2.h[5]\n"
      "ld1h { z0.h }, p5/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "fmla z10.h, z1.h, z7.h[5]\n"
      "fmla z14.h, z1.h, z6.h[5]\n"
      "fmla z18.h, z1.h, z5.h[5]\n"
      "fmla z22.h, z1.h, z4.h[5]\n"
      "fmla z26.h, z1.h, z3.h[5]\n"
      "fmla z30.h, z1.h, z2.h[5]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-8, MUL VL]\n"
      "fmla z11.h, z0.h, z7.h[5]\n"
      "fmla z15.h, z0.h, z6.h[5]\n"
      "fmla z19.h, z0.h, z5.h[5]\n"
      "fmla z23.h, z0.h, z4.h[5]\n"
      "fmla z27.h, z0.h, z3.h[5]\n"
      "fmla z31.h, z0.h, z2.h[5]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-7, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[6]\n"
      "fmla z12.h, z1.h, z6.h[6]\n"
      "fmla z16.h, z1.h, z5.h[6]\n"
      "fmla z20.h, z1.h, z4.h[6]\n"
      "fmla z24.h, z1.h, z3.h[6]\n"
      "fmla z28.h, z1.h, z2.h[6]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-6, MUL VL]\n"
      "fmla z9.h, z0.h, z7.h[6]\n"
      "fmla z13.h, z0.h, z6.h[6]\n"
      "fmla z17.h, z0.h, z5.h[6]\n"
      "fmla z21.h, z0.h, z4.h[6]\n"
      "fmla z25.h, z0.h, z3.h[6]\n"
      "fmla z29.h, z0.h, z2.h[6]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-5, MUL VL]\n"
      "fmla z10.h, z1.h, z7.h[6]\n"
      "fmla z14.h, z1.h, z6.h[6]\n"
      "fmla z18.h, z1.h, z5.h[6]\n"
      "fmla z22.h, z1.h, z4.h[6]\n"
      "fmla z26.h, z1.h, z3.h[6]\n"
      "fmla z30.h, z1.h, z2.h[6]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-4, MUL VL]\n"
      "fmla z11.h, z0.h, z7.h[6]\n"
      "fmla z15.h, z0.h, z6.h[6]\n"
      "fmla z19.h, z0.h, z5.h[6]\n"
      "fmla z23.h, z0.h, z4.h[6]\n"
      "fmla z27.h, z0.h, z3.h[6]\n"
      "fmla z31.h, z0.h, z2.h[6]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-3, MUL VL]\n"
      "fmla z8.h, z1.h, z7.h[7]\n"
      "fmla z12.h, z1.h, z6.h[7]\n"
      "fmla z16.h, z1.h, z5.h[7]\n"
      "fmla z20.h, z1.h, z4.h[7]\n"
      "fmla z24.h, z1.h, z3.h[7]\n"
      "fmla z28.h, z1.h, z2.h[7]\n"
      "ld1h { z1.h }, p5/Z, [x10, #-2, MUL VL]\n"
      "fmla z9.h, z0.h, z7.h[7]\n"
      "fmla z13.h, z0.h, z6.h[7]\n"
      "fmla z17.h, z0.h, z5.h[7]\n"
      "fmla z21.h, z0.h, z4.h[7]\n"
      "fmla z25.h, z0.h, z3.h[7]\n"
      "fmla z29.h, z0.h, z2.h[7]\n"
      "ld1h { z0.h }, p5/Z, [x10, #-1, MUL VL]\n"
      "fmla z10.h, z1.h, z7.h[7]\n"
      "fmla z14.h, z1.h, z6.h[7]\n"
      "fmla z18.h, z1.h, z5.h[7]\n"
      "fmla z22.h, z1.h, z4.h[7]\n"
      "fmla z26.h, z1.h, z3.h[7]\n"
      "fmla z30.h, z1.h, z2.h[7]\n"
      "fmla z11.h, z0.h, z7.h[7]\n"
      "fmla z15.h, z0.h, z6.h[7]\n"
      "fmla z19.h, z0.h, z5.h[7]\n"
      "fmla z23.h, z0.h, z4.h[7]\n"
      "fmla z27.h, z0.h, z3.h[7]\n"
      "fmla z31.h, z0.h, z2.h[7]\n"
      "bgt 74b\n"
      "75:"  // Height 6: Multiply loop: Single iteration only
      "whilelt p0.h, XZR, x27\n"
      "ld1rqh { z0.h }, p0/Z, [x26]\n"
      "ld1rqh { z1.h }, p0/Z, [x25]\n"
      "subs x27, x27, #0x1\n"
      "ld1rqh { z2.h }, p0/Z, [x24]\n"
      "ld1rqh { z3.h }, p0/Z, [x23]\n"
      "ld1rqh { z4.h }, p0/Z, [x22]\n"
      "ld1rqh { z5.h }, p0/Z, [x21]\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[0]\n"
      "fmla z12.h, z7.h, z1.h[0]\n"
      "fmla z16.h, z7.h, z2.h[0]\n"
      "fmla z20.h, z7.h, z3.h[0]\n"
      "fmla z24.h, z7.h, z4.h[0]\n"
      "fmla z28.h, z7.h, z5.h[0]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[0]\n"
      "fmla z13.h, z6.h, z1.h[0]\n"
      "fmla z17.h, z6.h, z2.h[0]\n"
      "fmla z21.h, z6.h, z3.h[0]\n"
      "fmla z25.h, z6.h, z4.h[0]\n"
      "fmla z29.h, z6.h, z5.h[0]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[0]\n"
      "fmla z14.h, z7.h, z1.h[0]\n"
      "fmla z18.h, z7.h, z2.h[0]\n"
      "fmla z22.h, z7.h, z3.h[0]\n"
      "fmla z26.h, z7.h, z4.h[0]\n"
      "fmla z30.h, z7.h, z5.h[0]\n"
      "fmla z11.h, z6.h, z0.h[0]\n"
      "fmla z15.h, z6.h, z1.h[0]\n"
      "fmla z19.h, z6.h, z2.h[0]\n"
      "fmla z23.h, z6.h, z3.h[0]\n"
      "fmla z27.h, z6.h, z4.h[0]\n"
      "fmla z31.h, z6.h, z5.h[0]\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[1]\n"
      "fmla z12.h, z7.h, z1.h[1]\n"
      "fmla z16.h, z7.h, z2.h[1]\n"
      "fmla z20.h, z7.h, z3.h[1]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z7.h, z4.h[1]\n"
      "fmla z28.h, z7.h, z5.h[1]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[1]\n"
      "fmla z13.h, z6.h, z1.h[1]\n"
      "fmla z17.h, z6.h, z2.h[1]\n"
      "fmla z21.h, z6.h, z3.h[1]\n"
      "fmla z25.h, z6.h, z4.h[1]\n"
      "fmla z29.h, z6.h, z5.h[1]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[1]\n"
      "fmla z14.h, z7.h, z1.h[1]\n"
      "fmla z18.h, z7.h, z2.h[1]\n"
      "fmla z22.h, z7.h, z3.h[1]\n"
      "fmla z26.h, z7.h, z4.h[1]\n"
      "fmla z30.h, z7.h, z5.h[1]\n"
      "fmla z11.h, z6.h, z0.h[1]\n"
      "fmla z15.h, z6.h, z1.h[1]\n"
      "fmla z19.h, z6.h, z2.h[1]\n"
      "fmla z23.h, z6.h, z3.h[1]\n"
      "fmla z27.h, z6.h, z4.h[1]\n"
      "fmla z31.h, z6.h, z5.h[1]\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[2]\n"
      "fmla z12.h, z7.h, z1.h[2]\n"
      "fmla z16.h, z7.h, z2.h[2]\n"
      "fmla z20.h, z7.h, z3.h[2]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z7.h, z4.h[2]\n"
      "fmla z28.h, z7.h, z5.h[2]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[2]\n"
      "fmla z13.h, z6.h, z1.h[2]\n"
      "fmla z17.h, z6.h, z2.h[2]\n"
      "fmla z21.h, z6.h, z3.h[2]\n"
      "fmla z25.h, z6.h, z4.h[2]\n"
      "fmla z29.h, z6.h, z5.h[2]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[2]\n"
      "fmla z14.h, z7.h, z1.h[2]\n"
      "fmla z18.h, z7.h, z2.h[2]\n"
      "fmla z22.h, z7.h, z3.h[2]\n"
      "fmla z26.h, z7.h, z4.h[2]\n"
      "fmla z30.h, z7.h, z5.h[2]\n"
      "fmla z11.h, z6.h, z0.h[2]\n"
      "fmla z15.h, z6.h, z1.h[2]\n"
      "fmla z19.h, z6.h, z2.h[2]\n"
      "fmla z23.h, z6.h, z3.h[2]\n"
      "fmla z27.h, z6.h, z4.h[2]\n"
      "fmla z31.h, z6.h, z5.h[2]\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[3]\n"
      "fmla z12.h, z7.h, z1.h[3]\n"
      "fmla z16.h, z7.h, z2.h[3]\n"
      "fmla z20.h, z7.h, z3.h[3]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z7.h, z4.h[3]\n"
      "fmla z28.h, z7.h, z5.h[3]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[3]\n"
      "fmla z13.h, z6.h, z1.h[3]\n"
      "fmla z17.h, z6.h, z2.h[3]\n"
      "fmla z21.h, z6.h, z3.h[3]\n"
      "fmla z25.h, z6.h, z4.h[3]\n"
      "fmla z29.h, z6.h, z5.h[3]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[3]\n"
      "fmla z14.h, z7.h, z1.h[3]\n"
      "fmla z18.h, z7.h, z2.h[3]\n"
      "fmla z22.h, z7.h, z3.h[3]\n"
      "fmla z26.h, z7.h, z4.h[3]\n"
      "fmla z30.h, z7.h, z5.h[3]\n"
      "fmla z11.h, z6.h, z0.h[3]\n"
      "fmla z15.h, z6.h, z1.h[3]\n"
      "fmla z19.h, z6.h, z2.h[3]\n"
      "fmla z23.h, z6.h, z3.h[3]\n"
      "fmla z27.h, z6.h, z4.h[3]\n"
      "fmla z31.h, z6.h, z5.h[3]\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[4]\n"
      "fmla z12.h, z7.h, z1.h[4]\n"
      "fmla z16.h, z7.h, z2.h[4]\n"
      "fmla z20.h, z7.h, z3.h[4]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z7.h, z4.h[4]\n"
      "fmla z28.h, z7.h, z5.h[4]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[4]\n"
      "fmla z13.h, z6.h, z1.h[4]\n"
      "fmla z17.h, z6.h, z2.h[4]\n"
      "fmla z21.h, z6.h, z3.h[4]\n"
      "fmla z25.h, z6.h, z4.h[4]\n"
      "fmla z29.h, z6.h, z5.h[4]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[4]\n"
      "fmla z14.h, z7.h, z1.h[4]\n"
      "fmla z18.h, z7.h, z2.h[4]\n"
      "fmla z22.h, z7.h, z3.h[4]\n"
      "fmla z26.h, z7.h, z4.h[4]\n"
      "fmla z30.h, z7.h, z5.h[4]\n"
      "fmla z11.h, z6.h, z0.h[4]\n"
      "fmla z15.h, z6.h, z1.h[4]\n"
      "fmla z19.h, z6.h, z2.h[4]\n"
      "fmla z23.h, z6.h, z3.h[4]\n"
      "fmla z27.h, z6.h, z4.h[4]\n"
      "fmla z31.h, z6.h, z5.h[4]\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[5]\n"
      "fmla z12.h, z7.h, z1.h[5]\n"
      "fmla z16.h, z7.h, z2.h[5]\n"
      "fmla z20.h, z7.h, z3.h[5]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z7.h, z4.h[5]\n"
      "fmla z28.h, z7.h, z5.h[5]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[5]\n"
      "fmla z13.h, z6.h, z1.h[5]\n"
      "fmla z17.h, z6.h, z2.h[5]\n"
      "fmla z21.h, z6.h, z3.h[5]\n"
      "fmla z25.h, z6.h, z4.h[5]\n"
      "fmla z29.h, z6.h, z5.h[5]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[5]\n"
      "fmla z14.h, z7.h, z1.h[5]\n"
      "fmla z18.h, z7.h, z2.h[5]\n"
      "fmla z22.h, z7.h, z3.h[5]\n"
      "fmla z26.h, z7.h, z4.h[5]\n"
      "fmla z30.h, z7.h, z5.h[5]\n"
      "fmla z11.h, z6.h, z0.h[5]\n"
      "fmla z15.h, z6.h, z1.h[5]\n"
      "fmla z19.h, z6.h, z2.h[5]\n"
      "fmla z23.h, z6.h, z3.h[5]\n"
      "fmla z27.h, z6.h, z4.h[5]\n"
      "fmla z31.h, z6.h, z5.h[5]\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[6]\n"
      "fmla z12.h, z7.h, z1.h[6]\n"
      "fmla z16.h, z7.h, z2.h[6]\n"
      "fmla z20.h, z7.h, z3.h[6]\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, z7.h, z4.h[6]\n"
      "fmla z28.h, z7.h, z5.h[6]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[6]\n"
      "fmla z13.h, z6.h, z1.h[6]\n"
      "fmla z17.h, z6.h, z2.h[6]\n"
      "fmla z21.h, z6.h, z3.h[6]\n"
      "fmla z25.h, z6.h, z4.h[6]\n"
      "fmla z29.h, z6.h, z5.h[6]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[6]\n"
      "fmla z14.h, z7.h, z1.h[6]\n"
      "fmla z18.h, z7.h, z2.h[6]\n"
      "fmla z22.h, z7.h, z3.h[6]\n"
      "fmla z26.h, z7.h, z4.h[6]\n"
      "fmla z30.h, z7.h, z5.h[6]\n"
      "fmla z11.h, z6.h, z0.h[6]\n"
      "fmla z15.h, z6.h, z1.h[6]\n"
      "fmla z19.h, z6.h, z2.h[6]\n"
      "fmla z23.h, z6.h, z3.h[6]\n"
      "fmla z27.h, z6.h, z4.h[6]\n"
      "fmla z31.h, z6.h, z5.h[6]\n"
      "ble 76f\n"
      "ld1h { z7.h }, p5/Z, [x10]\n"
      "ld1h { z6.h }, p5/Z, [x10, #1, MUL VL]\n"
      "fmla z8.h, z7.h, z0.h[7]\n"
      "fmla z12.h, z7.h, z1.h[7]\n"
      "fmla z16.h, z7.h, z2.h[7]\n"
      "fmla z20.h, z7.h, z3.h[7]\n"
      "fmla z24.h, z7.h, z4.h[7]\n"
      "fmla z28.h, z7.h, z5.h[7]\n"
      "ld1h { z7.h }, p5/Z, [x10, #2, MUL VL]\n"
      "fmla z9.h, z6.h, z0.h[7]\n"
      "fmla z13.h, z6.h, z1.h[7]\n"
      "fmla z17.h, z6.h, z2.h[7]\n"
      "fmla z21.h, z6.h, z3.h[7]\n"
      "fmla z25.h, z6.h, z4.h[7]\n"
      "fmla z29.h, z6.h, z5.h[7]\n"
      "ld1h { z6.h }, p5/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.h, z7.h, z0.h[7]\n"
      "fmla z14.h, z7.h, z1.h[7]\n"
      "fmla z18.h, z7.h, z2.h[7]\n"
      "fmla z22.h, z7.h, z3.h[7]\n"
      "fmla z26.h, z7.h, z4.h[7]\n"
      "fmla z30.h, z7.h, z5.h[7]\n"
      "fmla z11.h, z6.h, z0.h[7]\n"
      "fmla z15.h, z6.h, z1.h[7]\n"
      "fmla z19.h, z6.h, z2.h[7]\n"
      "fmla z23.h, z6.h, z3.h[7]\n"
      "fmla z27.h, z6.h, z4.h[7]\n"
      "fmla z31.h, z6.h, z5.h[7]\n"
      "76:"  // Height 6: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 71b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "add x21, x22, x20, LSL #1\n"
      "tbz %x[flags], #1, 77f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rh { z1.h }, p5/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z0.h }, p5/Z, [x20]\n"
      "fmin z8.h, p5/M, z8.h, z1.h\n"
      "fmin z9.h, p5/M, z9.h, z1.h\n"
      "fmin z10.h, p5/M, z10.h, z1.h\n"
      "fmin z11.h, p5/M, z11.h, z1.h\n"
      "fmin z12.h, p5/M, z12.h, z1.h\n"
      "fmin z13.h, p5/M, z13.h, z1.h\n"
      "fmin z14.h, p5/M, z14.h, z1.h\n"
      "fmin z15.h, p5/M, z15.h, z1.h\n"
      "fmin z16.h, p5/M, z16.h, z1.h\n"
      "fmin z17.h, p5/M, z17.h, z1.h\n"
      "fmin z18.h, p5/M, z18.h, z1.h\n"
      "fmin z19.h, p5/M, z19.h, z1.h\n"
      "fmin z20.h, p5/M, z20.h, z1.h\n"
      "fmin z21.h, p5/M, z21.h, z1.h\n"
      "fmin z22.h, p5/M, z22.h, z1.h\n"
      "fmin z23.h, p5/M, z23.h, z1.h\n"
      "fmin z24.h, p5/M, z24.h, z1.h\n"
      "fmin z25.h, p5/M, z25.h, z1.h\n"
      "fmin z26.h, p5/M, z26.h, z1.h\n"
      "fmin z27.h, p5/M, z27.h, z1.h\n"
      "fmin z28.h, p5/M, z28.h, z1.h\n"
      "fmin z29.h, p5/M, z29.h, z1.h\n"
      "fmin z30.h, p5/M, z30.h, z1.h\n"
      "fmin z31.h, p5/M, z31.h, z1.h\n"
      "fmax z8.h, p5/M, z8.h, z0.h\n"
      "fmax z9.h, p5/M, z9.h, z0.h\n"
      "fmax z10.h, p5/M, z10.h, z0.h\n"
      "fmax z11.h, p5/M, z11.h, z0.h\n"
      "fmax z12.h, p5/M, z12.h, z0.h\n"
      "fmax z13.h, p5/M, z13.h, z0.h\n"
      "fmax z14.h, p5/M, z14.h, z0.h\n"
      "fmax z15.h, p5/M, z15.h, z0.h\n"
      "fmax z16.h, p5/M, z16.h, z0.h\n"
      "fmax z17.h, p5/M, z17.h, z0.h\n"
      "fmax z18.h, p5/M, z18.h, z0.h\n"
      "fmax z19.h, p5/M, z19.h, z0.h\n"
      "fmax z20.h, p5/M, z20.h, z0.h\n"
      "fmax z21.h, p5/M, z21.h, z0.h\n"
      "fmax z22.h, p5/M, z22.h, z0.h\n"
      "fmax z23.h, p5/M, z23.h, z0.h\n"
      "fmax z24.h, p5/M, z24.h, z0.h\n"
      "fmax z25.h, p5/M, z25.h, z0.h\n"
      "fmax z26.h, p5/M, z26.h, z0.h\n"
      "fmax z27.h, p5/M, z27.h, z0.h\n"
      "fmax z28.h, p5/M, z28.h, z0.h\n"
      "fmax z29.h, p5/M, z29.h, z0.h\n"
      "fmax z30.h, p5/M, z30.h, z0.h\n"
      "fmax z31.h, p5/M, z31.h, z0.h\n"
      "77:"  // Height 6: No activation
      "st1h { z8.h }, p4, [x9]\n"
      "st1h { z9.h }, p3, [x9, #1, MUL VL]\n"
      "st1h { z10.h }, p2, [x9, #2, MUL VL]\n"
      "st1h { z11.h }, p1, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1h { z12.h }, p4, [x25]\n"
      "st1h { z13.h }, p3, [x25, #1, MUL VL]\n"
      "st1h { z14.h }, p2, [x25, #2, MUL VL]\n"
      "st1h { z15.h }, p1, [x25, #3, MUL VL]\n"
      "st1h { z16.h }, p4, [x24]\n"
      "st1h { z17.h }, p3, [x24, #1, MUL VL]\n"
      "st1h { z18.h }, p2, [x24, #2, MUL VL]\n"
      "st1h { z19.h }, p1, [x24, #3, MUL VL]\n"
      "st1h { z20.h }, p4, [x23]\n"
      "st1h { z21.h }, p3, [x23, #1, MUL VL]\n"
      "st1h { z22.h }, p2, [x23, #2, MUL VL]\n"
      "st1h { z23.h }, p1, [x23, #3, MUL VL]\n"
      "st1h { z24.h }, p4, [x22]\n"
      "st1h { z25.h }, p3, [x22, #1, MUL VL]\n"
      "st1h { z26.h }, p2, [x22, #2, MUL VL]\n"
      "st1h { z27.h }, p1, [x22, #3, MUL VL]\n"
      "st1h { z28.h }, p4, [x21]\n"
      "st1h { z29.h }, p3, [x21, #1, MUL VL]\n"
      "st1h { z30.h }, p2, [x21, #2, MUL VL]\n"
      "st1h { z31.h }, p1, [x21, #3, MUL VL]\n"
      "78:"  // Height 6: Writeback done
      "dech x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 67b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 80f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 79f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "79:"  // Update direct input
      "mov x20, #0xc\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "80:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
