package map.map25000.cell;

import java.awt.Rectangle;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.HashSet;
import java.util.Set;

import map.data.City;

/**
 * 都道府県セル
 * @author Masayasu Fujiwara
 */
public class Prefecture extends Rectangle {

	/**
	 * エンコーディング
	 */
	private final String ENCODING = "SJIS";

	private final int code;
	private Cell[][] cells;
	private int cellMinX;
	private int cellMinY;
	private int size;

	public Prefecture(int code, int x, int y, int width, int height) {
		super(x, y, width, height);
		this.code = code;
	}

	public Prefecture(int code, String[] state) {
		this(code, Integer.parseInt(state[0]), Integer.parseInt(state[1]), Integer.parseInt(state[2]), Integer.parseInt(state[3]));
	}

	public void ready() throws IOException {
		if(this.cells == null) {
			BufferedReader bi = null;
			try {
				bi = new BufferedReader(new InputStreamReader(Prefecture.class.getResourceAsStream("/data/" + City.prefectureFormat(this.code) + "/city.cell"), this.ENCODING));
				String line = bi.readLine();
				String[] param = line.split(",");
				this.cellMinX = Integer.parseInt(param[0]);
				this.cellMinY = Integer.parseInt(param[1]);
				this.size = Integer.parseInt(param[2]);
				this.cells = new Cell[Integer.parseInt(param[3])][Integer.parseInt(param[4])];
				for(int i = 0; i < this.cells.length; i++) {
					for(int j = 0; j < this.cells[i].length; j++) {
						line = bi.readLine();
						param = line.split(",");
						if ("".equals(param[0])) {
							this.cells[i][j] = null;
						} else {
							int[] codes = new int[param.length];
							for (int k = 0; k < param.length; k++) {
								codes[k] = Integer.parseInt(param[k]);
							}
							this.cells[i][j] = new Cell(codes);
						}
					}
				}
			} finally {
				if(bi != null) {
					bi.close();
				}
			}
		}
	}

	/**
	 * データを破棄
	 *
	 */
	public void dump() {
		this.cells = null;
	}
	
	/**
	 * 都道府県番号を取得します。
	 * @return 都道府県番号
	 */
	public int getCode() {
		return this.code;
	}

	/**
	 * 地域番号を取得する
	 * @param screen
	 * @return 地域番号の集合
	 */
	public Set<Integer> get(Rectangle screen) {
		int minX = screen.x / this.size - this.cellMinX;
		int minY = screen.y / this.size - this.cellMinY;
		int maxX = (screen.x + screen.width)  / this.size - this.cellMinX;
		int maxY = (screen.y + screen.height) / this.size - this.cellMinY;
		
		Set<Integer> codes = new HashSet<Integer>();

		for(int i = minX; i < this.cells.length && i <= maxX; i++) {
			for(int j = minY; j < this.cells[i].length && j <= maxY; j++) {
				
				if(this.cells[i][j] == null) {
					continue;
				}
				
				for(int code : this.cells[i][j].getCodes()) {
					codes.add(code);
				}
			}
		}
		return codes;
	}

	/**
	 * 地域検索を取得します。
	 * @param screen
	 * @return 地域番号の集合 
	 */
	public Set<Integer> search(Rectangle screen) {
		Set<Integer> codes = new HashSet<Integer>();
		int minX = screen.x / this.size - this.cellMinX;
		int minY = screen.y / this.size - this.cellMinY;
		int maxX = (screen.x + screen.width)  / this.size - this.cellMinX;
		int maxY = (screen.y + screen.height) / this.size - this.cellMinY;
		
		if(minX < 0) {
			minX = 0;
		}
		if(minY < 0) {
			minY = 0;
		}

		for(int i = minX; i < this.cells.length && i <= maxX; i++) {
			for(int j = minY; j < this.cells[i].length && j <= maxY; j++) {
				
				if(this.cells[i][j] == null) {
					continue;
				}
				
				for(int code : this.cells[i][j].getCodes()) {
					codes.add(code);
				}
			}
		}
		return codes;
	}

	/**
	 * 地域検索を取得します。
	 * @param codes
	 * @param screen
	 */
	public void search(Set<Integer> codes, Rectangle screen) {
		int minX = screen.x / this.size - this.cellMinX;
		int minY = screen.y / this.size - this.cellMinY;
		int maxX = (screen.x + screen.width)  / this.size - this.cellMinX;
		int maxY = (screen.y + screen.height) / this.size - this.cellMinY;
		
		if(minX < 0) {
			minX = 0;
		}
		if(minY < 0) {
			minY = 0;
		}

		for(int i = minX; i < this.cells.length && i <= maxX; i++) {
			for(int j = minY; j < this.cells[i].length && j <= maxY; j++) {
				
				if(this.cells[i][j] == null) {
					continue;
				}
				
				for(int code : this.cells[i][j].getCodes()) {
					codes.add(code);
				}
			}
		}
	}
}
