package map.data;

import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;


import util.Log;
import view.MapPanel;
import view.StatusBar;

import map.IsjFactory;
import map.KsjFactory;
import map.map25000.Map25000Factory;
import map.map25000.cell.CellSearch;
import map.route.SearchThread;
import map.store.Store;

/**
 * 市区町村管理クラス
 * 
 * @author Masayasu Fujiwara
 *
 */
public class CityMap implements Map<Integer, City>, Runnable {

	/**
	 * 地図パネルクラス
	 */
	private final MapPanel panel;

	/**
	 * セル型の地域検索クラス
	 */
	private final CellSearch cell;

	/**
	 * 地図データ作成クラス
	 */
	private final Map25000Factory map25kFactory;

	/**
	 * ステータスバー
	 */
	private final StatusBar statusbar;

	/**
	 * 都道府県番号の読み込み状態
	 * true  -> 既読
	 * false -> 未読
	 */
	private boolean[] prefecture;

	private Rectangle screen;

	/**
	 * 市区町村データのMap
	 */
	private final Map<Integer, City> cityMap;

	/**
	 * 街区レベル位置参照情報ファクトリー
	 */
	private final IsjFactory isjFactory;

	/**
	 * 店舗情報ファクトリー
	 */
	private final Collection<Store> shops;

	public CityMap(MapPanel panel, CellSearch cell, Map25000Factory map25kFactory, IsjFactory isjFactory, Collection<Store> shop, StatusBar statusbar) {
		this.cityMap = new HashMap<Integer, City>();
		this.panel = panel;
		this.cell = cell;
		this.shops = shop;
		this.map25kFactory = map25kFactory;
		this.isjFactory = isjFactory;
		this.statusbar = statusbar;
		this.prefecture = new boolean[47];
	}

	/**
	 * スレッドを開始します。
	 */
	public void start() {
		this.screen = this.panel.getScreen();
		Log.out(this, "start Thread.");
		Thread thread = new Thread(this);
		thread.setPriority(2);
		thread.start();
	}

	/**
	 * 地図情報を加える
	 * @param key 市区町村番号
	 * @param value 市区町村データ
	 * 
	 * @return 市区町村データ
	 */
	public synchronized City put(Integer key, City value) {
		return this.cityMap.put(key, value);
	}

	/**
	 * 指定した頂点番号の頂点を取得します。
	 * @param id 頂点番号
	 * @param isDetailRoad 詳細な道路区間データを取得する場合true
	 * @return 頂点
	 * @throws IOException 頂点を取得できなかった場合
	 */
	public Node getNode(long id, boolean isDetailRoad) throws IOException {
		City map = this.cityMap.get((int) (id / 1000000));
		if (map == null) {
			map = this.map25kFactory.productNode((int) (id / 1000000), isDetailRoad);
		} else if (!map.hasNode()) {
			this.map25kFactory.productNode(map, isDetailRoad);
		}
		Node node = map.getNode((int) (id % 1000000));
		if (node == null) {
			Log.out(this, " : node is not found, and reproduct node.");
			this.map25kFactory.productNode(map, isDetailRoad);
			node = map.getNode((int) (id % 1000000));
			if (node == null) {
				throw new IOException("node is not found.");
			}
		}
		return node;
	}

	/**
	 * 市区町村データの取得
	 * @param key 市区町村番号
	 * @return 市区町村データ
	 */
	public synchronized City get(Object key) {
		return this.cityMap.get(key);
	}

	public void removeNode(long id) {
		City map;
		map = this.get((int) (id / 1000000));
		if (!map.isLoaded()) {
			map.removeNode((int) (id % 1000000));
		}
	}

	/**
	 * 頂点のCollectionを取得
	 * @param code 市区町村番号
	 * @return 頂点のCollection
	 */
	public synchronized Collection<Node> getNodes(int code) {
		return this.cityMap.get(code).getNode();
	}

	public void run() {
		while (true) {
			Log.out(this, "running...");
			try {
				boolean isReadNewData = false;
				try {
					if (this.panel.mode() > 0) {
						Map<Integer, Set<Integer>> codes = this.cell.search(this.screen);
						Log.out(this, "codes = " + codes);
						if (!this.panel.isOperation()) {
							this.dumpPrefecture(codes.keySet());
						}
						for (Map.Entry<Integer, Set<Integer>> entry : codes.entrySet()) {
							Log.out(this, "read "+ City.cityFormat(entry.getKey()) +"ksj border.");
							this.productPrefecture(entry.getKey());
							isReadNewData &= this.readCity(entry.getValue());
						}
					}
				} catch (IOException e) {
					this.statusbar.ioException();
					isReadNewData = false;
					Log.err(this, e);
				} catch (OutOfMemoryError e) {
					this.statusbar.memoryError();
					isReadNewData = false;
				} finally {
					this.dump();
				}
				Log.out(this, "isReadNewData = " + isReadNewData);
				if (!isReadNewData) {
					Thread.sleep(5000L);
				}
			} catch (InterruptedException e) {
				Log.err(this, e);
			}
		}
	}

	/**
	 * 都道府県番号から市区町村番号と市区町村名のMapを取得します。
	 * @param code 都道府県番号
	 * @return 市区町村番号と市区町村名のMap
	 * @throws IOException
	 */
	public Map<Integer, String> getCityNameMap(int code) throws IOException {
		BufferedReader bi = null;
		Map<Integer, String> map = null;
		try {
			bi = new BufferedReader(new InputStreamReader(CityMap.class.getResourceAsStream("/data/" + City.prefectureFormat(code) + "/city.csv"), "SJIS"));
			map = new HashMap<Integer, String>();
			String line;
			while ((line = bi.readLine()) != null) {
				String[] param = line.split(",");
				if (param[0] != null && param[1] != null) {
					map.put(Integer.parseInt(param[1]), param[0]);
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return map;
	}
	/**
	 * 国土数値情報の行政界の読み込み市区町村データを生成
	 * @param code 市区町村番号
	 * @throws IOException 入出力エラー
	 */
	private void productPrefecture(int code) throws IOException {
		if (!this.prefecture[code - 1]) {
			String format = City.prefectureFormat(code);
			Map<Integer, List<Polygon>> polygonMap = KsjFactory.readSerializeKsj(format);
			Map<Integer, String> nameMap = this.getCityNameMap(code);
			for (Map.Entry<Integer, List<Polygon>> entry : polygonMap.entrySet()) {
				synchronized (this) {
					Integer key = entry.getKey();
					String name = nameMap.get(key);
					this.put(key, new City(key, name, entry.getValue().toArray(new Polygon[] {})));
				}
			}
			this.prefecture[code - 1] = true;
			this.panel.repaint();
		}
	}

	/**
	 * 市町村データの読み込み
	 * @param codes
	 * @return 新たなデータを読み込めばtrueを返します。
	 * @throws IOException
	 */
	private boolean readCity(Collection<Integer> codes) throws IOException {
		Log.out(this, "readCity started.");
		boolean flag = false;
		// セルメソッドから地図の読み込み
		Log.out(this, "codes = " + codes);
		for (int code : codes) {
			Log.out(this, "code = " + code);
			if (this.panel.mode() != 2) {
				Log.out(this, "break loop.");
				break;
			}
			City city = this.get(code);
			Log.out(this, "city = " + city);
			if (city == null) {
				Log.out(this, "city was null.");
				this.statusbar.setCheckCode(City.cityFormat(code));
				Log.out(this, "set check code.");
				city = this.map25kFactory.preproduct(code);
				Log.out(this, "preproduct " + code);
				this.statusbar.clearCheckCode();
				this.put(code, city);
				Log.out(this, "put " + code + ", " + city);
			} else if (city.getArea() == null) {
				Log.out(this, "city.getArea was null.");
				this.statusbar.setCheckCode(City.cityFormat(city.getCode()));
				Log.out(this, "set check code.");
				this.map25kFactory.preproduct(city);
				Log.out(this, "preproduct " + code);
				this.statusbar.clearCheckCode();
				Log.out(this, "cleared check code.");
			}
			if (this.screen.intersects(city.getArea())) {
				Log.out(this, "screen contains city.");
				if (!city.hasData()) {
					Log.out(this, "city has no data.");
					this.statusbar.setMessage(City.cityFormat(city.getCode()));
					Log.out(this, "start to read ISJ.");
					Map<String, Point> locationMap = null;
					locationMap = this.isjFactory.productStreaming(code);
					Log.out(this, "start to product " + code);
					this.map25kFactory.product(city);
					Log.out(this, "start to read Shop.");
					for (Store shop : this.shops) {
						city.addLabels(shop.getName(), shop.getLocation(city, locationMap));
					}
					this.statusbar.clearReading();
					this.panel.repaint();
					flag = true;
				}
			}
		}
		Log.out(this, "readCity finished. flag = " + flag);
		return flag;
	}

	/**
	 * 経路探索のために読み込んだ頂点を破棄します。
	 */
	public synchronized void clearNode() {
		for (City map : this.cityMap.values()) {
			if ((!map.hasData() && map.hasNode()) || (map.hasData() && !this.screen.intersects(map.getArea()))) {
				map.dump();
			}
		}
	}

	public void searchedRecover() {
		this.panel.repaint();
		this.clearNode();
	}

	public void setSearchThread(SearchThread thread) {
		this.statusbar.setThread(thread);
	}

	/**
	 * 不要なデータの解放
	 */
	public synchronized void dump() {
		for (City map : this.cityMap.values()) {
			if (map.hasData()
					&& (this.panel.mode() < 2 || !this.screen.intersects(map
							.getArea()))) {
				map.dump();
			}
		}
	}

	/**
	 * 引数の都道府県番号に含まれていないデータの解放
	 * @param codes 都道府県番号
	 */
	public synchronized void dumpPrefecture(Set<Integer> codes) {
		for (City map : this.cityMap.values()) {
			int code = map.getCode() / 1000;
			if (!codes.contains(code)) {
				System.out.println();
				map.dumpKsj();
				this.prefecture[code - 1] = false;
			}
		}
	}

	public int size() {
		return this.cityMap.size();
	}

	public boolean isEmpty() {
		return this.cityMap.isEmpty();
	}

	public boolean containsKey(Object key) {
		return this.cityMap.containsKey(key);
	}

	public boolean containsValue(Object value) {
		return this.cityMap.containsValue(value);
	}

	public City remove(Object key) {
		return this.cityMap.remove(key);
	}

	public void putAll(Map<? extends Integer, ? extends City> t) {
		// TODO Auto-generated method stub
	}

	public void clear() {
		this.cityMap.clear();
	}

	public Set<Integer> keySet() {
		return this.cityMap.keySet();
	}

	public Collection<City> values() {
		return this.cityMap.values();
	}

	public Set<Entry<Integer, City>> entrySet() {
		return this.cityMap.entrySet();
	}
}
