package map;

import java.awt.Polygon;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import util.Log;

import map.data.City;

/**
 * 
 * 国土数値情報のデータ読み込むクラス
 * @author Masayasu Fujiwara
 */
public class KsjFactory {
	
	public static void main(String[] args) throws IOException {
		KsjFactory.delete("./data/");
	}
	private KsjFactory() {
	}

	public static void delete(String cache) throws IOException {
		for (int i = 1; i <= 47; i++) {
			String code = City.prefectureFormat(i);
			File file = new File(cache + code + File.separatorChar + "ksj" + code + ".dat");
			Log.out(KsjFactory.class, "delete " + file.getCanonicalPath());
			file.delete();
		}
	}

	public static void translate() throws IOException {
		Polygon[] polygon = readPolygon("");
		ObjectOutputStream out = null;
		try {
			out = new ObjectOutputStream(new FileOutputStream("./data/prefecture.dat"));
			out.writeObject(polygon);
			out.flush();
		} finally {
			if (out != null) {
				out.close();
			}
		}
	}

	/**
	 * 
	 * @param name 入力ストリーム
	 * @return Polygon配列
	 * @throws IOException 入出力エラー
	 */
	public static Polygon[] readPolygon(String name) throws IOException {
		Polygon[] polygon = null;
		ObjectInputStream in = null;
		try {
			in = new ObjectInputStream(System.class.getResourceAsStream(name));
			polygon = (Polygon[]) in.readObject();
		} catch (ClassNotFoundException e) {
			// TODO
			Log.err(KsjFactory.class, e);
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return polygon;
	}

	/**
	 * 地図情報からPolygonデータ読み込む
	 * @param stream 入力ストリーム
	 * @return 読み込んだPolygon配列を返す
	 * @throws IOException 入出力エラー
	 */
	public static Polygon[] getPolygon(final InputStream stream) throws IOException {
		List<Polygon> list = new ArrayList<Polygon>();
		BufferedReader in = null;
		try {
			in = new BufferedReader(new InputStreamReader(stream));
			String data;
			while ((data = in.readLine()) != null) {
				String[] param = data.split(",");
				int length = param.length / 2;
				int[] x = new int[length];
				int[] y = new int[length];	
				for (int i = 0; i < length; i++) {
					int index = i * 2;
					x[i] = Integer.parseInt(param[index]);
					y[i] = Integer.parseInt(param[index + 1]);
				}
				if((param.length % 2) != 0) {
					throw new IllegalArgumentException("データの形式が不正です。"+ param.length);
				}
				list.add(new Polygon(x, y, length));
			}
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return list.toArray(new Polygon[]{});
	}
	
	public static void translateCityKsj() throws IOException {
		for (int i = 1; i <= 47; i++) {
			String code = City.prefectureFormat(i);
			Map<Integer, List<Polygon>> map = new LinkedHashMap<Integer, List<Polygon>>();
			BufferedReader in = null;
			try {
				in = new BufferedReader(new FileReader("./data/" + code + "/ksj" + code + ".dat"));
				String data;
				while ((data = in.readLine()) != null) {
					String[] param = data.split(",");
					int length = param.length / 2;
					int[] x = new int[length];
					int[] y = new int[length];
					for (int j = 0, index = j * 2 + 1; j < length; j++, index += 2) {
						x[j] = Integer.parseInt(param[index]);
						y[j] = Integer.parseInt(param[index + 1]);
					}
					if ((param.length % 2) == 0) {
						throw new IllegalArgumentException("データの形式が不正です。"+ param.length);
					}
					int key = Integer.valueOf(param[0]);
					if (!map.containsKey(key)) {
						map.put(key, new ArrayList<Polygon>());
					}
					map.get(key).add(new Polygon(x, y, length));
				}
			} finally {
				if (in != null) {
					in.close();
				}
			}
			ObjectOutputStream out = null;
			try {
				out = new ObjectOutputStream(new FileOutputStream("./data/" + code + "/ksj.dat"));
				out.writeObject(map);
				out.flush();
			} finally {
				if (out != null) {
					out.close();
				}
			}
			Log.out(KsjFactory.class, "finish translate.");
		}
	}

	/**
	 * 都道府県内の各市区町村に対応した国土数値情報を読み込む
	 * @param code 都道府県番号 
	 * @return 市区町村に対応した国土数値情報の行政界
	 * @throws IOException 入出力エラー
	 */
	public static Map<Integer, List<Polygon>> readKsj(String code) throws IOException {
		Map<Integer, List<Polygon>> map = new LinkedHashMap<Integer, List<Polygon>>();
		BufferedReader in = null;
		try {
			in = new BufferedReader(new InputStreamReader(KsjFactory.class.getResourceAsStream("/data/" + code + "/ksj" + code + ".dat")));
			String data;
			while ((data = in.readLine()) != null) {
				String[] param = data.split(",");
				int length = param.length / 2;
				int[] x = new int[length];
				int[] y = new int[length];
				for (int i = 0, index = i * 2 + 1; i < length; i++, index += 2) {
					x[i] = Integer.parseInt(param[index]);
					y[i] = Integer.parseInt(param[index + 1]);
				}
				if ((param.length % 2) == 0) {
					throw new IllegalArgumentException("データの形式が不正です。"+ param.length);
				}
				int key = Integer.valueOf(param[0]);
				if (!map.containsKey(key)) {
					map.put(key, new ArrayList<Polygon>());
				}
				map.get(key).add(new Polygon(x, y, length));
			}
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return map;
	}

	/**
	 * 都道府県内の各市区町村に対応した国土数値情報を読み込む（Serialize）
	 * @param code 都道府県番号
	 * @return 市区町村に対応した国土数値情報の行政界
	 * @throws IOException 入出力エラー
	 */
	public static Map<Integer, List<Polygon>> readSerializeKsj(String code) throws IOException {
		Map<Integer, List<Polygon>> map = null;
		ObjectInputStream in = null;
		try {
			in = new ObjectInputStream(KsjFactory.class.getResourceAsStream("/data/" + code + "/ksj.dat"));
			map = (LinkedHashMap<Integer, List<Polygon>>) in.readObject();
		} catch (Exception e) {
			map = KsjFactory.readKsj(code);
		} finally {
			if (in != null) {
				in.close();
			}
		}
		return map;
	}
}
