package map;

import isj.ISJUtil;

import java.awt.Shape;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;

import shop.Cocoichi;
import shop.Convenience;
import shop.Emap;
import shop.KFC;
import shop.Matsuya;
import shop.McDonalds;
import shop.MisterDonut;
import shop.Mizuho;
import shop.Mos;
import shop.Resona;
import shop.SEJ;
import shop.SMBC;
import shop.Skylark;
import shop.Times;
import shop.UFJ;
import shop.Yoshinoya;

/**
 * 国土数値情報の行政界・海岸線（面）から作成された1つの市区町村を管理するクラスです。
 * @author Kumano Tatsuo
 * 2005/11/11
 */
public class City {
	/**
	 * 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	private boolean has2500;

	/**
	 * 外接長方形
	 */
	private final Rectangle2D bounds;

	/**
	 * 高精度の領域
	 */
	private Shape fineShape;

	/**
	 * 市区町村コード
	 */
	private final String id;

	/**
	 * 市区町村名
	 */
	private final String label;

	/**
	 * 都道府県名
	 */
	private final String prefecture;

	/**
	 * 領域
	 */
	private final Shape shape;

	/**
	 * 数値地図2500（空間データ基盤）のURL
	 */
	private final URL url;

	/**
	 * 街区レベル位置参照情報
	 */
	private Map<String, Point2D> isj;

	/**
	 * 街区レベル位置参照情報のラベル位置
	 */
	private Map<Point2D, String> isjLabels;

	/**
	 * 銀行の一覧
	 * @since 3.16
	 */
	private Collection<PointData> banks;

	/**
	 * コンビニの一覧
	 * @since 3.16
	 */
	private Collection<PointData> convenience;

	/**
	 * 駐車場の一覧
	 * @since 4.04
	 */
	private Collection<PointData> parkings;

	/**
	 * 飲食店の一覧
	 */
	private Collection<PointData> fastFoods;

	/**
	 * 地図を描画するパネル。ステータスバーのメッセージを伝えるために持ちます。
	 */
	private final MapPanel panel;

	/**
	 * 市区町村を初期化します。
	 * @param shape 領域
	 * @param label 市区町村名
	 * @param id 市区町村コード
	 * @param url 数値地図2500（空間データ基盤）のURL
	 * @param prefecture 都道府県名
	 * @param panel 地図を描画するパネル。ステータスバーのメッセージを伝えるために持ちます。
	 */
	public City(final Shape shape, final String label, final String id, final URL url, final String prefecture,
			final MapPanel panel) {
		this.shape = shape;
		this.panel = panel;
		this.bounds = shape.getBounds2D();
		this.label = label;
		this.id = id;
		this.url = url;
		this.prefecture = prefecture;
		this.isjLabels = new HashMap<Point2D, String>();
	}

	/**
	 * 街区レベル位置参照情報のラベル位置を空にします。
	 */
	public void clearIsjLabels() {
		this.isjLabels.clear();
	}

	/**
	 * 高精度の領域を開放します。
	 */
	public void freeFineShape() {
		this.fineShape = null;
	}

	/**
	 * 街区レベル位置参照情報を開放します。
	 */
	public void freeIsj() {
		this.isj = null;
	}

	/**
	 * @return 外接長方形
	 */
	public Rectangle2D getBounds() {
		return this.bounds;
	}

	/**
	 * @return 高精度の領域
	 */
	public Shape getFineShape() {
		return this.fineShape;
	}

	/**
	 * @return 市区町村コード
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * @return 街区レベル位置参照情報
	 */
	public Map<String, Point2D> getIsj() {
		return this.isj;
	}

	/**
	 * @return 街区レベル位置参照情報のラベル位置
	 */
	public Map<Point2D, String> getIsjLabels() {
		return this.isjLabels;
	}

	/**
	 * @return 市区町村名
	 */
	public String getLabel() {
		return this.label;
	}

	/**
	 * @return 領域
	 */
	public Shape getShape() {
		return this.shape;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）のURL
	 */
	public URL getURL() {
		return this.url;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	public boolean has2500() {
		return this.has2500;
	}

	/**
	 * @return 高精度の領域を持っているかどうか
	 */
	public boolean hasFineShape() {
		return this.fineShape != null;
	}

	/**
	 * @return 街区レベル位置参照情報を持っているかどうか
	 */
	public boolean hasIsj() {
		return this.isj != null;
	}

	/**
	 * 街区レベル位置参照情報をダウンロードし、読み込みます。
	 * @throws IOException 
	 */
	public void loadIsj() throws IOException {
		this.isj = ISJUtil.loadIsj(this.id, this.panel);
	}

	/**
	 * @param shape 高精度の領域
	 */
	public void setFineShape(final Shape shape) {
		this.fineShape = shape;
	}

	/**
	 * @param has2500 数値地図2500（空間データ基盤）を読み込んだかどうか 
	 */
	public void setHas2500(final boolean has2500) {
		this.has2500 = has2500;
	}

	/**
	 * @since 3.16
	 * @return 銀行の一覧
	 */
	public Collection<PointData> getBanks() {
		return this.banks;
	}

	/**
	 * @since 3.16
	 * @return 飲食店の一覧
	 */
	public Collection<PointData> getFastFoods() {
		return this.fastFoods;
	}

	/**
	 * @since 3.16
	 * @return コンビニの一覧
	 */
	public Collection<PointData> getConvenience() {
		return this.convenience;
	}

	/**
	 * @since 4.04
	 * @return 駐車場の一覧
	 */
	public Collection<PointData> getParkings() {
		return this.parkings;
	}

	/**
	 * @return 銀行の一覧を持っているかどうか
	 * @since 3.16
	 */
	public boolean hasBanks() {
		return this.banks != null;
	}

	/**
	 * @return 駐車場の一覧を持っているかどうか
	 * @since 4.04
	 */
	public boolean hasParkings() {
		return this.parkings != null;
	}

	/**
	 * @return 飲食店の一覧を持っているかどうか
	 * @since 3.16
	 */
	public boolean hasFastFoods() {
		return this.fastFoods != null;
	}

	/**
	 * @since 3.16
	 * @return コンビニの一覧を持っているかどうか
	 */
	public boolean hasConvenience() {
		return this.convenience != null;
	}

	/**
	 * 銀行の一覧を読み込みます。
	 * @throws IOException 
	 * @throws MalformedURLException 
	 * @since 3.10
	 */
	public void loadBanks() throws MalformedURLException, IOException {
		this.banks = new ArrayList<PointData>();
		loadEmap(new SMBC(), this.banks, Const.Smbc.CACHE_DIR, Const.Smbc.PREFIX, Const.Smbc.SUFFIX, "三井住友");
		final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
		for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
			tempIsj.put(entry4.getKey().replaceAll(",", ""), entry4.getValue());
		}
		final Map<Point2D, String> points = UFJ.load(this.prefecture, this.label, this.id, tempIsj, this.panel);
		points.putAll(Mizuho.load(this.prefecture, this.id, tempIsj, this.panel));
		points.putAll(Resona.load(this.prefecture, this.id, tempIsj, this.panel));
		for (final Map.Entry<Point2D, String> entry : points.entrySet()) {
			final Point2D point = entry.getKey();
			final String attribute = entry.getValue();
			final PointData pointData = new PointData(attribute, PointData.CLASSIFICATION_UNKNOWN, point.getX(), point
					.getY());
			pointData.setAttribute(attribute);
			this.banks.add(pointData);
		}
	}

	/**
	 * 飲食店の一覧を読み込みます。
	 * @throws IOException
	 * @since 3.16 
	 */
	public void loadFastFoods() throws IOException {
		this.fastFoods = new ArrayList<PointData>();
		final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
		for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
			tempIsj.put(entry4.getKey().replaceAll(",", ""), entry4.getValue());
		}
		final Map<Point2D, String> points = McDonalds.load(this.id, this.prefecture, this.label, tempIsj, this.panel);
		points.putAll(Mos.load(this.id, tempIsj, this.panel));
		points.putAll(KFC.load(this.id, this.prefecture, tempIsj, this.panel));
		points.putAll(MisterDonut.load(this.id, tempIsj, this.panel));
		points.putAll(Cocoichi.load(this.id, this.label, this.prefecture, tempIsj, this.panel));
		points.putAll(Yoshinoya.load(this.id, this.label, tempIsj, this.panel));
		points.putAll(Matsuya.load(this.id, this.label, tempIsj, this.panel));
		points.putAll(Skylark.load(this.id, this.label, this.id.substring(0, 2), tempIsj, this.panel));
		for (final Map.Entry<Point2D, String> entry : points.entrySet()) {
			final Point2D point = entry.getKey();
			final String attribute = entry.getValue();
			final PointData pointData = new PointData(attribute, PointData.CLASSIFICATION_UNKNOWN, point.getX(), point
					.getY());
			pointData.setAttribute(attribute);
			this.fastFoods.add(pointData);
		}
	}

	/**
	 * コンビニの一覧を読み込みます。
	 * @throws IOException 
	 * @since 3.16
	 */
	public void loadConvenience() throws IOException {
		this.convenience = new ArrayList<PointData>();
		loadEmap(new SEJ(), this.convenience, Const.Sej.CACHE_DIR, Const.Sej.PREFIX, Const.Sej.SUFFIX, "セブン");
		final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
		for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
			tempIsj.put(entry4.getKey().replaceAll(",", ""), entry4.getValue());
		}
		// since 4.07
		final Map<Point2D, String> points = new Convenience().getConvenience(this.id, this.label, this.prefecture,
				this.isj, this.panel);
		for (final Map.Entry<Point2D, String> entry : points.entrySet()) {
			final Point2D point = entry.getKey();
			final String attribute = entry.getValue();
			final PointData pointData = new PointData(attribute, PointData.CLASSIFICATION_UNKNOWN, point.getX(), point
					.getY());
			pointData.setAttribute(attribute);
			this.convenience.add(pointData);
		}

	}

	/**
	 * 駐車場の一覧を読み込みます。
	 * @throws IOException
	 * @since 4.04
	 */
	public void loadParkings() throws IOException {
		this.parkings = new ArrayList<PointData>();
		final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
		for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
			tempIsj.put(entry4.getKey().replaceAll(",", ""), entry4.getValue());
		}
		final Map<Point2D, String> points = McDonalds.load(this.id, this.prefecture, this.label, tempIsj, this.panel);
		points.putAll(Times.load(this.id, this.label, this.prefecture, tempIsj, this.panel));
		for (final Map.Entry<Point2D, String> entry : points.entrySet()) {
			final Point2D point = entry.getKey();
			final String attribute = entry.getValue();
			final PointData pointData = new PointData(attribute, PointData.CLASSIFICATION_UNKNOWN, point.getX(), point
					.getY());
			pointData.setAttribute(attribute);
			this.parkings.add(pointData);
		}
	}

	/**
	 * 三井住友銀行、セブンイレブンの一覧を読み込みます。
	 * @param emap 解析エンジン
	 * @param points 結果を格納するオブジェクト
	 * @param cacheDir キャッシュディレクトリ
	 * @param prefix 接頭語
	 * @param suffix 接尾語
	 * @param attribute 表示する属性
	 * @throws MalformedURLException
	 * @throws IOException
	 * @throws FileNotFoundException
	 * @throws UnsupportedEncodingException
	 */
	private void loadEmap(final Emap emap, final Collection<PointData> points, final String cacheDir,
			final String prefix, final String suffix, final String attribute) throws MalformedURLException,
			IOException, FileNotFoundException, UnsupportedEncodingException {
		File csvFile = new File(cacheDir + File.separator + prefix + String.valueOf(this.id) + suffix);
		if (!csvFile.exists()) {
			if (!new File(cacheDir).exists()) {
				new File(cacheDir).mkdir();
			}
			final Map<String, URL> prefectures = emap.getPrefectures();
			for (final Map.Entry<String, URL> entry : prefectures.entrySet()) {
				if (this.prefecture.contains(entry.getKey())) {
					this.panel.addMessage(entry.getValue() + "をダウンロードしています。");
					final Map<String, URL> cities = emap.getCities(entry.getValue());
					for (final Map.Entry<String, URL> entry2 : cities.entrySet()) {
						final String[] strings = entry2.getKey().split(",");
						if (strings.length == 2) {
							final String city = strings[1];
							if (city.equals(this.label)) {
								final PrintWriter out = new PrintWriter(new File(cacheDir + File.separator + prefix
										+ String.valueOf(this.id) + suffix), "SJIS");
								final Map<String, String> addresses = emap.getAddresses(entry2.getValue());
								final Map<String, Point2D> tempIsj = new HashMap<String, Point2D>();
								for (final Map.Entry<String, Point2D> entry4 : this.isj.entrySet()) {
									tempIsj.put(entry4.getKey().replaceAll(",", ""), entry4.getValue());
								}
								ISJUtil.parseAddresses(addresses, out, tempIsj);
								out.close();
							}
						} else {
							System.out.println("WARNING: データ形式が不正です。" + entry2);
						}
					}
					this.panel.removeMessage();
					break;
				}
			}
			csvFile = new File(cacheDir + File.separator + prefix + String.valueOf(this.id) + suffix);
		}
		loadShops(points, attribute, csvFile);
	}

	/**
	 * ファイルからお店の座標を読み込んでPointDataのリストに加えます。
	 * @param points PointDataの一覧
	 * @param attribute 表示する属性
	 * @param csvFile ファイル
	 * @throws UnsupportedEncodingException
	 * @throws FileNotFoundException
	 * @throws NumberFormatException
	 */
	private void loadShops(final Collection<PointData> points, final String attribute, File csvFile)
			throws UnsupportedEncodingException, FileNotFoundException, NumberFormatException {
		if (csvFile.exists()) {
			final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(csvFile), "SJIS"));
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final String[] items = line.split(",");
				if (items.length == 4) {
					final double x = Double.parseDouble(items[2]);
					final double y = Double.parseDouble(items[3]);
					final PointData pointData = new PointData(attribute, PointData.CLASSIFICATION_UNKNOWN, x, y);
					pointData.setAttribute(attribute);
					points.add(pointData);
				} else {
					System.out.println("WARNING: データ形式が不正です。" + line);
				}
			}
			scanner.close();
		}
	}

	/**
	 * @return 都道府県名
	 */
	public String getPrefecture() {
		return this.prefecture;
	}
}
