package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * りそな銀行の住所を取得するクラスです。
 * @author Kumano Tatsuo
 * 2006/03/12
 */
public class Resona {
	/**
	 * 最初のURL
	 */
	private static final String URL1 = "http://www.mapion.co.jp/c/f?vp=20&p=1&grp=resona&uc=21&ob=0&mx=500&bool=%28%2Bkey1%29*admi2code*sales1*sales2*sales3*sales4*sales5*sales6*sales7*sales8*edit9*edit10&pg=&sfn=resona_search-partinfo-list_00&oi=edit4&sales1=1&key1=100&key1=200&admi2code=";

	/**
	 * 次のURL
	 */
	private static final String URL2 = "&admi3=";
	
	/**
	 * 最後のURL
	 */
	private static final String URL3 = "&%B8%A1%BA%F7=%B9%CA%A4%EA%B9%FE%A4%DF";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "EUC-JP";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "resona_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * りそな銀行の座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param prefectureLabel 都道府県名
	 * @param cityID 市区町村コード
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String prefectureLabel, final String cityID,
		final Map<String, Point2D> isj, final MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Resona.CACHE_DIR).exists()) {
			new File(Resona.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Resona.CACHE_DIR + File.separator + Resona.PREFIX + cityID
			+ Resona.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(Resona.getAddresses(prefectureLabel, cityID, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
			cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "りそな");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * りそな銀行の市区町村別ページを解析して住所の一覧を取得します。
	 * @param prefectureLabel 都道府県名
	 * @param cityID 市区町村コード
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String prefectureLabel,
		final String cityID, final MapPanel panel) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL1 + cityID.substring(0, 2) + URL2 + cityID + URL3);
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			String caption = null;
			final Pattern pattern2 = Pattern.compile("<b>([^<>]+)</b><br>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				if (line.startsWith("りそな銀行")) {
					caption = line;
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					if (caption != null) {
						final String address = matcher2.group(1);
						if (address.startsWith(prefectureLabel)) {
							ret.put(address, caption);
							caption = null;
						}
					}
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return ret;
	}
}
