import java.awt.Color;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Rectangle2D.Float;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.Map.Entry;
import javax.swing.JFrame;
import javax.swing.WindowConstants;

/**
 * 国土数値情報を読み込むユーティリティクラスです。
 * @author Kumano Tatsuo
 * 2006/03/17
 */
public class KSJ {
	/**
	 * 国土数値情報の鉄道データが記録されたファイル名
	 */
	private final static String FILENAME_RAILROAD = "N02-07L-2K.txt";

	/**
	 * 国土数値情報の土地利用メッシュが記録されたファイル名の接頭語
	 */
	private final static String FILENAME_USAGE_PREFIX = "L03-09M-";

	/**
	 * 国土数値情報の土地利用メッシュが記録されたファイル名の接尾語
	 */
	private final static String FILENAME_USAGE_SUFFIX = ".txt";

	/**
	 * 文字コードセット
	 */
	private final static String CHARSET = "SJIS";

	/**
	 * 線路の一覧を読み込みます。
	 * @return 線路
	 */
	public static Collection<Line2D> loadRailways() {
		final Collection<Line2D> ret = new ArrayList<Line2D>();
		final Map<String, Point2D> nodes = new LinkedHashMap<String, Point2D>();
		final Scanner scanner = new Scanner(KSJ.class.getResourceAsStream(KSJ.FILENAME_RAILROAD),
			KSJ.CHARSET);
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			if (line.startsWith("N")) {
				final String id = line.substring(3, 9).trim() + "_" + line.substring(9, 15).trim();
				final float x = Integer.parseInt(line.substring(15, 23).trim()) / 36000f;
				final float y = Integer.parseInt(line.substring(23, 31).trim()) / 36000f;
				nodes.put(id, new Point2D.Float(x, y));
			} else if (line.startsWith("L")) {
				final String mesh = line.substring(3, 9).trim();
				final String node1 = mesh + "_" + line.substring(9, 15).trim();
				final String node2 = line.substring(15, 21).trim() + "_"
					+ line.substring(21, 27).trim();
				final String id = mesh + "_" + line.substring(27, 33).trim();
				if (nodes.containsKey(node1) && nodes.containsKey(node2)) {
					final Point2D point1 = nodes.get(node1);
					final Point2D point2 = nodes.get(node2);
					ret.add(new Line2D.Float((float) point1.getX(), (float) point1.getY(),
						(float) point2.getX(), (float) point2.getY()));
				}
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * 駅の一覧を読み込みます。
	 * Celeron 2GHzで4.3秒くらいかかります。
	 * @return 駅
	 */
	public static Map<Point2D, String> loadStations() {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		final Map<String, Point2D> map = new LinkedHashMap<String, Point2D>();
		final Scanner scanner = new Scanner(KSJ.class.getResourceAsStream(KSJ.FILENAME_RAILROAD),
			KSJ.CHARSET);
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			if (line.startsWith("N")) {
				final String hasDP = line.substring(31, 33).trim();
				if (hasDP.equals("1")) {
					final float x = Integer.parseInt(line.substring(15, 23).trim()) / 36000f;
					final float y = Integer.parseInt(line.substring(23, 31).trim()) / 36000f;
					final String dp = line.substring(33, 43).trim();
					map.put(dp, new Point2D.Float(x, y));
				}
			} else if (line.startsWith("DP")) {
				final String dp = line.substring(3, 13);
				final String name = line.substring(16).trim();
				if (map.containsKey(dp)) {
					ret.put(map.get(dp), name);
				}
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * 土地利用メッシュを読み込みます。
	 * @param prefecture 都道府県コード
	 * @return 土地利用メッシュ
	 */
	public static Map<Rectangle2D, String> loadUsage(final String prefecture) {
		final Map<Rectangle2D, String> ret = new LinkedHashMap<Rectangle2D, String>();
		final Scanner scanner = new Scanner(KSJ.class.getResourceAsStream(KSJ.FILENAME_USAGE_PREFIX
			+ prefecture + KSJ.FILENAME_USAGE_SUFFIX));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			if (line.startsWith("M")) {
				final String mesh = line.substring(5, 13);
				final String usage = line.substring(109, 209);
				ret.put(meshToRectangle(mesh), usage);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * 3次メッシュコードから、対応する長方形の経度、緯度を求めます。
	 * @param mesh 3次メッシュコード
	 * @return 長方形（単位は経度、緯度）
	 */
	public static Rectangle2D meshToRectangle(final String mesh) {
		if (mesh.matches("[0-9]{4}[0-7]{2}[0-9]{2}")) {
			final float latitude = Integer.parseInt(mesh.substring(0, 2)) / 1.5f
				+ Integer.parseInt(mesh.substring(4, 5)) * 2f / 3 / 8
				+ Integer.parseInt(mesh.substring(6, 7)) * 2f / 3 / 8 / 10;
			final float longitude = Integer.parseInt(mesh.substring(2, 4)) + 100
				+ Integer.parseInt(mesh.substring(5, 6)) / 8f
				+ Integer.parseInt(mesh.substring(7, 8)) / 8f / 10;
			return new Rectangle2D.Float(longitude, latitude, 1f / 8 / 10, 2f / 3 / 8 / 10);
		} else {
			throw new IllegalArgumentException("3次メッシュコードの形式が不正です。");
		}
	}

	/**
	 * 土地利用メッシュの1/10細分区画を分解します。
	 * @param mesh メッシュが対応する長方形
	 * @param usage 1/10細分区画
	 * @return 1/10細分区画が対応する長方形と土地利用との対応表
	 */
	public static Map<Rectangle2D, String> splitUsage(final Rectangle2D mesh, final String usage) {
		if (usage.matches("[125679ABEFG]{100}")) {
			final Map<Rectangle2D, String> ret = new LinkedHashMap<Rectangle2D, String>();
			for (int i = 0; i < 100; i++) {
				final int row = i / 10;
				final int column = i % 10;
				final Rectangle2D rectangle = new Rectangle2D.Float((float) mesh.getX() + column
					/ 8f / 10 / 10, (float) mesh.getY() + row * 2f / 3 / 8 / 10 / 10,
					1f / 8 / 10 / 10, 2f / 3 / 8 / 10 / 10);
				final String value = usage.substring(i, i + 1);
				ret.put(rectangle, value);
			}
			return ret;
		} else {
			throw new IllegalArgumentException("土地利用メッシュにおける1/10細分区画の形式が不正です。");
		}
	}

	/**
	 * テスト用のメソッドです。
	 * @param args
	 */
	public static void main(String[] args) {
		final Map<Rectangle2D, String> usage = KSJ.loadUsage("28");
		final JFrame frame = new JFrame("test");
		frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
		frame.setSize(800, 600);
		final ShapePanel panel = new ShapePanel();
		final AffineTransform transform = new AffineTransform();
		transform.scale(1000, -1000);
		for (final Map.Entry<Rectangle2D, String> entry : usage.entrySet()) {
			final Rectangle2D mesh = entry.getKey();
			final String value = entry.getValue();
			for (final Map.Entry<Rectangle2D, String> entry2 : splitUsage(mesh, value).entrySet()) {
				final Shape shape = transform.createTransformedShape(entry2.getKey());
				panel.add(shape);
				panel.setBorderColor(shape, null);
				final String value2 = entry2.getValue();
				if (value2.contains("E")) {
					panel.setFillColor(shape, Color.YELLOW);
				} else if (value2.contains("F")) {
					panel.setFillColor(shape, Color.BLUE);
				} else if (value2.contains("5")) {
					panel.setFillColor(shape, Color.GREEN.darker());
				} else if (value2.contains("7")) {
					panel.setFillColor(shape, Color.RED);
				} else if (value2.contains("9")) {
					panel.setFillColor(shape, Color.LIGHT_GRAY);
				} else if (value2.contains("G")) {
					panel.setFillColor(shape, Color.GREEN.darker().darker());
				} else if (value2.contains("B")) {
					panel.setFillColor(shape, Color.CYAN);
				} else if (value2.contains("1")) {
					panel.setFillColor(shape, Color.GREEN);
				} else if (value2.contains("6")) {
					panel.setFillColor(shape, Color.ORANGE.darker());
				} else if (value2.contains("2")) {
					panel.setFillColor(shape, Color.GREEN);
				} else if (value2.contains("A")) {
					panel.setFillColor(shape, Color.ORANGE);
				} else {
					panel.setFillColor(shape, Color.WHITE);
				}
			}
		}
		frame.add(panel);
		frame.setVisible(true);
	}
}
