package map.data;

import java.awt.Polygon;
import java.awt.Rectangle;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import view.MapPanel;
import view.StatusBar;

import map.map25000.Map25000Factory;
import map.map25000.cell.CellSearch;
import map.route.SearchThread;

/**
 * 市区町村管理クラス
 * 
 * @author ma38su
 *
 */
public class CityMap implements Map<Integer, City>, Runnable {

	/**
	 * 地図パネルクラス
	 */
	private final MapPanel panel;

	/**
	 * セル型の地域検索クラス
	 */
	private final CellSearch cell;

	/**
	 * 地図データ作成クラス
	 */
	private final Map25000Factory factory;

	/**
	 * ステータスバー
	 */
	private final StatusBar statusbar;

	/**
	 * 都道府県番号の読み込み状態
	 * true  -> 既読
	 * false -> 未読
	 */
	private boolean[] prefecture;

	private Rectangle screen;

	/**
	 * 市区町村データのMap
	 */
	private final Map<Integer, City> cityMap;

	public CityMap(MapPanel panel, CellSearch cell, Map25000Factory factory,
			StatusBar statusbar) {
		this.cityMap = new HashMap<Integer, City>();
		this.panel = panel;
		this.cell = cell;
		this.factory = factory;
		this.statusbar = statusbar;
		this.prefecture = new boolean[47];
	}

	public void start() {
		this.screen = this.panel.getScreen();
		Thread thread = new Thread(this);
		thread.setPriority(3);
		thread.start();
	}

	/**
	 * 地図情報を加える
	 * @param key 
	 * @param value 
	 * 
	 * @return 市区町村
	 */
	public synchronized City put(Integer key, City value) {
		return this.cityMap.put(key, value);
	}

	/**
	 * 指定した頂点番号の頂点を取得
	 * @param id 頂点番号
	 * @param isDetailRoad 
	 * @return 頂点
	 * @throws IOException 
	 */
	public Node getNode(long id, boolean isDetailRoad) throws IOException {
		City map;
		synchronized (this) {
			map = this.cityMap.get((int) (id / 1000000));
		}
		if (map == null) {
			map = this.factory.productNode((int) (id / 1000000), isDetailRoad);
		} else if (!map.hasNode()) {
			this.factory.productNode(map, isDetailRoad);
		}
		Node node = map.getNode((int) (id % 1000000));
		if (node == null) {
			System.out.println("Search node is not found, and reproduct node. "
					+ this);
			this.factory.productNode(map, isDetailRoad);
			node = map.getNode((int) (id % 1000000));
			if (node == null) {
				throw new IOException();
			}
		}
		return node;
	}

	/**
	 * 市区町村データの取得
	 * @param key 市区町村番号
	 * @return 市区町村データ
	 */
	public synchronized City get(Object key) {
		return this.cityMap.get(key);
	}

	public void removeNode(long id) {
		City map;
		synchronized (this) {
			map = this.get((int) (id / 1000000));
		}
		if (!map.isLoaded()) {
			map.removeNode((int) (id % 1000000));
		}
	}

	/**
	 * 頂点のCollectionを取得
	 * @param code 取得する頂点の市町村番号
	 * @return 頂点のCollection
	 */
	public synchronized Collection<Node> getNodes(int code) {
		return this.cityMap.get(code).getNode();
	}

	public void run() {
		while (true) {
			System.out.println("CityMap: running...");
			try {
				boolean isReadNewData = false;
				try {
					if (this.panel.mode() > 0) {
						Map<Integer, Set<Integer>> codes = this.cell
								.search(this.screen);
						System.out.println("CityMap: codes = " + codes);
						if (!this.panel.isOperation()) {
							this.dump(codes.keySet());
							System.out.println("CityMap: dumped.");
						}
						for (Map.Entry<Integer, Set<Integer>> entry : codes
								.entrySet()) {
							this.readKsjBorder(entry.getKey());
							System.out.println("CityMap: read ksj border.");
							isReadNewData = this.readCity(entry.getValue());
							System.out.println("CityMap: isReadNewData = " + isReadNewData);
						}
					}
				} catch (IOException e) {
					this.statusbar.ioException();
					isReadNewData = false;
				} catch (OutOfMemoryError e) {
					this.statusbar.memoryError();
					isReadNewData = false;
				} finally {
					this.dump();
				}

				if (!isReadNewData) {
					Thread.sleep(2000L);
				}
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * 国土数値情報の行政界の読み込み
	 * @param code
	 * @throws IOException
	 */
	private void readKsjBorder(int code) throws IOException {
		if (!this.prefecture[code - 1]) {
			String format = City.prefectureFormat(code);
			Map<Integer, List<Polygon>> map = new LinkedHashMap<Integer, List<Polygon>>();
			BufferedReader in = null;
			try {
				in = new BufferedReader(new InputStreamReader(this.getClass()
						.getResourceAsStream(
								"/data/" + format + "/ksj" + format + ".dat")));
				String data;
				while ((data = in.readLine()) != null) {
					String[] param = data.split(",");
					int length = param.length / 2;
					int[] x = new int[length];
					int[] y = new int[length];
					for (int i = 0, index = i * 2 + 1; i < length; i++, index += 2) {
						x[i] = Integer.parseInt(param[index]);
						y[i] = Integer.parseInt(param[index + 1]);
					}
					if ((param.length % 2) == 0) {
						System.out.println("error!!");
					}
					int key = Integer.valueOf(param[0]);
					if (!map.containsKey(key)) {
						map.put(key, new ArrayList<Polygon>());
					}
					map.get(key).add(new Polygon(x, y, length));
				}
			} finally {
				if (in != null) {
					in.close();
				}
			}
			for (Map.Entry<Integer, List<Polygon>> entry : map.entrySet()) {
				this.put(entry.getKey(), new City(entry.getKey(), entry
						.getValue().toArray(new Polygon[] {})));
			}
			this.prefecture[code - 1] = true;
			this.panel.repaint();
		}
	}

	/**
	 * 市町村データの読み込み
	 * @param codes
	 * @return 新たなデータを読み込めばtrueを返します。
	 * @throws IOException
	 */
	private boolean readCity(Collection<Integer> codes) throws IOException {
		System.out.println("CityMap: readCity started.");
		boolean flag = false;
		// セルメソッドから地図の読み込み
		System.out.println("CityMap: codes = " + codes);
		for (int code : codes) {
			System.out.println("CityMap: code = " + code);
			if (this.panel.mode() != 2) {
				System.out.println("CityMap: break loop.");
				break;
			}
			City city = this.get(code);
			System.out.println("CityMap: city = " + city);
			if (city == null) {
				System.out.println("CityMap: city was null.");
				this.statusbar.setCheckCode(City.cityFormat(code));
				System.out.println("CityMap: set check code.");
				city = this.factory.preproduct(code);
				System.out.println("CityMap: preproduct " + code);
				this.statusbar.clearCheckCode();
				this.put(code, city);
				System.out.println("CityMap: put " + code + ", " + city);
			} else if (city.getArea() == null) {
				System.out.println("CityMap: city.getArea was null.");
				this.statusbar.setCheckCode(City.cityFormat(city.getCode()));
				System.out.println("CityMap: set check code.");
				this.factory.preproduct(city);
				System.out.println("CityMap: preproduct " + code);
				this.statusbar.clearCheckCode();
				System.out.println("CityMap: cleared check code.");
			}
			if (this.screen.intersects(city.getArea())) {
				System.out.println("CityMap: screen contains city.");
				if (!city.hasData()) {
					System.out.println("CityMap: city has no data.");
					this.statusbar.setReading(City.cityFormat(city.getCode()));
					this.factory.product(city);
					this.statusbar.clearReading();
					this.panel.repaint();
					flag = true;
				}
			}
		}
		System.out.println("CityMap: readCity finished. flag = " + flag);
		return flag;
	}

	public synchronized void clearNode() {
		for (City map : this.cityMap.values()) {
			if (!map.hasData() && map.hasNode()) {
				map.dump();
			}
			if (map.isLoaded() && !this.screen.intersects(map.getArea())) {
				map.dump();
			}
		}
	}

	public void searchedRecover() {
		this.panel.repaint();
		this.clearNode();
	}

	public void setSearchThread(SearchThread thread) {
		this.statusbar.setThread(thread);
	}

	/**
	 * 不要なデータの解放
	 */
	public synchronized void dump() {
		for (City map : this.cityMap.values()) {
			if (map.hasData()
					&& (this.panel.mode() < 2 || !this.screen.intersects(map
							.getArea()))) {
				map.dump();
			}
		}
	}

	/**
	 * 指定した市区町村データの解放
	 * @param codes 市区町村番号のSet
	 */
	public synchronized void dump(Set<Integer> codes) {
		for (City map : this.cityMap.values()) {
			int code = map.getCode() / 1000;
			if (!codes.contains(code)) {
				map.dumpKsj();
				this.prefecture[code - 1] = false;
			}
		}
	}

	public int size() {
		return this.cityMap.size();
	}

	public boolean isEmpty() {
		return this.cityMap.isEmpty();
	}

	public boolean containsKey(Object key) {
		return this.cityMap.containsKey(key);
	}

	public boolean containsValue(Object value) {
		return this.cityMap.containsValue(value);
	}

	public City remove(Object key) {
		return this.cityMap.remove(key);
	}

	public void putAll(Map<? extends Integer, ? extends City> t) {
		// TODO Auto-generated method stub

	}

	public void clear() {
		this.cityMap.clear();
	}

	public Set<Integer> keySet() {
		return this.cityMap.keySet();
	}

	public Collection<City> values() {
		return this.cityMap.values();
	}

	public Set<Entry<Integer, City>> entrySet() {
		return this.cityMap.entrySet();
	}
}
