package map;

import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Point;
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import search.CellSearch;
import search.Search;

/**
 * 属性を描画する座標を計算するクラスです。
 * @author Kumano Tatsuo
 * 作成日: 2004/01/16
 */
class FixAttributeLocation {
	/** 属性を描画する座標を決定します。
	 * @param maps 地図
	 * @param prefectures 都道府県の一覧
	 * @param panel パネル
	 * @throws IOException 
	 */
	public void fixAttributeLocation(final Map<String, MapData> maps,
		final Collection<Prefecture> prefectures, final MapPanel panel) throws IOException {
		final MapPreferences preferences = panel.getMapPreferences();
		final Font tatemonoFont = preferences.getTatemonoPreferences().getFont();
		final Font zyoutiFont = preferences.getZyoutiPreferences().getFont();
		final Font mizuFont = preferences.getMizuPreferences().getFont();
		final Font tyomeFont1 = new Font(Const.FONT_FAMILY_GOTHIC, Font.PLAIN, (int) (18 * panel
			.getFontZoom()));
		final Font tyomeFont2 = new Font(Const.FONT_FAMILY_GOTHIC, Font.PLAIN, (int) (16 * panel
			.getFontZoom()));
		final Font tyomeFont3 = new Font(Const.FONT_FAMILY_GOTHIC, Font.PLAIN, (int) (14 * panel
			.getFontZoom()));
		final Font ekiFont = preferences.getEkiPreferences().getFont();
		final Font roadFont = preferences.getNormalRoadPreferences().getFont();
		final Font tetudouFont = preferences.getRailwayPreferences().getFont();
		final Search search = new CellSearch(1000, 100);
		final Rectangle2D visibleRectangle = panel.getVisibleRectangle(); // 現在画面に表示されている範囲（仮想座標）
		final Collection<Point> usedPoints = new HashSet<Point>(); // 既に配置されたポリゴンの中心座標（仮想座標）
		final double tatemonoPointSize = 4; // 建物に表示する点の直径
		final double ekiPointSize = 8; // 駅に表示する点の直径
		final double zoom = panel.getZoom();
		if (zoom >= Const.Zoom.LOAD_2500) {
			// 駅の属性の表示位置を計算する
			for (MapData mapData : maps.values()) {
				if (mapData.hasEki()) {
					if (panel.isVisible(mapData.getBounds())) {
						for (PointData point : mapData.getEki().values()) {
							Rectangle2D pointRectangle = new Rectangle2D.Double(point.getX()
								- (ekiPointSize / 2 / zoom), point.getY()
								- (ekiPointSize / 2 / zoom), ekiPointSize / zoom, ekiPointSize
								/ zoom);
							search.insert(pointRectangle, null);
						}
					}
				}
			}
			for (MapData mapData : maps.values()) {
				if (mapData.hasEki()) {
					if (panel.isVisible(mapData.getBounds())) {
						FontMetrics metrics = panel.getFontMetrics(ekiFont);
						double attributeHeight = metrics.getHeight() / zoom;
						for (PointData point : mapData.getEki().values()) {
							fixPointAttributeLocation(point, usedPoints, visibleRectangle, zoom,
								attributeHeight, metrics, ekiPointSize, search);
						}
					}
				}
			}
		}
		// 店舗の表示位置を計算する
		if (zoom >= Const.Zoom.LOAD_ALL) {
			final FontMetrics metrics = panel.getFontMetrics(panel.getMapPreferences()
				.getTatemonoPreferences().getFont());
			final double attributeHeight = metrics.getHeight() / zoom;
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.hasCities()) {
					for (final City city : prefecture.getCities()) {
						if (panel.isVisible(city.hasFineShape() ? city.getFineShape() : city
							.getShape())) {
							if (city.hasBanks()) {
								for (final PointData point : city.getBanks()) {
									fixPointAttributeLocation(point, usedPoints, visibleRectangle,
										zoom, attributeHeight, metrics, tatemonoPointSize, search);
								}
							}
							if (city.hasConvenience()) {
								for (final PointData point : city.getConvenience()) {
									fixPointAttributeLocation(point, usedPoints, visibleRectangle,
										zoom, attributeHeight, metrics, tatemonoPointSize, search);
								}
							}
							if (city.hasFastFoods()) {
								for (final PointData point : city.getFastFoods()) {
									fixPointAttributeLocation(point, usedPoints, visibleRectangle,
										zoom, attributeHeight, metrics, tatemonoPointSize, search);
								}
							}
						}
					}
				}
			}
		}
		if (zoom >= Const.Zoom.LOAD_2500) {
			// 公共建物の属性の表示位置を計算する
			long startTime = System.currentTimeMillis();
			for (MapData mapData : maps.values()) {
				if (mapData.hasTatemono()) {
					if (panel.isVisible(mapData.getBounds())) {
						FontMetrics metrics = panel.getFontMetrics(tatemonoFont);
						double attributeHeight = metrics.getHeight() / zoom;
						for (PolygonData polygon : mapData.getTatemono().values()) {
							fixTatemonoAttributeLocation(polygon, usedPoints, visibleRectangle,
								zoom, attributeHeight, metrics, tatemonoPointSize, search);
						}
					}
				}
			}
			if ((System.currentTimeMillis() - startTime) > 200) {
				System.out.println("DEBUG: （建物の属性配置：" + (System.currentTimeMillis() - startTime)
					+ " ms）");
			}
			// 場地の属性の表示位置を計算する
			for (MapData mapData : maps.values()) {
				if (mapData.hasZyouti()) {
					if (panel.isVisible(mapData.getBounds())) {
						FontMetrics metrics = panel.getFontMetrics(zyoutiFont);
						double attributeHeight = metrics.getHeight() / zoom;
						for (PolygonData polygon : mapData.getZyouti().values()) {
							fixPolygonAttributeLocation(polygon, usedPoints, visibleRectangle,
								zoom, attributeHeight, metrics, tatemonoPointSize, search);
						}
					}
				}
			}
			// 内水面の属性の表示位置を計算する
			for (MapData mapData : maps.values()) {
				if (mapData.hasMizu()) {
					if (panel.isVisible(mapData.getBounds())) {
						FontMetrics metrics = panel.getFontMetrics(mizuFont);
						double attributeHeight = metrics.getHeight() / zoom;
						for (PolygonData polygon : mapData.getMizu().values()) {
							fixPolygonAttributeLocation(polygon, usedPoints, visibleRectangle,
								zoom, attributeHeight, metrics, tatemonoPointSize, search);
						}
					}
				}
			}
			// 道路の属性の表示位置を計算する
			{
				FontMetrics metrics = panel.getFontMetrics(roadFont);
				double attributeHeight = metrics.getHeight() / zoom;
				Collection<String> fixedAttributes = new HashSet<String>();
				for (MapData mapData : maps.values()) {
					if (mapData.hasRoadArc()) {
						if (panel.isVisible(mapData.getBounds())) {
							for (ArcData arc : mapData.getRoadArc().values()) {
								String attribute = arc.getAttribute();
								if (attribute != null) {
									arc.setAttributeLocation(0, 0);
									if (!fixedAttributes.contains(attribute)) {
										double attributeWidth = metrics.stringWidth(arc
											.getAttribute())
											/ zoom;
										Point2D currentPoint = arc.getPath().getCurrentPoint();
										Rectangle2D attributeRectangle = new Rectangle2D.Double(
											currentPoint.getX() - (attributeWidth / 2),
											currentPoint.getY() - attributeHeight, attributeWidth,
											attributeHeight);
										if (visibleRectangle.contains(attributeRectangle)
											&& search.search(attributeRectangle).isEmpty()) {
											arc.setAttributeLocation(currentPoint.getX()
												- (attributeWidth / 2), currentPoint.getY());
											fixedAttributes.add(attribute);
											search.insert(attributeRectangle, null);
										}
									}
								}
							}
						}
					} else if (mapData.hasLargeRoadArc()) {
						if (panel.isVisible(mapData.getBounds())) {
							for (ArcData arc : mapData.getLargeRoadArc().values()) {
								String attribute = arc.getAttribute();
								if (attribute != null) {
									arc.setAttributeLocation(0, 0);
									if (!fixedAttributes.contains(attribute)) {
										double attributeWidth = metrics.stringWidth(arc
											.getAttribute())
											/ zoom;
										Point2D currentPoint = arc.getPath().getCurrentPoint();
										Rectangle2D attributeRectangle = new Rectangle2D.Double(
											currentPoint.getX() - (attributeWidth / 2),
											currentPoint.getY() - attributeHeight, attributeWidth,
											attributeHeight);
										if (visibleRectangle.contains(attributeRectangle)
											&& search.search(attributeRectangle).isEmpty()) {
											arc.setAttributeLocation(currentPoint.getX()
												- (attributeWidth / 2), currentPoint.getY());
											fixedAttributes.add(attribute);
											search.insert(attributeRectangle, null);
										}
									}
								}
							}
						}
					}
				}
			}
			// 鉄道の属性の表示位置を計算する
			{
				FontMetrics metrics = panel.getFontMetrics(tetudouFont);
				double attributeHeight = metrics.getHeight() / zoom;
				Collection<String> fixedAttributes = new HashSet<String>();
				for (MapData mapData : maps.values()) {
					if (mapData.hasOthers()) {
						if (panel.isVisible(mapData.getBounds())) {
							for (ArcData arc : mapData.getOthers().values()) {
								String attribute = arc.getAttribute();
								if (attribute != null) {
									arc.setAttributeLocation(0, 0);
									if (!fixedAttributes.contains(attribute)) {
										double attributeWidth = metrics.stringWidth(arc
											.getAttribute())
											/ zoom;
										PathIterator iter3 = arc.getPath().getPathIterator(
											new AffineTransform());
										double[] coords = new double[6];
										while (!iter3.isDone()) {
											int currentSegment = iter3.currentSegment(coords);
											if (currentSegment == PathIterator.SEG_LINETO) {
												double currentX = coords[0];
												double currentY = coords[1];
												Rectangle2D attributeRectangle = new Rectangle2D.Double(
													currentX - (attributeWidth / 2), currentY
														- attributeHeight, attributeWidth,
													attributeHeight);
												if (visibleRectangle.contains(attributeRectangle)
													&& search.search(attributeRectangle).isEmpty()) {
													arc.setAttributeLocation(currentX
														- (attributeWidth / 2), currentY);
													fixedAttributes.add(attribute);
													search.insert(attributeRectangle, null);
													break;
												}
											}
											iter3.next();
										}
									}
								}
							}
						}
					}
				}
			}
			// 丁目の属性の表示位置を計算する
			startTime = System.currentTimeMillis();
			for (MapData mapData : maps.values()) {
				if (mapData.hasTyome()) {
					if (panel.isVisible(mapData.getBounds())) {
						for (PolygonData polygon : mapData.getTyome().values()) {
							polygon.setAttributeLocation(0, 0);
							fixTyomeAttributeLocation(polygon, tyomeFont1, visibleRectangle,
								usedPoints, panel, true, search);
							fixTyomeAttributeLocation(polygon, tyomeFont2, visibleRectangle,
								usedPoints, panel, true, search);
							fixTyomeAttributeLocation(polygon, tyomeFont3, visibleRectangle,
								usedPoints, panel, false, search);
						}
					}
				}
			}
			// 町丁目の読みの表示位置を計算する
			if (zoom >= Const.Zoom.LOAD_ALL) {
				for (MapData mapData : maps.values()) {
					if (mapData.hasTyome()) {
						if (panel.isVisible(mapData.getBounds())) {
							for (PolygonData polygon : mapData.getTyome().values()) {
								final Font yomiFont = preferences.getTyomePreferences().getFont();
								final FontMetrics yomiMetrics = panel.getFontMetrics(yomiFont);
								final Font tyomeFont = polygon.getTyomeFont();
								if (tyomeFont != null) {
									final FontMetrics tyomeMetrics = panel
										.getFontMetrics(tyomeFont);
									if (polygon.getClassificationCode() == PolygonData.CLASSIFICATION_TYOME) {
										if (polygon.getAttribute() != null) {
											if (panel.getVisibleRectangle().contains(
												polygon.getAttributeX(), polygon.getAttributeY())) {
												if (polygon.hasYomi()) {
													final double kanjiWidth = tyomeMetrics
														.stringWidth(polygon.getAttribute()
															.replaceFirst("[0-9０-９]+$", ""))
														/ zoom;
													final double kanjiHeight = tyomeMetrics
														.getHeight()
														/ zoom;
													final double yomiWidth = yomiMetrics
														.stringWidth(polygon.getYomi())
														/ zoom;
													final double yomiHeight = yomiMetrics
														.getHeight()
														/ zoom;
													final double yomiX = polygon.getAttributeX()
														+ (kanjiWidth - yomiWidth) / 2;
													final double yomiY = polygon.getAttributeY()
														- kanjiHeight;
													final Rectangle2D rectangle = new Rectangle2D.Double(
														yomiX, yomiY - yomiHeight, yomiWidth, yomiHeight);
													if (panel.getVisibleRectangle().contains(
														rectangle)) {
														if (search.search(rectangle).isEmpty()) {
															polygon.setYomiLocation(yomiX, yomiY);
															search.insert(rectangle, null);
														} else {
															polygon.setYomiLocation(0, 0);
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			if ((System.currentTimeMillis() - startTime) > 200) {
				System.out.println("DEBUG: （丁目の属性配置：" + (System.currentTimeMillis() - startTime)
					+ " ms）");
			}
		}
		// 街区レベル位置参照情報の表示位置を計算する
		if (zoom >= Const.Zoom.LOAD_ALL) {
			final FontMetrics metrics = panel.getFontMetrics(panel.getMapPreferences()
				.getIsjPreferences().getFont());
			final double attributeHeight = metrics.getHeight() / zoom;
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.hasCities()) {
					for (final City city : prefecture.getCities()) {
						if (panel.isVisible(city.hasFineShape() ? city.getFineShape() : city
							.getShape())) {
							city.clearIsjLabels();
							if (city.hasIsj()) {
								for (Map.Entry<String, Point2D> entry : city.getIsj().entrySet()) {
									final String key = entry.getKey();
									final String[] strings = key.split(",");
									if (strings.length == 4) {
										final Point2D point = entry.getValue();
										if (panel.getVisibleRectangle().contains(point)) {
											final double attributeWidth = metrics
												.stringWidth(strings[3])
												/ zoom;
											final Rectangle2D rectangle = new Rectangle2D.Double(
												point.getX() - attributeWidth / 2, point.getY()
													- attributeHeight / 2, attributeWidth,
												attributeHeight);
											if (panel.getVisibleRectangle().contains(rectangle)) {
												if (search.search(rectangle).isEmpty()) {
													city.getIsjLabels().put(
														new Point2D.Double(point.getX()
															- attributeWidth / 2, point.getY()
															+ attributeHeight / 2), strings[3]);
													search.insert(rectangle, null);
												}
											}
										}
									} else {
										System.out.println("WARNING: 街区レベル位置参照情報の形式が不正です。" + key);
									}
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * 町丁目の属性位置を決定します。
	 * @param polygon ポリゴン
	 * @param font フォント
	 * @param visibleRectangle 表示されている矩形
	 * @param usedPoints 既に属性が配置された点
	 * @param panel パネル
	 * @param contains 完全に含まれているかどうか
	 * @param search 既に属性が配置されている領域を検索するためのオブジェクト
	 */
	private void fixTyomeAttributeLocation(PolygonData polygon, Font font,
		Rectangle2D visibleRectangle, Collection<Point> usedPoints, MapPanel panel,
		boolean contains, final Search search) {
		if (polygon.getAttribute() != null) {
			double zoom = panel.getZoom();
			FontMetrics metrics = panel.getFontMetrics(font);
			double attributeHeight = metrics.getHeight() / zoom;
			double size = 4;
			double attributeWidth = metrics.stringWidth(polygon.getAttribute()) / zoom;
			if (contains) {
				if (polygon.getArea().getBounds().getWidth() < attributeWidth) {
					return;
				}
			}
			Rectangle2D pointRectangle = new Rectangle2D.Double(-polygon.getX() - (size / zoom),
				-polygon.getY() - (size / zoom), (size * 2) / zoom, (size * 2) / zoom);
			Rectangle2D attributeRectangle;
			// ポリゴンの中心
			attributeRectangle = new Rectangle2D.Double(polygon.getX() - (attributeWidth / 2),
				polygon.getY() - (attributeHeight / 2), attributeWidth, attributeHeight);
			Point point = new Point((int) polygon.getX(), (int) polygon.getY());
			if (isTyomePutable(polygon, point, usedPoints, attributeRectangle, visibleRectangle,
				contains, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				polygon.setTyomeFont(font);
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
			int div = 4; // 分割数
			double dx; // 動かす幅
			double dy; // 動かす高さ
			dx = (polygon.getArea().getBounds().getWidth() - attributeWidth) / 2 / div;
			dy = (polygon.getArea().getBounds().getHeight() - attributeHeight) / 2 / div;
			for (int i = 1; i <= div; i++) {
				// ポリゴンの中心から上
				attributeRectangle = new Rectangle2D.Double(polygon.getX() - (attributeWidth / 2),
					polygon.getY() - (attributeHeight / 2) - (dy * i), attributeWidth,
					attributeHeight);
				if (isTyomePutable(polygon, point, usedPoints, attributeRectangle,
					visibleRectangle, contains, search)) {
					polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
						.getMaxY());
					polygon.setTyomeFont(font);
					search.insert(pointRectangle, null);
					search.insert(attributeRectangle, null);
					usedPoints.add(point);
				}
				// ポリゴンの中心から下
				attributeRectangle = new Rectangle2D.Double(polygon.getX() - (attributeWidth / 2),
					polygon.getY() - (attributeHeight / 2) + (dy * i), attributeWidth,
					attributeHeight);
				if (isTyomePutable(polygon, point, usedPoints, attributeRectangle,
					visibleRectangle, contains, search)) {
					polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
						.getMaxY());
					polygon.setTyomeFont(font);
					search.insert(pointRectangle, null);
					search.insert(attributeRectangle, null);
					usedPoints.add(point);
				}
				// ポリゴンの中心から右
				attributeRectangle = new Rectangle2D.Double(polygon.getX() - (attributeWidth / 2)
					+ (dx * i), polygon.getY() - (attributeHeight / 2), attributeWidth,
					attributeHeight);
				if (isTyomePutable(polygon, point, usedPoints, attributeRectangle,
					visibleRectangle, contains, search)) {
					polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
						.getMaxY());
					polygon.setTyomeFont(font);
					search.insert(pointRectangle, null);
					search.insert(attributeRectangle, null);
					usedPoints.add(point);
				}
				// ポリゴンの中心から左
				attributeRectangle = new Rectangle2D.Double(polygon.getX() - (attributeWidth / 2)
					- (dx * i), polygon.getY() - (attributeHeight / 2), attributeWidth,
					attributeHeight);
				if (isTyomePutable(polygon, point, usedPoints, attributeRectangle,
					visibleRectangle, contains, search)) {
					polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
						.getMaxY());
					polygon.setTyomeFont(font);
					search.insert(pointRectangle, null);
					search.insert(attributeRectangle, null);
					usedPoints.add(point);
				}
			}
		}
	}

	/**
	 * 点の属性の表示位置を決定します。
	 * @param point 点
	 * @param usedPoints 既に属性が配置されたポリゴンの中心座標
	 * @param visibleRectangle 現在表示されている領域
	 * @param zoom 倍率
	 * @param attributeHeight 属性の高さ
	 * @param metrics フォントメトリクス
	 * @param size 点の直径
	 * @param search 属性によって使用済みの領域を検索するためのオブジェクト
	 */
	private void fixPointAttributeLocation(PointData point, Collection<Point> usedPoints,
		Rectangle2D visibleRectangle, double zoom, double attributeHeight, FontMetrics metrics,
		double size, final Search search) {
		if (point.getAttribute() != null) {
			point.setAttributeLocation(0, 0);
			double attributeWidth = metrics.stringWidth(point.getAttribute()) / zoom;
			Rectangle2D attributeRectangle;
			Rectangle2D pointRectangle = new Rectangle2D.Double(point.getX() - (size / zoom), point
				.getY()
				- (size / zoom), (size * 2) / zoom, (size * 2) / zoom);
			// 点の右
			attributeRectangle = new Rectangle2D.Double(point.getX() + (size / zoom), point.getY()
				- (attributeHeight / 2), attributeWidth, attributeHeight);
			Point location = new Point((int) point.getX(), (int) point.getY());
			if (isPointPutable(attributeRectangle, location, usedPoints, visibleRectangle, search)) {
				point.setAttributeLocation(attributeRectangle.getX(), attributeRectangle.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(location);
			}
			// 点の左
			attributeRectangle = new Rectangle2D.Double(point.getX() - attributeWidth
				- (size / zoom), point.getY() - (attributeHeight / 2), attributeWidth,
				attributeHeight);
			if (isPointPutable(attributeRectangle, location, usedPoints, visibleRectangle, search)) {
				point.setAttributeLocation(attributeRectangle.getX(), attributeRectangle.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(location);
			}
			// 点の右上
			attributeRectangle = new Rectangle2D.Double(point.getX() + (size / zoom), point.getY()
				- attributeHeight, attributeWidth, attributeHeight);
			if (isPointPutable(attributeRectangle, location, usedPoints, visibleRectangle, search)) {
				point.setAttributeLocation(attributeRectangle.getX(), attributeRectangle.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(location);
			}
			// 点の右下
			attributeRectangle = new Rectangle2D.Double(point.getX() + (size / zoom), point.getY(),
				attributeWidth, attributeHeight);
			if (isPointPutable(attributeRectangle, location, usedPoints, visibleRectangle, search)) {
				point.setAttributeLocation(attributeRectangle.getX(), attributeRectangle.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(location);
			}
			// 点の左上
			attributeRectangle = new Rectangle2D.Double(point.getX() - attributeWidth
				- (size / zoom), point.getY() - attributeHeight, attributeWidth, attributeHeight);
			if (isPointPutable(attributeRectangle, location, usedPoints, visibleRectangle, search)) {
				point.setAttributeLocation(attributeRectangle.getX(), attributeRectangle.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(location);
			}
			// 点の左下
			attributeRectangle = new Rectangle2D.Double(point.getX() - attributeWidth
				- (size / zoom), point.getY(), attributeWidth, attributeHeight);
			if (isPointPutable(attributeRectangle, location, usedPoints, visibleRectangle, search)) {
				point.setAttributeLocation(attributeRectangle.getX(), attributeRectangle.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(location);
			}
		}
	}

	/**
	 * 建物の属性の表示位置を決定します。
	 * @param polygon ポリゴン
	 * @param usedPoints 既に属性が配置されたポリゴンの中心座標
	 * @param visibleRectangle 現在表示されている領域
	 * @param zoom 倍率
	 * @param attributeHeight 属性の高さ
	 * @param metrics フォントメトリクス
	 * @param size 点の直径
	 * @param search 属性によって使用済みの領域を検索するためのオブジェクト
	 */
	private void fixTatemonoAttributeLocation(PolygonData polygon, Collection<Point> usedPoints,
		Rectangle2D visibleRectangle, double zoom, double attributeHeight, FontMetrics metrics,
		double size, final Search search) {
		if ((polygon.getAttribute() != null)
			&& (polygon.getTatemonoCode() != PolygonData.TATEMONO_STATION)) {
			polygon.setAttributeLocation(0, 0);
			double attributeWidth = metrics.stringWidth(polygon.getAttribute()) / zoom;
			Rectangle2D pointRectangle = new Rectangle2D.Double(polygon.getX() - (size / zoom),
				polygon.getY() - (size / zoom), (size * 2) / zoom, (size * 2) / zoom);
			Rectangle2D attributeRectangle;
			// 点の右
			attributeRectangle = new Rectangle2D.Double(polygon.getX() + (size / zoom), polygon
				.getY()
				- (attributeHeight / 2), attributeWidth, attributeHeight);
			Point point = new Point((int) polygon.getX(), (int) polygon.getY());
			if (isPutable(point, usedPoints, attributeRectangle, visibleRectangle, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
			// 点の左
			attributeRectangle = new Rectangle2D.Double(polygon.getX() - attributeWidth
				- (size / zoom), polygon.getY() - (attributeHeight / 2), attributeWidth,
				attributeHeight);
			if (isPutable(point, usedPoints, attributeRectangle, visibleRectangle, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
			// 点の右上
			attributeRectangle = new Rectangle2D.Double(polygon.getX() + (size / 2 / zoom), polygon
				.getY()
				- attributeHeight, attributeWidth, attributeHeight);
			if (isPutable(point, usedPoints, attributeRectangle, visibleRectangle, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
			// 点の右下
			attributeRectangle = new Rectangle2D.Double(polygon.getX() + (size / 2 / zoom), polygon
				.getY(), attributeWidth, attributeHeight);
			if (isPutable(point, usedPoints, attributeRectangle, visibleRectangle, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
			// 点の左上
			attributeRectangle = new Rectangle2D.Double(polygon.getX() - attributeWidth
				- (size / 2 / zoom), polygon.getY() - attributeHeight, attributeWidth,
				attributeHeight);
			if (isPutable(point, usedPoints, attributeRectangle, visibleRectangle, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
			// 点の左下
			attributeRectangle = new Rectangle2D.Double(polygon.getX() - attributeWidth
				- (size / 2 / zoom), polygon.getY(), attributeWidth, attributeHeight);
			if (isPutable(point, usedPoints, attributeRectangle, visibleRectangle, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
		}
	}

	/**
	 * ポリゴンの属性位置を決定します。
	 * @param polygon ポリゴン
	 * @param usedPoints 既に属性が配置されたポリゴンの中心座標
	 * @param visibleRectangle 現在表示されている矩形
	 * @param zoom 倍率
	 * @param attributeHeight 属性の高さ
	 * @param metrics フォントメトリクス
	 * @param size 点の大きさ
	 * @param search 属性が決定されている領域を検索するためのオブジェクト
	 */
	private void fixPolygonAttributeLocation(PolygonData polygon, Collection<Point> usedPoints,
		Rectangle2D visibleRectangle, double zoom, double attributeHeight, FontMetrics metrics,
		double size, final Search search) {
		if (polygon.getAttribute() != null) {
			polygon.setAttributeLocation(0, 0);
			double attributeWidth = metrics.stringWidth(polygon.getAttribute()) / zoom;
			Rectangle2D pointRectangle = new Rectangle2D.Double(polygon.getX() - (size / zoom),
				polygon.getY() - (size / zoom), (size * 2) / zoom, (size * 2) / zoom);
			Rectangle2D attributeRectangle;
			// ポリゴンの中心
			attributeRectangle = new Rectangle2D.Double(polygon.getX() - (attributeWidth / 2),
				polygon.getY() - (attributeHeight / 2), attributeWidth, attributeHeight);
			Point point = new Point((int) polygon.getX(), (int) polygon.getY());
			if (isPutable(point, usedPoints, attributeRectangle, visibleRectangle, search)) {
				polygon.setAttributeLocation(attributeRectangle.getX(), attributeRectangle
					.getMaxY());
				search.insert(pointRectangle, null);
				search.insert(attributeRectangle, null);
				usedPoints.add(point);
			}
		}
	}

	/**
	 * @param point 点
	 * @param usedPoints 既に属性が配置されたポリゴンの中心座標
	 * @param attributeRectangle 属性の矩形
	 * @param visibleRectangle 現在表示されている矩形
	 * @param search 既に属性が配置された領域を検索するためのオブジェクト
	 * @return 属性を配置することができるかどうか
	 */
	private boolean isPutable(Point point, Collection<Point> usedPoints,
		Rectangle2D attributeRectangle, Rectangle2D visibleRectangle, final Search search) {
		return !usedPoints.contains(point) && visibleRectangle.contains(point)
			&& visibleRectangle.contains(attributeRectangle) && search.search(point).isEmpty()
			&& search.search(attributeRectangle).isEmpty();
	}

	/**
	 * @param attributeRectangle 属性の矩形
	 * @param point 点
	 * @param usedPoints 既に属性が配置されたポリゴンの中心座標
	 * @param visibleRectangle 現在表示されている矩形
	 * @param search 既に属性が配置された領域を検索するためのオブジェクト
	 * @return 点を置くことができるかどうか
	 */
	private boolean isPointPutable(Rectangle2D attributeRectangle, Point point,
		Collection<Point> usedPoints, Rectangle2D visibleRectangle, final Search search) {
		return !usedPoints.contains(point) && visibleRectangle.contains(attributeRectangle)
			&& search.search(attributeRectangle).isEmpty();
	}

	/**
	 * @param polygon ポリゴン
	 * @param point 点
	 * @param usedPoints 既に属性が配置されたポリゴンの中心座標
	 * @param attributeRectangle 属性の矩形
	 * @param visibleRectangle 現在表示されている矩形
	 * @param contains ポリゴンが属性を完全に含むかどうか
	 * @param search 属性が配置された領域を検索するためのオブジェクト
	 * @return 町丁目の属性を配置することができるかどうか
	 */
	private boolean isTyomePutable(PolygonData polygon, Point point, Collection<Point> usedPoints,
		Rectangle2D attributeRectangle, Rectangle2D visibleRectangle, boolean contains,
		final Search search) {
		if (contains) {
			return !usedPoints.contains(point) && visibleRectangle.contains(attributeRectangle)
				&& polygon.getArea().contains(attributeRectangle)
				&& search.search(attributeRectangle).isEmpty();
		} else {
			return !usedPoints.contains(point) && visibleRectangle.contains(attributeRectangle)
				&& polygon.getArea().intersects(attributeRectangle)
				&& search.search(attributeRectangle).isEmpty();
		}
	}
}
