package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ローソンの住所を取得するクラスです。
 * @author Kumano Tatsuo
 * @since 3.16
 * 2005/12/04
 */
public class Lawson {
	/**
	 * 最初のURL
	 */
	private static final String URL1 = "http://map.lawson.co.jp/c/f/?uc=106&admi2=";

	/**
	 * 次のURL
	 */
	private static final String URL2 = "&admi3=";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "SJIS";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "lawson_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * ローソンの座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param city 市区町村コード
	 * @param isj 街区レベル位置参照情報
	 * @return 座標と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String city, final Map<String, Point2D> isj)
			throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Lawson.CACHE_DIR).exists()) {
			new File(Lawson.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Lawson.CACHE_DIR + File.separator + Lawson.PREFIX + city
				+ Lawson.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(Lawson.getAddresses(city), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
				cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "ローソン");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * ローソンの市区町村別ページを解析して住所の一覧を取得します。
	 * @param city 市区町村コード
	 * @return 住所と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String city) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL1 + city.substring(0, 2) + URL2 + city);
			System.out.println("DEBUG: getting " + url);
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			String caption = null;
			final Pattern pattern = Pattern.compile("<a href=\"[^<>]+\">([^<>]+)</a>");
			final Pattern pattern2 = Pattern.compile("<td class=\".+\">([^<>]+)</td>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					caption = matcher.group(1);
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					if (caption != null) {
						ret.put(matcher2.group(1), caption);
					}
				}
			}
			scanner.close();
		} catch (FileNotFoundException e) {
		}
		return ret;
	}
}
