package car.map;

import java.awt.Shape;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.Material;
import javax.media.j3d.PointLight;
import javax.media.j3d.PolygonAttributes;
import javax.media.j3d.QuadArray;
import javax.media.j3d.Shape3D;
import javax.media.j3d.TexCoordGeneration;
import javax.media.j3d.Texture;
import javax.media.j3d.TextureAttributes;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TriangleStripArray;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;
import javax.vecmath.Vector4f;

import car.CarUtil;

import com.sun.j3d.utils.geometry.GeometryInfo;
import com.sun.j3d.utils.geometry.NormalGenerator;
import com.sun.j3d.utils.geometry.Stripifier;
import com.sun.j3d.utils.image.TextureLoader;

/**
 * マップの情報を管理するクラスです。
 * @author Kumano Tatsuo
 * Created on 2004/12/12
 */
public class KinokoRoadMap implements RoadMap {

	/**
	 * @return チェックポイントの一覧
	 */
	public Line2D[] getCheckPoints() {
		if (this.cachedCheckPoints == null) {
			final Collection<Line2D> ret = new ArrayList<Line2D>();
			for (final int index : new int[] { 5, 6, 7, 8, 9, 10 }) {
				ret.add(CarUtil.toLines(CarUtil.toShape(this.data.DATA[index])).get(0));
			}
			this.cachedCheckPoints = ret.toArray(new Line2D[] {});
		}
		return this.cachedCheckPoints;
	}

	/**
	 * チェックポイントの一覧
	 */
	private Line2D[] cachedCheckPoints;

	/**
	 * 縮尺
	 */
	private final float zoom = 1f;

	/**
	 * @return 縮尺
	 */
	public float getZoom() {
		return this.zoom;
	}

	/**
	 * データ
	 */
	private KinokoData data;

	/**
	 * @return ガードレールの一覧
	 */
	public Line2D[] getBarriers() {
		if (this.cachedBarriers == null) {
			final Collection<Line2D> barriers = CarUtil.toLines(CarUtil.toShape(this.data.DATA[2]));
			barriers.addAll(CarUtil.toLines(CarUtil.toShape(this.data.DATA[3])));
			this.cachedBarriers = barriers.toArray(new Line2D[] {});
		}
		return this.cachedBarriers;
	}

	/**
	 * ガードレールの一覧
	 */
	private Line2D[] cachedBarriers;

	/**
	 * @return 情景
	 */
	public BranchGroup getScene() {
		// 3Dシーンを作る。
		final BranchGroup scene = new BranchGroup();
		scene.setCapability(BranchGroup.ALLOW_DETACH);
		final Point3d min = new Point3d(-256 * this.zoom, 256 * this.zoom, 0);
		final Point3d max = new Point3d(512 * this.zoom, -512 * this.zoom, 0);
		// 道路を作ってみる。
		{
			final Point3f[] roadVertices = CarUtil.toPolygonArray(CarUtil
					.toShape(this.data.DATA[0]), -0.03f);
			final GeometryInfo gi = new GeometryInfo(GeometryInfo.POLYGON_ARRAY);
			gi.setCoordinates(roadVertices);
			gi.setStripCounts(new int[] { roadVertices.length });
			final NormalGenerator ng = new NormalGenerator();
			ng.generateNormals(gi);
			final Stripifier st = new Stripifier();
			st.stripify(gi);
			final GeometryArray roadGeometry = gi.getGeometryArray();
			roadGeometry.setNormals(0, gi.getNormals());
			final Shape3D roadShape = new Shape3D(roadGeometry);
			final Appearance roadAppearance = new Appearance();
			final Material roadMaterial = new Material();
			roadMaterial.setDiffuseColor(new Color3f(0.3f, 0.3f, 0.3f));
			roadAppearance.setMaterial(roadMaterial);
			roadShape.setAppearance(roadAppearance);
			scene.addChild(roadShape);
		}
		// ゴールラインを読み込んでみる。
		{
			final Point3f[] goalVertices = CarUtil.createStrokedPolygonArray(this.data.DATA[4], 1,
					-0.02f);
			final TriangleStripArray goalGeometryArray = new TriangleStripArray(
					goalVertices.length, GeometryArray.COORDINATES | GeometryArray.NORMALS,
					new int[] { 4 });
			goalGeometryArray.setCoordinates(0, goalVertices);
			final Shape3D goalShape = new Shape3D(goalGeometryArray);
			final Appearance goalAppearance = new Appearance();
			final Texture texture = new TextureLoader(KinokoRoadMap.class.getResource("check.png"),
					null).getTexture();
			final TextureAttributes goalTextureAttributes = new TextureAttributes();
			final Transform3D goalTextureTransform = new Transform3D();
			goalTextureTransform.rotZ(-0);
			goalTextureAttributes.setTextureTransform(goalTextureTransform);
			goalAppearance.setTextureAttributes(goalTextureAttributes);
			final TexCoordGeneration goalTexCoordGeneration = new TexCoordGeneration(
					TexCoordGeneration.OBJECT_LINEAR, TexCoordGeneration.TEXTURE_COORDINATE_2);
			goalTexCoordGeneration.setPlaneS(new Vector4f(1.7f, 0.0f, 0.0f, 0.0f));
			goalTexCoordGeneration.setPlaneT(new Vector4f(0.0f, 1.7f, 0.0f, 0.0f));
			goalAppearance.setTexCoordGeneration(goalTexCoordGeneration);
			goalAppearance.setTexture(texture);
			goalShape.setAppearance(goalAppearance);
			scene.addChild(goalShape);
		}
		// 土を読み込む
		{
			final Appearance dirtAppearance = new Appearance();
			final PolygonAttributes dirtAttributes = new PolygonAttributes();
			dirtAttributes.setCullFace(PolygonAttributes.CULL_NONE);
			dirtAppearance.setPolygonAttributes(dirtAttributes);
			final Material dirtMaterial = new Material();
			dirtMaterial.setDiffuseColor(new Color3f(0.48f, 0.41f, 0.2f));
			dirtAppearance.setMaterial(dirtMaterial);
			for (final int index : new int[] { 2 }) {
				final float dirtHeight = -0.06f;
				final Point3f[] dirtVertices = CarUtil.toPolygonArray(CarUtil
						.toShape(this.data.DATA[index]), dirtHeight);
				final GeometryInfo dirtGeometryInfo = new GeometryInfo(GeometryInfo.POLYGON_ARRAY);
				dirtGeometryInfo.setCoordinates(dirtVertices);
				dirtGeometryInfo.setStripCounts(new int[] { dirtVertices.length });
				final NormalGenerator dirtNormalGenerator = new NormalGenerator();
				dirtNormalGenerator.generateNormals(dirtGeometryInfo);
				final Stripifier dirtStripfier = new Stripifier();
				dirtStripfier.stripify(dirtGeometryInfo);
				final GeometryArray dirtGeometry = dirtGeometryInfo.getGeometryArray();
				dirtGeometry.setNormals(0, dirtGeometryInfo.getNormals());
				final Shape3D dirtShape = new Shape3D(dirtGeometry);
				dirtShape.setAppearance(dirtAppearance);
				scene.addChild(dirtShape);
			}
		}
		// 芝生を読み込む
		{
			final Appearance greenAppearance = new Appearance();
			final PolygonAttributes greenAttributes = new PolygonAttributes();
			greenAttributes.setCullFace(PolygonAttributes.CULL_NONE);
			greenAppearance.setPolygonAttributes(greenAttributes);
			final Material greenMaterial = new Material();
			greenMaterial.setDiffuseColor(new Color3f(0.1f, 0.4f, 0));
			greenAppearance.setMaterial(greenMaterial);
			for (final int index : new int[] { 3 }) {
				final float greenHeight = -0.01f;
				final Point3f[] greenVertices = CarUtil.toPolygonArray(CarUtil
						.toShape(this.data.DATA[index]), greenHeight);
				final GeometryInfo greenGeometryInfo = new GeometryInfo(GeometryInfo.POLYGON_ARRAY);
				greenGeometryInfo.setCoordinates(greenVertices);
				greenGeometryInfo.setStripCounts(new int[] { greenVertices.length });
				greenGeometryInfo.reverse();
				final NormalGenerator greenNormalGenerator = new NormalGenerator();
				greenNormalGenerator.generateNormals(greenGeometryInfo);
				final Stripifier greenStripfier = new Stripifier();
				greenStripfier.stripify(greenGeometryInfo);
				final GeometryArray greenGeometry = greenGeometryInfo.getGeometryArray();
				greenGeometry.setNormals(0, greenGeometryInfo.getNormals());
				final Shape3D greenShape = new Shape3D(greenGeometry);
				greenShape.setAppearance(greenAppearance);
				scene.addChild(greenShape);
			}
		}
		// 赤と白の縞を作ってみる。
		{
			final Appearance whiteAppearance = new Appearance();
			final Material whiteMaterial = new Material();
			whiteMaterial.setDiffuseColor(new Color3f(0.7f, 0.7f, 0.7f));
			whiteAppearance.setMaterial(whiteMaterial);
			final Appearance redAppearance = new Appearance();
			final Material redMaterial = new Material();
			redMaterial.setDiffuseColor(new Color3f(0.6f, 0, 0));
			redAppearance.setMaterial(redMaterial);
			for (final int index : new int[] { 11, 12, 13, 14, 15, 16, 17 }) {
				final float height = -0.02f;
				final double width = 1.2;
				final Point3f[] whiteVertices = CarUtil.createStrokedPolygonArray(
						this.data.DATA[index], width, height);
				final int[] whiteStripVertexCounts = new int[whiteVertices.length / 4];
				Arrays.fill(whiteStripVertexCounts, 4);
				final TriangleStripArray whiteGeometry = new TriangleStripArray(
						whiteVertices.length, GeometryArray.COORDINATES | GeometryArray.NORMALS,
						whiteStripVertexCounts);
				whiteGeometry.setCoordinates(0, whiteVertices);
				final Vector3f[] whiteNormals = new Vector3f[whiteVertices.length];
				Arrays.fill(whiteNormals, new Vector3f(0, 0, 1));
				whiteGeometry.setNormals(0, whiteNormals);
				final Shape3D whiteShape = new Shape3D(whiteGeometry);
				whiteShape.setAppearance(whiteAppearance);
				scene.addChild(whiteShape);
				if (whiteVertices.length > 2) {
					final Point3f[] redVertices = new Point3f[whiteVertices.length - 2];
					System.arraycopy(whiteVertices, 2, redVertices, 0, redVertices.length);
					final int[] redStripVertexCounts = new int[redVertices.length / 4];
					Arrays.fill(redStripVertexCounts, 4);
					final TriangleStripArray redGeometry = new TriangleStripArray(
							redVertices.length, GeometryArray.COORDINATES | GeometryArray.NORMALS,
							redStripVertexCounts);
					redGeometry.setCoordinates(0, redVertices);
					final Vector3f[] redNormals = new Vector3f[redVertices.length];
					Arrays.fill(redNormals, new Vector3f(0, 0, 1));
					redGeometry.setNormals(0, redNormals);
					final Shape3D redShape = new Shape3D(redGeometry);
					redShape.setAppearance(redAppearance);
					scene.addChild(redShape);
				}
			}
		}
		// 白線を引いてみる。
		{
			final Material whiteMaterial = new Material();
			whiteMaterial.setDiffuseColor(new Color3f(0.6f, 0.6f, 0.6f));
			final Appearance whiteAppearance = new Appearance();
			whiteAppearance.setMaterial(whiteMaterial);
			final float height = -0.01f;
			final double width = 0.3;
			{
				final Point2D[] points = new Point2D[this.data.DATA[0].length / 2 + 9];
				System.arraycopy(this.data.DATA[0], 0, points, 0, points.length);
				final Point3f[] vertices = CarUtil.createStrokedPolygonArray(points, width, height);
				final int[] stripVertexCounts = new int[] { vertices.length };
				final TriangleStripArray geometry = new TriangleStripArray(vertices.length,
						GeometryArray.COORDINATES | GeometryArray.NORMALS, stripVertexCounts);
				geometry.setCoordinates(0, vertices);
				final Vector3f[] normals = new Vector3f[vertices.length];
				Arrays.fill(normals, new Vector3f(0, 0, 1));
				geometry.setNormals(0, normals);
				final Shape3D shape = new Shape3D(geometry);
				shape.setAppearance(whiteAppearance);
				scene.addChild(shape);
			}
			{
				final Point2D[] points = new Point2D[this.data.DATA[0].length / 2 - 8];
				System.arraycopy(this.data.DATA[0], this.data.DATA[0].length - points.length,
						points, 0, points.length);
				final Point3f[] vertices = CarUtil.createStrokedPolygonArray(points, width, height);
				final int[] stripVertexCounts = new int[] { vertices.length };
				final TriangleStripArray geometry = new TriangleStripArray(vertices.length,
						GeometryArray.COORDINATES | GeometryArray.NORMALS, stripVertexCounts);
				geometry.setCoordinates(0, vertices);
				final Vector3f[] normals = new Vector3f[vertices.length];
				Arrays.fill(normals, new Vector3f(0, 0, 1));
				geometry.setNormals(0, normals);
				final Shape3D shape = new Shape3D(geometry);
				shape.setAppearance(whiteAppearance);
				scene.addChild(shape);
			}
		}
		// ガードレールを作ってみる。
		{
			final float barrierHeight = 0.6f;
			final float barrierThickness = 0.2f;
			final Appearance barrierAppearance = new Appearance();
			final Material barrierMaterial = new Material();
			barrierMaterial.setDiffuseColor(new Color3f(1, 1, 1));
			barrierAppearance.setMaterial(barrierMaterial);
			for (final Line2D line : getBarriers()) {
				CarUtil.addBarrier(line.getX1(), line.getY1(), line.getX2(), line.getY2(),
						barrierThickness, barrierHeight, -0.04f, barrierAppearance, scene);
			}
		}
		// 草を作ってみる。
		{
			final Point3d[] greenVertices = new Point3d[] { new Point3d(min.x, min.y, -0.1),
					new Point3d(max.x, min.y, -0.1), new Point3d(max.x, max.y, -0.1),
					new Point3d(min.x, max.y, -0.1) };
			final QuadArray greenGeometry = new QuadArray(greenVertices.length,
					GeometryArray.COORDINATES | GeometryArray.NORMALS);
			final Vector3f[] greenNormals = new Vector3f[] { new Vector3f(0, 0, 1),
					new Vector3f(0, 0, 1), new Vector3f(0, 0, 1), new Vector3f(0, 0, 1) };
			greenGeometry.setCoordinates(0, greenVertices);
			greenGeometry.setNormals(0, greenNormals);
			final Shape3D greenShape = new Shape3D(greenGeometry);
			final Appearance greenAppearance = new Appearance();
			final PolygonAttributes greenPolygonAttributes = new PolygonAttributes();
			greenPolygonAttributes.setCullFace(PolygonAttributes.CULL_NONE);
			greenAppearance.setPolygonAttributes(greenPolygonAttributes);
			final Material greenMaterial = new Material();
			greenMaterial.setDiffuseColor(new Color3f(0.1f, 0.4f, 0));
			greenAppearance.setMaterial(greenMaterial);
			greenShape.setAppearance(greenAppearance);
			scene.addChild(greenShape);
		}
		// 背景色を設定する。
		final Background background = new Background(new Color3f(0, 0.5f, 1));
		background.setApplicationBounds(new BoundingSphere(min, 2000));
		scene.addChild(background);
		// 照明を作る。
		final float lightHeight = 512;
		final float lightPower = 0.0025f;
		final PointLight pointLight1 = new PointLight(new Color3f(1, 1, 1), new Point3f(
				(float) min.x, (float) min.y, lightHeight), new Point3f(0, lightPower, 0));
		pointLight1.setInfluencingBounds(new BoundingSphere(min, 1000));
		scene.addChild(pointLight1);
		final PointLight pointLight2 = new PointLight(new Color3f(1, 1, 1), new Point3f(
				(float) max.x, (float) max.y, lightHeight), new Point3f(0, lightPower, 0));
		pointLight2.setInfluencingBounds(new BoundingSphere(max, 1000));
		scene.addChild(pointLight2);
		final PointLight pointLight3 = new PointLight(new Color3f(1, 1, 1), new Point3f(
				(float) min.x, (float) max.y, lightHeight), new Point3f(0, lightPower, 0));
		pointLight3.setInfluencingBounds(new BoundingSphere(new Point3d(min.x, max.y, 0), 1000));
		scene.addChild(pointLight3);
		final PointLight pointLight4 = new PointLight(new Color3f(1, 1, 1), new Point3f(
				(float) max.x, (float) min.y, lightHeight), new Point3f(0, lightPower, 0));
		pointLight4.setInfluencingBounds(new BoundingSphere(new Point3d(max.x, min.y, 0), 1000));
		scene.addChild(pointLight4);
		return scene;
	}

	/**
	 * コンストラクタです。
	 */
	public KinokoRoadMap() {
		this.cachedBarriers = null;
		this.cachedCheckPoints = null;
		this.data = new KinokoData(this.zoom, -this.zoom);
	}

	/**
	 * @return 道路を平面に投影したもの
	 */
	public Shape getRoadShape() {
		return CarUtil.toShape(this.data.DATA[0]);
	}

	/**
	 * @return 赤白の縁石の領域
	 */
	public Shape[] getStripes() {
		if (this.cachedStripes == null) {
			final Collection<Shape> stripes = new ArrayList<Shape>();
			for (final int index : new int[] { 11, 12, 13, 14, 15, 16, 17 }) {
				stripes.add(CarUtil.createStrokedShape(this.data.DATA[index], 1.2));
			}
			this.cachedStripes = stripes.toArray(new Shape[] {});
		}
		return this.cachedStripes;
	}

	/**
	 * 赤白の縁石の領域
	 */
	private Shape[] cachedStripes = null;

	/**
	 * @return プレイヤ1の初期位置
	 */
	public Point2D getInitialLocation1() {
		return this.data.DATA[1][0];
	}

	/**
	 * @return プレイヤ2の初期位置
	 */
	public Point2D getInitialLocation2() {
		return this.data.DATA[1][1];
	}

	/**
	 * @return プレイヤの初期向き
	 */
	public double getInitialDirection() {
		return Math.atan2(this.data.DATA[1][1].getY() - this.data.DATA[1][0].getY(),
				this.data.DATA[1][1].getX() - this.data.DATA[1][0].getX())
				+ Math.PI / 2;
	}

	/**
	 * @return コースマップ
	 */
	public TransformGroup getCourseMap() {
		final TransformGroup ret = new TransformGroup();
		final Transform3D roadTransform = new Transform3D();
		roadTransform.setTranslation(new Vector3d(-0.8, 0.21, -2));
		final Transform3D roadScaleTransform = new Transform3D();
		roadScaleTransform.setScale(0.001);
		roadTransform.mul(roadScaleTransform);
		ret.setTransform(roadTransform);
		final Point3f[] roadVertices = CarUtil.toPolygonArray(this.getRoadShape(), -0.05f);
		final GeometryInfo gi = new GeometryInfo(GeometryInfo.POLYGON_ARRAY);
		gi.setCoordinates(roadVertices);
		gi.setStripCounts(new int[] { roadVertices.length });
		final NormalGenerator ng = new NormalGenerator();
		ng.generateNormals(gi);
		final Stripifier st = new Stripifier();
		st.stripify(gi);
		final GeometryArray roadGeometry = gi.getGeometryArray();
		roadGeometry.setNormals(0, gi.getNormals());
		final Shape3D roadShape = new Shape3D(roadGeometry);
		ret.addChild(roadShape);
		final Point3f[] goalVertices = CarUtil.createStrokedPolygonArray(this.data.DATA[4], 10,
				-0.02f);
		final TriangleStripArray goalGeometryArray = new TriangleStripArray(goalVertices.length,
				GeometryArray.COORDINATES | GeometryArray.COLOR_3, new int[] { 4 });
		goalGeometryArray.setCoordinates(0, goalVertices);
		final Shape3D goalShape = new Shape3D(goalGeometryArray);
		ret.addChild(goalShape);
		return ret;
	}

	public int getLap() {
		return 5;
	}

}
