package bubble;
import java.awt.Point;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Random;
import java.util.StringTokenizer;
import sound.SE;

/**
 * ユーティリティクラスです。
 * @author Kumano Tatsuo
 * 作成日：2004/10/14
 */
public class Util {
    /**
     * 指定した行、列に対応する座標を取得します。
     * 座標系は、左上のグリッドの中心が原点です。
     * @param row 行
     * @param col 列
     * @return 座標
     */
    public static Point toLocation(int row, int col) {
        int y = Const.GRID_HEIGHT * row;
        int x = Const.GRID_WIDTH * col + (row % 2) * Const.GRID_WIDTH / 2;
        return new Point(x, y);
    }

    /**
     * 指定した座標に対応するグリッド位置を取得します。
     * @param x x座標
     * @param y y座標
     * @return グリッド位置
     */
    public static Point toGridLocation(double x, double y) {
        int row = (int)((y + Const.GRID_HEIGHT / 2) / Const.GRID_HEIGHT);
        int col = (int)((x + Const.GRID_WIDTH / 2 - (row % 2) * Const.GRID_WIDTH / 2) / Const.GRID_WIDTH);
        return new Point(col, row);
    }

    /**
     * 指定した座標に隣接する固定した泡のうち、最も近い泡までの距離の2乗を求めます。
     * @param x x座標
     * @param y y座標
     * @param game ゲームの情報
     * @return 距離
     */
    public static int getMinDistannceSq(int x, int y, Game game) {
        int ret = Integer.MAX_VALUE;
        Point gridPoint = toGridLocation(x, y);
        int col = gridPoint.x;
        int row = gridPoint.y;
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            if (game.getData(p.y, p.x) != Const.NO_BUBBLE) {
                    int distanceSq = (int) toLocation(p.y, p.x).distanceSq(x, y);
                    if (distanceSq < ret) {
                        ret = distanceSq;
                    }
            }
        }
        return ret;
    }

    /**
     * 指定した座標に隣接する固定した泡のうち、最も近い泡までの距離の2乗を求めます。
     * @param x x座標
     * @param y y座標
     * @param game ゲームの情報
     * @return 距離
     */
    public static int getMinDistanceSq(int x, int y, ReadOnlyGame game) {
        int ret = Integer.MAX_VALUE;
        Point gridPoint = toGridLocation(x, y);
        int col = gridPoint.x;
        int row = gridPoint.y;
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            if (game.getData(p.y, p.x) != Const.NO_BUBBLE) {
                int distanceSq = (int) toLocation(p.y, p.x).distanceSq(x, y);
                if (distanceSq < ret) {
                    ret = distanceSq;
                }
            }
        }
        return ret;
    }

    /**
     * 指定した座標に隣接する固定した泡のうち、最も近い泡の色を取得します。
     * @param x x座標
     * @param y y座標
     * @param game ゲームの情報
     * @return 色
     */
    public static int getMinDistanceColor(int x, int y, ReadOnlyGame game) {
        int ret = Const.NO_BUBBLE;
        int min = Integer.MAX_VALUE;
        Point gridPoint = toGridLocation(x, y);
        int col = gridPoint.x;
        int row = gridPoint.y;
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            int color = game.getData(p.y, p.x);
            if (color != Const.NO_BUBBLE) {
                int distanceSq = (int) toLocation(p.y, p.x).distanceSq(x, y);
                if (distanceSq < min) {
                    min = distanceSq;
                    ret = color;
                }
            }
        }
        return ret;
    }
    
    /**
     * ステージをランダムに設定します。
     * @param game ゲームの状態
     * @param height 高さ
     */
    public static void setRandomData(Game game, int height) {
        Random random = new Random();
        for (int row = 0; row < height; ++row) {
            for (int col = 0; col < Const.STAGE_COLS - row % 2; ++col) {
                game.setData(random.nextInt(Const.COLOR_NUMBER), row, col);
            }
        }
    }

    /**
     * 指定した値の2乗を求めます。
     * @param value
     * @return 2乗した値
     */
    public static int square(int value) {
        return value * value;
    }

    /**
     * 指定した位置にある固定した泡に連結な、同じ色の泡を消去します。
     * @param row 行
     * @param col 列
     * @param game ゲーム
     */
    public static void eraseSameColor(int row, int col, Game game) {
        int number = getSameColorNumberRecursively(row, col, game,
                new boolean[Const.STAGE_ROWS][Const.STAGE_ROWS]);
        if (number >= Const.MIN_ERASE_NUMBER) {
            SE.remove();
            eraseRecursively(row, col, game.getData(row, col), game);
            if (number > Const.MIN_ERASE_NUMBER) {
                if (game.getRule() != Const.RULE_FROZEN_1P) {
                    game.sendJam(number - Const.MIN_ERASE_NUMBER);
                }
            }
        }else{
            SE.put();
        }
    }

    /**
     * 指定した位置の泡に隣接した、同じ色の泡の個数を求めます。
     * @param row 行
     * @param col 列
     * @param game ゲーム
     * @param hasDone その場所を調べ終ったかどうか
     * @return 同じ色の泡の個数
     */
    private static int getSameColorNumberRecursively(int row, int col, Game game,
            boolean[][] hasDone) {
        int ret = 1;
        int color = game.getData(row, col);
        hasDone[row][col] = true;
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            if (game.getData(p.y, p.x) == color) {
                if (!hasDone[p.y][p.x]) {
                    ret += getSameColorNumberRecursively(p.y, p.x, game, hasDone);
                }
            }
        }
        return ret;
    }
    /**
     * 指定した位置の泡に隣接した、同じ色の泡の個数を求めます。
     * @param row 行
     * @param col 列
     * @param color 色
     * @param game ゲーム
     * @param hasDone その場所を調べ終ったかどうか
     * @return 同じ色の泡の個数
     */
    public static int getSameColorNumberRecursively(int row, int col, int color, ReadOnlyGame game,
            boolean[][] hasDone) {
        int ret = 1;
        hasDone[row][col] = true;
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            if (game.getData(p.y, p.x) == color) {
                if (!hasDone[p.y][p.x]) {
                    ret += getSameColorNumberRecursively(p.y, p.x, color, game, hasDone);
                }
            }
        }
        return ret;
    }

    /**
     * 指定した位置に隣接した、同じ色の泡を消去します。
     * @param row 行
     * @param col 列
     * @param color 色
     * @param game ゲーム
     */
    private static void eraseRecursively(int row, int col, int color, Game game) {
        game.getFallingBubbles().add(new MovingBubble(toLocation(row, col), color));
        game.setData(Const.NO_BUBBLE, row, col);
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            if (game.getData(p.y, p.x) == color) {
                eraseRecursively(p.y, p.x, color, game);
            }
        }
    }

    /**
     * 指定した位置に隣接した位置を取得します。
     * @param direction 方向を0から5の整数で指定します。
     * 0: 左上
     * 1: 右上
     * 2: 左
     * 3: 右
     * 4: 左下
     * 5: 右下
     * @param row 行
     * @param col 列
     * @return 位置
     */
    public static Point getAdjacent(int direction, int row, int col) {
        switch (direction) {
        case 0:
            return new Point(col - 1 + row % 2, row - 1);
        case 1:
            return new Point(col + row % 2, row - 1);
        case 2: // 左
            return new Point(col - 1, row);
        case 3: // 右
            return new Point(col + 1, row);
        case 4:
            return new Point(col - 1 + row % 2, row + 1);
        case 5:
            return new Point(col + row % 2, row + 1);
        default:
            return null;
        }
    }

    /**
     * ぶら下がれなくなった泡を落とします。 
     * @param game ゲーム
     * @return 落とした泡の一覧
     */
    public static Collection<MovingBubble> dropBubble(Game game) {
        Collection<MovingBubble> ret = new LinkedList<MovingBubble>();
        boolean[][] isAdjacent = new boolean[Const.STAGE_ROWS][Const.STAGE_COLS];
        for (int col = 0; col < Const.STAGE_COLS; ++col) {
            int row = 0;
            markAdjacentRecursively(row, col, isAdjacent, game);
        }
        for (int row = 0; row < Const.STAGE_ROWS; ++row) {
            for (int col = 0; col < Const.STAGE_COLS; ++col) {
                if (game.getData(row, col) != Const.NO_BUBBLE && !isAdjacent[row][col]) {
                    MovingBubble bubble = new MovingBubble(toLocation(row, col), game.getData(row,
                            col));
                    game.getFallingBubbles().add(bubble);
                    ret.add(bubble);
                    game.setData(Const.NO_BUBBLE, row, col);
                }
            }
        }
        if (ret.size() > 0) {
            if (game.getRule() != Const.RULE_FROZEN_1P) {
                game.sendJam(ret.size());
            }
        }
        return ret;
    }

    /**
     * 指定した位置に隣接した位置を再帰的に取得します。
     * @param row 行
     * @param col 列
     * @param isAdjacent 隣接しているかどうか
     * @param game ゲーム
     */
    private static void markAdjacentRecursively(int row, int col, boolean[][] isAdjacent, Game game) {
        if (game.getData(row, col) != Const.NO_BUBBLE) {
            if (!isAdjacent[row][col]) {
                isAdjacent[row][col] = true;
                for (int direction = 0; direction < 6; ++direction) {
                    Point p = getAdjacent(direction, row, col);
                    if (game.getData(p.y, p.x) != Const.NO_BUBBLE) {
                        if (!isAdjacent[p.y][p.x]) {
                            markAdjacentRecursively(p.y, p.x, isAdjacent, game);
                        }
                    }
                }
            }
        }
    }

    /**
     * 連鎖反応を起こします。
     * @param game ゲーム
     * @param droppedBubbles 落とされた泡の一覧
     */
    public static void chainReaction(Game game, Collection<MovingBubble> droppedBubbles) {
        boolean[][] hasDone = new boolean[Const.STAGE_ROWS][Const.STAGE_COLS];
        boolean[] isUsed = new boolean[droppedBubbles.size()];
        int index = 0;
        for (MovingBubble bubble : droppedBubbles) {
            Point space = null;
            for (int row = 0; row < Const.STAGE_ROWS; ++row) {
                for (int col = 0; col < Const.STAGE_COLS - row % 2; ++col) {
                    if (game.getData(row, col) == bubble.getColor()) {
                        boolean isAnySpace = false;
                        for (int direction = 0; direction < 6; ++direction) {
                            Point p = getAdjacent(direction, row, col);
                            if (p.x >= 0 && p.x < Const.STAGE_COLS - p.y % 2 && p.y >= 0) {
                                if (game.getData(p.y, p.x) == Const.NO_BUBBLE) {
                                    space = p;
                                    isAnySpace = true;
                                    break;
                                }
                            }
                        }
                        if (isAnySpace) {
                            if (getSameColorNumberRecursively(row, col, game,
                                    new boolean[Const.STAGE_ROWS][Const.STAGE_COLS]) >= Const.MIN_ERASE_NUMBER - 1) {
                                if (!hasDone[row][col]) {
                                    if (!hasDone[space.y][space.x]) {
                                        if (!isUsed[index]) {
                                            markSameColor(row, col, hasDone, game);
                                            hasDone[space.y][space.x] = true;
                                            game.getChainBubbles().add(bubble);
                                            game.getFallingBubbles().remove(bubble);
                                            bubble.setDestination(space);
                                            isUsed[index] = true;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            ++index;
        }
    }

    /**
     * 指定した位置にある固定した泡に連結な、同じ色の泡にマークをつけます。
     * @param row 行
     * @param col 列
     * @param isSameColor 同じ色かどうか
     * @param game ゲーム
     */
    public static void markSameColor(int row, int col, boolean[][] isSameColor, Game game) {
        int color = game.getData(row, col);
        isSameColor[row][col] = true;
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            if (game.getData(p.y, p.x) == color) {
                if (!isSameColor[p.y][p.x]) {
                    markSameColor(p.y, p.x, isSameColor, game);
                }
            }
        }
    }

    /**
     * 隣接した同じ色があるかどうかを調べます。
     * @param row 行
     * @param col 列
     * @param game ゲーム
     * @param color 連鎖中の色
     * @return 同じ色があるかどうか
     */
    public static boolean canJoin(int row, int col, Game game, int color) {
        boolean ret = false;
        for (int direction = 0; direction < 6; ++direction) {
            Point p = getAdjacent(direction, row, col);
            if (game.getData(p.y, p.x) == color) {
                ret = true;
                break;
            }
        }
        return ret;
    }

    /**
     * おじゃまぷよを設置します。
     * @param game ゲーム
     */
    public static void jam(Game game) {
        Random random = new Random();
        int count = 0; // 降らせたおじゃまぷよの数
        boolean[][] hasDone = new boolean[Const.STAGE_ROWS][Const.STAGE_COLS];
        for (int i = 0; i < 100000 && game.getGotJam() > 0 && count < Const.ONE_TIME_JAM_NUMBER; ++i) {
            int row = random.nextInt(Const.STAGE_ROWS);
            int col = random.nextInt(Const.STAGE_COLS - row % 2);
            if (canPutJum(row, col, game, hasDone)) {
                int color = 0;
                for (int j = 0; j < 100000; ++j) {
                    color = random.nextInt(Const.COLOR_NUMBER);
                    if (!game.isBad(color)) {
                        break;
                    }
                }
                MovingBubble bubble = new MovingBubble(new Point(0, Const.GRID_WIDTH
                        * Const.STAGE_ROWS), color);
                bubble.setDestination(new Point(col, row));
                game.getJumBubbles().add(bubble);
                game.setGotJam(game.getGotJam() - 1);
                ++count;
            }
        }
    }

    /**
     * おじゃまぷよを置けるかどうかを取得します。
     * @param row
     * @param col
     * @param game ゲーム
     * @param hasDone すでにおじゃまぷよの予定地かどうか
     * @return おじゃまぷよを置けるかどうか
     */
    public static boolean canPutJum(int row, int col, Game game, boolean[][] hasDone) {
        if (row == Const.STAGE_ROWS - 1) {
            return false;
        }
        if (row == 0) {
            if (game.getData(row, col) == Const.NO_BUBBLE) {
                if (!hasDone[row][col]) {
                    hasDone[row][col] = true;
                    return true;
                }
            }
        }
        if (game.getData(row, col) == Const.NO_BUBBLE) {
            if (!hasDone[row][col]) {
                for (int direction = 0; direction < 6; ++direction) {
                    Point p = getAdjacent(direction, row, col);
                    if (p.x >= 0 && p.x < Const.STAGE_COLS - p.y % 2 && p.y >= 0) {
                        if (game.getData(p.y, p.x) != Const.NO_BUBBLE) {
                            hasDone[row][col] = true;
                            hasDone[p.y][p.x] = true;
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * 1段下げます。
     * @param game ゲームの情報
     */
    public static void scrollDown(Game game) {
        for (int row = Const.STAGE_ROWS - 2; row >= 0; --row) {
            for (int col = 0; col < Const.STAGE_COLS - 1; ++col) {
                game.setData(game.getData(row, col), row + 1, col);
            }
        }
    }

    /**
     * ファイルからランダムにステージ情報を読み込みます。
     * @param game1 ゲームの情報
     * @param game2 ゲームの情報
     * @param file 読み込むファイル
     * @throws IOException 例外
     */
    public static void readFromFile(Game game1, Game game2, String file) throws IOException {
        boolean hasDone = false;
        while (!hasDone) {
            int count = 0;
            BufferedReader in = new BufferedReader(new InputStreamReader(Util.class.getResource(
                    file).openStream()));
            while (!hasDone && count < 1000) {
                if (new Random().nextDouble() < 0.01) {
                    int row = 0;
                    while (in.ready()) {
                        String line = in.readLine();
                        StringTokenizer tokenizer = new StringTokenizer(line);
                        if (tokenizer.countTokens() == 0) {
                            break;
                        }
                        for (int col = 0; col < Const.STAGE_COLS; ++col) {
                            if (tokenizer.hasMoreTokens()) {
                                String string = tokenizer.nextToken();
                                if (string.equals("-")) {
                                    game1.setData(Const.NO_BUBBLE, row, col);
                                    game2.setData(Const.NO_BUBBLE, row, col);
                                } else {
                                    game1.setData(Integer.parseInt(string), row, col);
                                    game2.setData(Integer.parseInt(string), row, col);
                                }
                            }
                        }
                        hasDone = true;
                        ++row;
                    }
                } else {
                    while (in.ready()) {
                        String line = in.readLine();
                        StringTokenizer tokenizer = new StringTokenizer(line);
                        if (tokenizer.countTokens() == 0) {
                            break;
                        }
                    }
                }
                ++count;
            }
            in.close();
        }
    }

    /**
     * 全消しかどうかを調べます。
     * @param game ゲーム
     * @return 全消しかどうか
     */
    public static boolean isAllNoBubble(Game game) {
        for (int row = 0; row < Const.STAGE_ROWS; ++row) {
            for (int col = 0; col < Const.STAGE_COLS - row % 2; ++col) {
                if (game.getData(row, col) != Const.NO_BUBBLE) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * 今ステージにない色を使用禁止にします。
     * @param game ゲーム
     */
    public static void setBadColors(Game game) {
        boolean badColors[] = new boolean[Const.COLOR_NUMBER];
        Arrays.fill(badColors, true);
        for (int row = 0; row < Const.STAGE_ROWS; ++row) {
            for (int col = 0; col < Const.STAGE_COLS - row % 2; ++col) {
                if (game.getData(row, col) != Const.NO_BUBBLE) {
                    badColors[game.getData(row, col)] = false;
                }
            }
        }
        game.setBadColors(badColors);
    }
}