package server;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import common.CommonConstants;

/**
 * ゲームの情報を扱うクラスです。
 * @author Kumano Tatsuo
 * Created on 2005/01/18 22:18:03
 */
public class Game extends Thread {
    /**
     * ブロックの状態 
     */
    private byte[][] data;

    /**
     * プレイヤの一覧
     */
    private List<Player> players;

    /**
     * 爆弾の一覧
     */
    private List<Bomb> bombs;

    /**
     * 爆弾の通し番号
     */
    private int bombCount;

    /**
     * ゲームを初期化します。
     */
    public Game() {
        this.players = new LinkedList<Player>();
        this.bombs = new LinkedList<Bomb>();
        this.data = new byte[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
        this.bombCount = 0;
        for (int i = 0; i < CommonConstants.STAGE_HEIGHT; ++i) {
            this.data[i][0] = CommonConstants.DATA_HARD_BLOCK;
            this.data[i][CommonConstants.STAGE_WIDTH - 1] = CommonConstants.DATA_HARD_BLOCK;
        }
        for (int i = 0; i < CommonConstants.STAGE_WIDTH; ++i) {
            this.data[0][i] = CommonConstants.DATA_HARD_BLOCK;
            this.data[CommonConstants.STAGE_WIDTH - 1][i] = CommonConstants.DATA_HARD_BLOCK;
        }
        Random random = new Random();
        for (int i = 1; i < CommonConstants.STAGE_HEIGHT - 1; ++i) {
            for (int j = 1; j < CommonConstants.STAGE_WIDTH - 1; ++j) {
                if (random.nextDouble() < ServerConstants.SERVER_HARD_BLOCK_DENSITY) {
                    this.data[i][j] = CommonConstants.DATA_HARD_BLOCK;
                } else if (random.nextDouble() < ServerConstants.SERVER_SOFT_BLOCK_DENSITY) {
                    this.data[i][j] = CommonConstants.DATA_SOFT_BLOCK;
                }
            }
        }
    }

    /**
     * プレイヤを追加します。
     * @param player
     * @throws IOException 
     */
    public void add(Player player) throws IOException {
        this.players.add(player);
        for (byte i = 0; i < CommonConstants.STAGE_HEIGHT; ++i) {
            for (byte j = 0; j < CommonConstants.STAGE_WIDTH; ++j) {
                switch (this.data[i][j]) {
                case CommonConstants.DATA_HARD_BLOCK:
                    player.tellHardBlock(i, j);
                    break;
                case CommonConstants.DATA_SOFT_BLOCK:
                    player.tellSoftBlock(i, j);
                    break;
                case CommonConstants.DATA_ITEM:
                    player.tellItem(i, j);
                    break;
                }
            }
        }
        for (Player player2 : this.players) {
            if (player2.getName() != null && player2.getPass() != null) {
                player.tellName(player2.getNumber(), player2.getName());
                if (player2.isPlaying()) {
                    player.tellLocation(player2.getNumber(), 1, player2.getRow(), player2.getCol());
                    player.tellRate(player2.getNumber(), (int) Rate.getRate(player2.getName(),
                            player2.getPass()));
                }
            }
        }
        for (Bomb bomb : this.bombs) {
            player.tellBombLocation(bomb.getNumber(), 1, bomb.getRow(), bomb.getCol());
        }
    }

    public void run() {
        long lastFixedTime = System.currentTimeMillis();
        long lastExchangeTime = System.currentTimeMillis();
        while (true) {
            try {
                {
                    Collection<Player> disconnectPlayers = new ArrayList<Player>();
                    for (Player player : this.players) {
                        if (player.isDisconnected) {
                            disconnectPlayers.add(player);
                        }
                    }
                    if (!disconnectPlayers.isEmpty()) {
                        this.players.removeAll(disconnectPlayers);
                        for (Player player : this.players) {
                            for (Player disconnectPlayer : disconnectPlayers) {
                                if (disconnectPlayer.isPlaying()) {
                                    player.tellDead(disconnectPlayer.getNumber());
                                    if (player.getName() != null && player.getPass() != null) {
                                        Rate.record(player.getName(), player.getPass(),
                                                disconnectPlayer.getName(), disconnectPlayer
                                                        .getPass());
                                        player.tellRate(player.getNumber(), (int) Rate.getRate(
                                                player.getName(), player.getPass()));
                                    }
                                }
                            }
                        }
                    }
                }
                for (Player player : this.players) {
                    if (player.getName() == null || player.getPass() == null) {
                        player.setBomb(false);
                    }
                    if (player.isBomb() && !player.isPlaying() && player.getName() != null
                            && player.getPass() != null) {
                        int bestRow = new Random().nextInt(CommonConstants.STAGE_HEIGHT - 2) + 1;
                        int bestCol = new Random().nextInt(CommonConstants.STAGE_WIDTH - 2) + 1;
                        int maxDistance = 0;
                        for (int row = 1; row < CommonConstants.STAGE_HEIGHT - 1; ++row) {
                            for (int col = 1; col < CommonConstants.STAGE_WIDTH - 1; ++col) {
                                if (this.data[row][col] == CommonConstants.DATA_NO_BLOCK) {
                                    if ((this.data[row - 1][col] == CommonConstants.DATA_NO_BLOCK && this.data[row][col + 1] == CommonConstants.DATA_NO_BLOCK)
                                            || (this.data[row][col + 1] == CommonConstants.DATA_NO_BLOCK && this.data[row + 1][col] == CommonConstants.DATA_NO_BLOCK)
                                            || (this.data[row + 1][col] == CommonConstants.DATA_NO_BLOCK && this.data[row][col - 1] == CommonConstants.DATA_NO_BLOCK)
                                            || (this.data[row][col - 1] == CommonConstants.DATA_NO_BLOCK && this.data[row - 1][col] == CommonConstants.DATA_NO_BLOCK)) {
                                        int distance = Integer.MAX_VALUE;
                                        for (Player player2 : this.players) {
                                            if (player2 != player) {
                                                if (Math.abs(player2.getRow() - row)
                                                        + Math.abs(player2.getCol() - col) < distance) {
                                                    distance = Math.abs(player2.getRow() - row)
                                                            + Math.abs(player2.getCol() - col);
                                                }
                                            }
                                        }
                                        if (distance > maxDistance) {
                                            bestRow = row;
                                            bestCol = col;
                                            maxDistance = distance;
                                        }
                                    }
                                }
                            }
                        }
                        player.setRow((byte) bestRow);
                        player.setCol((byte) bestCol);
                        for (Player player2 : this.players) {
                            player2.tellName(player.getNumber(), player.getName());
                            player2.tellLocation(player.getNumber(), 1, player.getRow(), player
                                    .getCol());
                            player2.tellRate(player.getNumber(), (int) Rate.getRate(player
                                    .getName(), player.getPass()));
                        }
                        player.setPlaying(true);
                        player.setBomb(false);
                    }
                    if (player.isPlaying() && player.getStopTime() <= System.currentTimeMillis()) {
                        if (player.isBomb()) {
                            boolean[][] isAnyBomb = new boolean[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
                            for (Bomb bomb2 : this.bombs) {
                                isAnyBomb[bomb2.getRow()][bomb2.getCol()] = true;
                            }
                            if (player.getBombNumber() > 0) {
                                if (!isAnyBomb[player.getRow()][player.getCol()]) {
                                    Bomb bomb = new Bomb(this.bombCount++, player, player
                                            .getPower());
                                    bomb.setRow(player.getRow());
                                    bomb.setCol(player.getCol());
                                    this.bombs.add(bomb);
                                    player.setBombNumber(player.getBombNumber() - 1);
                                    for (Player player2 : this.players) {
                                        player2.tellBombLocation(bomb.getNumber(), 1,
                                                bomb.getRow(), bomb.getCol());
                                    }
                                }
                            }
                            player.setBomb(false);
                        }
                        if (player.isLeft && !player.isRight) {
                            movePlayer(player, 0, -1);
                        } else if (!player.isLeft && player.isRight) {
                            movePlayer(player, 0, 1);
                        } else if (player.isUp && !player.isDown) {
                            movePlayer(player, -1, 0);
                        } else if (!player.isUp && player.isDown) {
                            movePlayer(player, 1, 0);
                        }
                    }
                }
                Collection<Bomb> blowBombs = new ArrayList<Bomb>();
                for (Bomb bomb : this.bombs) {
                    if (bomb.getStopTime() <= System.currentTimeMillis()
                            && bomb.getBlowTime() <= System.currentTimeMillis()) {
                        blowBombs.add(bomb);
                        bomb.getParent().setBombNumber(bomb.getParent().getBombNumber() + 1);
                        for (Player player : this.players) {
                            player.tellBombBlow(bomb.getNumber());
                        }
                        boolean[][] isAnyBomb = new boolean[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
                        for (Bomb bomb2 : this.bombs) {
                            isAnyBomb[bomb2.getRow()][bomb2.getCol()] = true;
                        }
                        boolean[][] isAnyPlayer = new boolean[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
                        for (Player player : this.players) {
                            isAnyPlayer[player.getRow()][player.getCol()] = true;
                        }
                        blowBomb(0, 0, bomb, isAnyBomb, isAnyPlayer);
                        int rightLength = blowBomb(0, 1, bomb, isAnyBomb, isAnyPlayer);
                        int downLength = blowBomb(1, 0, bomb, isAnyBomb, isAnyPlayer);
                        int upLength = blowBomb(-1, 0, bomb, isAnyBomb, isAnyPlayer);
                        int leftLength = blowBomb(0, -1, bomb, isAnyBomb, isAnyPlayer);
                        for (Player player : this.players) {
                            player.tellHorizontalBlow(bomb.getRow(),
                                    (byte) (bomb.getCol() - leftLength),
                                    (byte) (leftLength + rightLength + 1));
                            player.tellVerticalBlow((byte) (bomb.getRow() - upLength), bomb
                                    .getCol(), (byte) (upLength + downLength + 1));
                        }
                    }
                }
                this.bombs.removeAll(blowBombs);
                for (Bomb bomb : this.bombs) {
                    if (bomb.getStopTime() <= System.currentTimeMillis()) {
                        boolean[][] isAnyBomb = new boolean[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
                        for (Bomb bomb2 : this.bombs) {
                            isAnyBomb[bomb2.getRow()][bomb2.getCol()] = true;
                        }
                        boolean[][] isAnyPlayer = new boolean[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
                        for (Player player : this.players) {
                            isAnyPlayer[player.getRow()][player.getCol()] = true;
                        }
                        byte row = bomb.getRow();
                        byte col = bomb.getCol();
                        int dr = bomb.getDr();
                        int dc = bomb.getDc();
                        if (this.data[row + dr][col + dc] == CommonConstants.DATA_NO_BLOCK
                                && !isAnyBomb[row + dr][col + dc]
                                && !isAnyPlayer[row + dr][col + dc]) {
                            bomb.setRow((byte) (row + dr));
                            bomb.setCol((byte) (col + dc));
                            bomb.setStopTime(System.currentTimeMillis() + ServerConstants.SERVER_BOMB_SPEED);
                            for (Player player : this.players) {
                                player.tellBombLocation(bomb.getNumber(),
                                        (int) (bomb.getStopTime() - System.currentTimeMillis()),
                                        bomb.getRow(), bomb.getCol());
                            }
                            for (Player player : this.players) {
                                if (player.getRow() == bomb.getRow()
                                        && player.getCol() == bomb.getCol()) {
                                    player.setPlaying(false);
                                    Rate.record(bomb.getParent().getName(), bomb.getParent()
                                            .getPass(), player.getName(), player.getPass());
                                    for (Player player2 : this.players) {
                                        player2.tellDead(player.getNumber());
                                        player2.tellRate(player.getNumber(), (int) Rate.getRate(
                                                player.getName(), player.getPass()));
                                        player2.tellRate(bomb.getParent().getNumber(), (int) Rate
                                                .getRate(bomb.getParent().getName(), bomb
                                                        .getParent().getPass()));
                                    }
                                }
                            }
                        } else {
                            bomb.setDr(0);
                            bomb.setDc(0);
                        }
                    }
                }
                if (System.currentTimeMillis() - lastFixedTime > ServerConstants.SERVER_SOFT_BLOCK_FIX_WAIT_TIME) {
                    int count = 0;
                    int softBlockCount = 0;
                    for (int i = 1; i < CommonConstants.STAGE_HEIGHT; ++i) {
                        for (int j = 1; j < CommonConstants.STAGE_WIDTH; ++j) {
                            if (this.data[i][j] != CommonConstants.DATA_HARD_BLOCK) {
                                ++count;
                                if (this.data[i][j] == CommonConstants.DATA_SOFT_BLOCK) {
                                    ++softBlockCount;
                                }
                            }
                        }
                    }
                    if ((double) softBlockCount / count < ServerConstants.SERVER_SOFT_BLOCK_DENSITY) {
                        byte row = (byte) (new Random().nextInt(CommonConstants.STAGE_HEIGHT - 2) + 1);
                        byte col = (byte) (new Random().nextInt(CommonConstants.STAGE_WIDTH - 2) + 1);
                        boolean canPut = true;
                        if (this.data[row][col] != CommonConstants.DATA_NO_BLOCK) {
                            canPut = false;
                        } else {
                            for (Player player : this.players) {
                                if (player.isPlaying()) {
                                    if (Math.abs(row - player.getRow()) < 2
                                            && Math.abs(col - player.getCol()) < 2) {
                                        canPut = false;
                                        break;
                                    }
                                }
                            }
                        }
                        if (canPut) {
                            this.data[row][col] = CommonConstants.DATA_SOFT_BLOCK;
                            for (Player player : this.players) {
                                player.tellSoftBlock(row, col);
                            }
                            lastFixedTime = System.currentTimeMillis();
                        }
                    }
                }
                if (System.currentTimeMillis() - lastExchangeTime > ServerConstants.SERVER_EXCHANGE_SOFT_BLOCK_AND_HARD_BLOCK_WAIT_TIME) {
                    for (int i = 0; i < 1000000; ++i) {
                        byte row1 = (byte) (new Random().nextInt(CommonConstants.STAGE_HEIGHT - 2) + 1);
                        byte col1 = (byte) (new Random().nextInt(CommonConstants.STAGE_WIDTH - 2) + 1);
                        byte row2 = (byte) (new Random().nextInt(CommonConstants.STAGE_HEIGHT - 2) + 1);
                        byte col2 = (byte) (new Random().nextInt(CommonConstants.STAGE_WIDTH - 2) + 1);
                        if (this.data[row1][col1] == CommonConstants.DATA_HARD_BLOCK
                                && this.data[row2][col2] == CommonConstants.DATA_SOFT_BLOCK) {
                            this.data[row1][col1] = CommonConstants.DATA_SOFT_BLOCK;
                            this.data[row2][col2] = CommonConstants.DATA_HARD_BLOCK;
                            for (Player player : this.players) {
                                player.tellSoftBlock(row1, col1);
                                player.tellHardBlock(row2, col2);
                            }
                            lastExchangeTime = System.currentTimeMillis();
                            break;
                        }
                    }
                }
                if (this.players.isEmpty()) {
                    Thread.sleep(1000);
                }
                Thread.sleep(ServerConstants.SERVER_THREAD_WAIT_TIME);
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 爆弾を爆発させます。
     * @param dr 行方向の距離
     * @param dc 列方向の距離
     * @param bomb 爆弾
     * @param isAnyBomb 爆弾があるかどうか
     * @param isAnyPlayer プレイヤが居るかどうか
     * @return 炎の長さ
     * @throws IOException
     */
    private int blowBomb(int dr, int dc, Bomb bomb, boolean[][] isAnyBomb, boolean[][] isAnyPlayer)
            throws IOException {
        int ret = 0;
        int row = bomb.getRow();
        int col = bomb.getCol();
        for (byte i = 1; i <= bomb.getPower(); ++i) {
            if (this.data[row + dr * i][col + dc * i] == CommonConstants.DATA_SOFT_BLOCK) {
                this.data[row + dr * i][col + dc * i] = CommonConstants.DATA_NO_BLOCK;
                for (Player player2 : this.players) {
                    player2.tellNoBlock((byte) (bomb.getRow() + dr * i), (byte) (bomb.getCol() + dc
                            * i));
                }
                break;
            } else if (this.data[row + dr * i][col + dc * i] == CommonConstants.DATA_HARD_BLOCK) {
                break;
            } else if (isAnyPlayer[row + dr * i][col + dc * i]) {
                for (Player player : this.players) {
                    if (player.isPlaying()) {
                        if (player.getRow() == row + dr * i && player.getCol() == col + dc * i) {
                            player.setPlaying(false);
                            Player winner = (bomb.getKicker() == null) ? bomb.getParent() : bomb
                                    .getKicker();
                            Rate.record(winner.getName(), winner.getPass(), player.getName(),
                                    player.getPass());
                            for (Player player2 : this.players) {
                                player2.tellDead(player.getNumber());
                                player2.tellRate(winner.getNumber(), (int) Rate.getRate(winner
                                        .getName(), winner.getPass()));
                                player2.tellRate(player.getNumber(), (int) Rate.getRate(player
                                        .getName(), player.getPass()));
                            }
                        }
                    }
                }
                break;
            } else if (isAnyBomb[row + dr * i][col + dc * i]) {
                for (Bomb bomb2 : this.bombs) {
                    if (bomb2.getRow() == row + dr * i && bomb2.getCol() == col + dc * i) {
                        bomb2.setBlowTime(System.currentTimeMillis());
                        break;
                    }
                }
                break;
            }
            ++ret;
        }
        return ret;
    }

    /**
     * プレイヤを動かします。
     * @param player プレイヤ
     * @param dr 行方向に動かす距離
     * @param dc 列方向に動かす距離
     * @throws IOException
     */
    private void movePlayer(Player player, int dr, int dc) throws IOException {
        if (this.data[player.getRow() + dr][player.getCol() + dc] == CommonConstants.DATA_NO_BLOCK) {
            boolean[][] isAnyBomb = new boolean[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
            for (Bomb bomb : this.bombs) {
                isAnyBomb[bomb.getRow()][bomb.getCol()] = true;
            }
            for (Bomb bomb : this.bombs) {
                if (bomb.getRow() == player.getRow() + dr && bomb.getCol() == player.getCol() + dc) {
                    if (this.data[bomb.getRow() + dr][bomb.getCol() + dc] == CommonConstants.DATA_NO_BLOCK
                            && !isAnyBomb[bomb.getRow() + dr][bomb.getCol() + dc]) {
                        bomb.setDr(dr);
                        bomb.setDc(dc);
                        bomb.setStopTime(System.currentTimeMillis());
                        bomb.setKicker(player);
                        player.setStopTime(System.currentTimeMillis()
                                + ServerConstants.SERVER_PLAYER_KICK_TIME);
                        break;
                    }
                }
            }
            if (!isAnyBomb[player.getRow() + dr][player.getCol() + dc]) {
                player.setRow((byte) (player.getRow() + dr));
                player.setCol((byte) (player.getCol() + dc));
                player.setStopTime(System.currentTimeMillis() + ServerConstants.SERVER_PLAYER_DEFAULT_SPEED);
                for (Player player2 : this.players) {
                    player2.tellLocation(player.getNumber(), (int) (player.getStopTime() - System
                            .currentTimeMillis()), player.getRow(), player.getCol());
                }
            }
        }
    }
}
