package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * UFJ銀行の住所を取得するプログラムです。
 * @author Kumano Tatsuo
 * 2005/12/04
 * @since 3.16
 */
public class UFJ {
	/**
	 * ベースURL
	 */
	private static final String BASE_URL = "http://www.e-map.co.jp/standard/11044020/";

	/**
	 * 都道府県別ページのURL
	 */
	private static final String PREFECTURE_URL = "http://www.e-map.co.jp/standard/11044020/ssmtop.htm?SEL_COND=1&SEL_SRV=7&SEL_SUBMIT_GO=1&ARA_TODID=";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "EUC-JP";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "ufj_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * テスト用のメインメソッドです。
	 * @param args コマンドライン引数
	 * @throws IOException 
	 */
	public static void main(String[] args) throws IOException {
		System.out.println(getAddresses("大阪市中央区", "27"));
	}

	/**
	 * UFJ銀行の座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param city 市区町村名
	 * @param cityID 市区町村コード
	 * @param prefectureID 都道府県コード
	 * @param isj 街区レベル位置参照情報
	 * @return 座標と店舗名の対応表
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String city, final String cityID, final String prefectureID,
			final Map<String, Point2D> isj) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(UFJ.CACHE_DIR).exists()) {
			new File(UFJ.CACHE_DIR).mkdirs();
		}
		final String cacheFile = UFJ.CACHE_DIR + File.separator + UFJ.PREFIX + cityID + UFJ.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(UFJ.getAddresses(city, prefectureID), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
				cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "UFJ");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * UFJ銀行の市区町村別ページを解析して住所と店舗名の一覧を取得します。
	 * @param city 市区町村名
	 * @param prefectureID 都道府県コード
	 * @return 住所と店舗名の一覧
	 * @throws IOException 
	 */
	public static Map<String, String> getAddresses(final String city, final String prefectureID)
			throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		for (final Map.Entry<String, URL> entry : getCities(prefectureID).entrySet()) {
			if (entry.getKey().contains(city)) {
				final Collection<URL> shops = getShops(entry.getValue());
				if (shops.isEmpty()) {
					final Pair pair = getAddresses(entry.getValue());
					if (pair != null) {
						ret.put(pair.address, pair.caption);
					}
				} else {
					for (final URL url : shops) {
						final Pair pair = getAddresses(url);
						if (pair != null) {
							ret.put(pair.address, pair.caption);
						}
					}
				}
				break;
			}
		}
		return ret;
	}

	/**
	 * 店舗名と住所をカプセル化するクラスです。
	 * @author Kumano Tatsuo
	 * 2005/12/04
	 */
	private static class Pair {
		/**
		 * 店舗名
		 */
		String caption;

		/**
		 * 住所
		 */
		String address;

		/**
		 * コンストラクタです。
		 * @param caption 店舗名
		 * @param address 住所
		 */
		public Pair(final String caption, final String address) {
			this.caption = caption;
			this.address = address;
		}
	}

	/**
	 * 店舗別ページを解析して、店舗名と住所の対応表を取得します。
	 * @param url URL
	 * @return 店舗名と住所の対応表
	 * @throws IOException
	 */
	public static Pair getAddresses(final URL url) throws IOException {
		final Scanner scanner = new Scanner(url.openStream(), UFJ.ENCODING);
		boolean isFirst = true;
		boolean isAddress = false;
		String caption = null;
		final Pattern pattern = Pattern.compile("<TD [^<>]+><FONT [^<>]+>([^<>]+)</FONT></TD>");
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final Matcher matcher = pattern.matcher(line);
			if (matcher.find()) {
				if (isFirst) {
					caption = matcher.group(1).replace("　", "");
					isFirst = false;
				} else if (isAddress) {
					if (!line.contains("〒")) {
						final String address = matcher.group(1).replace("　", "");
						if (caption != null) {
							return new Pair(caption, address);
						}
					}
				}
			}
			if (line.contains("住　所")) {
				isAddress = true;
			}
		}
		scanner.close();
		return null;
	}

	/**
	 * 市区町村別ページを解析して店舗別ページのURLの一覧を取得します。
	 * @param url URL
	 * @return 店舗別ページのURLの一覧
	 * @throws IOException
	 */
	public static Collection<URL> getShops(final URL url) throws IOException {
		final Collection<URL> ret = new ArrayList<URL>();
		final Scanner scanner = new Scanner(url.openStream(), UFJ.ENCODING);
		final Pattern pattern = Pattern
				.compile("<A HREF=\"/standard/11044020/(ssactl.htm\\?ENC=[^<>]+)\">");
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final Matcher matcher = pattern.matcher(line);
			if (matcher.find()) {
				ret.add(new URL(UFJ.BASE_URL + matcher.group(1)));
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * 都道府県別ページを解析して市区町村名とURLの対応表を取得します。
	 * @param id 都道府県コード
	 * @return 市区町村名とURLの対応表
	 * @throws IOException 
	 */
	public static Map<String, URL> getCities(final String id) throws IOException {
		final Map<String, URL> ret = new LinkedHashMap<String, URL>();
		final URL url = new URL(UFJ.PREFECTURE_URL + id);
		final Scanner scanner = new Scanner(url.openStream(), UFJ.ENCODING);
		String prefecture = null;
		final Pattern pattern = Pattern
				.compile("<IMG SRC=\"img/ken/[0-9]+.gif\" WIDTH=\"[0-9]+\" ALT=\"([^> ]+) +\">");
		final Pattern pattern2 = Pattern
				.compile("<A HREF=\"/standard/11044020/(ssactl.htm\\?ENC=[^<>#]+)\">(.+)</A></TD>");
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final Matcher matcher = pattern.matcher(line);
			if (matcher.find()) {
				prefecture = matcher.group(1);
			}
			final Matcher matcher2 = pattern2.matcher(line);
			String city = null;
			if (matcher2.find()) {
				city = matcher2.group(2);
				final URL url2 = new URL(UFJ.BASE_URL + matcher2.group(1));
				ret.put(prefecture + "," + city, url2);
			}
		}
		scanner.close();
		return ret;
	}
}
