package map;

import java.awt.GraphicsEnvironment;
import java.awt.Toolkit;
import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Arrays;

/**
 * 定数を集めたクラスです。
 * @author Kumano Tatsuo
 * Created on 2005/05/03 9:07:02
 */
public class Const {
	/**
	 * 街区レベル位置参照情報に関した定数を集めたクラスです。
	 * @author Kumano Tatsuo
	 * 2005/11/27
	 */
	public static class Isj {
		/**
		 * 街区レベル位置参照情報の配布URL
		 */
		public static final String BASE_URL = "http://nlftp.mlit.go.jp/isj/dls/data/";

		/**
		 * キャッシュディレクトリの相対パス
		 */
		public static final String CACHE_DIR = ".map" + File.separator + "isj";

		/**
		 * 圧縮ファイルの接頭語
		 */
		public static final String ZIP_PREFIX = "03.0a/";

		/**
		 * 圧縮ファイルの接尾語
		 */
		public static final String ZIP_SUFFIX = "-03.0a.zip";

		/**
		 * 展開済みファイルの接尾語
		 */
		public static final String CSV_SUFFIX = "_2004.csv";

		/**
		 * 古い圧縮ファイルの接頭語
		 */
		public static final String OLD_ZIP_PREFIX = "02.0a/";

		/**
		 * 古い圧縮ファイルの接尾語
		 */
		public static final String OLD_ZIP_SUFFIX = "-02.0a.zip";

		/**
		 * 古い展開済みファイルの接尾語
		 */
		public static final String OLD_CSV_SUFFIX = "_2003.csv";
	}

	/**
	 * 国土数値情報関係の定数を集めたクラスです。
	 * @author Kumano Tatsuo
	 * 2005/11/14
	 */
	public static class Ksj {
		/**
		 * 国土数値情報の配布URL
		 */
		public static final String BASE_URL = "http://nlftp.mlit.go.jp/ksj/dls/data/";

		/**
		 * キャッシュディレクトリの相対パス
		 */
		public static final String CACHE_DIR = ".map" + File.separator + "ksj";

		/**
		 * 国土数値情報の行政界・海岸線の展開済みファイルの接頭語
		 */
		public static final String TXT_PREFIX = "N03-11A-2K_";

		/**
		 * 国土数値情報の行政界・海岸線の展開済みファイルの接尾語
		 */
		public static final String TXT_SUFFIX = ".txt";

		/**
		 * 国土数値情報の行政界・海岸線の圧縮ファイルの接頭語
		 */
		public static final String ZIP_PREFIX = "N03-11A-";

		/**
		 * 国土数値情報の行政界・海岸線の圧縮ファイルの接尾語
		 */
		public static final String ZIP_SUFFIX = "-01.0a.zip";
	}

	/**
	 * 数値地図2500関係の定数を集めたクラスです。
	 * @author Kumano Tatsuo
	 * 2005/11/14
	 */
	public static class Sdf2500 {
		/**
		 * 数値地図2500（空間データ基盤）の配布URL 
		 */
		public static final String BASE_URL = "http://sdf.gsi.go.jp/";

		/**
		 * キャッシュディレクトリの相対パス 
		 */
		public static final String CACHE_DIR = ".map" + File.separator + "cache";

		/**
		 * 展開済みファイルの一覧を記録したファイル
		 */
		public static final String EXTRACTED_LOG_FILE = CACHE_DIR + File.separator
				+ "extractedfiles.txt";
	}

	/**
	 * セブンイレブン関連の定数を集めたクラスです。
	 * @author Kumano Tatsuo
	 * 2005/11/29
	 */
	public static class Sej {
		/**
		 * ベースURL
		 */
		public static final String BASE_URL = "http://gis.e-map.co.jp/standard/13125010/";
		
		/**
		 * トップページのURL
		 */
		public static final String TOP_PAGE = "http://gis.e-map.co.jp/standard/13125010/ssactl.htm?X_shop_id=40000&X_cancel_url=http%3A%2F%2Fwww.sej.co.jp%2F&GO.x=16&GO.y=5";

		/**
		 * エンコーディング
		 */
		public static final String ENCODING = "EUC-JP";

		/**
		 * キャッシュディレクトリの相対パス
		 */
		public static final String CACHE_DIR = ".map" + File.separator + "sej";
		
		/**
		 * CSVファイルの接頭語
		 */
		public static final String PREFIX = "sej_";
		
		/**
		 * CSVファイルの接尾語
		 */
		public static final String SUFFIX = ".csv";
}
	
	/**
	 * 三井住友銀行関連の定数を集めたクラスです。
	 * @author Kumano Tatsuo
	 * 2005/11/29
	 */
	public static class Smbc {
		/**
		 * ベースURL
		 */
		public static final String BASE_URL = "http://map.e-map.co.jp/standard/11034020/";

		/**
		 * トップページのURL
		 */
		public static final String TOP_PAGE = "http://map.e-map.co.jp/standard/11034020/ssmtop.htm";

		/**
		 * エンコーディング
		 */
		public static final String ENCODING = "EUC-JP";
		
		/**
		 * キャッシュディレクトリの相対パス
		 */
		public static final String CACHE_DIR = ".map" + File.separator + "smbc";
		
		/**
		 * CSVファイルの接頭語
		 */
		public static final String PREFIX = "smbc_";
		
		/**
		 * CSVファイルの接尾語
		 */
		public static final String SUFFIX = ".csv";
	}

	/**
	 * 表示倍率関係の定数を集めたクラスです。
	 * @author Kumano Tatsuo
	 * 2005/11/26
	 */
	public static class Zoom {
		/**
		 * 表示倍率の変化率
		 */
		public static final double RATE = 1.1;

		/**
		 * ディスプレイの解像度（dpi）
		 */
		public static final int RESOLUTION = Toolkit.getDefaultToolkit().getScreenResolution();

		/**
		 * 国土数値情報の市区町村データを読み込む倍率
		 */
		public static final double LOAD_CITIES = Const.Zoom.RESOLUTION / 2.5 * 100 / 900000;

		/**
		 * 高精度の国土数値情報の市区町村データを読み込む倍率
		 */
		public static final double LOAD_FINE_CITIES = Const.Zoom.RESOLUTION / 2.5 * 100 / 150000;

		/**
		 * 数値地図2500（空間データ基盤）を読み込む表示倍率
		 */
		public static final double LOAD_2500 = Const.Zoom.RESOLUTION / 2.5 * 100 / 75000;

		/**
		 * 丁目のポリゴンデータを読み込む表示倍率
		 */
		public static final double LOAD_GYOUSEI = Const.Zoom.RESOLUTION / 2.5 * 100 / 21000;

		/**
		 * 全てのデータを読み込む表示倍率 
		 */
		public static final double LOAD_ALL = Const.Zoom.RESOLUTION / 2.5 * 100 / 10000;

		/**
		 * 表示倍率の最大値
		 */
		public static final double MAX_VALUE = Const.Zoom.RESOLUTION / 2.5 * 100 / 2500;
	}

	/**
	 * ファイルの一覧が記載されているファイル
	 */
	public static final URL FILE_LIST = Const.class.getResource("files.csv");

	/**
	 * ゴシック体のフォント
	 */
	public static final String FONT_FAMILY_GOTHIC = Arrays.asList(
			GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames())
			.contains("MS UI Gothic") ? "MS UI Gothic" : Arrays.asList(
			GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames())
			.contains("Kochi Gothic") ? "Kochi Gothic" : "SansSerif";

	/**
	 * 明朝体のフォント
	 */
	public static final String FONT_FAMILY_MINCHO = Arrays.asList(
			GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames())
			.contains("ＭＳ Ｐ明朝") ? "ＭＳ Ｐ明朝" : Arrays.asList(
			GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames())
			.contains("Kochi Mincho") ? "Kochi Mincho" : "Serif";

	/**
	 * コンポーネント間の隙間[px]
	 */
	public static final int GAP = 4;

	/**
	 * 一度読み込んだ地図を捨てずに置いておく範囲
	 */
	public static final double KEEP_COFFICIENT = 1;

	/**
	 * 地図を先読みする範囲
	 */
	public static final double PRE_LOAD_COEFFICIENT = 0;

	/**
	 * 高精度の都道府県の情報が記載されているファイルの接頭語
	 */
	public static final String PREFECTURE_PREFIX = "prefecture_";

	/**
	 * 高精度の都道府県の情報が記載されているファイルの拡張子
	 */
	public static final String PREFECTURE_SUFFIX = ".csv";

	/**
	 * 全ての都道府県の情報が記載されているファイル
	 */
	public static final InputStream PREFECTURES = Const.class
			.getResourceAsStream("prefectures.csv");
}
