package map;

import java.awt.Shape;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import web.WebUtilities;

/**
 * 国土数値情報の行政界・海岸線（面）から作成された1つの市区町村を管理するクラスです。
 * @author Kumano Tatsuo
 * 2005/11/11
 */
public class City {
	/**
	 * 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	private boolean has2500;

	/**
	 * 外接長方形
	 */
	private final Rectangle2D bounds;

	/**
	 * 高精度の領域
	 */
	private Shape fineShape;

	/**
	 * 市区町村コード
	 */
	private final String id;

	/**
	 * 市区町村名
	 */
	private final String label;

	/**
	 * 領域
	 */
	private final Shape shape;

	/**
	 * 数値地図2500（空間データ基盤）のURL
	 */
	private final URL url;

	/**
	 * 市区町村を初期化します。
	 * @param shape 領域
	 * @param label 市区町村名
	 * @param id 市区町村コード
	 * @param url 数値地図2500（空間データ基盤）のURL
	 */
	public City(final Shape shape, final String label, final String id, final URL url) {
		this.shape = shape;
		this.bounds = shape.getBounds2D();
		this.label = label;
		this.id = id;
		this.url = url;
		this.isjLabels = new HashMap<Point2D, String>();
	}

	/**
	 * 高精度の領域を開放します。
	 */
	public void freeFineShape() {
		this.fineShape = null;
	}

	/**
	 * @return 外接長方形
	 */
	public Rectangle2D getBounds() {
		return this.bounds;
	}

	/**
	 * @return 高精度の領域
	 */
	public Shape getFineShape() {
		return this.fineShape;
	}

	/**
	 * @return 市区町村コード
	 */
	public String getId() {
		return this.id;
	}

	/**
	 * @return 市区町村名
	 */
	public String getLabel() {
		return this.label;
	}

	/**
	 * @return 領域
	 */
	public Shape getShape() {
		return this.shape;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）のURL
	 */
	public URL getURL() {
		return this.url;
	}

	/**
	 * @return 高精度の領域を持っているかどうか
	 */
	public boolean hasFineShape() {
		return this.fineShape != null;
	}

	/**
	 * @param shape 高精度の領域
	 */
	public void setFineShape(final Shape shape) {
		this.fineShape = shape;
	}

	/**
	 * @return 数値地図2500（空間データ基盤）を読み込んだかどうか
	 */
	public boolean has2500() {
		return this.has2500;
	}

	/**
	 * @param has2500 数値地図2500（空間データ基盤）を読み込んだかどうか 
	 */
	public void setHas2500(final boolean has2500) {
		this.has2500 = has2500;
	}

	/**
	 * 街区レベル位置参照情報をダウンロードし、読み込みます。
	 * @throws IOException 
	 */
	public void loadIsj() throws IOException {
		File csvFile = new File(Const.Isj.CACHE_DIR + File.separator + this.id
				+ Const.Isj.CSV_SUFFIX);
		final File oldCsvFile = new File(Const.Isj.CACHE_DIR + File.separator + this.id
				+ Const.Isj.OLD_CSV_SUFFIX);
		final URL url = new URL(Const.Isj.BASE_URL + Const.Isj.ZIP_PREFIX + this.id
				+ Const.Isj.ZIP_SUFFIX);
		final URL oldUrl = new URL(Const.Isj.BASE_URL + Const.Isj.OLD_ZIP_PREFIX + this.id
				+ Const.Isj.OLD_ZIP_SUFFIX);
		if (csvFile.exists()) {
			// 平成16年の展開済みファイルがあるとき
			System.out.println("DEBUG: skipped getting " + url);
		} else {
			if (oldCsvFile.exists()) {
				// 平成15年の展開済みファイルがあるとき
				System.out.println("DEBUG: skipped getting " + oldUrl);
				csvFile = oldCsvFile;
			} else {
				final File cacheDir = new File(Const.Isj.CACHE_DIR);
				if (!cacheDir.exists()) {
					cacheDir.mkdir();
				}
				try {
					url.openStream();
					// 平成16年の圧縮ファイルをダウンロードできるとき
					final File file = new File(Const.Isj.CACHE_DIR + File.separator + this.id
							+ Const.Isj.ZIP_SUFFIX);
					file.createNewFile();
					System.out.println("DEBUG: getting " + url);
					get(url, file);
					csvFile = new File(Const.Isj.CACHE_DIR + File.separator + this.id
							+ Const.Isj.CSV_SUFFIX);
				} catch (FileNotFoundException e) {
					try {
						System.out.println("WARNING: failed to get " + url);
						oldUrl.openStream();
						// 平成15年の圧縮ファイルをダウンロードできるとき
						final File file = new File(Const.Isj.CACHE_DIR + File.separator + this.id
								+ Const.Isj.OLD_ZIP_SUFFIX);
						file.createNewFile();
						System.out.println("DEBUG: getting " + oldUrl);
						get(oldUrl, file);
						csvFile = new File(Const.Isj.CACHE_DIR + File.separator + this.id
								+ Const.Isj.OLD_CSV_SUFFIX);
					} catch (FileNotFoundException e1) {
						System.out.println("WARNING: failed to get " + oldUrl);
						e1.printStackTrace();
					}
				}
			}
		}
		this.isj = new HashMap<String, Point2D>();
		{
			final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(csvFile),
					"SJIS"));
			boolean isFirst = true;
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				if (isFirst) {
					isFirst = false;
				} else {
					final Scanner scanner2 = new Scanner(line);
					scanner2.useDelimiter(",");
					final StringBuilder string = new StringBuilder();
					string.append(scanner2.next().replaceAll("\"", "") + ",");
					string.append(scanner2.next().replaceAll("\"", "") + ",");
					string.append(scanner2.next().replaceAll("\"", "") + ",");
					string.append(scanner2.next().replaceAll("\"", ""));
					scanner2.next();
					scanner2.next();
					scanner2.next();
					final String latitude = scanner2.next();
					final String longitude = scanner2.next();
					scanner2.next();
					final int rep = scanner2.nextInt();
					if (rep == 1) {
						if (longitude.length() == 10 && latitude.length() == 9) {
							final Point2D point = UTMUtil.toUTM(Double.parseDouble(longitude),
									-Double.parseDouble(latitude));
							this.isj.put(string.toString(), point);
						} else {
							System.out.println("WARNING: invalid longitude or latitude: " + line);
						}
					}
				}
			}
			scanner.close();
		}
	}

	/**
	 * ZIPファイルを取得して展開します。
	 * @param url URL
	 * @param file 出力ファイル
	 * @throws IOException
	 * @throws FileNotFoundException
	 * @throws ZipException
	 */
	private void get(final URL url, final File file) throws IOException, FileNotFoundException,
			ZipException {
		WebUtilities.copy(url.openStream(), new FileOutputStream(file));
		final ZipFile zipFile = new ZipFile(file);
		for (final Enumeration<? extends ZipEntry> enumeration = zipFile.entries(); enumeration
				.hasMoreElements();) {
			final ZipEntry entry = enumeration.nextElement();
			if (entry.getName().endsWith(".csv")) {
				WebUtilities.copy(zipFile.getInputStream(entry), new FileOutputStream(
						Const.Isj.CACHE_DIR + File.separator + new File(entry.getName())));
			}
		}
	}

	/**
	 * 街区レベル位置参照情報を開放します。
	 */
	public void freeIsj() {
		this.isj = null;
	}

	/**
	 * @return 街区レベル位置参照情報を持っているかどうか
	 */
	public boolean hasIsj() {
		return this.isj != null;
	}

	/**
	 * 街区レベル位置参照情報
	 */
	private Map<String, Point2D> isj;

	/**
	 * 街区レベル位置参照情報のラベル位置
	 */
	private Map<Point2D, String> isjLabels;

	/**
	 * @return 街区レベル位置参照情報
	 */
	public Map<String, Point2D> getIsj() {
		return this.isj;
	}

	/**
	 * @return 街区レベル位置参照情報のラベル位置
	 */
	public Map<Point2D, String> getIsjLabels() {
		return this.isjLabels;
	}

	/**
	 * 街区レベル位置参照情報のラベル位置を空にします。
	 */
	public void clearIsjLabels() {
		this.isjLabels.clear();
	}
}
