package map;

import java.awt.Shape;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import ksj.ShapeIO;

/**
 * 国土数値情報の行政界・海岸線（面）から作成された都道府県に関するユーティリティクラスです。
 * @author Kumano Tatsuo
 * 2005/11/11
 */
public class Prefectures {
	/**
	 * 必要に応じて市区町村のデータを読み込んだり、開放したりします。
	 * ついでに高精度のと道府県界も読み込んだり開放したりします。
	 * @param prefectures 都道府県の一覧
	 * @param panel 地図を表示するパネル
	 * @throws IOException 入出力例外
	 * @throws UnsupportedEncodingException サポート外エンコーディング例外
	 */
	public static void loadCities(final Collection<Prefecture> prefectures, final MapPanel panel) throws UnsupportedEncodingException, IOException {
		if (panel.getZoom() >= Const.LOAD_CITIES_ZOOM) {
			final Rectangle2D visibleRectangle = panel.getVisibleRectangle();
			for (final Prefecture prefecture : prefectures) {
				if (prefecture.getBounds().intersects(visibleRectangle)) {
					final Shape shape = prefecture.hasFineShape() ? prefecture.getFineShape()
							: prefecture.getShape();
					if (shape.intersects(visibleRectangle)) {
						if (!prefecture.hasCities()) {
							prefecture.loadFineShape();
							prefecture.loadCities();
							panel.setNeedsRepaint(true);
							panel.repaint();
						}
					} else {
						prefecture.freeCities();
						prefecture.freeFineShape();
					}
				} else {
					prefecture.freeCities();
					prefecture.freeFineShape();
				}
			}
		} else {
			for (final Prefecture prefecture : prefectures) {
				prefecture.freeCities();
				prefecture.freeFineShape();
			}			
		}
	}
	
	/**
	 * 都道府県を読み込みます。
	 * @param preferences 色の設定
	 * @return 都道府県の一覧
	 */
	public static Collection<Prefecture> loadPrefectures(final MapPreferences preferences) {
		final Collection<Prefecture> ret = new ArrayList<Prefecture>();
		for (final Map.Entry<Shape, String> entry : ShapeIO.readShapes(Const.PREFECTURES)
				.entrySet()) {
			final String[] values = entry.getValue().split("_");
			if (values.length == 2) {
				final String idString = values[0];
				final int id = Integer.parseInt(idString);
				final String label = values[1];
				ret.add(new Prefecture(entry.getKey(), label, idString, preferences
						.getTyomeFillColor(id == 30 || id == 13 ? (id + 2) % 6 + 1 : id % 6 + 1)));
			} else {
				System.out.println("WARNING: 都道府県名の表記がおかしいです。" + entry.getValue());
			}
		}
		return ret;
	}
}
