package map;

import java.awt.Color;
import java.awt.Shape;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import ksj.ShapeIO;

/**
 * 1つの都道府県を管理するクラスです。
 * @author Kumano Tatsuo
 * 2005/11/10
 */
public class Prefecture {
	/**
	 * 領域
	 */
	private final Shape shape;

	/**
	 * 外接長方形
	 */
	private final Rectangle2D bounds;

	/**
	 * 都道府県名
	 */
	private final String label;

	/**
	 * 都道府県コード
	 */
	private final String id;

	/**
	 * 都道府県内の市区町村の一覧
	 */
	private Collection<City> cities;

	/**
	 * 色
	 */
	private final Color color;

	/**
	 * 新しい都道府県を初期化します。
	 * @param shape 領域
	 * @param label 都道府県名
	 * @param id 都道府県コード
	 * @param color 色
	 */
	public Prefecture(final Shape shape, final String label, final String id, final Color color) {
		this.shape = shape;
		this.bounds = shape.getBounds2D();
		this.label = label;
		this.id = id;
		this.color = color;
		this.cities = new ArrayList<City>();
	}

	/**
	 * @return 市区町村の一覧
	 * @throws IOException 入出力例外
	 * @throws UnsupportedEncodingException サポート外エンコーディング例外
	 */
	public Collection<City> getCities() throws UnsupportedEncodingException, IOException {
		if (this.cities.isEmpty()) {
			loadCities();
		}
		return this.cities;
	}

	/**
	 * 市区町村の一覧を読み込みます。
	 * @throws IOException 入出力例外
	 * @throws UnsupportedEncodingException サポート外エンコーディング例外
	 */
	public void loadCities() throws UnsupportedEncodingException, IOException {
		final Map<Shape, String> shapes = ShapeIO.readShapes(Prefecture.class
				.getResourceAsStream(Const.CITIES_PREFIX + this.id + Const.CITIES_SUFFIX));
		final Map<String, URL> urls = new HashMap<String, URL>();
		final java.util.Scanner scanner = new java.util.Scanner(new InputStreamReader(
				Const.FILE_LIST.openStream(), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			if (!line.startsWith("#")) {
				final String[] strings = line.split(",");
				if (strings.length == 3) {
					final String filename = strings[2];
					final Pattern pattern = Pattern.compile(".+/([0-9][0-9][0-9][0-9][0-9])\\.lzh");
					final Matcher matcher = pattern.matcher(filename);
					if (matcher.matches()) {
						final String cityID = matcher.group(1);
						final URL url = new URL(Const.BASE_URL + filename);
						urls.put(cityID, url);
					} else {
						System.out.println("WARNING: ファイル名の形式が不正です。" + filename);
					}
				} else {
					System.out.println("WARNING: ファイル一覧の形式が不正です。" + line);
				}
			}
		}
		for (final Map.Entry<Shape, String> entry : shapes.entrySet()) {
			final String[] values = entry.getValue().split("_");
			if (values.length == 4) {
				this.cities.add(new City(entry.getKey(), values[3], values[2], urls.get(values[2])));
			} else {
				System.out.println("WARNING: 市区町村名の表記がおかしいです。" + entry.getValue());
			}
		}
	}

	/**
	 * 市区町村の一覧を開放します。
	 */
	public void freeCities() {
		if (!this.cities.isEmpty()) {
			this.cities.clear();
		}
	}

	/**
	 * @return 市区町村の一覧を持っているかどうか
	 */
	public boolean hasCities() {
		return !this.cities.isEmpty();
	}

	/**
	 * @return 領域
	 */
	public Shape getShape() {
		return this.shape;
	}

	/**
	 * @return 都道府県名
	 */
	public String getLabel() {
		return this.label;
	}

	/**
	 * @return 外接長方形
	 */
	public Rectangle2D getBounds() {
		return this.bounds;
	}

	/**
	 * @return 色
	 */
	public Color getColor() {
		return this.color;
	}

	/**
	 * @return 都道府県コード
	 */
	public String getId() {
		return this.id;
	}
}
