package sidepanel;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.swing.JButton;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;

import jp.gr.java_conf.dangan.util.lha.LhaHeader;
import jp.gr.java_conf.dangan.util.lha.LhaInputStream;
import map.Const;
import map.MapData;
import map.MapPanel;
import web.WebUtilities;

/**
 * 読み込む地図を選択するパネルです。 
 * @author Kumano Tatsuo
 * Created on 2005/05/13 8:27:58
 */
public class LoadMapPanel extends JPanel {
	/**
	 * 地図
	 */
	private final Map<String, MapData> maps;

	/**
	 * 地図を表示するパネル
	 */
	private final MapPanel panel;

	/**
	 * 地区町村名とURLの対応表 
	 */
	final Map<String, URL> urls;

	/**
	 * 読み込む地図を選択するパネルを初期化します。
	 * @param maps 地図
	 * @param panel 地図を表示するパネル
	 * @throws IOException 入出力例外
	 */
	public LoadMapPanel(final Map<String, MapData> maps, final MapPanel panel) throws IOException {
		this.maps = maps;
		this.panel = panel;
		this.urls = new LinkedHashMap<String, URL>();
		this.setBackground(Color.WHITE);
		final JPanel keywordPanel = new JPanel();
		keywordPanel.setLayout(new BorderLayout(Const.GAP, Const.GAP));
		keywordPanel.setBackground(Color.WHITE);
		final JTextField keywordField = new JTextField();
		final JButton searchButton = new JButton("検索");
		keywordPanel.add(keywordField, BorderLayout.CENTER);
		keywordPanel.add(searchButton, BorderLayout.EAST);
		final JPanel resultPanel = new JPanel();
		resultPanel.setLayout(new BorderLayout(Const.GAP, Const.GAP));
		final JList resultList = new JList();
		resultPanel.add(new JScrollPane(resultList));
		final JButton loadButton = new JButton("読み込み");
		this.setLayout(new GridBagLayout());
		this.add(keywordPanel, new GridBagConstraints(0, 0, 1, 1, 1, 0,
				GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH, new Insets(Const.GAP,
						Const.GAP, 0, Const.GAP), 0, 0));
		this.add(resultPanel, new GridBagConstraints(0, 1, 1, 1, 1, 1,
				GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH, new Insets(Const.GAP,
						Const.GAP, Const.GAP, Const.GAP), 0, 0));
		this.add(loadButton, new GridBagConstraints(0, 2, 1, 1, 1, 0, GridBagConstraints.NORTHWEST,
				GridBagConstraints.BOTH, new Insets(0, Const.GAP, Const.GAP, Const.GAP), 0, 0));
		final Map<String, Map<String, String>> files = WebUtilities.loadFileList(Const.FILE_LIST);
		keywordField.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				searchButton.doClick(200);
			}
		});
		searchButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				try {
					final String keyword = keywordField.getText();
					final String baseURL = Const.BASE_URL;
					LoadMapPanel.this.urls.clear();
					for (final Map.Entry<String, Map<String, String>> entry : files.entrySet()) {
						final String prefecture = entry.getKey();
						for (final Map.Entry<String, String> entry2 : entry.getValue().entrySet()) {
							final String city = entry2.getKey();
							final String filename = entry2.getValue();
							if (prefecture.contains(keyword) || city.contains(keyword)) {
								LoadMapPanel.this.urls.put(prefecture + city, new URL(baseURL
										+ filename));
							}
						}
					}
					resultList.setListData(LoadMapPanel.this.urls.keySet().toArray());
					if (LoadMapPanel.this.urls.size() == 1) {
						resultList.setSelectedIndex(0);
						loadButton.doClick(200);
					}
				} catch (IOException exception) {
					exception.printStackTrace();
				}
			}
		});
		resultList.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent e) {
				if (e.getClickCount() == 2) {
					loadButton.doClick();
				}
			}
		});
		resultList.addKeyListener(new KeyAdapter() {
			public void keyTyped(KeyEvent e) {
				if (e.getKeyChar() == '\n') {
					loadButton.doClick(200);
				}
			}
		});
		loadButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				try {
					final String cacheDir = Const.CACHE_DIR;
					final Collection<URL> selectedURLs = new ArrayList<URL>();
					for (final Object o : resultList.getSelectedValues()) {
						selectedURLs.add(LoadMapPanel.this.urls.get(o));
					}
					final Collection<File> cachedFiles = new HashSet<File>();
					final Collection<String> loadedMaps = new ArrayList<String>();
					String baseDir = null;
					String basePrefecture = null;
					maps.clear();
					for (final URL url : selectedURLs) {
						final String[] separetedPath = url.getPath().split("/");
						if (separetedPath.length == 4) { // こういう数字がバグの種。
							final String prefecture = separetedPath[2];
							final String filename = separetedPath[3];
							new File(cacheDir + File.separator + prefecture).mkdirs();
							final File outFile = new File(cacheDir + File.separator + prefecture
									+ File.separator + filename);
							if (outFile.exists()
									&& url.openConnection().getContentLength() == outFile.length()) {
								System.out.println("INFO: skipped " + url + " -> " + outFile);
							} else {
								System.out.println("INFO: downloading " + url + " -> " + outFile);
								WebUtilities.copy(url.openStream(), new FileOutputStream(outFile));
							}
							if (baseDir == null) {
								baseDir = cacheDir + File.separator + prefecture;
								basePrefecture = prefecture;
							}
							cachedFiles.add(outFile);
						}
					}
					for (final File file : cachedFiles) {
						System.out.println("INFO: extracting " + file);
						final String[] separatedPath = file.getPath().split("\\" + File.separator);
						if (separatedPath.length == 4) { // こういう数字がバグの種。
							final String prefecture = separatedPath[separatedPath.length - 2];
							final LhaInputStream in = new LhaInputStream(new FileInputStream(file));
							LhaHeader entry;
							while ((entry = in.getNextEntry()) != null) {
								final String entryPath = entry.getPath();
								final String path = cacheDir + File.separator + prefecture
										+ File.separator + entryPath;
								final File outFile = new File(path);
								if (path.endsWith(File.separator)) {
									outFile.mkdir();
									if (prefecture.equals(basePrefecture)) {
										if (entryPath.indexOf(File.separator) == entryPath.length() - 1) {
											loadedMaps.add(entryPath.substring(0,
													entryPath.length() - 1).toLowerCase());
										}
									}
								} else {
									if (!outFile.exists()
											|| entry.getOriginalSize() != outFile.length()) {
										WebUtilities.copy(in, new FileOutputStream(outFile));
									}
								}
							}
						}
					}
					if (baseDir != null) {
						final List<String> list = new ArrayList<String>();
						final String[] files = new File(baseDir).list();
						final Pattern pattern = Pattern
								.compile("[0-9][0-9][a-zA-Z][a-zA-Z][0-9][0-9][0-9]");
						for (int i = 0; i < files.length; i++) {
							if (pattern.matcher(files[i]).matches()) {
								list.add(files[i].toLowerCase());
							}
						}
						for (final String mapName : list) {
							final MapData map = new MapData(baseDir, mapName);
							maps.put(mapName, map);
						}
						panel.setMinMaxXY(loadedMaps);
						panel.zoomAutomaticaly();
						panel.moveToCenter();
						panel.setNeedsRepaint(true);
						panel.repaint();
					}
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});
	}
}
