package swt;

import java.awt.Dimension;
import java.awt.Graphics2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Pattern;

import jp.gr.java_conf.dangan.util.lha.LhaHeader;
import jp.gr.java_conf.dangan.util.lha.LhaInputStream;
import map.Const;
import map.MapData;
import map.MapPanel;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;

import web.WebUtilities;

/**
 * SWT版のメインウィンドウです。
 * @author Kumano Tatsuo
 * Created on 2005/05/14 16:35:51
 */
public class SWTMainFrame extends Thread {
    /**
     * 地図
     */
    final Map<String, MapData> maps;

    /**
     * 地図を表示するパネル
     */
    final MapPanel panel;

    /**
     * 地図を描画するキャンバス
     */
    Canvas canvas;

    /**
     * 初期化が終了したかどうか
     */
    private boolean isReady;

    /**
     * ディスプレイ 
     */
    private Display display;

    /**
     * キャンバスの大きさ 
     */
    Dimension canvasSize;

    /**
     * ツールを表す列挙型です。
     * @author Kumano Tatsuo
     * Created on 2005/05/15 17:02:19
     */
    enum Tool {
        /**
         * 手のひらツール 
         */
        HAND,
        /**
         * 縮尺変更ツール
         */
        ZOOM
    }

    /**
     * 選択されているツール
     */
    Tool tool;

    /**
     * キャンバス上でマウスが押された座標
     */
    Point lastMouseLocation;

    /**
     * キャンバス上にあるマウスの座標
     */
    Point nowMouseLocation;

    /**
     * マウスがキャンバス上にあるかどうか
     */
    boolean isMouseInCanvas;

    /**
     * マウスが動いた幅
     */
    double mouseMotionWidth;

    /**
     * マウスが動いた高さ
     */
    double mouseMotionHeight;

    /**
     * 縮尺が変化したかどうか
     */
    boolean isChanged;

    /**
     * 地区町村名とURLの対応表 
     */
    final Map<String, URL> urls;

    /**
     * メインウィンドウを初期化します。
     * @param maps 地図
     * @param panel 地図を表示するパネル
     */
    public SWTMainFrame(final Map<String, MapData> maps, MapPanel panel) {
        this.maps = maps;
        this.panel = panel;
        this.isReady = false;
        this.isChanged = false;
        this.urls = new LinkedHashMap<String, URL>();
    }

    public void run() {
        try {
            // 定数を設定する
            final int MARGIN = 6;
            this.display = new Display();
            final Shell shell = new Shell(this.display);
            shell.setText("Map");
            final GridLayout layout = new GridLayout(1, true);
            layout.marginWidth = 0;
            layout.marginHeight = 0;
            layout.verticalSpacing = 0;
            shell.setLayout(layout);
            // メニューバー
            final Menu menubar = new Menu(shell, SWT.BAR);
            shell.setMenuBar(menubar);
            final MenuItem fileItem = new MenuItem(menubar, SWT.CASCADE);
            fileItem.setText("ファイル(&F)");
            final Menu fileMenu = new Menu(fileItem);
            fileItem.setMenu(fileMenu);
            // まだ実装されていません。
            //final MenuItem exportItem = new MenuItem(fileMenu, SWT.CASCADE);
            //exportItem.setText("エクスポート(&E)");
            //final Menu exportMenu = new Menu(exportItem);
            //exportItem.setMenu(exportMenu);
            //final MenuItem rasterItem = new MenuItem(exportMenu, SWT.PUSH);
            //rasterItem.setText("ラスタ画像（PNG、JPG、BMPファイル）(&I)...");
            //final MenuItem psItem = new MenuItem(exportMenu, SWT.PUSH);
            //psItem.setText("&PSファイル...");
            //final MenuItem separator = new MenuItem(fileMenu, SWT.SEPARATOR);
            final MenuItem exitItem = new MenuItem(fileMenu, SWT.PUSH);
            exitItem.setText("終了(&X)");
            exitItem.addSelectionListener(new SelectionListener() {
                public void widgetDefaultSelected(SelectionEvent arg0) {
                }

                public void widgetSelected(SelectionEvent arg0) {
                    System.out.println("DEBUG: メニューから終了コマンドが実行されました。");
                    System.exit(0);
                }
            });
            final MenuItem toolItem = new MenuItem(menubar, SWT.CASCADE);
            toolItem.setText("ツール(&T)");
            final Menu toolMenu = new Menu(toolItem);
            toolItem.setMenu(toolMenu);
            final MenuItem handMenuItem = new MenuItem(toolMenu, SWT.RADIO);
            handMenuItem.setText("手(&H)");
            final MenuItem zoomMenuItem = new MenuItem(toolMenu, SWT.RADIO);
            zoomMenuItem.setText("拡大(&Z)");
            // ツールバー
            final ToolBar toolBar = new ToolBar(shell, SWT.PUSH);
            final ToolItem handToolItem = new ToolItem(toolBar, SWT.RADIO);
            handToolItem.setImage(new Image(this.display, SWTMainFrame.class
                    .getResourceAsStream("hand.gif")));
            handToolItem.setText("手(&H)");
            handMenuItem.setSelection(true);
            handToolItem.setSelection(true);
            SWTMainFrame.this.tool = Tool.HAND;
            final ToolItem zoomToolItem = new ToolItem(toolBar, SWT.RADIO);
            zoomToolItem.setText("拡大(&Z)");
            zoomToolItem.setImage(new Image(this.display, SWTMainFrame.class
                    .getResourceAsStream("zoom.gif")));
            final SelectionListener handSelectionListener = new SelectionListener() {
                public void widgetDefaultSelected(SelectionEvent arg0) {
                }

                public void widgetSelected(SelectionEvent e) {
                    boolean selection = false;
                    if (e.widget instanceof MenuItem) {
                        selection = ((MenuItem) e.widget).getSelection();
                    } else if (e.widget instanceof ToolItem) {
                        selection = ((ToolItem) e.widget).getSelection();
                    }
                    if (selection) {
                        System.out.println("DEBUG: 手のひらツールが選択されました。");
                        SWTMainFrame.this.tool = Tool.HAND;
                        handMenuItem.setSelection(true);
                        handToolItem.setSelection(true);
                        zoomMenuItem.setSelection(false);
                        zoomToolItem.setSelection(false);
                    }
                }
            };
            handMenuItem.addSelectionListener(handSelectionListener);
            handToolItem.addSelectionListener(handSelectionListener);
            final SelectionListener zoomSelectionListener = new SelectionListener() {
                public void widgetDefaultSelected(SelectionEvent arg0) {
                }

                public void widgetSelected(SelectionEvent e) {
                    boolean selection = false;
                    if (e.widget instanceof MenuItem) {
                        selection = ((MenuItem) e.widget).getSelection();
                    } else if (e.widget instanceof ToolItem) {
                        selection = ((ToolItem) e.widget).getSelection();
                    }
                    if (selection) {
                        System.out.println("DEBUG: 縮尺変更ツールが選択されました。");
                        SWTMainFrame.this.tool = Tool.ZOOM;
                        zoomMenuItem.setSelection(true);
                        zoomToolItem.setSelection(true);
                        handMenuItem.setSelection(false);
                        handToolItem.setSelection(false);
                    }
                }
            };
            zoomMenuItem.addSelectionListener(zoomSelectionListener);
            zoomToolItem.addSelectionListener(zoomSelectionListener);
            final GridData toolBarGridData = new GridData(
                    GridData.FILL_HORIZONTAL);
            toolBar.setLayoutData(toolBarGridData);
            // サッシ
            final SashForm sash = new SashForm(shell, SWT.NULL);
            sash.SASH_WIDTH = MARGIN;
            final GridData sashGridData = new GridData(GridData.FILL_BOTH);
            sash.setLayoutData(sashGridData);
            // サイドパネル
            final Composite sideComposite = new Composite(sash, SWT.NULL);
            sideComposite.setLayout(new FillLayout(SWT.VERTICAL));
            final TabFolder tabFolder = new TabFolder(sideComposite, SWT.NULL);
            final TabItem tabItem = new TabItem(tabFolder, SWT.NULL);
            tabItem.setText("読み込み");
            final Composite loadComposite = new Composite(tabFolder, SWT.NULL);
            final GridLayout loadLayout = new GridLayout(1, true);
            loadComposite.setLayout(loadLayout);
            tabItem.setControl(loadComposite);
            final Label searchLabel = new Label(loadComposite, SWT.NULL);
            searchLabel.setText("市区町村名(&C)");
            final Map<String, Map<String, String>> files = WebUtilities
                    .loadFileList(Const.FILE_LIST);
            final Text searchText = new Text(loadComposite, SWT.BORDER);
            searchText.setFocus();
            final Label resultLabel = new Label(loadComposite, SWT.NULL);
            resultLabel.setText("検索結果(&R)");
            final List resultList = new List(loadComposite, SWT.BORDER
                    | SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
            final Button loadButton = new Button(loadComposite, SWT.BORDER);
            loadButton.setText("読み込み(&L)");
            loadButton.addSelectionListener(new SelectionListener() {
                public void widgetDefaultSelected(SelectionEvent arg0) {
                }

                public void widgetSelected(SelectionEvent arg0) {
                    System.out.println("DEBUG: 検索ボタンが押されました。");
                    System.out.println("DEBUG: 選択されているものは"
                            + Arrays.toString(resultList.getSelection()));
                    try {
                        final String cacheDir = Const.CACHE_DIR;
                        final Collection<URL> selectedURLs = new ArrayList<URL>();
                        for (final String string : resultList.getSelection()) {
                            selectedURLs
                                    .add(SWTMainFrame.this.urls.get(string));
                        }
                        final Collection<File> cachedFiles = new HashSet<File>();
                        final Collection<String> loadedMaps = new ArrayList<String>();
                        String baseDir = null;
                        String basePrefecture = null;
                        SWTMainFrame.this.maps.clear();
                        for (final URL url : selectedURLs) {
                            final String[] separetedPath = url.getPath().split(
                                    "/");
                            if (separetedPath.length == 4) { // こういう数字がバグの種。
                                final String prefecture = separetedPath[2];
                                final String filename = separetedPath[3];
                                new File(cacheDir + File.separator + prefecture)
                                        .mkdirs();
                                final File outFile = new File(cacheDir
                                        + File.separator + prefecture
                                        + File.separator + filename);
                                if (outFile.exists()
                                        && url.openConnection()
                                                .getContentLength() == outFile
                                                .length()) {
                                    System.out.println("INFO: skipped " + url
                                            + " -> " + outFile);
                                } else {
                                    System.out.println("INFO: downloading "
                                            + url + " -> " + outFile);
                                    WebUtilities.copy(url.openStream(),
                                            new FileOutputStream(outFile));
                                }
                                if (baseDir == null) {
                                    baseDir = cacheDir + File.separator
                                            + prefecture;
                                    basePrefecture = prefecture;
                                }
                                cachedFiles.add(outFile);
                            }
                        }
                        for (final File file : cachedFiles) {
                            System.out.println("INFO: extracting " + file);
                            final String[] separatedPath = file.getPath()
                                    .split("\\" + File.separator);
                            if (separatedPath.length == 4) { // こういう数字がバグの種。
                                final String prefecture = separatedPath[separatedPath.length - 2];
                                final LhaInputStream in = new LhaInputStream(
                                        new FileInputStream(file));
                                LhaHeader entry;
                                while ((entry = in.getNextEntry()) != null) {
                                    final String entryPath = entry.getPath();
                                    final String path = cacheDir
                                            + File.separator + prefecture
                                            + File.separator + entryPath;
                                    final File outFile = new File(path);
                                    if (path.endsWith(File.separator)) {
                                        outFile.mkdir();
                                        if (prefecture.equals(basePrefecture)) {
                                            if (entryPath
                                                    .indexOf(File.separator) == entryPath
                                                    .length() - 1) {
                                                loadedMaps
                                                        .add(entryPath
                                                                .substring(
                                                                        0,
                                                                        entryPath
                                                                                .length() - 1)
                                                                .toLowerCase());
                                            }
                                        }
                                    } else {
                                        if (!outFile.exists()
                                                || entry.getOriginalSize() != outFile
                                                        .length()) {
                                            WebUtilities.copy(in,
                                                    new FileOutputStream(
                                                            outFile));
                                        }
                                    }
                                }
                            }
                        }
                        if (baseDir != null) {
                            final Collection<String> list = new ArrayList<String>();
                            final String[] files = new File(baseDir).list();
                            final Pattern pattern = Pattern
                                    .compile("[0-9][0-9][a-zA-Z][a-zA-Z][0-9][0-9][0-9]");
                            for (int i = 0; i < files.length; i++) {
                                if (pattern.matcher(files[i]).matches()) {
                                    list.add(files[i].toLowerCase());
                                }
                            }
                            for (final String mapName : list) {
                                final MapData map = new MapData(baseDir,
                                        mapName);
                                SWTMainFrame.this.maps.put(mapName, map);
                            }
                            SWTMainFrame.this.canvas.getVerticalBar()
                                    .setSelection(
                                            SWTMainFrame.this.canvas
                                                    .getVerticalBar()
                                                    .getMinimum());
                            SWTMainFrame.this.panel.setMinMaxXY(loadedMaps);
                            SWTMainFrame.this.panel.zoomAutomaticaly();
                            SWTMainFrame.this.panel.moveToCenter();
                            SWTMainFrame.this.panel.setNeedsRepaint(true);
                            SWTMainFrame.this.panel.repaint();
                        }
                    } catch (Exception exception) {
                        exception.printStackTrace();
                    }
                }
            });
            searchText.addModifyListener(new ModifyListener() {
                public void modifyText(ModifyEvent arg0) {
                    System.out.println("DEBUG: テキストが変更されました。");
                    try {
                        final String keyword = searchText.getText();
                        if (keyword.length() > 0) {
                            final String baseURL = Const.BASE_URL;
                            SWTMainFrame.this.urls.clear();
                            resultList.removeAll();
                            for (final Map.Entry<String, Map<String, String>> entry : files
                                    .entrySet()) {
                                final String prefecture = entry.getKey();
                                for (final Map.Entry<String, String> entry2 : entry
                                        .getValue().entrySet()) {
                                    final String city = entry2.getKey();
                                    final String filename = entry2.getValue();
                                    if (prefecture.contains(keyword)
                                            || city.contains(keyword)) {
                                        SWTMainFrame.this.urls.put(prefecture
                                                + city, new URL(baseURL
                                                + filename));
                                    }
                                }
                            }
                            for (final String key : SWTMainFrame.this.urls
                                    .keySet()) {
                                resultList.add(key);
                            }
                            if (SWTMainFrame.this.urls.size() == 1) {
                                resultList.selectAll();
                            }
                        }
                    } catch (IOException exception) {
                        exception.printStackTrace();
                    }
                }
            });
            final GridData searchTextGridData = new GridData(
                    GridData.FILL_HORIZONTAL);
            searchText.setLayoutData(searchTextGridData);
            final GridData resultListGridData = new GridData(GridData.FILL_BOTH);
            resultList.setLayoutData(resultListGridData);
            final GridData loadGridData = new GridData(GridData.FILL_HORIZONTAL);
            loadButton.setLayoutData(loadGridData);
            // 地図を表示するパネル
            final Composite centerComposite = new Composite(sash, SWT.NULL);
            centerComposite.setLayout(new FillLayout());
            final Graphics2DRenderer renderer = new Graphics2DRenderer();
            this.canvas = new Canvas(centerComposite, SWT.BORDER | SWT.V_SCROLL);
            this.canvas.addPaintListener(new PaintListener() {
                public void paintControl(PaintEvent e) {
                    System.out.println("DEBUG: キャンバスが再描画されました。");
                    final GC gc = e.gc;
                    renderer.prepareRendering(gc);
                    final Graphics2D g2d = renderer.getGraphics2D();
                    SWTMainFrame.this.panel.drawMap(200, g2d);
                    renderer.render(gc);
                }
            });
            this.canvas.addControlListener(new ControlListener() {
                public void controlResized(ControlEvent arg0) {
                    System.out.println("DEBUG: キャンバスの大きさが変更されました。");
                    SWTMainFrame.this.canvasSize = new Dimension(
                            SWTMainFrame.this.canvas.getClientArea().width,
                            SWTMainFrame.this.canvas.getClientArea().height);
                    SWTMainFrame.this.panel
                            .setSWTSize(SWTMainFrame.this.canvasSize);
                    SWTMainFrame.this.isChanged = true;
                }

                public void controlMoved(ControlEvent arg0) {
                }
            });
            final ScrollBar scrollBar = this.canvas.getVerticalBar();
            scrollBar.setMinimum(10);
            scrollBar.setMaximum(1000);
            scrollBar.addSelectionListener(new SelectionListener() {
                public void widgetDefaultSelected(SelectionEvent arg0) {
                }

                public void widgetSelected(SelectionEvent e) {
                    scroll(scrollBar);
                }

            });
            this.canvas.addMouseListener(new MouseListener() {
                public void mouseUp(MouseEvent e) {
                    System.out.println("DEBUG: キャンバスでマウスが離されました。");
                    if (SWTMainFrame.this.tool == Tool.HAND) {
                        SWTMainFrame.this.isChanged = true;
                    } else if (SWTMainFrame.this.tool == Tool.ZOOM) {
                        SWTMainFrame.this.isChanged = true;
                    }
                }

                public void mouseDown(MouseEvent e) {
                    System.out.println("DEBUG: キャンバスでマウスが押されました。" + e);
                    if (SWTMainFrame.this.tool == Tool.HAND) {

                    } else if (SWTMainFrame.this.tool == Tool.ZOOM) {
                        if (e.button == 1) {
                            scrollBar.setSelection(scrollBar.getSelection()
                                    + scrollBar.getPageIncrement());
                        } else {
                            scrollBar.setSelection(scrollBar.getSelection()
                                    - scrollBar.getPageIncrement());
                        }
                        scroll(scrollBar);
                    }
                    SWTMainFrame.this.lastMouseLocation = new Point(e.x, e.y);
                }

                public void mouseDoubleClick(MouseEvent arg0) {
                }
            });
            this.canvas.addMouseMoveListener(new MouseMoveListener() {
                public void mouseMove(MouseEvent e) {
                    if (e.stateMask == SWT.BUTTON1
                            || e.stateMask == SWT.BUTTON2
                            || e.stateMask == SWT.BUTTON3) {
                        System.out.println("DEBUG: マウスがドラッグされました。");
                        if (SWTMainFrame.this.tool == Tool.HAND) {
                            SWTMainFrame.this.panel
                                    .setOffset(
                                            SWTMainFrame.this.panel
                                                    .getOffsetX()
                                                    - (e.x - SWTMainFrame.this.lastMouseLocation.x),
                                            SWTMainFrame.this.panel
                                                    .getOffsetY()
                                                    - (e.y - SWTMainFrame.this.lastMouseLocation.y));
                            SWTMainFrame.this.lastMouseLocation = new Point(
                                    e.x, e.y);
                            SWTMainFrame.this.mouseMotionWidth = e.x
                                    - SWTMainFrame.this.lastMouseLocation.x;
                            SWTMainFrame.this.mouseMotionHeight = e.y
                                    - SWTMainFrame.this.lastMouseLocation.y;
                            repaint();
                        } else if (SWTMainFrame.this.tool == Tool.ZOOM) {

                        }
                    }
                    SWTMainFrame.this.nowMouseLocation = new Point(e.x, e.y);
                }
            });
            this.canvas.addMouseTrackListener(new MouseTrackListener() {
                public void mouseHover(MouseEvent arg0) {
                }

                public void mouseExit(MouseEvent arg0) {
                    SWTMainFrame.this.isMouseInCanvas = false;
                }

                public void mouseEnter(MouseEvent arg0) {
                    SWTMainFrame.this.isMouseInCanvas = true;
                }
            });
            // ウィンドウを表示する
            final int width = 640;
            final int sideWidth = 200;
            final int height = 480;
            sash.setWeights(new int[] { sideWidth, width - sideWidth });
            shell.setSize(width, height);
            shell.open();
            this.isReady = true;
            while (!shell.isDisposed()) {
                if (!this.display.readAndDispatch()) {
                    this.display.sleep();
                }
            }
            // 「org.eclipse.swt.SWTException: Widget is disposed」って言われます。
            //this.display.dispose();
            System.exit(0); // ええんか？こんなんで。
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 地図を描画するキャンバスを取得します。
     * @return 地図を描画するキャンバス
     */
    public Canvas getCanvas() {
        return this.canvas;
    }

    /**
     * @return 初期化が終了したかどうか
     */
    public boolean isReady() {
        return this.isReady;
    }

    /**
     * キャンバスを再描画します。
     */
    public void repaint() {
        System.out.println("DEBUG: フレームの再描画が要求されました。");
        this.display.asyncExec(new Repainter());
    }

    /**
     * キャンバスを再描画するためのクラスです。
     * @author Kumano Tatsuo
     * Created on 2005/05/15 11:16:29
     */
    class Repainter implements Runnable {
        public void run() {
            SWTMainFrame.this.canvas.redraw();
        }
    }

    /**
     * @return キャンバスの大きさ
     */
    public Dimension getCancasSize() {
        return this.canvasSize;
    }

    /**
     * @return 縮尺が変化したかどうか
     */
    public boolean isChanged() {
        return this.isChanged;
    }

    /**
     * @param isZoomChanged 縮尺が変化したかどうか
     */
    public void setChanged(boolean isZoomChanged) {
        this.isChanged = isZoomChanged;
    }

    /**
     * スクロールバーの値に応じて地図の縮尺を変更します。
     * @param scrollBar スクロールバー
     */
    void scroll(final ScrollBar scrollBar) {
        System.out
                .println("DEBUG: スクロールバーが操作されました。" + scrollBar.getSelection());
        if (SWTMainFrame.this.isMouseInCanvas) {
            System.out.println("DEBUG: ホイールなり。");
        } else {
            System.out.println("DEBUG: スクロールバー直接操作なり。");
        }
        final int mouseX = (SWTMainFrame.this.isMouseInCanvas) ? SWTMainFrame.this.nowMouseLocation.x
                : SWTMainFrame.this.canvasSize.width / 2;
        final int mouseY = (SWTMainFrame.this.isMouseInCanvas) ? SWTMainFrame.this.nowMouseLocation.y
                : SWTMainFrame.this.canvasSize.height / 2;
        final double newZoom = SWTMainFrame.this.panel.getMinZoom()
                * scrollBar.getSelection() / scrollBar.getMinimum();
        final double newX = ((SWTMainFrame.this.panel.getOffsetX() + mouseX)
                / SWTMainFrame.this.panel.getZoom() * newZoom)
                - mouseX;
        final double newY = ((SWTMainFrame.this.panel.getOffsetY() + mouseY)
                / SWTMainFrame.this.panel.getZoom() * newZoom)
                - mouseY;
        SWTMainFrame.this.panel.setOffset(newX, newY);
        SWTMainFrame.this.panel.setZoom(newZoom);
        repaint();
        SWTMainFrame.this.isChanged = true;
    }

}
