// http://www-6.ibm.com/jp/developerworks/java/040702/j_j-2dswt.html
package swt;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;

import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.widgets.Display;

/**
 * Helper class allowing the use of Java 2D on SWT or Draw2D graphical
 * context.
 * @author Yannick Saillet
 */
public class Graphics2DRenderer {
    /**
     * パレットデータ 
     */
    private static final PaletteData PALETTE_DATA = new PaletteData(0xFF0000,
            0xFF00, 0xFF);

    /**
     * AWTのBufferedImage 
     */
    private BufferedImage awtImage;

    /**
     * SWTのImage
     */
    private Image swtImage;

    /**
     * SWTのImageData
     */
    private ImageData swtImageData;

    /**
     * SWTのBufferedImageの内容を配列にダンプしたもの
     */
    private int[] awtPixels;

    /** RGB value to use as transparent color */
    private static final int TRANSPARENT_COLOR = 0x123456;

    /**
     * Prepare to render on a SWT graphics context.
     * @param gc グラフィクスコンテキスト
     */
    public void prepareRendering(GC gc) {
        org.eclipse.swt.graphics.Rectangle clip = gc.getClipping();
        prepareRendering(clip.x, clip.y, clip.width, clip.height);
    }

    /**
     * Prepare to render on a Draw2D graphics context.
     * @param graphics 描画対象のオブジェクト
     */
    public void prepareRendering(org.eclipse.draw2d.Graphics graphics) {
        org.eclipse.draw2d.geometry.Rectangle clip = graphics
                .getClip(new org.eclipse.draw2d.geometry.Rectangle());
        prepareRendering(clip.x, clip.y, clip.width, clip.height);
    }

    /**
     * Prepare the AWT offscreen image for the rendering of the rectangular
     * region given as parameter.
     * @param clipX クリッピング領域の左上のx座標
     * @param clipY クリッピング領域の左上のy座標
     * @param clipW クリッピング領域の幅
     * @param clipH クリッピング領域の高さ
     */
    private void prepareRendering(int clipX, int clipY, int clipW, int clipH) {
        // check that the offscreen images are initialized and large enough
        checkOffScreenImages(clipW, clipH);
        // fill the region in the AWT image with the transparent color
        java.awt.Graphics awtGraphics = this.awtImage.getGraphics();
        awtGraphics.setColor(new java.awt.Color(TRANSPARENT_COLOR));
        awtGraphics.fillRect(clipX, clipY, clipW, clipH);
    }

    /**
     * Returns the Graphics2D context to use.
     * @return Graphics2Dオブジェクト
     */
    public Graphics2D getGraphics2D() {
        if (this.awtImage == null)
            return null;
        return (Graphics2D) this.awtImage.getGraphics();
    }

    /**
     * Complete the rendering by flushing the 2D renderer on a SWT graphical
     * context.
     * @param gc グラフィクスコンテキスト
     */
    public void render(GC gc) {
        if (this.awtImage == null)
            return;

        org.eclipse.swt.graphics.Rectangle clip = gc.getClipping();
        transferPixels(clip.x, clip.y, clip.width, clip.height);
        gc.drawImage(this.swtImage, clip.x, clip.y, clip.width, clip.height,
                clip.x, clip.y, clip.width, clip.height);
    }

    /**
     * Complete the rendering by flushing the 2D renderer on a Draw2D
     * graphical context.
     * @param graphics 描画対象
     */
    public void render(org.eclipse.draw2d.Graphics graphics) {
        if (this.awtImage == null)
            return;

        org.eclipse.draw2d.geometry.Rectangle clip = graphics
                .getClip(new org.eclipse.draw2d.geometry.Rectangle());
        transferPixels(clip.x, clip.y, clip.width, clip.height);
        graphics.drawImage(this.swtImage, clip.x, clip.y, clip.width,
                clip.height, clip.x, clip.y, clip.width, clip.height);
    }

    /**
     * Transfer a rectangular region from the AWT image to the SWT image.
     * @param clipX クリッピング領域の左上のx座標
     * @param clipY クリッピング領域の左上のy座標
     * @param clipW クリッピング領域の幅
     * @param clipH クリッピング領域の高さ
     */
    private void transferPixels(int clipX, int clipY, int clipW, int clipH) {
        int step = this.swtImageData.depth / 8;
        byte[] data = this.swtImageData.data;
        this.awtImage.getRGB(clipX, clipY, clipW, clipH, this.awtPixels, 0, clipW);
        for (int i = 0; i < clipH; i++) {
            int idx = (clipY + i) * this.swtImageData.bytesPerLine + clipX * step;
            for (int j = 0; j < clipW; j++) {
                int rgb = this.awtPixels[j + i * clipW];
                for (int k = this.swtImageData.depth - 8; k >= 0; k -= 8) {
                    data[idx++] = (byte) ((rgb >> k) & 0xFF);
                }
            }
        }
        if (this.swtImage != null)
            this.swtImage.dispose();
        this.swtImage = new Image(Display.getDefault(), this.swtImageData);
    }

    /**
     * Dispose the resources attached to this 2D renderer.
     */
    public void dispose() {
        if (this.awtImage != null)
            this.awtImage.flush();
        if (this.swtImage != null)
            this.swtImage.dispose();
        this.awtImage = null;
        this.swtImageData = null;
        this.awtPixels = null;
    }

    /**
     * Ensure that the offscreen images are initialized and are at least
     * as large as the size given as parameter.
     * @param width 幅
     * @param height 高さ
     */
    private void checkOffScreenImages(int width, int height) {
        int currentImageWidth = 0;
        int currentImageHeight = 0;
        if (this.swtImage != null) {
            currentImageWidth = this.swtImage.getImageData().width;
            currentImageHeight = this.swtImage.getImageData().height;
        }

        // if the offscreen images are too small, recreate them
        if (width > currentImageWidth || height > currentImageHeight) {
            dispose();
            width = Math.max(width, currentImageWidth);
            height = Math.max(height, currentImageHeight);
            this.awtImage = new BufferedImage(width, height,
                    BufferedImage.TYPE_INT_ARGB);
            this.swtImageData = new ImageData(width, height, 24, PALETTE_DATA);
            this.swtImageData.transparentPixel = TRANSPARENT_COLOR;
            this.awtPixels = new int[width * height];
        }
    }
}