package map;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.WindowConstants;
import javax.swing.filechooser.FileFilter;

/**
 * 地図を表示するフレームです。
 * @author Kumano Tatsuo
 * Created on 2003/11/01, 16:00
 */
public class MainFrame extends JFrame {
    /**
     * 地図を表示するパネル
     */
    final MapPanel panel;

    /**
     * 出力するファイル
     */
    File file;

    /**
     * 出力するファイルの種類を表す定数です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 17:35:07
     */
    enum FileType {
        /**
         * PSファイル
         */
        PS, /**
         * PNGファイル
         */
        PNG, /**
         * JPEGファイル
         */
        JPG, /**
         * BMPファイル
         */
        BMP
    }

    /**
     * ファイルの種類
     */
    FileType fileType;

    /**
     * 新しくフレームを初期化します。
     * @param panel 地図を表示するパネル
     */
    public MainFrame(final MapPanel panel) {
        this.panel = panel;
        this.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        this.setSize(800, 600);
        this.setTitle("Map");
        this.setLayout(new BorderLayout());
        this.add(panel, BorderLayout.CENTER);
        JMenuBar menuBar = new JMenuBar();
        this.add(menuBar, BorderLayout.NORTH);
        JMenu fileMenu = new JMenu("ファイル(F)");
        fileMenu.setMnemonic('F');
        menuBar.add(fileMenu);
        JMenuItem saveAsItem = new JMenuItem("エクスポート(E)...");
        saveAsItem.setMnemonic('E');
        saveAsItem.addActionListener(new ExportListener());
        fileMenu.add(saveAsItem);
        fileMenu.addSeparator();
        JMenuItem exitItem = new JMenuItem("終了(X)");
        exitItem.setMnemonic('X');
        exitItem.addActionListener(new ExitListener());
        fileMenu.add(exitItem);
    }

    /**
     * エクスポートの実装です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 15:39:42
     */
    private class ExportListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                JFileChooser chooser = new JFileChooser(new File("."));
                chooser.addChoosableFileFilter(new FileFilter() {
                    public boolean accept(File file) {
                        return file.getName().toLowerCase().endsWith(".ps");
                    }

                    public String getDescription() {
                        return "PSファイル（*.ps）";
                    }
                });
                chooser.addChoosableFileFilter(new FileFilter() {
                    public boolean accept(File file) {
                        return file.getName().toLowerCase().endsWith(".bmp");
                    }

                    public String getDescription() {
                        return "BMPファイル（*.bmp）";
                    }
                });
                chooser.addChoosableFileFilter(new FileFilter() {
                    public boolean accept(File file) {
                        return file.getName().toLowerCase().endsWith(".jpg");
                    }

                    public String getDescription() {
                        return "JPEGファイル（*.jpg）";
                    }
                });
                chooser.addChoosableFileFilter(new FileFilter() {
                    public boolean accept(File file) {
                        return file.getName().toLowerCase().endsWith(".png");
                    }

                    public String getDescription() {
                        return "PNGファイル（*.png）";
                    }
                });
                if (MainFrame.this.file != null) {
                    chooser.setSelectedFile(MainFrame.this.file);
                }
                chooser.showDialog(MainFrame.this, "エクスポート");
                final File selectedFile = chooser.getSelectedFile();
                if (selectedFile != null) {
                    final String fileName = selectedFile.getName();
                    if (fileName != null) {
                        if (fileName.toLowerCase().endsWith(".ps")) {
                            MainFrame.this.file = new File(fileName);
                            MainFrame.this.panel.printPS(MainFrame.this.file);
                            MainFrame.this.fileType = FileType.PS;
                            MainFrame.this.setTitle(MainFrame.this.file.getName() + " - Map");
                        } else if (fileName.toLowerCase().endsWith(".bmp")) {
                            MainFrame.this.file = new File(fileName);
                            MainFrame.this.panel.printBitmap(MainFrame.this.file, "bmp");
                            MainFrame.this.fileType = FileType.BMP;
                            MainFrame.this.setTitle(MainFrame.this.file.getName() + " - Map");
                        } else if (fileName.toLowerCase().endsWith(".jpg")) {
                            MainFrame.this.file = new File(fileName);
                            MainFrame.this.panel.printBitmap(MainFrame.this.file, "jpg");
                            MainFrame.this.fileType = FileType.JPG;
                            MainFrame.this.setTitle(MainFrame.this.file.getName() + " - Map");
                        } else if (fileName.toLowerCase().endsWith(".png")) {
                            MainFrame.this.file = new File(fileName);
                            MainFrame.this.panel.printBitmap(MainFrame.this.file, "png");
                            MainFrame.this.fileType = FileType.PNG;
                            MainFrame.this.setTitle(MainFrame.this.file.getName() + " - Map");
                        }
                    }
                }
            } catch (PrinterException e1) {
                e1.printStackTrace();
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }
    }

    /**
     * 終了の実装です。
     * @author Kumano Tatsuo
     * Created on 2005/02/28 17:49:02
     */
    private class ExitListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            System.exit(0);
        }
    }
}
