// $Id$
//
// Copyright (C) 2006 BEE Co.,Ltd.
//
// Author : Fukasawa Mitsuo
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
//

#include "srgs/xml_schema.hxx"

#include <xercesc/util/XercesDefs.hpp>
#include <xercesc/util/RuntimeException.hpp>
#include <xercesc/util/XMLUri.hpp>
#include <xercesc/util/XMLException.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/XMLUni.hpp>
#include <xercesc/util/XMLURL.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <iostream>

#include "boost/filesystem/operations.hpp"
#include "boost/filesystem/path.hpp"
#include <boost/algorithm/string.hpp>


XERCES_CPP_NAMESPACE_USE

bool is_allspace(const std::string& text)
{
    bool result = false;
    XMLCh * fLocalForm = XMLString::transcode(text.c_str());
    if (XMLString::isAllWhiteSpace(fLocalForm))
    {
        result = true;
    }
    XMLString::release(&fLocalForm);
    return result;
}

std::string trim(const std::string& text)
{
    std::string tmp(text);
    boost::trim(tmp);
    return tmp;
}

// copy from Xerces-C++ sample program
class StrX
{
public :
    // -----------------------------------------------------------------------
    //  Constructors and Destructor
    // -----------------------------------------------------------------------
    StrX(const XMLCh* const toTranscode)
    {
        // Call the private transcoding method
        fLocalForm = XMLString::transcode(toTranscode);
    }

    ~StrX()
    {
        XMLString::release(&fLocalForm);
    }


    // -----------------------------------------------------------------------
    //  Getter methods
    // -----------------------------------------------------------------------
    const char* localForm() const
    {
        return fLocalForm;
    }

private :
    // -----------------------------------------------------------------------
    //  Private data members
    //
    //  fLocalForm
    //      This is the local code page form of the string.
    // -----------------------------------------------------------------------
    char*   fLocalForm;
};


//-----------------------------------------------------------------------------
//
//
namespace xml_schema
{

namespace fs = boost::filesystem;

int init()
{
    // init Xercers-C++ Library
    try
    {
        XMLPlatformUtils::Initialize();
    }
    catch(const XMLException& toCatch)
    {
        XERCES_STD_QUALIFIER wcout << L"Parser Init Failed!\n   INFO: ("
            << toCatch.getSrcFile() << L"." << toCatch.getSrcLine()
            << L") -" << toCatch.getMessage()
            << XERCES_STD_QUALIFIER endl;
        return -1;
    }
    return 0;
}

int fini()
{
    // terminate Xercers-C++ Library
    XMLPlatformUtils::Terminate();
    return 0;
}

//
// Identifier Class
//
class URI
{
public:
    URI() {}
    URI(const char * urich)
    {
    	XMLCh * uri = XMLString::transcode(urich);
    	XMLUri tmp(uri);
    	uri_ = tmp;
    	XMLString::release(&uri);
    }
    URI(const std::string& urich)
    {
    	XMLCh * uri = XMLString::transcode(urich.c_str());
    	XMLUri tmp(uri);
    	uri_ = tmp;
    	XMLString::release(&uri);
    }
    URI(const URI& rhs) : uri_(rhs.uri_) {}
    ~URI() {}

    URI& operator=(const URI& rhs)
    {
    	if (this != &rhs)
    	    uri_ = rhs.uri_;
    	return *this;
    }
    
    std::string path() const
    {
        StrX s(uri_.getPath());
        const char * ch = s.localForm();
    	return ((ch != NULL) ? ch : "");
    }
    std::string fragment() const
    {
        StrX s(uri_.getFragment());
        const char * ch = s.localForm();
    	return ((ch != NULL) ? ch : "");
    }
     
private:
	XMLUri uri_;
};

//
// parse pathname to uri
//
URI * make_uri(const char * pathname)
{
	if (*pathname == '#')
	{   // only fragment
		return NULL;
	}
	std::string tmp(pathname);
	if (strchr(pathname, ':') == NULL)
	{
	    fs::path full_path;
        full_path = fs::system_complete(fs::path(pathname, fs::portable_posix_name));
        full_path.normalize();
	    tmp = "file://" + full_path.string();
	}
	URI * uri = new URI(tmp);
    return uri;
}

//
// URI
//
uri::uri() : name_(""), path_(""), fragment_(""), uri_(NULL)
{
}

uri::uri(const char * uri) : name_(uri), path_(""), fragment_(""), uri_(NULL)
{
	if (uri != NULL && strlen(uri) > 0)
	{
	    uri_ = make_uri(uri);
	    if (uri_ != NULL)
        {
	        path_ = uri_->path();
	        fragment_ = uri_->fragment();
	    }
	    else
	    {   // Only fragment
		    path_ = "";
		    fragment_ = name_;
	    }
	}
}

uri::uri(const std::string& uri) : name_(uri), path_(""), fragment_(""),
                                   uri_(NULL)
{
	if (uri.size() > 0)
	{
	    uri_ = make_uri(uri.c_str());
	    if (uri_ != NULL)
	    {
            path_ = uri_->path();
	        fragment_ = uri_->fragment();
	    }
	    else
	    {   // Only fragment
		    path_ = "";
		    fragment_ = name_;
	    }
	}
}

uri::uri(const uri& rhs) : name_(rhs.name_), path_(rhs.path_),
                           fragment_(rhs.fragment_)
{
	if (rhs.uri_)
	    uri_ = new URI(*rhs.uri_);
    else
        uri_ = NULL;
}

uri::~uri()
{
	if (uri_ != NULL)
	    delete uri_;
}
    
uri& uri::operator=(const uri& rhs)
{
	if (this != &rhs)
	{
	    name_ = rhs.name_;
	    path_ = rhs.path_;
	    fragment_ = rhs.fragment_;
	    if (uri_ != NULL)
	        delete uri_;
        if (rhs.uri_ != NULL)
	        uri_ = new URI(*rhs.uri_);
        else
            uri_ = NULL;
	}
	return *this;
}

// Get/Set
void  uri::name(const std::string& uri) 
{
	name_ = uri;
	if (uri.size() > 0)
	{
        if (uri_ != NULL)
            delete uri_;
            
	    uri_ = make_uri(uri.c_str());
	    if (uri_ != NULL)
	    {
            path_ = uri_->path();
	        fragment_ = uri_->fragment();
	    }
	    else
	    {   // Only fragment
		    path_ = "";
		    fragment_ = name_.substr(1);
	    }
	}
}

    	
}  // namespace
