// $Id$
//
// Copyright (C) 2006 BEE Co.,Ltd.
//
// Author : Fukasawa Mitsuo
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
//

#include "bee/booster.hpp"
#include "srgs/srgs-parser.hxx"
#include "srgs/xml_schema-parser.hxx"
#include "srgs/rule_visitor.hpp"
using namespace srgs;

#include <boost/program_options.hpp>
using namespace boost;
namespace po = boost::program_options;
namespace fs = boost::filesystem;
using std::cerr;
using std::endl;

namespace xml_schema
{
    extern int init();
    extern int fini();
}

namespace srgs
{
extern grammar * make_grammar(const std::string& fileName);  // see grammar.cxx
extern void resolve_ruleref(grammar * gram, const std::string& fileName);
}

// grobal variables
std::string opt_infile("");
std::string opt_outfile("");
std::string opt_outdir("");
std::string opt_dotname("");
std::string opt_version("0.0.0");
fs::path _outfile;
bool opt_tree = false;
bool opt_graphviz = false;
int  opt_trace = bee::trace::L_MIDDLE;
std::string opt_tag_format("");
std::string opt_lang("");
std::string opt_recognize("");

std::string& getopt_recognize()
{
    return opt_recognize;
}

bool getopt_tree()
{
    return opt_tree;
}

int getopt_trace()
{
    return opt_trace;
}

const char * getopt_outfile()
{
    return _outfile.string().c_str();
}

const char * getopt_infile()
{
    return opt_infile.c_str();
}

void setopt_ofile(const std::string& fname)
{
    opt_infile = fname;
}

const std::string& getopt_tag_format()
{
    return opt_tag_format;
}

const std::string& getopt_lang()
{
    return opt_lang;
}

bool getopt_graphviz()
{
    return opt_graphviz;
}

const std::string& getopt_dotname()
{
    return opt_dotname;
}

//
// parse command options
//
int parse_options(int argc, char * argv[])
{
    try
    {
        //po::options_description desc("Allowed options");
        po::options_description desc("Option");
        //    ("help", "produce help message")
        desc.add_options()
            ("help,h", "Help")
            ("trace,t", po::value<int>(&opt_trace)->default_value(3), 
                "Trace Level(0..7)")      // 3 = bee::trace::L_MIDDLE
            ("output-directory,D", po::value<std::string>(&opt_outdir),
                "Output Directory")
            ("tree,T", "Enable Tree")
            ("output-file,o", po::value<std::string>(&opt_outfile),
                "Output File(.txt)")
            ("input-file,i", po::value<std::string>(&opt_infile),
                "Input File(.grxml)")
            ("tag-format,f", po::value<std::string>(&opt_tag_format),
                "effective tag-format attribute")
            ("lang,l", po::value<std::string>(&opt_lang),
                "effective lang attribute")
            ("recognize,r", po::value<std::string>(&opt_recognize),
                "recognize string")
            ("graphviz,g", "write graphvis's dot file")
            ("version,v", "view version")
        ;

        po::positional_options_description p;
        p.add("input-file", -1);
        
        po::variables_map vm;
        po::store(po::command_line_parser(argc, argv).
                  options(desc).positional(p).run(), vm);
        po::notify(vm);
   
        if (argc < 2 || vm.count("help")) 
        {
            std::cout << "Usage: " << argv[0] << " [options] input-file\n";
            std::cout << desc;
            return -1;
        }
        if (vm.count("version"))
        {
            std::cout << opt_version << "\n";
            if (argc <= 2)
                return -1; 
        }

        // check input file
        //if (vm.count("input-file"))
        //{
        //    //opt_infile = vm["input-file"].as<std::string>();
        //}
        if (vm.count("output-directory"))
        {
            if (! fs::exists(opt_outdir))
            {
                std::cerr << "Directory not found: " << opt_outdir << "\n";
                return -1;
            }
            if (! fs::is_directory(opt_outdir))
            {
                std::cerr << "\"" << opt_outdir << "\" is not directory.\n";
                return -1;
            }
        }
        if (vm.count("output-file"))
        {
            if (fs::exists(opt_outfile) && fs::is_directory(opt_outfile))
            {
                std::cerr << "\"" << opt_outfile << "\" is directory.\n";
                return -1;
            }
            //else if (fs::exists(opt_outfile))
            //{
            //    std::cerr << "Warning: over write \"" << opt_outfile << "\".\n";
            //}
        }
        if (vm.count("tree"))
        {
            opt_tree = true; 
        }
        if (vm.count("graphviz"))
        {
            opt_graphviz = true; 
            size_t s = opt_infile.rfind('/');
            opt_dotname = (s == std::string::npos) ? opt_infile :
                                                     opt_infile.substr(s + 1);
            size_t e = opt_dotname.rfind('.');
            if (e != std::string::npos)
                opt_dotname = opt_dotname.substr(0, e);
            opt_dotname += ".dot";
        }
    }
    catch (std::exception& e)
    {
        std::cout << e.what() << "\n";
        return -1;
    }    
    return 0;
}

int check_options()
{
    if (! fs::exists(opt_infile))
    {
        std::cerr << "\"" << opt_infile << "\" is not found.\n";
        return -1;
    }
    if (fs::is_directory(opt_infile))
    {
        std::cerr << "\"" << opt_infile << "\" is directory.\n";
        return -1;
    }
    // output file
    if (opt_outfile.empty())
    {
        size_t p = opt_infile.rfind('.');
        if (p == std::string::npos)
            opt_outfile = opt_infile;
        else
            opt_outfile = opt_infile.substr(0, p);
        opt_outfile += ".txt";
    }
    if (opt_outdir.size() > 0)
    {
        _outfile = opt_outdir;
        _outfile /= opt_outfile;
    }
    else
    {
        _outfile = opt_outfile;
    }
    return 0;
}


//
// Main procedure
//
int sub_main(int argc, char* argv[])
{
    // get command option
    if (parse_options(argc, argv) < 0)
        return 1;
    else if (check_options() < 0)
        return 1;
   
    try
    {
        xml_schema::init();
        bee::trace::init(getopt_trace());
        
        grammar * gram = make_grammar(opt_infile);
        if (gram == NULL)
        {
        	std::cerr << "SRGS PARSER: Error (" << opt_infile
        	          << ")" << std::endl;
        	return 1;
        }

        // check rule reference in root grammar
        srgs::resolve_ruleref(gram, opt_infile);
              
        // Let's print what we've got.
        //
        gram->dump(std::cout);

        srgs::grammar_visitor<> visitor;
        visitor.visit(gram);
        
        //std::stringstream sout;
        visitor.dump(std::cout);
        //std::cerr << sout.str() << std::endl;
        
        //sout.str("");   // reset ostream
        visitor.print_graph(std::cout);
        //std::cerr << sout.str() << std::endl;

        if (! getopt_recognize().empty())
        {
            std::cout << getopt_recognize() << std::endl;
            std::stringstream sout;
            visitor.recognize(gram, sout, getopt_recognize());
            std::cout << sout.str() << std::endl;
        }
        if (getopt_graphviz())
        {
            fs::path grpath(getopt_dotname());
            fs::ofstream grofs(grpath, std::ios_base::binary);
            //std::stringstream grout;
            visitor.print_graphviz(grofs, opt_infile);
            //grofs << grout.str();
            grofs.close();
        }
    }
    catch (const srgs::exception& e)
    {
        std::cerr << e.what() << std::endl;
        xml_schema::fini();
        return 1;
    }
    catch (const xml_schema::parser::exception& e)
    {
        std::cerr << e << std::endl;
        xml_schema::fini();
        return 1;
    }
    catch (const std::ios_base::failure&)
    {
        std::cerr << "io failure" << std::endl;
        xml_schema::fini();
        return 1;
    }
    catch (const std::exception& e)
    {
        std::cerr << e.what() << std::endl;
        xml_schema::fini();
        return 1;
    }
    xml_schema::fini();
    return 0;
}


