/* $Id$ */

//=============================================================================
/**
 *  @file    srgs_wrapper.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#include "bee/booster.hpp"
#include "srgs/srgs-parser.hxx"
#include "srgs/xml_schema-parser.hxx"
#include "srgs/grammar_visitor.hpp"
#include <boost/python/class.hpp>
#include <boost/python/module.hpp>
#include <boost/python/def.hpp>
#include <boost/shared_ptr.hpp>

using namespace boost;
using std::cerr;
using std::endl;

namespace xml_schema
{
extern int init();
extern int fini();
}

namespace srgs
{
extern grammar * make_grammar(const std::string& fileName);  // see grammar.cxx
extern void init_ruleref();
extern void resolve_ruleref(grammar * gram, const std::string& fileName);
extern void setopt_infile(const std::string& fname);
}

extern int  parse_options(int argc, char * argv[]);


namespace srgs
{

using namespace boost::python;

struct gram_wrapper
{
    boost::shared_ptr<srgs::grammar> grammar_;

    gram_wrapper() {}
    gram_wrapper(const gram_wrapper& rhs) : grammar_(rhs.grammar_) {}
    ~gram_wrapper() {}

    gram_wrapper& operator=(const gram_wrapper& rhs)
    {
        if (this != &rhs)
            grammar_ = rhs.grammar_;
        return *this; 
    }

    std::string recognize(const std::string& w)
    {
        TRACE_FUNCTION("srgs::gram_wrapper::recognize");
        if (grammar_.get() == NULL)
            return "REJECT";
        srgs::grammar_visitor<srgs::rule_visitor> visitor;
        visitor.visit(grammar_.get());

        //visitor.dump(std::cout);
        //visitor.print_graph(std::cout);
        
        std::stringstream sout;
        int result = visitor.recognize(grammar_.get(), sout, w);
        if (result < 0)
        {
            //TRACE_DEBUG_FMT("unfunished=%1%", sout.str());
            sout.str(""); // reset buffer
            sout << "REJECT";
        }
        return sout.str(); 
    }

    std::string graph()
    {
        if (grammar_.get() == NULL)
            return "";
        srgs::grammar_visitor<srgs::rule_visitor> visitor;
        visitor.visit(grammar_.get());
        
        std::stringstream sout;
        visitor.dump(sout);
        //std::cerr << sout.str() << std::endl;
        sout << "\n";
        
        //sout.str("");   // reset ostream
        visitor.print_graph(sout);
        //std::cerr << sout.str() << std::endl;
        return sout.str(); 
    }

    std::string dump()
    {
        if (grammar_.get() == NULL)
            return "";
        std::stringstream sout;
        grammar_->dump(sout); 
        return sout.str(); 
    }

    static void wrap()
    {
        class_<srgs::gram_wrapper>("grammar", init<>())
            .def("recognize", &srgs::gram_wrapper::recognize)
            .def("graph",     &srgs::gram_wrapper::graph)
            .def("dump",      &srgs::gram_wrapper::dump)
            ;
    }
};

// A function taking a hello object as an argument.
gram_wrapper make_dictionary(const std::string& fname) 
{
    gram_wrapper result;

    try
    {
        srgs::init_ruleref();   

        result.grammar_ = boost::shared_ptr<srgs::grammar>(srgs::make_grammar(fname));
        if (result.grammar_.get() != NULL)
            srgs::resolve_ruleref(result.grammar_.get(), fname);
    }
    catch (const srgs::exception& e)
    {
        std::cerr << e.what() << std::endl;
    }
    catch (const xml_schema::parser::exception& e)
    {
        std::cerr << e << std::endl;
    }
    catch (const std::exception& e)
    {
        std::cerr << e.what() << std::endl;
    }
           
    return result;
}

int init(/*const std::string& opt*/) 
{
    //int argc = 3;
    //char * argv[] = { "srgs", "-t", "4" , NULL};
    //parse_options(argc, argv);
    xml_schema::init();
    bee::trace::init(getopt_trace());
    return 0;
}

void init_module()
{
    def("init", init);
    def("make_dictionary", make_dictionary);
    gram_wrapper::wrap();
}

} // *namespace* - srgs

BOOST_PYTHON_MODULE(srgs)
{
    srgs::init_module();
}
