/* $Id$ */

//=============================================================================
/**
 *  @file    srgs.hxx
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef SRGS_SRGS_HXX
#define SRGS_SRGS_HXX

#include "bee/trace.hpp"
#include "bee/uri.hpp"

#include "srgs/xml_schema.hxx"
#include "srgs/repeat.hpp"
#include "srgs/element.hpp"
#include "srgs/rule_visitor.hpp"
#include "bee/bee_xsd.hpp"
#include <boost/algorithm/string/trim.hpp>

// function prototype
extern int  getopt_trace();
extern bool getopt_tree();
extern const char * getopt_outfile();
extern const char * getopt_infile();

namespace srgs
{

struct exception: std::runtime_error
{
	exception(const std::string& msg) : std::runtime_error(msg) {}

};

typedef std::string Id_datatype;
typedef std::string Scope_datatype;
typedef std::string Special_datatype;
typedef long double Repeat_prob_datatype;
typedef std::string Repeat_datatype;
typedef std::string Weight_datatype;
typedef std::string Tag_format_datatype;
typedef std::string Version_datatype;
typedef std::string Root_datatype;
typedef std::string Mode_datatype;
typedef std::string Type_datatype;

typedef boost::tokenizer<boost::escaped_list_separator<char> > Tok;

// class prototype
class grammar;
class rule;
class item;

//
// rule reference element
//
class ruleref : public srgs::element
{
public:
    ruleref() : element(element::RULEREF),
                mtype_(""), uri_(""), special_(""), available_(true) {}
  	ruleref(const ruleref& rhs) : element(rhs), mtype_(rhs.mtype_),
  	                              uri_(rhs.uri_), 
  	                              special_(rhs.special_),
                                  rule_(rhs.rule_),
                                  available_(rhs.available_)  {}
    virtual ~ruleref() {}
    
    ruleref& operator=(const ruleref& rhs)
    {
    	ruleref tmp(rhs);
        this->element::swap(tmp);
    	std::swap(mtype_, tmp.mtype_);
    	std::swap(uri_, tmp.uri_);
    	std::swap(special_, tmp.special_);
        std::swap(rule_, tmp.rule_);
        std::swap(available_, tmp.available_);
    	return *this;
    }

    virtual srgs::element * clone() const
    {
    	return new ruleref(*this);
    }
    void clear()
    {
    	this->element::clear();
    	mtype_ = "";
    	uri_ = "";
    	special_ = "";
        rule_.reset();
        available_ = true;
    }
    
    // Get/Set
    bool available() const { return available_; }
    const Type_datatype& media_type() const  { return mtype_; }
    //const xml_schema::uri& uri() const { return uri_; }
    const bee::uri& uri() const { return uri_; }
  	const Special_datatype& special() const { return special_; }
    const boost::shared_ptr<srgs::rule>& rule() const { return rule_; }
    void media_type(const Type_datatype& type) { mtype_ = type; }
  	void uri(const std::string& uri);
    void special(const Special_datatype& special)
    {
        TRACE_FUNCTION("srgs::ruleref::special");
        if (! uri_.name().empty())
        {
            TRACE_ERROR_FMT("illegal to provide both special(%1%) and uri(%2%)",
                            special % uri_.name());
            available_ = false;
        }
        special_ = special;
    }
    void rule(const boost::shared_ptr<srgs::rule>& rule) { rule_ = rule; }

    virtual int accept(rule_visitor& vis)
    {
        TRACE_FUNCTION("srgs::ruleref::visit");
        vis.visit(*this);
        return 0;
    }

    virtual void dump(std::ostream& out, int ident) const
    {
    	std::string space(ident * 2, ' ');
    	if (uri_.name().empty() && (! special_.empty()))
    	{
   		    out << space << "ruleref: " << special_ << "\n";
    	}
    	else
    	{
   		    out << space << "ruleref: " << uri_.name() << "\n";
    	}
    }
    	
private:
    Type_datatype    mtype_;
    //xml_schema::uri  uri_;
    bee::uri         uri_;
  	Special_datatype special_;
    boost::shared_ptr<srgs::rule> rule_;
    bool available_;
};

//
typedef bee::xsd::element_ptr<srgs::ruleref> ruleref_ptr;

//
// manager of rule references
//
class ruleref_manager
{
public:
    struct comp_ruleref
    {
        bool operator()(const ruleref * lhs, const ruleref * rhs)
        {
            if (lhs->str() < rhs->str())
                return true;
            return false;
        }
    };

    typedef std::set<const ruleref *> references;

    ruleref_manager() : current_grammar_("") {}
    void clear() { refs_.clear(); current_grammar_ = ""; }

    // Get/Set
    void  grammar(const std::string& gram) { current_grammar_ = gram; }
    const std::string& grammar() const     { return current_grammar_; }
    //
    bool insert(const ruleref * ref)
    {
        std::pair<references::iterator, bool> result;
        result = refs_.insert(ref);
        return result.second;
    }
    
    const ruleref * get(const std::string& name)
    {
        ruleref dummy;
        dummy.str(name);
        references::iterator iter = refs_.find(&dummy);
        if (iter == refs_.end())
            return NULL;
        return *iter;
    }

    int getall(std::vector<const ruleref *>& ref_v)
    {
        references::iterator iter = refs_.begin();
        for ( ; iter != refs_.end(); ++iter)
        {
            ref_v.push_back(*iter);
        }
        return ref_v.size();
    }

    static ruleref_manager * instance();

    void dump(std::ostream& out) const
    {
        out << "*** rule reference list ***\n";
        references::const_iterator iter = refs_.begin();
        for ( ; iter != refs_.end(); ++iter)
        {
            (*iter)->dump(out, 1);
        }
    }

private:
    references refs_;
    std::string current_grammar_;
};

//
// Meta Data (ANY)
//
class metadata
{
public:
    metadata() : metadata_("") {}
    metadata(const metadata& rhs) : metadata_(rhs.metadata_) {}
    ~metadata() {}
    
    metadata& operator=(const metadata& rhs)
    {
    	metadata tmp(rhs);
    	metadata_ = tmp.metadata_;
    	return *this;
    }

    void clear() { metadata_ = ""; }

    // Get/Set
    const std::string& value() const { return metadata_; }
    void value(const std::string& metadata) { metadata_ = metadata; }

private:
    std::string metadata_;
};

//
// tag element
//
class tag: public srgs::element
{
public:
    tag() : element(element::TAG) {}
    tag(const char * t) : element(t, element::TAG) {}
    tag(const std::string& t) : element(t, element::TAG) {}
    tag(const tag& rhs) : element(rhs) {}
    virtual ~tag() {}
    
    tag& operator=(const tag& rhs)
    {
    	tag tmp(rhs);
    	this->element::swap(tmp);
    	return *this;
    }
    
    virtual element * clone() const { return new tag(*this); }
    
    void clear() { this->element::clear(); }

    // Get/Set
   
    // accept vistor
    virtual int accept(rule_visitor& vis) 
    {
        return vis.visit(*this);
    }
       
private:
};

typedef bee::xsd::element_ptr<srgs::tag> tag_ptr;
typedef boost::shared_ptr<srgs::tag> shared_tag;

//
// token element
//
class token: public srgs::element
{
public:
    token() : element(element::TOKEN) {}
    token(const token& rhs) : element(rhs), lang_(rhs.lang_) {}
    virtual ~token() {}
    
    token& operator=(const token& rhs)
    {
    	token tmp(rhs);
    	this->element::swap(tmp);
    	std::swap(lang_, tmp.lang_);
    	return *this;
    }

    virtual element * clone() const
    {
    	return new token(*this);
    }

    void clear()
    {
    	this->element::clear();
        lang_ = "";
    }
    
    // Get/Set
    const xml_schema::lang& lang() const { return lang_; }
    void lang(const xml_schema::lang& lang) { lang_ = lang; }

    // accept vistor
    virtual int accept(rule_visitor& vis) 
    {
        return vis.visit(*this);
    }

    virtual void dump(std::ostream& out, int indent) const
    {
        std::string space(indent * 2, ' '); 
        out << space << "token: " << this->str() << "\n";
    }
    
private:
    xml_schema::lang lang_;
};

//
typedef bee::xsd::element_ptr<srgs::token> token_ptr;

//
// one-of element
//
typedef bee::xsd::element_ptr<srgs::item> item_ptr;
//
class one_of : public srgs::element
{
public:
    typedef std::vector<shared_ptr<srgs::item> > items;

    one_of() : element(element::ONE_OF) {}
    one_of(const one_of& rhs) : element(rhs), lang_(rhs.lang_),
                                items_(rhs.items_) {}
    virtual ~one_of() {}

    one_of& operator=(const one_of& rhs)
    {
    	one_of tmp(rhs);
    	this->element::swap(tmp);
    	std::swap(lang_, tmp.lang_);
    	std::swap(items_, tmp.items_);
    	return *this;
    }

    virtual srgs::element * clone() const
    {
    	return new one_of(*this);
    }

    void clear();

    // Get/Set
    items& item_table() { return items_; }
    const xml_schema::lang& lang() const { return lang_; }
    const srgs::item& at(size_t idx) const
    {
    	if (idx >= items_.size())
    	    throw std::invalid_argument("srgs::one_of::at: out of range error");
    	 return *(items_[idx].get());
    }
    void lang(const xml_schema::lang& lang) { lang_ = lang; }
    void add(const srgs::item_ptr& item);

    bool available() const
    {
        //* empty is checked in srgs-parser-templates.txx
        //return (items_.size() > 0);
        return true;
    }

    // accept visitor
    virtual int accept(rule_visitor& vis)
    {
        // TRACE_FUNCTION("srgs::one_of::visit");
        vis.visit(*this);
        return 0;
    }
    
    virtual void dump(std::ostream& out, int indent) const;
 
private:
  	xml_schema::lang  lang_;
  	items             items_;
};

//
typedef bee::xsd::element_ptr<srgs::one_of> one_of_ptr;

//
// rule expantion container
//
class rule_expantions
{
public:
    typedef std::vector<shared_elm> elements;

    rule_expantions() {}
    rule_expantions(const rule_expantions& rhs)
    {
        for (size_t i = 0; i < rhs.expantions_.size(); i++)
        {
            expantions_.push_back(rhs.expantions_[i]);
        }
    }
    ~rule_expantions() { this->clear(); }
    
    rule_expantions& operator=(const rule_expantions& rhs)
    {
        rule_expantions tmp(rhs);
        std::swap(expantions_, tmp.expantions_);
        return *this;
    }

    void swap(rule_expantions& rhs)
    {
        std::swap(expantions_, rhs.expantions_);
    }


    // Get/Set
    const srgs::element * expantion(unsigned int idx) const
    {
        if (idx >= expantions_.size())
            return NULL;
        return expantions_[idx].get();
    }
    void expantion(const shared_ptr<srgs::element>& elm)
    {
         expantions_.push_back(elm);
    }

    elements& expantions() { return expantions_; }
    size_t size() const    { return expantions_.size(); }

    // erase all rule expantion
    void clear()
    {
        for (size_t i = 0; i < expantions_.size(); i++)
        {
            expantions_[i].reset();
        }
        expantions_.clear();
    }
    
    // accept vistor
    void accept(rule_visitor& vis)
    {
        vis.visit(*this);
    }

    void dump(std::ostream& out, int indent) const
    {
        for (size_t i = 0; i < expantions_.size(); i++)
        {
            srgs::element * elm = expantions_[i].get();
            elm->dump(out, indent);
        }
    }
   
private:
    elements expantions_;
};


//
// item element
//
typedef bee::xsd::element_ptr<srgs::item> item_ptr;

class item : public srgs::element
{
public:

    item();
    item(const item& rhs);
    virtual ~item() { this->clear(); }

    item& operator=(const item& rhs)
    {
    	item tmp(rhs);
    	this->element::swap(tmp);
    	tag_ = tmp.tag_;
        std::swap(weight_, tmp.weight_);
        repeat_ = tmp.repeat_;
        std::swap(repeat_prob_, tmp.repeat_prob_);
        std::swap(lang_, tmp.lang_); 
        expantions_.swap(tmp.expantions_);
        terminate_ = tmp.terminate_;
        return *this;
    } 

    virtual srgs::element * clone() const
    {
    	return new item(*this); 
    }

    void clear()
    {
    	this->element::clear();
        tag_.reset();
        weight_ = "";
        repeat_.set("");
        repeat_prob_ = 0.0;
        lang_ = "";
        expantions_.clear();
        terminate_ = false;
    }

    // Get/Set
    srgs::rule_expantions& expantions() { return expantions_; }
    const srgs::element * expantion(unsigned int idx) const
    {
        return expantions_.expantion(idx);
    }

    const srgs::Repeat_prob_datatype& repeat_prob() const { return repeat_prob_; }
    const srgs::repeat& repeat() const { return repeat_; }
    const srgs::Weight_datatype& weight() const { return weight_; }   
    const xml_schema::lang& lang() const { return lang_; }
    const srgs::tag& tag() const   { return *tag_; }   
    const std::string& tag_string() const 
    {
        if (tag_.get() == NULL)
            return bee::empty_string;
        return tag_->str();
    }   
    void token(const srgs::token& token)
    {
        this->str(token.str());
        //shared_ptr<srgs::token> tmp(new srgs::token(token));
        //expantions_.expantion(tmp);
    }
    void ruleref(const srgs::ruleref_ptr& ruleref)
    {
        TRACE_FUNCTION("srgs::item::ruleref");
        std::string refname = ruleref->uri().fragment();
        if (refname == "NULL" || refname == "VOID" || refname == "GARBAGE")
        {
            TRACE_ERROR_FMT("special rule name (%1%)", refname);
        }
    	expantions_.expantion(ruleref.shared());
    }
    void item1(const srgs::item_ptr& item)
    {
        TRACE_FUNCTION("srgs::item::item1");
        if (! item->available())
        {
            TRACE_DEBUG("ignore item tag (probabbly, empty)");
            return ;
        }
    	expantions_.expantion(item.shared());
    	terminate_ = false;
    }
    void one_of(const srgs::one_of_ptr& one_of)
    {
        //TRACE_FUNCTION("srgs::item::one_of");
        expantions_.expantion(one_of.shared());
    }
    void tag(const srgs::tag_ptr& tag)
    {
        //TRACE_FUNCTION("srgs::item::tag");
    	tag_ = tag.shared();
        if (tag->str().size() > 0)
    	    terminate_ = true;
    }
    void repeat_prob(const srgs::Repeat_prob_datatype& repeat_prob)
    {
        repeat_prob_ = repeat_prob;
    }
    void repeat(const srgs::Repeat_datatype& repeat)
    {
        int result = repeat_.set(repeat);
        if (result < 0)
        {
            std::string msg = "srgs::item::repeat: illegal range(";
            msg += repeat;
            msg += ")";
            throw srgs::exception(msg);
        }
    }
    void weight(const srgs::Weight_datatype& weight) { weight_ = weight; }   
    void lang(const xml_schema::lang& lang) { lang_ = lang; }

    static item Null;
    bool isNull() const { return (this == &item::Null); }

    bool terminate() const { return terminate_; }
    void terminate(bool tof) { terminate_ = tof; }

    bool available() const
    {
        return (terminate_ || this->str().size() > 0 ||
                expantions_.size() > 0); 
    }


    // accept vistor
    virtual int accept(rule_visitor& vis) 
    {
        TRACE_FUNCTION("srgs::item::visit");
        vis.visit(*this);
   	    return 0;
    }

    virtual void dump(std::ostream& out, int indent) const
    {
    	std::string space(indent * 2, ' '); 
    	if (this->terminate())
    	{
            out << space << "item: "
                << this->str() << " (" << this->tag_string() << ") ";
            repeat_.dump(out);
            out << "\n";
    	}
    	else
    	{
    		out << space << "item: ";
            repeat_.dump(out);
            out << "\n";
            expantions_.dump(out, indent + 1);
    	}   	
    }

private:
    shared_tag tag_;
    srgs::Weight_datatype weight_;
    srgs::repeat repeat_;
    srgs::Repeat_prob_datatype repeat_prob_;
    xml_schema::lang lang_;
    srgs::rule_expantions expantions_;
    bool terminate_;
    
};

//
// Pronunciation Lexicon
//
class lexicon
{
public:
    lexicon() : uri_(""), type_("") {}
    lexicon(const std::string& uri, const std::string& type)
        : uri_(uri), type_(type) {}
  	lexicon(const lexicon& rhs) : uri_(rhs.uri_), type_(rhs.type_) {}
    ~lexicon() {}
    
    lexicon& operator=(const lexicon& rhs)
    {
    	lexicon tmp(rhs);
    	std::swap(uri_, tmp.uri_);
    	std::swap(type_, tmp.type_);
    	return *this;
    }

    void clear()
    {
    	uri_.name("");
    	type_ = "";
    }
    
    // Get/Set
    const std::string& uri() const { return uri_.name(); }
    const srgs::Type_datatype& type() const { return type_; }
    void uri(const std::string& uri) { uri_.name(uri); }
    void type(const srgs::Type_datatype& type) { type_ = type; }

private:
  	xml_schema::uri     uri_;
  	srgs::Type_datatype type_;
};

//
// meta element
//
class meta
{
public:
    meta() : name_(""), content_(""), http_equiv_("") {}
  	meta(const meta& rhs) : name_(rhs.name_), content_(rhs.content_),
  	                        http_equiv_(rhs.http_equiv_) {}
    ~meta() {}
    
    meta& operator=(const meta& rhs)
    {
    	srgs::meta tmp(rhs);
    	name_ = tmp.name_;
    	content_ = tmp.content_;
    	http_equiv_ = tmp.http_equiv_;
    	return *this;
    }

    void clear()
    {
    	name_ = "";
    	content_ = "";
    	http_equiv_ = "";
    }

    // Get/Set
    const std::string& name() const { return name_; }
    const std::string& content() const { return content_; }
    const std::string& http_equiv() const { return http_equiv_; }
    void name(const std::string& name) { name_ = name; }
    void content(const std::string& content) { content_ = content; }
    void http_equiv(const std::string& http_equiv) { http_equiv_ = http_equiv; }

private:
  	std::string name_;
  	std::string content_;
  	std::string http_equiv_;
};

}  // namespace

#include "srgs/rule.hxx"
#include "srgs/grammar.hxx"

namespace srgs
{

//
// after grammar.hxx
//
template<typename VISITOR>
void grammar_visitor<VISITOR>::visit(srgs::grammar * gram)
{
    //TRACE_FUNCTION("srgs::grammar_visitor::visit");
    gram->accept(*this);                      
}

}  // *namespace* - srgs

#endif // SRGS_SRGS_HXX
