// $Id$
//
// Copyright (C) 2006 BEE Co.,Ltd.
//
// Author : Fukasawa Mitsuo
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
//

#ifndef SRGS_PARSER_HXX
#define SRGS_PARSER_HXX

#include "srgs/srgs.hxx"
#include "srgs/srgs-parser-templates.hxx"
#include "srgs/xml_schema-parser.hxx"

extern bool is_allspace(const std::string& text);  // see xml_schema.cxx
extern std::string trim(const std::string& text);  // see xml_schema.cxx


namespace srgs { namespace parser
{

//
// parser of base type
typedef xml_schema::parser::string Id_datatype;
typedef xml_schema::parser::string Scope_datatype;
typedef xml_schema::parser::string Special_datatype;
typedef xml_schema::parser::decimal Repeat_prob_datatype;
typedef xml_schema::parser::string Repeat_datatype;
typedef xml_schema::parser::string Weight_datatype;
typedef xml_schema::parser::string Tag_format_datatype;
typedef xml_schema::parser::string Version_datatype;
typedef xml_schema::parser::string Root_datatype;
typedef xml_schema::parser::string Mode_datatype;
typedef xml_schema::parser::string Type_datatype;

//-----------------------------------------------------------------------------
//
struct tag: templates::tag<srgs::tag_ptr>
{
	typedef templates::tag<srgs::tag> base_type;
	
    virtual void pre()
    {
    	//TRACE_FUNCTION("tag::pre");
        tag_.pre();
    }
    virtual void _any_characters(const ::xsd::cxx::parser::ro_string< char >& v)
    {
        //TRACE_FUNCTION("tag::_any_characters");
        //TRACE_DEBUG_FMT("%1%\n", v);
        tag_->str(v);
    }
    virtual void _characters(const ::xsd::cxx::parser::ro_string< char >& v)
    {
        //TRACE_FUNCTION("tag::_characters");
        //TRACE_DEBUG_FMT("%1%\n", v);
        tag_->str(v);
    }
    virtual srgs::tag_ptr post()
    {
    	return tag_.post();
    }

private:
    srgs::tag_ptr tag_;
};

//-----------------------------------------------------------------------------
//
struct ruleref: templates::ruleref<srgs::ruleref_ptr,
                                   srgs::Type_datatype,     // type
                                   std::string,             // uri
                                   srgs::Special_datatype>  // special
{
	typedef templates::ruleref<srgs::ruleref_ptr, srgs::Type_datatype,
                               std::string, srgs::Special_datatype> base_type;
	
    virtual void pre()
    {
    	//TRACE_FUNCTION("srgs::parser::ruleref::pre");
    	ruleref_.pre();
    }

    virtual void type(const std::string& type)
    {
    	ruleref_->media_type(type);
    }
    
    virtual void uri(const std::string& uri)
    {
    	ruleref_->uri(uri);
    }
    
    virtual void special(const srgs::Special_datatype& special)
    {
    	ruleref_->special(special);
    }
    
    virtual srgs::ruleref_ptr post() { return ruleref_.post(); }

private:
    srgs::ruleref_ptr ruleref_;
};


//-----------------------------------------------------------------------------
// Meta Data (ANY)
//
struct metadata: templates::metadata<srgs::metadata>
{
	typedef templates::metadata<srgs::metadata> base_type;
	
    virtual void pre() { metadata_.clear(); }
    virtual srgs::metadata post() { return metadata_; }

private:
    srgs::metadata metadata_;
};


struct token: templates::token<srgs::token,
                               xml_schema::lang>
{
    typedef templates::token<srgs::token, xml_schema::lang> base_type;

    virtual void pre() {}

    virtual void _any_characters(const ::xsd::cxx::parser::ro_string< char >& v)
    {
        //TRACE_FUNCTION("token::_any_characters");
        //TRACE_DEBUG_FMT("%1%\n", v);
        std::string tmp("\"");
        tmp += v;
        tmp += "\"";
        token_.str(tmp);
    }
    virtual void _characters(const ::xsd::cxx::parser::ro_string< char >& v)
    {
        //TRACE_FUNCTION("token::_characters");
        //TRACE_DEBUG_FMT("%1%\n", v);
        std::string tmp("\"");
        tmp += v;
        tmp += "\"";
        token_.str(tmp);
    }
    
    virtual void lang(const xml_schema::lang& lang)
    {
    	token_.lang(lang);
    }

    virtual srgs::token post() { return token_; }
    
private:
    srgs::token token_;
};

//-----------------------------------------------------------------------------
//
//
struct one_of: templates::one_of<srgs::one_of_ptr,
                                 srgs::item_ptr,
                                 xml_schema::lang>
{
    typedef templates::one_of<srgs::one_of_ptr, srgs::item_ptr, xml_schema::lang>
            base_type;

    virtual void pre()
    {
    	//TRACE_FUNCTION("srgs::parser::one-of::pre");
    	one_of_.pre();
    }

    virtual void item(const srgs::item_ptr& item)
    {
    	//TRACE_FUNCTION("srgs::parser::one-of::item");
    	one_of_->add(item);
    }
    
    virtual void lang(const xml_schema::lang& lang)
    {
    	one_of_->lang(lang);
    }

    virtual srgs::one_of_ptr post() { return one_of_.post(); }

private:
    srgs::one_of_ptr one_of_;
};

//-----------------------------------------------------------------------------
//
//
struct item: templates::item<srgs::item_ptr,
                             srgs::token,
                             srgs::ruleref_ptr,
                             srgs::item_ptr,
                             srgs::one_of_ptr,
                             srgs::tag_ptr,
                             srgs::Repeat_prob_datatype,
                             srgs::Repeat_datatype,
                             srgs::Weight_datatype,
                             xml_schema::lang>
{
    typedef templates::item<srgs::item_ptr, srgs::token, srgs::ruleref_ptr,
                            srgs::item_ptr, srgs::one_of_ptr, srgs::tag_ptr,
                            srgs::Repeat_prob_datatype,
                            srgs::Repeat_datatype, srgs::Weight_datatype,
                            xml_schema::lang> base_type;

    virtual void pre()
    {
    	//TRACE_FUNCTION("srgs::parser::item::pre");
  	    item_.pre();
    }

    virtual void _any_characters(const ::xsd::cxx::parser::ro_string< char >& v)
    {
    	//TRACE_FUNCTION("srgs::parser::item::_any_characters");
    	//TRACE_DEBUG("[%s]\n", v.c_str());
        if (is_allspace(v))
        {
            //TRACE_DEBUG("data are all white space.\n");
            //item_->str("");
        }
        else
        {
            item_->str(trim(v));
            if (item_->str().size() > 0)
                item_->terminate(true);
        }
    }

    virtual void token(const srgs::token& token)
    {
    	item_->token(token);
    }
    
    virtual void ruleref(const srgs::ruleref_ptr& ruleref)
    {
    	item_->ruleref(ruleref);
    }
    
    virtual void item1(const srgs::item_ptr& item1)
    {
        item_->item1(item1);
    }
    
    virtual void one_of(const srgs::one_of_ptr& one_of)
    {
    	item_->one_of(one_of);
    }
    
    virtual void tag(const srgs::tag_ptr& tag)
    {
    	item_->tag(tag);
    }
    
    virtual void repeat_prob(const srgs::Repeat_prob_datatype& repeat_prob)
    {
    	item_->repeat_prob(repeat_prob);
    }
    
    virtual void repeat(const srgs::Repeat_datatype& repeat)
    {
    	item_->repeat(repeat);
    }
    
    virtual void weight(const srgs::Weight_datatype& weight)
    {
    	item_->weight(weight);
    }
    
    virtual void lang(const xml_schema::lang& lang)
    {
    	item_->lang(lang);
    }

    virtual srgs::item_ptr post()
    {
        //TRACE_FUNCTION("srgs::parser::item::post");
    	return item_.post();
    }

private:
    srgs::item_ptr item_;
};

//-----------------------------------------------------------------------------
//
struct rule: templates::rule<srgs::rule_ptr,
                             srgs::token,
                             srgs::ruleref_ptr,
                             srgs::item_ptr,
                             srgs::one_of_ptr,
                             srgs::tag_ptr,
                             std::string,         // example
                             srgs::Id_datatype,
                             srgs::Scope_datatype>
{
    typedef templates::rule<srgs::rule_ptr, srgs::token, srgs::ruleref_ptr,
                            srgs::item_ptr, srgs::one_of_ptr, srgs::tag_ptr,
                            std::string, srgs::Id_datatype,
                            srgs::Scope_datatype> base_type;
    virtual void pre()
    {
        //TRACE_FUNCTION("srgs::parser::rule::pre");
        rule_.pre();
    }

    virtual void _any_characters(const ::xsd::cxx::parser::ro_string< char >& v)
    {
        //TRACE_FUNCTION("srgs::parser::rule::_any_characters");
        if (is_allspace(v))
        {
            //TRACE_DEBUG("data are all white space.\n");
            rule_->str("");
        }
        else
        {
            //TRACE_DEBUG_FMT("data = %1%.\n", v);
            rule_->str(trim(v));
        }
    }

    virtual void token(const srgs::token& token)
    {
        //TRACE_FUNCTION("srgs::parser::rule::token");
        rule_->token(token);
    }
    
    virtual void ruleref(const srgs::ruleref_ptr& ruleref)
    {
        //TRACE_FUNCTION("srgs::parser::rule::ruleref");
        rule_->ruleref(ruleref);
    }
    virtual void item(const srgs::item_ptr& item)
    {
        //TRACE_FUNCTION("srgs::parser::rule::item");
        rule_->item(item);
    }
    virtual void one_of(const srgs::one_of_ptr& one_of)
    {
        //TRACE_FUNCTION("srgs::parser::rule::one_of");
        rule_->one_of(one_of);
    }
    virtual void tag(const srgs::tag_ptr& tag)
    {
        rule_->tag(tag);
    }
    
    virtual void example(const std::string& example)
    {
        rule_->example(example);
    }
    
    virtual void id(const srgs::Id_datatype& id)
    {
        rule_->id(id);
    }
    
    virtual void scope(const srgs::Scope_datatype& scope)
    {
        rule_->scope(scope);
    }   

    virtual srgs::rule_ptr post()
    {
        return rule_.post(); 
    }

private:
    srgs::rule_ptr rule_;
};


//-----------------------------------------------------------------------------
//
struct lexicon: templates::lexicon<srgs::lexicon,
                                   std::string,          // uri
                                   srgs::Type_datatype>  // type
{
    typedef templates::lexicon<srgs::lexicon, std::string, srgs::Type_datatype>
            base_type;
            
    void pre() { lexicon_.clear(); }

    virtual void uri(const std::string& uri)
    {
    	lexicon_.uri(uri);
    }
    
    virtual void type(const srgs::Type_datatype& type)
    {
    	lexicon_.type(type);
    }

    virtual srgs::lexicon post() { return lexicon_; }
    
private:
    srgs::lexicon lexicon_;
};

//-----------------------------------------------------------------------------
//
struct meta: templates::meta<srgs::meta,
                             std::string,    // name
                             std::string,    // content
                             std::string>    // http_equiv
{
    typedef templates::meta<srgs::meta, std::string, std::string, std::string>
            base_type;

    void pre() { meta_.clear(); }

    virtual void name(const std::string& name)
    {
    	meta_.name(name);
    }
    
    virtual void content(const std::string& content)
    {
    	meta_.content(content);
    }

    virtual void http_equiv(const std::string& http_equiv)
    {
    	meta_.http_equiv(http_equiv);
    }

    virtual srgs::meta post() { return meta_; }

private:
    srgs::meta meta_;
};


//-----------------------------------------------------------------------------
// parse grammar element
//
struct grammar: templates::grammar<srgs::grammar *,
                                   srgs::rule_ptr,
                                   srgs::lexicon,
                                   srgs::meta,
                                   std::string,               // metadata
                                   srgs::tag_ptr,
                                   srgs::Tag_format_datatype,
                                   srgs::Version_datatype,
                                   xml_schema::lang,
                                   std::string,               // base,
                                   srgs::Root_datatype,
                                   srgs::Mode_datatype>
{
    typedef templates::grammar<srgs::grammar *,
                               srgs::rule_ptr,
                               srgs::lexicon, srgs::meta, srgs::metadata,
                               srgs::tag_ptr, srgs::Tag_format_datatype,
                               srgs::Version_datatype, xml_schema::lang,
                               std::string, srgs::Root_datatype,
                               srgs::Mode_datatype> base_type;
    virtual void pre()
    {
    	//TRACE_FUNCTION("srgs::parser::grammar::pre");
    	grammar_ = new srgs::grammar(); 
    }

    virtual void _any_characters(const ::xsd::cxx::parser::ro_string< char >& v)
    {
    	//TRACE_FUNCTION("srgs::parser::grammar::_any_characters");
    }

    virtual void rule(const srgs::rule_ptr& rule)
    {
        //TRACE_FUNCTION("grammar::rule");
        grammar_->add(rule.shared());
        //const_cast<rule_wrap&>(rule).set(NULL);
    }

    virtual void lexicon(const srgs::lexicon& lexicon)
    {
    	grammar_->lexicon(lexicon);
    }
    
    virtual void meta(const srgs::meta& meta)
    {
    	grammar_->meta(meta);
    }
    
    virtual void metadata(const std::string& metadata)
    {
    	grammar_->metadata(metadata);
    }

    virtual void tag(const srgs::tag_ptr& tag)
    {
    	grammar_->tag(tag);
    }
    
    virtual void tag_format(const srgs::Tag_format_datatype& tag_format)
    {
    	grammar_->tag_format(tag_format);
    }
    
    virtual void version(const srgs::Version_datatype& version)
    {
    	grammar_->version(version);
    }
    
    virtual void lang(const xml_schema::lang& lang)
    {
    	grammar_->lang(lang);
    }
    
    virtual void base(const std::string& base)
    {
    	grammar_->base(base);
    }
    
    virtual void root(const srgs::Root_datatype& root)
    {
    	grammar_->root(root);
    }
    
    virtual void mode(const srgs::Mode_datatype& mode)
    {
    	grammar_->mode(mode);
    }
    
    virtual srgs::grammar * post() { return grammar_; }

private:
    srgs::grammar * grammar_;
};


} }  // namespace

#endif // SRGS_PARSER_HXX
