/* $Id$ */

//=============================================================================
/**
 *  @file    rule_visitor.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef SRGS_RULE_VISITOR_HPP
#define SRGS_RULE_VISITOR_HPP

#include <list>
#include <deque>
#include "bee/beeconf.hpp"
#include "srgs/repeat.hpp"
#include <boost/tokenizer.hpp>

	
#define NONE_ONE_OF
//#define FRONT_ONE_OF

namespace srgs
{

//
// prototype
//
class grammar;
class element;
class root_element;
class item;
class rule;
class ruleref;
class one_of;
class token;
class tag;
class rule_expantions;
class rule_visitor;
struct graph_base;
struct graph_data;

//
struct repeater : public repeat
{
    repeater() : repeat(), repeat_cur_(0), repeat_do_(false) {}
    repeater(const repeater& rhs) :
        repeat(rhs), repeat_cur_(rhs.repeat_cur_), repeat_do_(rhs.repeat_do_) {}
    ~repeater() {}

    repeater& operator=(const repeater& rhs)
    {
        repeater tmp(rhs);
        this->repeat::operator=(tmp);
        std::swap(repeat_cur_, tmp.repeat_cur_);
        std::swap(repeat_do_, tmp.repeat_do_);
        return *this;
    }

    repeater& operator=(const repeat& rhs)
    {
        this->repeat::operator=(rhs);
        repeat_cur_ = 0;
        repeat_do_ = false;
        return *this;
    }

    void dump(std::ostream& out) const
    {
        this->repeat::dump(out);
        //out << repeat_cur_ << (repeat_do_ ? "T" : "F");
    }
//
    int  repeat_cur_;
    bool repeat_do_;
};

//
//
typedef boost::shared_ptr<element> shared_elm;
typedef boost::shared_ptr<graph_base> shared_gb;
typedef boost::shared_ptr<graph_data> shared_gd;
//
//
#ifdef NONE_ONE_OF
struct node_base
{
    enum { UNKNOWN, LEXIS, REPEAT, ONE_OF,
           RULEREF, _NULL, _VOID, GARBAGE, ROOT };

    node_base() : elm_(NULL), name_(""), special_(UNKNOWN) {}
    node_base(element * elm, int spc = UNKNOWN)
        : elm_(elm), name_(""), special_(spc) {}
    node_base(element * elm, const std::string& n, int spc = UNKNOWN)
        : elm_(elm), name_(n), special_(spc) {}
    node_base(const node_base& rhs)
        : elm_(rhs.elm_), name_(rhs.name_), special_(rhs.special_),
          rule_stack_(rhs.rule_stack_), froms_of_oneof_(rhs.froms_of_oneof_),
          repeater_(rhs.repeater_) {}
    ~node_base() {}

    node_base& operator=(const node_base& rhs)
    {
        if (this != &rhs)
        {
            elm_ = rhs.elm_;
            name_ = rhs.name_;
            special_ = rhs.special_;
            rule_stack_ = rhs.rule_stack_;
            froms_of_oneof_ = rhs.froms_of_oneof_;
            repeater_ = rhs.repeater_;
        }
        return *this;
    }
    element * get() { return elm_; }
    const element * get() const  { return elm_; }

    void rule_name(const std::string& nm) { elm_->rule(nm); }
    void str(const std::string& data) { elm_->str(data); }
    void name(const std::string& name) { name_ = name; }
    void type(int n) { special_ = n; }
    void from_clear() { froms_of_oneof_.clear(); }
    void from_oneof(int oof) { froms_of_oneof_.push_back(oof); }
    void from_ids(const std::vector<int>& ids) { froms_of_oneof_ = ids; }
    void repeater(const srgs::repeat& rep) { repeater_ = rep; }
    void set(const std::deque<std::string>& rule_stack)
    {
         rule_stack_ = rule_stack; 
    }   
    const std::string& name() const { return name_; }
    const std::string& rule_name() const { return elm_->rule(); }
    const std::string& str() const { return elm_->str(); }
    int elm_type() const { return elm_->type(); }
    int type() const { return special_; }
    int from_oneof(int idx = 0) const
    {
        assert(idx < froms_of_oneof_.size());
        return froms_of_oneof_[idx]; 
    }
    size_t from_size() const { return froms_of_oneof_.size(); }
    std::vector<int>& from_ids() { return froms_of_oneof_; }
    const std::deque<std::string>& rule_stack() const { return rule_stack_; }   
    srgs::repeater& repeater() { return repeater_; }
//
private:
    element * elm_;
    std::string name_;
    int special_;
    std::deque<std::string> rule_stack_;
    std::vector<int> froms_of_oneof_;
    srgs::repeater repeater_;
};
#else
struct node_base
{
    enum { UNKNOWN, LEXIS, REPEAT, ONE_OF,
           RULEREF, _NULL, _VOID, GARBAGE, ROOT };

    node_base() : elm_(NULL), name_(""), special_(UNKNOWN),
                  top_of_oneof_(-1) {}
    node_base(element * elm, int spc = UNKNOWN)
        : elm_(elm), name_(""), special_(spc), top_of_oneof_(-1) {}
    node_base(element * elm, const std::string& n, int spc = UNKNOWN)
        : elm_(elm), name_(n), special_(spc), top_of_oneof_(-1) {}
    node_base(const node_base& rhs)
        : elm_(rhs.elm_), name_(rhs.name_), special_(rhs.special_),
          rule_stack_(rhs.rule_stack_), top_of_oneof_(rhs.top_of_oneof_),
          repeater_(rhs.repeater_) {}
    ~node_base() {}

    node_base& operator=(const node_base& rhs)
    {
        if (this != &rhs)
        {
            elm_ = rhs.elm_;
            name_ = rhs.name_;
            special_ = rhs.special_;
            rule_stack_ = rhs.rule_stack_;
            top_of_oneof_ = rhs.top_of_oneof_;
            repeater_ = rhs.repeater_;
        }
        return *this;
    }
    element * get() { return elm_; }
    const element * get() const  { return elm_; }

    void rule_name(const std::string& nm) { elm_->rule(nm); }
    void str(const std::string& data) { elm_->str(data); }
    void name(const std::string& name) { name_ = name; }
    void type(int n) { special_ = n; }
    void top_of_oneof(int oof) { top_of_oneof_ = oof; }
    void repeater(const srgs::repeat& rep) { repeater_ = rep; }
    void set(const std::deque<std::string>& rule_stack)
    {
         rule_stack_ = rule_stack; 
    }   
    const std::string& name() const { return name_; }
    const std::string& rule_name() const { return elm_->rule(); }
    const std::string& str() const { return elm_->str(); }
    int elm_type() const { return elm_->type(); }
    int type() const { return special_; }
    int top_of_oneof() const { return top_of_oneof_; }
    const std::deque<std::string>& rule_stack() const { return rule_stack_; }   
    srgs::repeater& repeater() { return repeater_; }
//
private:
    element * elm_;
    std::string name_;
    int special_;
    std::deque<std::string> rule_stack_;
    int top_of_oneof_;
    srgs::repeater repeater_;
};
#endif

struct node : public node_base
{
    node() : node_base(), num_(-1), pass_count_(0) {}
    node(element * elm, int spc = UNKNOWN) : node_base(elm, spc), num_(-1),
        pass_count_(0) {}
    node(element * elm, const std::string& n, int spc = UNKNOWN) 
        : node_base(elm, n, spc), num_(-1), pass_count_(0) {}
    node(const node& rhs)
        : node_base(rhs), num_(rhs.num_), graf_(rhs.graf_),
          pass_count_(rhs.pass_count_) {}
    ~node() {}

    node& operator=(const node& rhs)
    {
        node tmp(rhs);
        this->node_base::operator=(tmp);
        std::swap(num_, tmp.num_);
        graf_.swap(tmp.graf_);
        std::swap(pass_count_, tmp.pass_count_);
        return *this;
    }

    // Get/Set
    int number() const  { return num_; }
    int pass_count() const { return pass_count_; }
    graph_base * graphbase()   { return graf_.get(); }
    const graph_base * graphbase() const { return graf_.get(); }
    void number(int n)
    {
        num_ = n;
        if (this->name() == "NS" || this->name() == "NE" ||
            this->name() == "NH" || this->name() == "NT")
            return ;
        char buf[64];
        sprintf(buf, "%d", num_);
        this->name(buf);
    }
    void graphbase(const shared_gb& sgb)   { graf_ = sgb; }
    void inc_pass_count() { pass_count_++; }
    void dec_pass_count() { pass_count_--; }
    bool isNull() const { return (this->get() == NULL); }

    void dump(std::ostream& out);

//
private:
    int num_;
    shared_gb graf_;
    int pass_count_;
};

struct arc_data : public node_base
{
    arc_data(element * elm, int spc = UNKNOWN) : node_base(elm, spc) {}
    arc_data(element * elm, const std::string& n, int spc = UNKNOWN)
        : node_base(elm, n, spc) {}
    arc_data(const arc_data& rhs) : node_base(rhs) {}
    arc_data(const node& rhs) : node_base(rhs) {}
    ~arc_data() {}

    arc_data& operator=(const arc_data& rhs)
    {
        arc_data tmp(rhs);
        this->node_base::operator=(tmp);
        return *this;
    }
    
    arc_data& operator=(const node& rhs)
    {
        arc_data tmp(rhs);
        this->node_base::operator=(tmp);
        return *this;
    }

    const char * value() const;
    void dump(std::ostream& out);
};

//
// edges of words
//
struct edge_info : public std::pair<int, int>
{
    typedef std::pair<int, int> base_type;

    edge_info(int from, int to, element * elm)
        : base_type(from, to), data_(elm) {}
    edge_info(int from, int to, const arc_data& a)
        : base_type(from, to), data_(a) {}
    edge_info(const edge_info& rhs)
        : base_type(rhs), data_(rhs.data_) {}
    ~edge_info() {}

    edge_info& operator=(const edge_info& rhs)
    {
        edge_info tmp(rhs);
        this->first = tmp.first;
        this->second = tmp.second;
        data_ = tmp.data_;
        return *this;
    }

    void dump(std::ostream& out)
    {
        out << this->first << " -[";
	    data_.dump(std::cout);
	    out << "]-> " << this->second << "\n";
    }
    arc_data data_;
};

struct vedges : std::vector<edge_info>
{
    typedef edge_info value_type;
    typedef std::vector<value_type> base_type;
    typedef std::list<value_type>   reservations;

    int add(int f, int t, const arc_data& a)
    {
        //TRACE_FUNCTION("srgs::vedges::add");
        //TRACE_DEBUG_FMT("from=%1% to=%2%\n", f % t);
        base_type::iterator iter;        
        for (iter = this->begin(); iter != this->end(); ++iter)
        {
            if (iter->first == f && iter->second == t)
            {
                std::stringstream err;
                err << "srgs::vedges::add: duplicate node pair(" << f
                    << "," << t << ")" << std::endl;
                throw std::invalid_argument(err.str());
            }
        }
        push_back(value_type(f, t, a));
        return 0;
    }
     
    //
    // edge's reservation of queue
    //
    void make_reserve(int from, int to, const arc_data& a)
    {
        //TRACE_FUNCTION("srgs::vedges::make_reserve");
        //TRACE_DEBUG_FMT("%1% -> %2%", from % to);
        reservations_.push_back(edge_info(from, to, a));
    }
    void make_reserve(const reservations::value_type& q)
    {
        reservations_.push_back(q);
    }
    bool exist_idx(int idx) const
    {
        reservations::const_iterator iter = reservations_.begin();
        for ( ; iter != reservations_.end(); ++iter)
        {
            if (iter->second == idx)
                return true;
        }
        return false;
    }
    int erase_reserve(int idx)
    {
        reservations::iterator iter = reservations_.begin();
        for ( ; iter != reservations_.end(); ++iter)
        {
            if (iter->second == idx)
            {
                int from = iter->first;
                reservations_.erase(iter);
                return from;
            }
        }
        return -1;
    }

    int put_on(int idx, const node& node)
    {
        //TRACE_FUNCTION("srgs::vedges::put_on");
        int result = 0;
        arc_data a(node);
        reservations::iterator iter = reservations_.begin();
        while (iter != reservations_.end())
        {
            reservations::iterator next = iter;
            ++next;
            if (iter->second == idx)
            {
                this->add(iter->first, iter->second, a /*iter->data_*/);
                reservations_.erase(iter);
                result++;
                //return result;
            }
            iter = next;
        }
        return result;
    }

    int put_reverse(int idx, int to, const element * elm)
    {
        //TRACE_FUNCTION("srgs::vedges::put_reverse");
        int result = 0;
        arc_data a(const_cast<element *>(elm));
        reservations::iterator iter = reservations_.begin();
        while (iter != reservations_.end())
        {
            reservations::iterator next = iter;
            ++next;
            if (iter->second == idx)
            {
                this->add(iter->first, to, a /*iter->data_*/);
                result++;
                //return result;
            }
            iter = next;
        }
        return result;
    }

    void dump(std::ostream& out)
    {
        for (size_t i = 0; i < this->size(); i++)
        {
            out << "  (" << (*this)[i].first << " -> " 
                << (*this)[i].second << ")\n";
        }    	
    }

    reservations reservations_;
};


//
//
struct graph_base
{
    typedef std::vector<node> nodes;

    int node_size() const { return nodes_.size(); }
    node& node_at(int pos) { assert(pos < nodes_.size()); return nodes_[pos]; }
    node& tail_node() { return nodes_.back(); }
    const node& tail_node() const
    {
    	if (nodes_.empty())
    	    return *nodes_.end();
    	return nodes_[nodes_.size() - 1];
    }
    void add_node(const node& node)
    {
        nodes_.push_back(node);
    }
    void add_edge(int f, int t, const arc_data& a)
    {
        edges_.add(f, t, a);
    }

    void dump(std::ostream& out)
    {
        for (size_t i = 0; i < nodes_.size(); i++)
        {
            nodes_[i].dump(out);
        }
        for (size_t i = 0; i < edges_.size(); i++)
        {
            out << "  (" << edges_[i].first << " -[";
            edges_[i].data_.dump(out);
            out << "]-> " << edges_[i].second << ")\n";
        }    	
    }
    
    nodes   nodes_;
    vedges  edges_;
    //graph_t graph_;
    std::deque<std::string> called_;   // called rules for recognize
};

//
struct graph_data : graph_base
{
    graph_data() : graph_base(), head_(element::HEAD), tail_(element::TAIL) {}

    // before element
	void set(const node& p, const node& b) { parent_ = p; brother_ = b; }
	node& parent() { return parent_; }
	node& brother() { return brother_; }
	node parent(const node& p)
    {
        node old = parent_;
        parent_ = p;
        return old;
    }
	node brother(const node& b)
    {
        node old = brother_;
        brother_ = b;
        return old;
    }

    //
    void reserve_edge(int f, int t, const arc_data& a)
    {
        edges_.make_reserve(f, t, a);
    }

    bool empty() const;
    void link_sequential(const node& node);
    void link_branch(const node& node);
    void link_option(const rule_visitor * visitor, int from, const node& node);
    void link_unlimited(const rule_visitor * visitor, int top, const node& node);

//
    root_element head_;
    root_element tail_;

    node brother_;
    node parent_;
};

//
// visitor of rule
//
class rule_visitor
{
public:
    // Constructor/Destructor
	rule_visitor() : rule_(NULL), error_(false), refq_(0) {}
	virtual ~rule_visitor() {}

public:
    bool error() const { return error_; }
    int  current_num() const { return gd_->nodes_.size(); }
    int  before_num() const { return gd_->nodes_.size() - 1; }
    graph_data * graphdata() { return main_gd_.get(); }
    const graph_data * main_graphdata() const { return main_gd_.get(); }
    const std::string& name() const { return name_; }
    const rule * get_rule() const { return rule_; }

    int  refcount() const    { return refq_; }
    void set_refcount(int q) { refq_ = q; }
    int  inc_refcount()      { refq_++; return refq_; }

    virtual int visit(srgs::rule& rule);
    virtual int visit(srgs::ruleref& ruleref);
    virtual int visit(srgs::rule_expantions& rulexp);
    virtual int visit(srgs::item& item);
    virtual int visit(srgs::one_of& one_of);
    virtual int visit(srgs::token& token);
    virtual int visit(srgs::tag& tag);
   
    void dump(std::ostream& out) const;

protected:
    void init(node& rule, node& snode);
    void fini(node& enode);

    void append(node& n);
    void append_child(node& n);
    void append_end(node& n);

    void nest_in(const node& parent, const node& brother)
    {
        //TRACE_FUNCTION("srgs::rule_visitor::nest_in");
        if ((gd_ != NULL) && (! gd_->parent().isNull()))
        {
            shared_gd pgd_ = gd_;
            branches_.push(gd_);
            gd_ = shared_gd(new srgs::graph_data);
            node head_node(&gd_->head_, node_base::ROOT);
            this->append(head_node);
            node& p = pgd_->tail_node();
            p.graphbase(gd_);   // to save the graph data of child

            //gd_->set(parent, brother);
        }
        else
        {
            gd_->set(parent, brother);
        }
    }

    void nest_out()
    {
        //TRACE_FUNCTION("srgs::rule_visitor::nest_out");
        if (branches_.size())
        {
            node tail_node(&gd_->tail_, "NT", node_base::ROOT);
            this->append_end(tail_node);
            gd_ = branches_.top();
            branches_.pop();
        }
    }

    void link_node(int from, int to, const arc_data& arc)
    {
        gd_->add_edge(from, to, arc); 
    }
          

//
protected:
    rule * rule_;	
    std::string name_;
    bool   error_;
    shared_gd gd_;
    shared_gd main_gd_;
    std::stack<shared_gd> branches_;
    int refq_;   // reference count

};

}

#endif /* SRGS_RULE_VISITOR_HPP */
