/* $Id$ */

//=============================================================================
/**
 *  @file    rule_visitor.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#include "srgs.hxx"
#include "rule_visitor.hpp"
#include <boost/graph/breadth_first_search.hpp>

namespace srgs
{

void node::dump(std::ostream& out)
{
#ifdef NONE_ONE_OF
    out << "  " << num_ << "(";
    for (size_t i = 0; i < this->from_ids().size(); i++)
    {
        if (i > 0)
            out << ',';
        out << this->from_ids()[i];
    }
    out << ")";
#else
    out << "  " << num_ << "(" << top_of_oneof() << ")";
#endif
    //repeater_.dump(out);
    out << ": ";
    if (this->type() == node_base::LEXIS ||
        this->type() == node_base::RULEREF)
    {
        get()->dump(out);
    }
    else
    {
        out << get()->type_name();
        out << "\n";
    }
    if (graf_.get() != NULL)
        graf_->dump(out);
}

const char * arc_data::value() const
{
    const char * result;
    if (this->type() == node_base::LEXIS)
    {
        result = get()->str().c_str();
    }
    else
    {
        result = get()->type_name();
    }
    return result;
}

void arc_data::dump(std::ostream& out)
{
    if (this->type() == node_base::LEXIS)
    {
        get()->element::dump(out);
    }
    else if (this->type() == node_base::RULEREF)
    {
        get()->element::dump(out);
    }
    else if (this->type() == node_base::_NULL ||
             this->type() == node_base::GARBAGE ||
             this->type() == node_base::_VOID)
    {
        srgs::ruleref * ruleref_elm = dynamic_cast<srgs::ruleref *>(get());
        out << ruleref_elm->special();
    }
    else
    {
        out << get()->type_name();
    }
    repeater().dump(out);
}

//
// graph_data class
//
bool graph_data::empty() const
{
    if (nodes_.size() <= 1)
        return true;
    else if (nodes_.size() == 2 && nodes_[1].elm_type() == srgs::element::END)
        return true;
    return false;
}

void graph_data::link_sequential(const node& node)
{
    if (! brother_.isNull())  // As brother is NULL, element is first.
    {
        add_edge(brother_.number(), node.number(), arc_data(node));
    }
    else if (! parent_.isNull() &&
             parent_.elm_type() != element::RULE)
    {   // top node in nest of one-of
        add_edge(parent_.number(), node.number(), arc_data(node));
    }
}

void graph_data::link_branch(const node& node)
{
    //TRACE_FUNCTION("srgs::graph_data::link_branch");
    if (! parent_.isNull())
    {
        add_edge(parent_.number(), node.number(), arc_data(node));
    }
}

void graph_data::link_option(const rule_visitor * visitor, int from,
                             const node& node)
{
    //TRACE_FUNCTION("srgs::graph_data::link_option");
    int to = visitor->current_num();
#ifdef NONE_ONE_OF
    edges_.make_reserve(from, to, arc_data(node));
#else
    add_edge(from, to, arc_data(node));
#endif
}

void graph_data::link_unlimited(const rule_visitor * visitor, int top,
                                const node& node)
{
    //TRACE_FUNCTION("srgs::graph_data::link_unlimited");
    int from = visitor->current_num() - 1;
    edges_.add(from, top, arc_data(node));
}

//
//
//
void rule_visitor::init(node& rule, node& snode)
{
    gd_ = shared_gd(new srgs::graph_data); 
    gd_->parent(rule);
    this->append(snode);

    main_gd_ = gd_;        // to save graph of rule main
}

void rule_visitor::fini(node& enode)
{
    this->append_end(enode);
}

//
//
//
int rule_visitor::visit(srgs::rule& rule)
{
    //TRACE_FUNCTION("srgs::rule_visitor::visit(rule)");
    rule_ = &rule;
    name_ = rule.ext_id();
    node rule_n(&rule);
    node start_n(&rule.start_elm(), srgs::node::ROOT);
    this->init(rule_n, start_n);

    rule.expantions().accept(*this);

    node end_n(&rule.end_elm(), "NE", srgs::node::ROOT);
    this->fini(end_n);
    return 0;
}

//
//
//
int rule_visitor::visit(srgs::ruleref& ruleref)
{
    TRACE_FUNCTION("srgs::rule_visitor::visit(ruleref)");
    node n(&ruleref);
    if (ruleref.special().empty())
    {
        n.type(node_base::RULEREF);
    }
    else
    {
    	if (ruleref.special() == "NULL")
    	{
    		n.type(node_base::_NULL);
    	}
    	else if (ruleref.special() == "GARBAGE")
    	{
    		n.type(node_base::GARBAGE);
    	}
    	else if (ruleref.special() == "VOID")
    	{
    		n.type(node_base::_VOID);
    	}
    	else
    	{
            TRACE_ERROR_FMT("Illegal type attibute(%1%) in %2%",
                            ruleref.special() % name_);
            error_ = true;
            return -1;
    	}
    }
    this->append(n);
    return 0;
}


//
//
//
int rule_visitor::visit(srgs::rule_expantions& rulexp)
{
    //TRACE_FUNCTION("srgs::rule_visitor::visit(rule_expantions)");
    rule_expantions::elements& expantions = rulexp.expantions();
    rule_expantions::elements::iterator iter = expantions.begin();
    for ( ; iter != expantions.end(); ++iter)
    {
        srgs::element * elm = iter->get();
        elm->accept(*this);
    }
    return 0;
}

//
//
//
int rule_visitor::visit(srgs::item& item)
{
    TRACE_FUNCTION("srgs::rule_visitor::visit(item)");
    node item_node(&item);
    item_node.repeater(item.repeat());
    if (item.terminate())
    {
    	item_node.type(node_base::LEXIS);
        this->append(item_node);
    }
    else
    {

        int before = this->before_num();

#if defined(NONE_ONE_OF)
        if (item.repeat().max() > 1 || item.repeat().unlimited())
        {
            item_node.type(node_base::REPEAT);
            this->append(item_node);

            if (gd_->brother().isNull())
            {   // within one-of
                this->nest_in(item_node, gd_->parent());
            }
            else
            {
                this->nest_in(item_node, gd_->brother());
            }
            item.expantions().accept(*this);
            this->nest_out();
        }
#else
        // check repeat count
        if (item.repeat().unlimited())
        {
        	int repeat_top = this->current_num();
        	item_node.type(node_base::REPEAT);
        	this->append(item_node);    // put repeat node
            item.expantions().accept(*this);
            gd_->link_unlimited(this, repeat_top, item_node);
        }
        else if (item.repeat().max() > 1)
        {
            item_node.type(node_base::REPEAT);
            this->append(item_node);

            if (gd_->brother().isNull())
            {   // within one-of
                this->nest_in(item_node, gd_->parent());
            }
            else
            {
                this->nest_in(item_node, gd_->brother());
            }
            item.expantions().accept(*this);
            this->nest_out();
        }
#endif
        else if (item.repeat().none())
        {   // repeat="0-0"
            // skip element;
        }
        else
        {
            item.expantions().accept(*this);
        }
        // check repeat count (option)
        if (item.repeat().option())
        {
            item_node.type(node_base::REPEAT);
            gd_->link_option(this, before, item_node); 
        }
    }       
    return 0;
}

//
//
//
#ifdef NONE_ONE_OF
int rule_visitor::visit(srgs::one_of& one_of)
{
    TRACE_FUNCTION("srgs::rule_visitor::visit(one_of)");

    srgs::node save_parent, save_brother;
    srgs::element * null_element = NULL;

    std::vector<int> from_nums;    
    int from = gd_->edges_.erase_reserve(this->current_num());
    while (from >= 0)
    {
        //TRACE_DEBUG_FMT("[%1%]reserved from num = %2%", name_ % from);
        from_nums.push_back(from);
        from = gd_->edges_.erase_reserve(this->current_num());
    }
    //TRACE_DEBUG_FMT("reserved num = %1%", from);
    int before = this->before_num();  // for <ruleref type="NULL"/>
    //
    struct node_pair
    {
        int head_;
        int tail_;
    };
    srgs::one_of::items& items = one_of.item_table();
    save_parent = gd_->parent();
    save_brother = gd_->brother();
    int brother_num = save_brother.number();
    //
    node null_node;
    if (gd_->brother().isNull())
    {
        gd_->set(gd_->parent(), null_node);
    }
    else
    {
        gd_->set(gd_->brother(), null_node);
    }
    int q = items.size();
    node_pair * ids = new node_pair[q];
    for (int i = 0; i < q; i++)
    {
        if (items[i]->terminate())
        {
            node item_node(items[i].get(), node_base::LEXIS);
            this->append_child(item_node);
            ids[i].head_ = gd_->tail_node().number();
            ids[i].tail_ = ids[i].head_;
        }
        else
        {
            int top = this->current_num();

            srgs::node parent_backup = gd_->parent();
            srgs::node brother_backup = gd_->brother();

            items[i]->accept(*this);

            gd_->set(parent_backup, brother_backup);

            ids[i].tail_ = gd_->tail_node().number();
            ids[i].head_ = -1;
            for (int k = top; k < gd_->node_size(); k++)
            {
                const node * n = &(gd_->node_at(k));
                if (n->type() == srgs::node_base::LEXIS || 
                    n->type() == srgs::node_base::RULEREF ||
                    n->type() == srgs::node_base::GARBAGE)
                {   // exist token 
                    ids[i].head_ = n->number();
                    break;
                }
                else if (n->type() == node_base::_NULL)
                {
                    null_element = const_cast<srgs::element *>(n->get());
                    break;
                }
            }
        }
    }
    
    if (from_nums.size() > 0)
    {
        for (int n = 0; n < q; n++)
        {
#if 0
            if (ids[n].head_ == -1)
            {
                TRACE_ERROR_FMT("Panic! (logic error: %1%)", n);
                error_ = true;
                return -1;
            }
#endif
            int to = ids[n].head_;
            gd_->node_at(to).from_ids(from_nums);
        } 
        for (int m = 0; m < from_nums.size(); m++)
        {
            if (from_nums[m] == brother_num)
                continue ;                // linked above
            for (int n = 0; n < q; n++)
            {
                int to = ids[n].head_;
                //TRACE_DEBUG_FMT("reserved from %1% to %2%", from_nums[m] % to);
                gd_->add_edge(from_nums[m], to, arc_data(gd_->node_at(to)));
            } 
        }
    }
    
    int next = this->current_num();
    for (int j = 0; j < q; j++)
    {
        int from1 = ids[j].tail_;
        gd_->reserve_edge(from1, next, arc_data(gd_->node_at(from1))); 
    }

    // <ruleref type="NULL"/>
    if (null_element)
    {
        arc_data null_arc(null_element, node_base::_NULL);
        if (from_nums.size() > 0)
        {   // before ellement is <one-of> or option
            for (int m = 0; m < from_nums.size(); m++)
            {
                gd_->reserve_edge(from_nums[m], next, null_arc);
            }
        }
        else
        {
            gd_->reserve_edge(before, next, null_arc);
        } 
    }
    
    gd_->set(save_parent, save_brother);
    
    delete[] ids;

    return 0;
}

#else  // NONE_ONE_OF

int rule_visitor::visit(srgs::one_of& one_of)
{
    TRACE_FUNCTION("srgs::rule_visitor::visit(one_of)");

    srgs::node save_parent, save_brother;
    srgs::element * null_element = NULL;
    
#ifdef FRONT_ONE_OF
    srgs::node one_node(&one_of, node_base::ONE_OF);
    this->append(one_node);
#else
    int from = gd_->edges_.erase_reserve(this->current_num());
    //TRACE_DEBUG_FMT("reserved num = %1%", from);
#endif
    int before = this->before_num();  // for <ruleref type="NULL"/>
    //
    struct node_pair
    {
        int head_;
        int tail_;
    };
    srgs::one_of::items& items = one_of.item_table();
    save_parent = gd_->parent();
    save_brother = gd_->brother();
    //
    node null_node;
    if (gd_->brother().isNull())
    {
        gd_->set(gd_->parent(), null_node);
    }
    else
    {
        gd_->set(gd_->brother(), null_node);
    }
    int q = items.size();
    node_pair * ids = new node_pair[q];
    for (int i = 0; i < q; i++)
    {
        if (items[i]->terminate())
        {
            node item_node(items[i].get(), node_base::LEXIS);
            this->append_child(item_node);
            ids[i].head_ = gd_->tail_node().number();
            ids[i].tail_ = ids[i].head_;
        }
        else
        {
            int top = this->current_num();

            srgs::node parent_backup = gd_->parent();
            srgs::node brother_backup = gd_->brother();

            items[i]->accept(*this);

            gd_->set(parent_backup, brother_backup);

            ids[i].tail_ = gd_->tail_node().number();
            ids[i].head_ = -1;
            for (int k = top; k < gd_->node_size(); k++)
            {
                const node * n = &(gd_->node_at(k));
                if (n->type() == srgs::node_base::LEXIS || 
                    n->type() == srgs::node_base::RULEREF ||
                    n->type() == srgs::node_base::GARBAGE)
                {   // exist token 
                    ids[i].head_ = n->number();
                    break;
                }
                else if (n->type() == node_base::_NULL)
                {
                	null_element = const_cast<srgs::element *>(n->get());
                	break;
                }
            }
        }
    }
    
#ifdef FRONT_ONE_OF
    int next = this->current_num();
    for (int j = 0; j < q; j++)
    {
        int from1 = ids[j].tail_;
        gd_->reserve_edge(from1, next, arc_data(gd_->node_at(from1))); 
    }

    if (null_element)
    {
       arc_data null_arc(null_element, node_base::_NULL);
       gd_->edges_.make_reserve(before, next, null_arc); 
    }
    
    gd_->set(save_parent, save_brother);
#else
    int next = this->current_num();
    if (from >= 0)
    {   // optional link
        arc_data one_of_arc(&one_of, node_base::ONE_OF);
        gd_->edges_.make_reserve(from, next, one_of_arc); 
    }
    else if (null_element != NULL)
    {
       arc_data null_arc(null_element, node_base::_NULL);
       gd_->edges_.make_reserve(before, next, null_arc); 
    }
    //
    for (int j = 0; j < q; j++)
    {
        int from1 = ids[j].tail_;
        gd_->edges_.make_reserve(from1, next, arc_data(gd_->node_at(from1))); 
    }

    gd_->set(save_parent, save_brother);

    srgs::node one_node(&one_of, node_base::ONE_OF);
    this->append(one_node);
    
    // set id in top member of <one-of> element
    for (int h = 0; h < q; h++)
    {
        int head_num = ids[h].head_;
        //TRACE_NOTICE_FMT("member head id = %1%/%2%",
        //                 head_num % gd_->node_size() - 1);
        if (head_num < 0)
            continue;
        const srgs::node * headp = &(gd_->node_at(head_num));
        const_cast<srgs::node *>(headp)->top_of_oneof(this->before_num());
    }
#endif

    delete[] ids;

    return 0;
}
#endif
//
//
//
int rule_visitor::visit(srgs::token& token)
{
    TRACE_FUNCTION("srgs::rule_visitor::visit(token)");
    node token_node(&token, node_base::LEXIS);
    this->append(token_node);
    return 0;
}

int rule_visitor::visit(srgs::tag& tag)
{
    TRACE_FUNCTION("srgs::rule_visitor::visit(tag)");
    return 0;
}
//
//
//
void rule_visitor::append(node& node)
{
    TRACE_FUNCTION("srgs::rule_visitor::append");
    std::string ndname;
    char namebuf[64];
   
   	node.number(this->current_num());
    sprintf(namebuf, "%d", node.number());
    node.rule_name(name_);         // set rule name
    
    switch (node.elm_type())
    {
    case element::ITEM :
        {
            //srgs::item * item_elm = dynamic_cast<srgs::item *>(node.get());
            //std::string reading = item_elm->str();
            //std::string text = item_elm->tag().str();
            //printf("word = {%s,%s}\n", reading.c_str(), text.c_str());
            ndname = namebuf;
        }
        break;
    case element::RULEREF :
        {
            //srgs::ruleref * ruleref_elm = dynamic_cast<srgs::ruleref *>(node.get());
            //std::string ref = ruleref_elm->uri().fragment();
            //printf("ruleref = %s\n", ref.c_str());
            ndname = namebuf;
        }
        break;
    case element::ONE_OF :
        {
            //srgs::one_of * one_of_elm = dynamic_cast<srgs::one_of *>(node.get());
           ndname = namebuf;
        }
        break;
    case element::TOKEN :
        ndname = namebuf;
        break;
    case element::START :
        ndname = "NS";
        break;
    case element::END :
        ndname = "NE";
        break;
    case element::HEAD :
        ndname = "NH";
        break;
    case element::TAIL :
        ndname = "NT";
        break;
    default :
        TRACE_ERROR_FMT("illegal element type = %1% in %2%",
                        node.elm_type() % name_);
        break;
    }

    node.name(ndname);
    gd_->add_node(node);

    int ret = gd_->edges_.put_on(node.number(), node);
    if (ret == 0) 
    {
        gd_->link_sequential(node);
    }
    gd_->brother(node);
}

void rule_visitor::append_child(node& node)
{
    TRACE_FUNCTION("srgs::rule_visitor::append_child");
    std::string ndname;
    char namebuf[64];

    node.number(this->current_num());
    sprintf(namebuf, "%d", node.number());
    node.rule_name(name_);           // set rule name

    switch (node.elm_type())
    {
    case element::ITEM :
        {
            //srgs::item * item_elm = dynamic_cast<srgs::item *>(node.get());
            //std::string reading = item_elm->str();
            //std::string text = item_elm->tag().str();
            //printf("word = {%s,%s}\n", reading.c_str(), text.c_str());
            ndname = namebuf;
        }
        break;
    case element::RULEREF :
        {
            //srgs::ruleref * ruleref_elm = dynamic_cast<srgs::ruleref *>(node.get());
            //std::string ref = ruleref_elm->uri().fragment();
            //printf("ruleref = %s\n", ref.c_str());
            ndname = namebuf;
        }
        break;
    case element::ONE_OF :
        {
            //srgs::one_of * one_of_elm = dynamic_cast<srgs::one_of *>(node.get());
            ndname = namebuf;
        }
        break;
    case element::START :
        ndname = "NS";
        break;
    case element::END :
        ndname = "NE";
        break;
    case element::HEAD :
        ndname = "NH";
        break;
    case element::TAIL :
        ndname = "NT";
        break;
    default :
        TRACE_ERROR_FMT("illegal element type = %1% in %2%",
                        node.elm_type() % name_);
        break;
    }

    node.name(ndname);
    gd_->add_node(node);

    if (gd_->edges_.put_on(node.number(), node) == 0)
    {
       	gd_->link_branch(node);
   	}
    //gd_->brother(node);
}

void rule_visitor::append_end(node& node)
{
    TRACE_FUNCTION("srgs::rule_visitor::append_end");

    node.number(this->current_num());
    node.rule_name(name_);           // set rule name

    gd_->add_node(node);
    
    if (gd_->edges_.put_on(node.number(), node) == 0)
    {
        if (gd_->empty())
        {   // the rule is empty
            gd_->link_sequential(node);
        }
        else
        {   
            gd_->link_sequential(node);
        }
    }
}


void rule_visitor::dump(std::ostream& out) const
{
    //TRACE_FUNCTION("srgs::rule_visitor::dump");
    out << "[" << name_ << "]\n";
    main_gd_->dump(out);
    out << "\n";
}

}  // namespace
