/* $Id$ */

//=============================================================================
/**
 *  @file    rule.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef SRGS_RULE_HPP
#define SRGS_RULE_HPP

namespace srgs
{

class grammar;

//
// rule element
//
class rule : public srgs::element
{
public:
    typedef std::vector<std::string> examples;
   
    rule(): srgs::element(element::RULE), id_(), scope_(),
            start_(element::START), end_(element::END), grammar_(NULL),
            ext_id_("") {}
    rule(const rule& rhs)
        : srgs::element(rhs), id_(rhs.id_), scope_(rhs.scope_),
          examples_(rhs.examples_), expantions_(rhs.expantions_),
          start_(rhs.start_), end_(rhs.end_), grammar_(rhs.grammar_),
          ext_id_(ext_id_) {}
    virtual ~rule() {}
    
    rule& operator=(const rule& rhs)
    {
        rule tmp(rhs);
        this->element::swap(tmp);
        std::swap(id_, tmp.id_);
        std::swap(scope_, tmp.scope_);
        std::swap(examples_, tmp.examples_);
        std::swap(tags_, tmp.tags_);
        expantions_ = tmp.expantions_;
        start_ = tmp.start_;
        end_ = tmp.end_;
        grammar_ = tmp.grammar_;
        ext_id_ = tmp.ext_id_;
        return *this;
    }

    virtual srgs::element * clone() const
    {
        return new rule(*this); 
    }

    void clear()
    {
        this->element::clear();
        id_ = "";
        scope_ = "";
        examples_.clear();
        tags_.clear();
        expantions_.clear();
        grammar_ = NULL;
        ext_id_ = "";
    }
    
    // Get/Set
    srgs::rule_expantions& expantions()   { return expantions_; }
    srgs::root_element& start_elm() { return start_; }
    srgs::root_element& end_elm()   { return end_; }
    const srgs::grammar * grammar() const { return grammar_; }
    const srgs::element * expantion(unsigned int idx) const
    {
        return expantions_.expantion(idx);
    }
    const srgs::Id_datatype& id() const { return id_; }
    const std::string& ext_id() const { return ext_id_; }
    const srgs::Scope_datatype& scope() const { return scope_; }   
    const std::string& example(unsigned int idx = 0) const
    {
        if (idx >= examples_.size())
            throw std::invalid_argument("srgs::rule::example: out of range error");
        return examples_[idx];
    }
    const srgs::tag& tag(int idx = 0) const
    {
        if (idx >= tags_.size())
            throw std::invalid_argument("srgs::rule::tag: out of range error");
        return *(tags_[idx]);
    }   

    void grammar(srgs::grammar * gram) { grammar_ = gram; }
    void token(const srgs::token& token)
    {
        //this->str(token.str());
        shared_ptr<srgs::token> tmp(new srgs::token(token));
        expantions_.expantion(tmp);
    }
    void ruleref(const srgs::ruleref_ptr& ruleref)
    {
        TRACE_FUNCTION("srgs::rule::ruleref");
        std::string refname = ruleref->uri().fragment();
        if (refname == "NULL" || refname == "VOID" || refname == "GARBAGE")
        {
            TRACE_ERROR_FMT("special rule name (%1%)", refname);
        }
        expantions_.expantion(ruleref.shared());
    }
    void item(const srgs::item_ptr& item)
    {
        TRACE_FUNCTION("srgs::rule::item");
        if (! item->available())
        {
            TRACE_DEBUG("ignore <item> element (probabbly, empty)");
            return ;
        }
        expantions_.expantion(item.shared());
    }
    void one_of(const srgs::one_of_ptr& one_of)
    {
        //TRACE_FUNCTION("srgs::rule::one_of");
        expantions_.expantion(one_of.shared());
    }
    void tag(const srgs::tag_ptr& tag)
    {
        if (expantions_.size() > 0)
        {
            expantions_.expantions().back()->relation(tag.shared());
        }
        else
        {
            start_.relation(tag.shared());
        }
        tags_.push_back(tag.shared());
    }
    void id(const srgs::Id_datatype& id) { id_ = id; ext_id_ = id; }
    void ext_id(const std::string& id) { ext_id_ = id; }
    void scope(const srgs::Scope_datatype& scope) { scope_ = scope; }   
    void example(const std::string& example)
    {
        examples_.push_back(example);
    }
    virtual void  str(const std::string& v)
    {
        if (v.size() > 0)
        {
            shared_ptr<srgs::token> tmp(new srgs::token());
            tmp->str(v);
            expantions_.expantion(tmp);
            //value_.push_back(v);
        }
    }
    
    bool available() const
    {
        return ((this->element::str().size() > 0) || (expantions_.size() > 0)); 
    }
    
    // accept visitor    
    virtual int accept(rule_visitor& vis)
    {
        TRACE_FUNCTION("srgs::rule::visit");
        vis.visit(*this);
        //if (vis.error())
        //{
        //    std::string msg = "visitor found rule(";
        //    msg += id_;
        //    msg += ") error.";
        //    throw srgs::exception(msg);
        //}
        return 0;
    }
    
     virtual void dump(std::ostream& out, int indent = 1) const
    {
        out << "rule: " << id_ << "(" << ext_id_ << ")" << "\n";
        expantions_.dump(out, indent);
    }
    
private:        
    srgs::Id_datatype    id_;
    srgs::Scope_datatype scope_;

    examples examples_;
    srgs::rule_expantions expantions_;
    std::vector<shared_tag> tags_;

    root_element start_;
    root_element end_;
    
    srgs::grammar * grammar_;
    std::string ext_id_;
};

//
typedef bee::xsd::element_ptr<srgs::rule> rule_ptr;

}  // namespace


#endif /* SRGS_RULE_HPP */
