/* $Id$ */

//=============================================================================
/**
 *  @file    grammar_visitor.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef SRGS_GRAMMAR_VISITOR_HPP
#define SRGS_GRAMMAR_VISITOR_HPP

#include <boost/config.hpp>
#include <boost/graph/copy.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/graph_utility.hpp>

#include "srgs/rule_visitor.hpp"

enum vertex_srgs_t { vertex_srgs };
enum edge_srgs_t   { edge_srgs };

namespace boost
{
    BOOST_INSTALL_PROPERTY(vertex, srgs);
    BOOST_INSTALL_PROPERTY(edge, srgs);
}

namespace srgs 
{

// declare type
typedef boost::adjacency_list< boost::vecS, boost::vecS, boost::directedS,
        boost::property< vertex_srgs_t, srgs::node >,
        boost::property< edge_srgs_t, srgs::arc_data > >
        graph_t;
typedef boost::property_traits< boost::property_map< graph_t,
        vertex_srgs_t >::type >::value_type
        NodeType;
typedef boost::property_map< graph_t, vertex_srgs_t >::type
        NodeMap;
typedef boost::property_map< graph_t, vertex_srgs_t >::const_type
        ConstNodeMap;
typedef boost::property_map< graph_t, edge_srgs_t >::type
        ArcMap;
typedef boost::property_map< graph_t, edge_srgs_t >::const_type
        ConstArcMap;

//
//
struct grammar_graph : graph_base
{
    grammar_graph() : graph_base(), depth_(0) {}
    
    int make_graph()
    {
        node_map_ = boost::get(vertex_srgs, graph_);
        for (size_t i = 0; i < nodes_.size(); i++)
        {
            graph_t::vertex_descriptor v = add_vertex(graph_);
            boost::put(node_map_, v, nodes_[i]);
        }

        for (size_t i = 0; i < edges_.size(); ++i)
        {
            boost::add_edge(edges_[i].first, edges_[i].second,
                            edges_[i].data_, graph_);
        }

        arc_map_ = boost::get(edge_srgs, graph_);
        return 0;
    }
    
    void dump_graph(std::ostream& out)
    {
        boost::graph_traits<graph_t>::vertex_iterator vi;
        for (vi = vertices(graph_).first; vi != vertices(graph_).second; ++vi)
            out << node_map_[*vi].name() <<  ": " << node_map_[*vi].str()
                << std::endl;
        out << std::endl;

        boost::graph_traits< graph_t >::edge_iterator ei, ei_end;
        for (boost::tie(ei, ei_end) = boost::edges(graph_); ei != ei_end; ++ei)
        {
            out << boost::get(node_map_, boost::source(*ei, graph_)).name()
            << " -> "
            << boost::get(node_map_, boost::target(*ei, graph_)).name()
            << std::endl;
        }
        out << std::endl;
    }

    //
    void make_graphviz(std::ostream& out, const std::string& grfname);

    graph_t graph_;
    NodeMap node_map_;
    ArcMap  arc_map_;
    int     depth_;

};


class recognition_visitor: public boost::default_bfs_visitor
{
public:
    recognition_visitor(NodeMap& node_map, ArcMap& arc_map, std::ostream& out,
                        std::vector<std::string>& str, int& current):
        boost::default_bfs_visitor(), node_map_(node_map), arc_map_(arc_map),
        out_(out), str_(str), current_(current), last_rule_("")
    {
    }

    //template <class Vertex, class Graph>
    //void initialize_vertex(Vertex s, const Graph & g)
    //{
    //    TRACE_FUNCTION("srgs::recognition_visitor::initialize_vertex");
    //    TRACE_NOTICE_FMT("node = %1%", node_map_[s].name());
    //}

    template <class Vertex, class Graph>
    void discover_vertex(Vertex u, const Graph & g)
    {
        TRACE_FUNCTION("srgs::recognition_visitor::discover_vertex");
        TRACE_NOTICE_FMT("node = %1%", node_map_[u].name());
#ifdef NONE_ONE_OF
        if (node_map_[u].type() == node_base::REPEAT)
        {
            grammar_graph * gg =
                static_cast<srgs::grammar_graph *>(node_map_[u].graphbase()); 
            recognition_visitor vis(gg->node_map_, gg->arc_map_, out_,
                                    str_, current_);
            boost::graph_traits<srgs::graph_t>::vertex_descriptor s(0);
            breadth_first_search(gg->graph_, s, visitor(vis));  
        }
#endif
    }
    
    //template <class Edge, class Graph>
    //void tree_edge(Edge e, const Graph & g)
    //{
    //    TRACE_FUNCTION("srgs::recognition_visitor::tree_edge");
    //    TRACE_NOTICE_FMT("arc = %1%", arc_map_[e].name());
    //}
    
    template <class Edge, class Graph>
    void examine_edge(Edge e, const Graph & g)
    {
        TRACE_FUNCTION("srgs::recognition_visitor::examine_edge");

        typedef typename boost::graph_traits<srgs::graph_t>::vertex_descriptor
                Vertex;

        int curpos = current_;
        Vertex src = boost::source(e, g);
        Vertex targ = boost::target(e, g);
        
        TRACE_NOTICE_FMT("arc = %1% (%2% -> %3%)", arc_map_[e].value() %
            node_map_[src].name() % node_map_[targ].name());

        //if (current_ >= str_.size())
        //{
        //    TRACE_NOTICE_FMT("recognize over (%1%)", current_);
        //    return ;
        //}

        if (str_.empty())
        {   // called from print_graph() as debug
            return ;
        }
                 
        if (arc_map_[e].type() == node_base::LEXIS)
        {
            srgs::element * elm = arc_map_[e].get();
            //TRACE_NOTICE_FMT("compare item data \"%1%\" \"%2%\"", 
            //                 str_[current_] % elm->str());
            std::string tag("");
            if (arc_map_[e].elm_type() == element::ITEM)
            {
                srgs::item * item_elm =
                    dynamic_cast<srgs::item *>(arc_map_[e].get());
                assert(item_elm != NULL);
                if ((curpos = item_elm->compare(str_, current_)) < 0)
                {   // different words
                    return;
                }
                tag = item_elm->tag_string();
            }
            else if (arc_map_[e].elm_type() == element::TOKEN)
            {
                srgs::token * tk_elm =
                    dynamic_cast<srgs::token *>(arc_map_[e].get());
                assert(tk_elm != NULL);
                if ((curpos = tk_elm->compare(str_, current_)) < 0)
                {   // different words
                    return;
                }
                if (elm->relation() != NULL)
                {
                    tag = elm->relation()->str();
                } 
            }
            current_ = curpos;

            if (node_map_[targ].rule_stack().empty())
            {
                out_ << "$" << elm->rule() << "[";
                int count = elm->tokens(out_);
                if (! tag.empty())
                {
                    if (count > 0)
                        out_ << ",";
                    out_ << "{!{" << tag << "}!}";
                }
                out_ << "]";
                last_rule_ = elm->rule();
                out_ << " ";
            }
            else
            {
                std::deque<std::string>::const_reverse_iterator iter =
                    node_map_[targ].rule_stack().rbegin();
                for ( ; iter != node_map_[targ].rule_stack().rend(); ++iter)
                {
                    out_ << "$" << *iter << "[";
                }

                out_ << "$" << elm->rule() << "[";
                int count = elm->tokens(out_);
                if (! tag.empty())
                {
                    if (count > 0)
                        out_ << ",";
                    out_ << "{!{" << tag << "}!}";
                }
                out_ << "]";
                last_rule_ = elm->rule();

                for (size_t i = 0; i < node_map_[targ].rule_stack().size(); i++)
                {
                    out_ << "]";
                }
                out_ << " ";
            }
        }
        else if (arc_map_[e].type() == node_base::RULEREF)
        {   // ruleref is empty
            srgs::ruleref * ref_elm =
                    dynamic_cast<srgs::ruleref *>(arc_map_[e].get());
            assert(ref_elm != NULL);
            TRACE_NOTICE_FMT("ruleref: %1%", ref_elm->str());
            if (node_map_[targ].rule_stack().empty())
            {
                last_rule_ = ref_elm->uri().fragment();
                out_ << "$" << last_rule_ << "[]";
                out_ << " ";
            }
            else
            {
                std::deque<std::string>::const_reverse_iterator iter =
                    node_map_[targ].rule_stack().rbegin();
                for ( ; iter != node_map_[targ].rule_stack().rend(); ++iter)
                {
                    out_ << "$" << *iter << "[";
                }

                last_rule_ = ref_elm->uri().fragment();
                out_ << "$" << last_rule_ << "[]";

                for (size_t i = 0; i < node_map_[targ].rule_stack().size(); i++)
                {
                    out_ << "]";
                }
                out_ << " ";
            }
        }
    }
    
    //template <class Vertex, class Graph>
    //void examine_vertex(Vertex u, const Graph & g)
    //{
    //    TRACE_FUNCTION("srgs::recognition_visitor::examine_vertex");
    //}

    //template <class Vertex, class Graph>
    //void finish_vertex(Vertex u, const Graph& g)
    //{
    //    TRACE_FUNCTION("srgs::recognition_visitor::finish_vertex");
    //}

    NodeMap& node_map_;
    ArcMap&  arc_map_;
    std::ostream& out_;
    std::vector<std::string>& str_;
    int& current_;
    std::string last_rule_;
};

//
//
//
template<typename VISITOR = srgs::rule_visitor>
class grammar_visitor
{
public:
    grammar_visitor() : current_visitor_(NULL), current_(0) {}

    VISITOR& rule_visitor() { return *current_visitor_; }
    int size() const { return visitors_.size(); }
    VISITOR& at(int offset)
    {
        assert(offset < visitors_.size());
        return visitors_[offset];
    }

    VISITOR * create()
    {
        make_rule_visitor();
        return current_visitor_;
    }

    const graph_data * find(const std::string& name)
    {
        for (size_t i = 0; i < visitors_.size(); i++)
        {
        	if (visitors_[i].name() == name)
        	    return  visitors_[i].main_graphdata();
        }
        return NULL;
    }
    int find_position(const std::string& name)
    {
        for (size_t i = 0; i < visitors_.size(); i++)
        {
            if (visitors_[i].name() == name)
                return i;
        }
        return -1;
    }

    const std::vector<std::string>& recognition_words() const
    {
        return str_;   // recognized words
    }
    
    void visit(srgs::grammar * gram);   // see srgs.hxx

    int  recognize(srgs::grammar * gram, std::ostream& out,
                   const std::string& recog)
    {
        TRACE_FUNCTION("srgs::grammar_visitor::recognize");
        int result;
        boost::tokenizer<> tok(recog);
        for (boost::tokenizer<>::iterator beg = tok.begin();
             beg != tok.end(); ++beg)
        {
            str_.push_back(*beg);
        }
 
        srgs::grammar_graph root_gd;
      
        srgs::graph_data * gd = visitors_[0].graphdata();
        visitors_[0].inc_refcount();   // set reference count of root rule     
        collect_graphdata(root_gd, *gd);       
        root_gd.make_graph();

        recognition_visitor vis(root_gd.node_map_, root_gd.arc_map_, out,
                                str_, current_);
        boost::graph_traits<graph_t>::vertex_descriptor s(0);
        breadth_first_search(root_gd.graph_, s, visitor(vis));  
        if (vis.current_ == str_.size())
        {
            //TRACE_DEBUG_FMT("PASS = %1%", vis.current_);
            result = 0;
        }
        else
        {
            //TRACE_DEBUG_FMT("FAIL = %1%/%2%", vis.current_ % str_.size());
            result = -1;
        }
        return result;
    }

    void print_graph(std::ostream& out)
    {
        if (visitors_.empty())
            return;
        srgs::graph_data * gd = visitors_[0].graphdata();
        srgs::grammar_graph root_gd;
        collect_graphdata(root_gd, *gd);
        root_gd.dump(out);
        
        root_gd.make_graph();
        out << std::endl;
        root_gd.dump_graph(out);

#if 1      
        recognition_visitor vis(root_gd.node_map_, root_gd.arc_map_, out,
                                str_, current_);
        boost::graph_traits<graph_t>::vertex_descriptor s(0);
        breadth_first_search(root_gd.graph_, s, visitor(vis));  
#endif        
    }

    //
    void print_graphviz(std::ostream& out, const std::string& grxml)
    {
        srgs::grammar_graph root_gd;
      
        srgs::graph_data * gd = visitors_[0].graphdata();
        visitors_[0].inc_refcount();   // set reference count of root rule     
        collect_graphdata(root_gd, *gd);       
        root_gd.make_graph();

        bee::pathname grname(grxml);
        root_gd.make_graphviz(out, grname.name());
    }

    //
    // collect graph data of all rules to root
    //
    int collect_graphdata(srgs::grammar_graph& root_gd,
                          srgs::graph_data& rule_gd)
    {
        TRACE_FUNCTION("srgs::grammar_visitor::collect_graphdata");
        int result = 0;
        // copy nodes
        srgs::node node(rule_gd.nodes_[0]);  // start node
        root_gd.nodes_.push_back(node);
 
        std::vector<srgs::edge_info> ending;
        result = copy_graphdata(root_gd, rule_gd, ending, -1);  // *****
            // result is count that reserve to node number 

        node = rule_gd.nodes_.back();    // end node (*)
        node.number(root_gd.nodes_.size());
        root_gd.nodes_.push_back(node);

        assert(ending.size() > 0);
        for (int i = 0; i < ending.size(); i++)
        {
            int old = ending[i].second;
            assert(old == -1);
            ending[i].second = node.number();
            root_gd.edges_.push_back(ending[i]);
        }
        return result;
    }

    //
#ifdef NONE_ONE_OF
    struct remap
    {
        remap() : type_(node_base::UNKNOWN), absid_(-1), fh_(0), ft_(0),
                  nh_(0), nt_(0), eh_(0), et_(0) {}
 
        int type_;
        int absid_;
        std::vector<int> fromids_oneof_;
        int fh_;
        int ft_;
        int nh_;
        int nt_;
        int eh_;
        int et_;
    };
#else
    struct remap
    {
        remap() : type_(node_base::UNKNOWN), absid_(-1), mbr_oneof_(0),
                  nh_(0), nt_(0), eh_(0), et_(0) {}
 
        int type_;
        int absid_;
        int mbr_oneof_;
        int nh_;
        int nt_;
        int eh_;
        int et_;
    };
#endif
    //
    // copy graph data from rule to root
    //
    int copy_graphdata(srgs::grammar_graph& root_gd,
                       srgs::graph_data& rule_gd,
                       std::vector<srgs::edge_info>& ending, int link_from)
    {
        TRACE_FUNCTION("srgs::grammar_visitor::copy_graphdata");
        int result = 0;
        //ending.clear();
        root_gd.depth_++;
        if (root_gd.depth_ >= 16)
        {
            TRACE_ERROR_FMT("refer rule depth over (%1%)",
                            root_gd.depth_);
            throw srgs::exception("ruleref depth over.\n");
        }
        
        std::vector<remap> idxmap(rule_gd.nodes_.size());
        idxmap[0].type_ = node_base::LEXIS;
        if (link_from >= 0)
            idxmap[0].absid_ = -(link_from + 1);  // can't use -1 
        else
            idxmap[0].absid_ = root_gd.nodes_.size() - 1;    // marking (START)
            
        // copy nodes
        int idx = root_gd.nodes_.size();
        for (size_t i = 1; i < rule_gd.nodes_.size() - 1; i++) 
        {   // ignore star/end node
            srgs::node node(rule_gd.nodes_[i]);           
            int node_id = node.number();
            idxmap[node_id].type_ = node.type();   // marking
            node.set(root_gd.called_);
            if (node.type() == node_base::_NULL)   // ignore node
            {
                continue;
            }
            else if (node.type() == node_base::RULEREF)
            {
                srgs::ruleref * ruleref_elm =
                    dynamic_cast<srgs::ruleref *>(node.get());
                std::string path = ruleref_elm->uri().path();
                std::string ref = ruleref_elm->uri().fragment();
                if (path.empty() && ruleref_elm->element::rule() == ref)
                {
                    TRACE_ERROR_FMT("can't refer to own rule (%1%)", ref);
                    throw srgs::exception("there is a reference of own rule.\n");                    
                }
                if (! path.empty())
                    ref = path + "#" + ruleref_elm->uri().fragment();       
                int pos = this->find_position(ref);
                //assert(pos >= 0);
                if (pos < 0)
                {
                    TRACE_ERROR_FMT("the rule (%1%) is not found", ref);
                    throw srgs::exception("there isn't the rule.\n");                    
                }
                if (visitors_[pos].get_rule()->available())
                {
                    srgs::graph_data * sub_rule = visitors_[pos].graphdata();
                    visitors_[pos].inc_refcount();   // set reference count of rule
                    //rule_gd.nodes_[i].inc_pass_count();

                    idxmap[node_id].absid_ = idx;
#ifdef NONE_ONE_OF
                    if (rule_gd.nodes_[i-1].type() == node_base::RULEREF &&
                        ending.size() > 1)
                    {
                        node.from_clear();
                        for (size_t b = 0; b < ending.size(); b++)
                        {
                            node.from_oneof(ending[b].first);
                        }
                        ending.clear();
                    }
                    idxmap[node_id].fh_ = ending.size();
                    idxmap[node_id].fromids_oneof_ = node.from_ids();
                    int mbr_oneof = node.from_ids()[0];
                    if (node.from_size() == 0)
                        mbr_oneof = -1;
#else
                    int mbr_oneof = node.top_of_oneof();
                    idxmap[node_id].mbr_oneof_ = mbr_oneof;
#endif
                    idxmap[node_id].nh_ = root_gd.nodes_.size();
                    idxmap[node_id].eh_ = root_gd.edges_.size();
                    // for recognize test
                    root_gd.called_.push_front(node.rule_name()); 
                    //TRACE_NOTICE_FMT("call rule = %1% (%2%)",
                    //                 ref % node.rule_name());
                    result = this->copy_graphdata(root_gd, *sub_rule, ending,
                                                  mbr_oneof);
                    root_gd.called_.pop_front();   // for recognize test
                    idxmap[node_id].nt_ = root_gd.nodes_.size() - 1;
                    idxmap[node_id].et_ = root_gd.edges_.size() - 1;
                    idx = root_gd.nodes_.size();   // renew index
                    //rule_gd.nodes_[i].dec_pass_count();
#ifdef NONE_ONE_OF
                    idxmap[node_id].ft_ = ending.size();
#endif
                }
                else
                {   // empty rule
                    //TRACE_DEBUG_FMT("the rule (%1%) is empty", ref);
                    root_gd.called_.push_front(node.rule_name()); 
                    node.set(root_gd.called_);
                    idxmap[node_id].type_ = node_base::LEXIS;
                    idxmap[node_id].absid_ = idx;
                    node.number(idx);  // relative id to absolute
                    idx++;
                    root_gd.nodes_.push_back(node);
                    root_gd.called_.pop_front();   // for recognize test
                }
            }
            else if (node.type() == node_base::REPEAT)
            {
                TRACE_NOTICE_FMT("repeat node [%1%-%2%]", 
                    node.repeater().min() % node.repeater().max());
#ifdef NONE_ONE_OF                
                //graph_data * repeat_gd = dynamic_cast<graph_data *>(node.graphbase());
                graph_data * repeat_gd =
                    static_cast<graph_data *>(node.graphbase());
                srgs::grammar_graph * repeat_gg = new srgs::grammar_graph;
                repeat_gg->called_ = root_gd.called_;  // for recognize test 
                collect_graphdata(*repeat_gg, *repeat_gd);  // ###
                repeat_gg->make_graph();
                shared_gb srg(repeat_gg);
                node.graphbase(srg);
                if (getopt_tree())
                {
                    repeat_gg->dump(std::cout);
                }
#endif
                idxmap[node_id].absid_ = idx;
                node.number(idx);  // relative id to absolute
                idx++;
                root_gd.nodes_.push_back(node);
            }
            else
            {
                idxmap[node_id].absid_ = idx;
                node.number(idx);  // relative id to absolute
                idx++;
                root_gd.nodes_.push_back(node);
            }
        }
        idxmap[rule_gd.nodes_.size()-1].type_ = node_base::ROOT;  // marking (END)

        //for (size_t x = 0; x < rule_gd.nodes_.size(); x++)
        //{
        //    TRACE_NOTICE_FMT("absolute node id = %1%", idxmap[x]);
        //}
         
#ifdef NONE_ONE_OF
        // re-number one-of node number that is in top node of <one-of> member
        for (size_t i = 1; i < rule_gd.nodes_.size() - 1; i++)
        {
            srgs::node node(rule_gd.nodes_[i]);
            int tid = idxmap[node.number()].absid_;
            if (node.from_size() == 0)
                continue;
            if (node.type() == node_base::RULEREF && tid < 0)
                continue;    // empty rule
            //TRACE_NOTICE_FMT("node id = %1% -> %2%", relid % absid);
            if (node.type() == node_base::RULEREF)
            {
                int relid = idxmap[node.number()].fromids_oneof_[0];
                for (size_t f = 0;
                     f < idxmap[node.number()].fromids_oneof_.size(); f++)
                {
                    int oid = idxmap[node.number()].fromids_oneof_[f];
                    int aid = (oid < 0) ? -1 : idxmap[oid].absid_;
                    idxmap[node.number()].fromids_oneof_[f] = aid;
                }
                for (size_t ni = idxmap[node.number()].nh_;
                     ni <= idxmap[node.number()].nt_; ni++)
                {
                    int chkid = root_gd.nodes_[ni].from_oneof();
                    if (root_gd.nodes_[ni].from_size() == 0)
                        continue;   // not menber

                    assert(chkid == -(relid + 1));
                    // aboslute re-number
                    root_gd.nodes_[ni].from_clear();
                    root_gd.nodes_[ni].from_ids(idxmap[node.number()].fromids_oneof_);
                }
            }
            else
            {
                std::vector<int>& ids = root_gd.nodes_[tid].from_ids();
                for (size_t f = 0; f < ids.size(); f++)
                {
                    int aid = idxmap[ids[f]].absid_;
                    root_gd.nodes_[tid].from_ids()[f] = aid;  // aboslute number
                }
            }
        } 
#else
        // re-number one-of node number that is in top node of <one-of> member
        for (size_t i = 1; i < rule_gd.nodes_.size() - 1; i++)
        {
            srgs::node node(rule_gd.nodes_[i]);
            int tid = idxmap[node.number()].absid_;
            int relid = node.top_of_oneof();
            if (relid == -1)
                continue;
            if (node.type() == node_base::RULEREF && tid < 0)
                continue;    // empty rule
            int absid = idxmap[relid].absid_;
            //TRACE_NOTICE_FMT("node id = %1% -> %2%", relid % absid);
            if (node.type() == node_base::RULEREF)
            {
                for (size_t ni = idxmap[node.number()].nh_;
                     ni <= idxmap[node.number()].nt_; ni++)
                {
                    int chkid = root_gd.nodes_[ni].top_of_oneof();
                    if (chkid == -1)
                        continue;   // not menber
                    if (chkid >= 0)
                        continue;   // re-menbered
                    assert(chkid == -(relid + 1));
                    root_gd.nodes_[ni].top_of_oneof(absid);  // aboslute re-number
                }
            }
            else
            {
                root_gd.nodes_[tid].top_of_oneof(absid);  // aboslute number
            }
        } 
#endif
        // copy edges
        bool cleared = false;
        for (size_t j = 0; j < rule_gd.edges_.size(); j++)
        {
            srgs::edge_info edge(rule_gd.edges_[j]);
            int ty1st = idxmap[edge.first].type_;
            int ty2nd = idxmap[edge.second].type_;
            if (ty1st == node_base::RULEREF &&
                (ty2nd == node_base::_NULL || ty2nd == node_base::ROOT))
            {   // ignore: ruleref --> end/null
                //TRACE_NOTICE_FMT("ignore ruleref edge [%1%-%2%]",
                //                 edge.first % edge.second);
            }
            else if (ty1st == node_base::RULEREF &&
                     (ty2nd == node_base::LEXIS || ty2nd == node_base::ONE_OF))
            {   // ruleref --> lexis
#ifdef NONE_ONE_OF
                for (int l = idxmap[edge.first].fh_;
                     l < idxmap[edge.first].ft_; l++)
                {
                    edge.first = ending[l].first;  // last id in calling rule
                    edge.second = idxmap[edge.second].absid_;
                    root_gd.edges_.push_back(edge);                
                }
#else
                edge.first = idxmap[edge.first].nt_;  // last id in calling rule
                edge.second = idxmap[edge.second].absid_;
                root_gd.edges_.push_back(edge);                
                //TRACE_NOTICE_FMT("pushed edge [%1%-%2%]",
                //                 edge.first % edge.second);
#endif
            }
            else if (ty2nd == node_base::RULEREF)
            {
#ifdef NONE_ONE_OF
                if (ty1st == node_base::RULEREF &&
                    idxmap[edge.second].fromids_oneof_.size() > 0)
                {
                    for (size_t ei = idxmap[edge.second].eh_;
                         ei <= idxmap[edge.second].et_; ei++)
                    {
                        if (root_gd.edges_[ei].first < 0)
                        {
                            assert(root_gd.edges_[ei].first ==
                                -(idxmap[edge.second].fromids_oneof_[0] + 1));

                            srgs::edge_info edge1(root_gd.edges_[ei]);
                            for (int l = 1;
                                 l < idxmap[edge.second].fromids_oneof_.size(); l++)
                            {
                                edge1.first = idxmap[edge.second].fromids_oneof_[l];  // last id in calling rule
                                root_gd.edges_.push_back(edge1);                
                                TRACE_NOTICE_FMT("append edge = %1% -> %2%", 
                                             edge1.first % edge1.second);
                            }
                        }
                    }
                }
#endif
                if (edge.data_.type() == node_base::REPEAT)
                {   // repeat: ? --> ruleref
                    //TRACE_NOTICE_FMT("repeat edge to ruleref [%1%-%2%]",
                    //                 edge.first % edge.second);
                    edge.first = idxmap[edge.first].nt_;
                    edge.second = idxmap[edge.second].nh_;
                    root_gd.edges_.push_back(edge);
                    TRACE_NOTICE_FMT("pushed repeat edge [%1%-%2%]",
                                     edge.first % edge.second);
                }
                
#ifdef NONE_ONE_OF
                if (idxmap[edge.second].fromids_oneof_.size() > 0)
                {
                    for (size_t n = 0; 
                         n < idxmap[edge.second].fromids_oneof_.size(); n++)
                    {
                        int old_first = idxmap[edge.second].fromids_oneof_[n];
                        int first = idxmap[old_first].absid_;
                        for (size_t ei = idxmap[edge.second].eh_;
                             ei <= idxmap[edge.second].et_; ei++)
                        {
                            if (root_gd.edges_[ei].first < 0)
                            {
                                TRACE_NOTICE_FMT("change first = %1% -> %2%", 
                                                 root_gd.edges_[ei].first % first);
                                assert(root_gd.edges_[ei].first ==
                                   -(idxmap[edge.second].fromids_oneof_[0] + 1));
                                root_gd.edges_[ei].first = first;
                            }
                            assert(root_gd.edges_[ei].second >= 0);
                        }
                    }
                }
#else
                if (idxmap[edge.second].mbr_oneof_ >= 0)
                {
                    int first = idxmap[edge.first].absid_;
                    for (size_t ei = idxmap[edge.second].eh_;
                         ei <= idxmap[edge.second].et_; ei++)
                    {
                        if (root_gd.edges_[ei].first < 0)
                        {
                            TRACE_NOTICE_FMT("change first = %1% -> %2%", 
                                             root_gd.edges_[ei].first % first);
                            assert(root_gd.edges_[ei].first ==
                                   -(idxmap[edge.second].mbr_oneof_ + 1));
                            root_gd.edges_[ei].first = first;
                        }
                        assert(root_gd.edges_[ei].second >= 0);
                    }
                }
                //TRACE_NOTICE_FMT("ignore edge to ruleref [%1%-%2%]",
                //                 edge.first % edge.second);
#endif
            }
            else if (ty1st == node_base::_NULL || ty1st == node_base::ROOT)
            {   // NULL or END -->
                //TRACE_NOTICE_FMT("ignore null edge [%1%->%2%]",
                //                 edge.first % edge.second);
            }
            else if (ty2nd == node_base::ROOT)
            {   // link to end node
                //std::cout << "unlink edge: "; edge.dump(std::cout);
                if (! cleared)
                {
                    ending.clear();
                    cleared = true;
                }
                edge.first = idxmap[edge.first].absid_;
                edge.second = -1;
                ending.push_back(edge);  // require link end node
                //TRACE_NOTICE_FMT("pushed edge [%1%-%2%]",
                //                 edge.first % edge.second);
            }
            else if (ty2nd == node_base::_NULL)
            {   // --> NULL rule
                int k = 0;
                while ((idxmap[edge.second + k].type_ == node_base::_NULL) &&
                       ((edge.first + k) < rule_gd.nodes_.size()))
                    k++;
                assert(edge.second + k < rule_gd.nodes_.size());
                if (idxmap[edge.second + k].type_ == node_base::ROOT)
                {   // END
                    if (! cleared)
                    {
                        ending.clear();
                        cleared = true;
                    }
                    //edge.first = idxmap[edge.first];
                    edge.second = -1;
                    ending.push_back(edge);  // require link end node
                    //break;
                }
#ifdef NONE_ONE_OF
                else if (root_gd.nodes_[edge.second].from_oneof() == -1)
#else
                else if (root_gd.nodes_[edge.second].top_of_oneof() == -1)
#endif
                {   // sequential
                    edge.second = idxmap[edge.second + k].absid_;
                    edge.data_ = root_gd.nodes_[edge.second];
                    root_gd.edges_.push_back(edge);
                    //TRACE_NOTICE_FMT("re-link null edge [%1%->%2%]",
                    //                 edge.first % edge.second);
                }
            }
            else if (ty1st == node_base::UNKNOWN || ty2nd == node_base::UNKNOWN)
            {   // NULL or END -->
                TRACE_NOTICE_FMT("ignore unknown edge [%1%->%2%]",
                                 edge.first % edge.second);
            }
            else
            {
                edge.first = idxmap[edge.first].absid_;
                edge.second = idxmap[edge.second].absid_;
                root_gd.edges_.push_back(edge);
                //TRACE_NOTICE_FMT("pushed edge [%1%-%2%]",
                //                 edge.first % edge.second);
            }
        }
        root_gd.depth_--;
        return result;
    }

    //
    //
    //
    void dump(std::ostream& out) const
    {
        //TRACE_FUNCTION("srgs::grammar_visitor::dump");
        for (size_t i = 0; i < visitors_.size(); i++)
        {
            //TRACE_DEBUG_FMT("rule %1%: %2%", i % &(visitors_[i]));
            visitors_[i].dump(out);
        }
    }

//
private:
    void make_rule_visitor()
    {
        VISITOR visitor;
        visitors_.push_back(visitor);
        current_visitor_ = &(visitors_.back());
    }
//
protected:
    VISITOR * current_visitor_;
    std::vector<VISITOR> visitors_;

    // for compatibility test
    std::vector<std::string> str_;   // recognized words
    int current_;
};


}  // *namespace* - srgs

#endif
