/* $Id$ */

//=============================================================================
/**
 *  @file    grammar.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef SRGS_GRAMMAR_HPP
#define SRGS_GRAMMAR_HPP

#include <boost/tokenizer.hpp>
#include "bee/sequential_map.hpp"
#include "srgs/grammar_visitor.hpp"

namespace srgs
{

//
// manager of grammars
//
class grammar_manager
{
public:
    struct grampath
    {
        grampath(const std::string path, const grammar * gram) 
            : path_(path), grammar_(gram) {}
        grampath(const grampath& rhs) 
            : path_(rhs.path_), grammar_(rhs.grammar_) {}

        grampath& operator=(const grampath& rhs)
        {
            grampath tmp(rhs); 
            std::swap(path_, tmp.path_);
            std::swap(grammar_, tmp.grammar_);
            return *this;
        }
        bool operator<(const grampath& rhs) const
        {
            if (path_ < rhs.path_)
                return true;
            return false;
        }
           
        std::string path_;
        const grammar * grammar_;
    };

    typedef std::set<grampath> grammars;

    bool insert(const std::string& path, const grammar * gram)
    {
        std::pair<grammars::iterator, bool> result;
        result = grams_.insert(grampath(path, gram));
        return result.second;
    }
    
    const grammar * get(const std::string& path)
    {
        grammars::iterator iter = grams_.find(grampath(path, NULL));
        if (iter == grams_.end())
            return NULL;
        return iter->grammar_;
    }

    static grammar_manager * instance();

private:
    grammars grams_;
};

//
// grammar element
//
class grammar
{
public:
    enum { VOICE, DTMF };

    typedef bee::sequential_map<srgs::rule, std::string,
        std::less<std::string>, &srgs::rule::ext_id> rules;
    
    grammar(): lexicon_(), meta_(), metadata_(), tags_(), tag_format_(),
               version_(""), lang_(), base_(), root_(), mode_() {}

    grammar(const grammar& rhs): lexicon_(rhs.lexicon_), meta_(rhs.meta_),
                                 metadata_(rhs.metadata_), tags_(rhs.tags_),
                                 tag_format_(rhs.tag_format_),
                                 version_(rhs.version_), lang_(rhs.lang_),
                                 base_(rhs.base_), root_(rhs.root_),
                                 mode_(rhs.mode_)
    {
        rules::const_iterator iter = rhs.rules_.begin();
        for ( ; iter != rhs.rules_.end(); ++iter)
        {
            boost::shared_ptr<rule> dst = *iter;
            rules_.add(dst);
        } 
    }
    ~grammar() {}

    grammar& operator=(const grammar& rhs)
    {
        grammar tmp(rhs);
        lexicon_ = tmp.lexicon_;
        meta_ = tmp.meta_;
        metadata_ = tmp.metadata_;
        tags_ = tmp.tags_;
        tag_format_ = tmp.tag_format_;
        version_ = tmp.version_;
        lang_ = tmp.lang_;
        base_ = tmp.base_;
        root_ = tmp.root_;
        mode_ = tmp.mode_;
        std::swap(rules_, tmp.rules_);
        return *this;
    }
    
    // Get/Set
    const srgs::grammar::rules& get_rules() const { return rules_; }
    const srgs::lexicon& lexicon(unsigned int idx = 0) const
    {
        if (idx >= lexicon_.size())
            throw std::invalid_argument("srgs::grammar::lexicon: out of range error");
        return lexicon_[idx]; 
    }
    const srgs::meta& meta(unsigned int idx = 0) const
    {
        if (idx >= meta_.size())
            throw std::invalid_argument("srgs::grammar::meta: out of range error");
        return meta_[idx];
    }
    const std::string& metadata() const { return metadata_.value(); }
    const srgs::tag& tag(unsigned int idx = 0) const
    {
        if (idx >= tags_.size())
            throw std::invalid_argument("srgs::grammar::tag: out of range error");
        return *(tags_[idx]);
    }
    std::string tag_string() const
    {
        std::string result("");
        for (size_t i = 0; i < tags_.size(); i++)
        {
            if (i > 0)
                result += "\n";
            result += tags_[i]->str();
        }
        return result;
    }   
    const srgs::Tag_format_datatype& tag_format() const { return tag_format_; }
    const srgs::Version_datatype& version() const { return version_; }
    const xml_schema::lang& lang() const { return lang_; }
    const std::string& base() const      { return base_.name(); }
    const srgs::Root_datatype& root() const { return root_; }
    const srgs::Mode_datatype& mode() const { return mode_; }
    rules::pointer search(const std::string& id) const
    {
        return rules_.search(id);
    }
    rules::pointer search_root() const
    {
        return this->search(root_);
    }
    const srgs::rule * find(const std::string& id) const
    {
        srgs::rule * rp = rules_.find(id);
        return rp;
    }
    const srgs::rule * root_rule() const
    {
        return this->find(root_);
    }
    
    void name(const std::string& name) { name_ = name; }
    void lexicon(const srgs::lexicon& lexicon)
    {
        lexicon_.push_back(lexicon);
    }
    void meta(const srgs::meta& meta) { meta_.push_back(meta); }
    void metadata(const std::string& metadata) { metadata_.value(metadata); }
    void tag(const srgs::tag_ptr& tag)
    {
        tags_.push_back(tag.shared()); 
    }
    void tag_format(const srgs::Tag_format_datatype& tag_format)
    {
        tag_format_ = tag_format; 
    }
    void version(const srgs::Version_datatype& version) { version_ = version; }
    void lang(const xml_schema::lang& lang) { lang_ = lang; }
    void base(const std::string& base) { base_.name(base); }
    void root(const srgs::Root_datatype& root) { root_ = root; }
    void mode(const srgs::Mode_datatype& mode) { mode_ = mode; }
    const std::string& name() const { return name_; }
    bool is_main() const;

    void add(const shared_ptr<srgs::rule>& rule)
    {
        TRACE_FUNCTION("srgs::grammar::add");
        
        if (rule->id() == "NULL" || rule->id() == "VOID" || rule->id() == "GARBAGE")
        {
            TRACE_ERROR_FMT("rule id doesn't special rule name (%1%)", rule->id());
            throw srgs::exception("srgs::manager::add: special rule id");            
        }
        if (! rule->available())
        {
            TRACE_DEBUG_FMT("ignore <rule> element \"%1%\" (probabbly empty)",
                            rule->id());
            //return ;
        }
        int result = rules_.add(rule);
        if (result < 0)
        {
            TRACE_DEBUG_FMT("duplicate rule (id=\"%1%\")", rule->id());
            throw srgs::exception("srgs::manager::add: duplicate rule id");            
        }
        if (root_.empty())
            root_ = rule->id();  // first rule is root
    }
    
    template<typename VISITOR>
    void accept(grammar_visitor<VISITOR>& visitor)
    {
        //TRACE_FUNCTION("srgs::grammar::accept");
        std::string error_rule("");
        
        int root_position = rules_.position(root_);
        assert(root_position >= 0);
        rule_visitor * rv = visitor.create();
        rules_[root_position]->accept(*rv);
        if (rv->error())
        {
            //std::string msg = "visitor found rule(";
            //msg += rv->name();
            //msg += ") error.";
            //throw srgs::exception(msg);
            error_rule += " " + rv->name();
        }
        //     
        for (size_t i = 0; i < rules_.size(); i++)
        {
            if (i == root_position)
                continue;
            rv = visitor.create();
            rules_[i]->accept(*rv);
            if (rv->error())
            {
                //std::string msg = "visitor found rule(";
                //msg += rv->name();
                //msg += ") error.";
                //throw srgs::exception(msg);
                error_rule += " " + rv->name();
            }
        }
        //TRACE_DEBUG("end.");
        if (! error_rule.empty())
        {
            std::string msg = "visitor found rules(";
            msg += error_rule;
            msg += " ) error.";
            throw srgs::exception(msg);
        }
    }

    //
    // Dump srgs tree that are elements and attributes
    //
    void dump(std::ostream& out) const
    {
        out << "*** srgs grammar tree ***\n";
        for (size_t i = 0; i < rules_.size(); i++)
        {
            const shared_ptr<rule>& rule_elm = rules_[i];
            //out << "id: " << rule_elm->id() << "\n";
            rule_elm->dump(out);
        }
        out << "\n";
    }
    
private:
    std::string    name_;
    std::vector<srgs::lexicon>  lexicon_;
    std::vector<srgs::meta>  meta_;
    srgs::metadata metadata_;
    std::vector<srgs::shared_tag> tags_;
    srgs::Tag_format_datatype tag_format_;
    srgs::Version_datatype version_;
    xml_schema::lang    lang_;
    xml_schema::base    base_;
    srgs::Root_datatype root_;
    srgs::Mode_datatype mode_;

    rules    rules_;
}; 

}  // namespace

#endif /* SRGS_GRAMMAR_HPP */
