// $Id$
//
// Copyright (C) 2006 BEE Co.,Ltd.
//
// Author : Fukasawa Mitsuo
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
//

#include <iostream>

#include "srgs/srgs-parser.hxx"
#include "srgs/xml_schema-parser.hxx"
using namespace srgs;

using std::cerr;
using std::endl;

extern const std::string& getopt_tag_format();
extern const std::string& getopt_lang();

namespace srgs
{
    
grammar * make_grammar(const std::string& fileName)
{
    TRACE_FUNCTION("srgs::make_grammar");
	grammar * gram = NULL;
	
    try
    {
        // Construct the parser.
        //
        xml_schema::parser::string  string_p;
        xml_schema::parser::boolean boolean_p;
        xml_schema::parser::lang    lang_p;

        parser::Id_datatype Id_datatype_p;
        parser::Scope_datatype Scope_datatype_p;
        parser::Special_datatype Special_datatype_p;
        parser::Repeat_prob_datatype Repeat_prob_datatype_p;
        parser::Repeat_datatype Repeat_datatype_p;
        parser::Weight_datatype Weight_datatype_p;
        parser::Tag_format_datatype Tag_format_datatype_p;
        parser::Version_datatype Version_datatype_p;
        parser::Root_datatype Root_datatype_p;
        parser::Mode_datatype Mode_datatype_p;
        parser::Type_datatype Type_datatype_p;

        parser::tag tag_p;

        parser::ruleref ruleref_p;
        ruleref_p.parsers(Type_datatype_p,     // type
                          string_p,            // uri
                          Special_datatype_p); // special

        parser::metadata metadata_p;
        
        srgs::parser::token token_p;
        token_p.lang_parser(lang_p);
        
        parser::item item_p;

        srgs::parser::one_of one_of_p;
        one_of_p.parsers(item_p,
                         lang_p);

        item_p.parsers(token_p,
                       ruleref_p,
                       item_p,
                       one_of_p,
                       tag_p,
                       Repeat_prob_datatype_p,
                       Repeat_datatype_p,
                       Weight_datatype_p,
                       lang_p);
                            
        parser::rule rule_p;
        rule_p.parsers(token_p,
                       ruleref_p,
                       item_p,
                       one_of_p,
                       tag_p,
                       string_p,
                       Id_datatype_p,
                       Scope_datatype_p);

        srgs::parser::lexicon lexicon_p;
        lexicon_p.parsers(string_p,           // uni
                          Type_datatype_p);   // type

        srgs::parser::meta meta_p;
        meta_p.parsers(string_p,    // name
                       string_p,    // content
                       string_p);   // http_equiv

        parser::grammar grammar_p;
        grammar_p.parsers(rule_p,
                          lexicon_p,
                          meta_p,
                          string_p,
                          tag_p,
                          Tag_format_datatype_p,
                          Version_datatype_p,
                          lang_p,
                          string_p,        // base
                          Root_datatype_p,
                          Mode_datatype_p);


        // Parse the XML instance document.
        //
        xml_schema::parser::document<grammar *> doc_p(
            grammar_p,
            "http://www.w3.org/2001/06/grammar", // root element namespace
            "grammar");                          // root element name

        unsigned long flags = (xsd::cxx::parser::xerces::flags::dont_validate);

        //gram = new grammar(doc_p.parse(fileName, flags));
        gram = doc_p.parse(fileName, flags);

        assert(gram != NULL);
        gram->name(fileName);
        grammar_manager::instance()->insert(fileName, gram);

        // Check attributes of grammar
        std::string exception_msg("");
        if (gram->version() != "1.0")
        {
            TRACE_ERROR_FMT("version is not \"1.0\" (\"%1%\")",
                            gram->version());
            exception_msg += "grammar's version is not \"1.0\"\n";
        }       
        if ((! gram->tag_format().empty()) &&
            (gram->tag_format() != "bee-test/1.0") &&
            (gram->tag_format() != "semantics/1.0") &&
            (gram->tag_format() != getopt_tag_format()))
        {
            TRACE_ERROR_FMT("tag-format is not \"bee-test/1.0\" (\"%1%\")",
                            gram->tag_format());
            exception_msg += "grammar's tag-format is not \"bee-test/1.0\"\n";
        }       
        if ((gram->mode().size() > 0) &&
            (gram->mode() != "voice" && gram->mode() != "dtmf"))
        {
            TRACE_ERROR_FMT("mode is not \"voice\" (\"%1%\")", gram->mode());
            exception_msg += "grammar's mode is not \"voice\" or \"dtmf\"\n";
        }       
        if ((gram->lang().name() != "ja") && (gram->lang().name() != "en-US") &&
            (gram->lang().name() != "en") &&
            (gram->lang().name() != getopt_lang()))
        {
            TRACE_ERROR_FMT("lang is not \"ja\" (\"%1%\")", gram->lang().name());
            exception_msg += "grammar's lang is not \"ja\"\n";
        }       
        if (! exception_msg.empty())
        {
            throw srgs::exception(exception_msg);            
        }
        // check rule count
        if (gram->get_rules().empty())
        {
            TRACE_ERROR("Error: no rule !");
            throw srgs::exception("no rule in grammar");            
        }
        if (gram->find(gram->root()) == NULL)
        {
            TRACE_ERROR_FMT("root rule is not found (\"%1%\")", gram->root());
            throw srgs::exception("grammar's root is not found");            
        }
    }
    catch (const srgs::exception& e)
    {
        std::cerr << e.what() << std::endl;
        delete gram;
        return NULL;
    }
    catch (const xml_schema::parser::exception& e)
    {
        std::cerr << e << std::endl;
        delete gram;
        return NULL;
    }
    catch (const std::ios_base::failure&)
    {
        std::cerr << "io failure" << std::endl;
        delete gram;
        return NULL;
    }
    catch (const std::exception& e)
    {
        std::cerr << e.what() << std::endl;
        delete gram;
        return NULL;
    }
    return gram;
}

// initialize rule reference in root grammar
void init_ruleref()
{
    TRACE_FUNCTION("srgs::init_ruleref");
    ruleref_manager::instance()->clear();
}

// check rule reference in root grammar
void resolve_ruleref(grammar * main, const std::string& rootgram)
{
    TRACE_FUNCTION("srgs::resolve_ruleref");

    typedef srgs::grammar::rules::pointer shared_rule;

    size_t rule_count;
    std::vector<const ruleref *> rulerefs;
    if (ruleref_manager::instance()->getall(rulerefs) <= 0)
        return;
    do
    {
        rule_count = 0;
        rulerefs.clear();
        ruleref_manager::instance()->getall(rulerefs);

        bool undefined_rule = false;
        for (size_t i = 0; i < rulerefs.size(); i++)
        {
            const ruleref * ref = rulerefs[i];
            if (! ref->available())
            {
                throw srgs::exception("illegal ruleref is found");
                //continue;
            }
            if (ref->rule() != NULL)
            {   // resolved undefine rule
                rule_count++;
                continue;
            }

            std::string path("");
            if (! ref->uri().has_root_path())
                path += main->base();
            path += ref->uri().path();
            if (path.empty())
                path = rootgram;

            shared_rule theRule;
            std::string fragment = ref->uri().fragment();
            std::string ext_ruleid = (ref->uri().path().empty()) ?
                                     fragment : path + "#" + fragment;
            const grammar * gram = grammar_manager::instance()->get(path);
            if (gram == NULL)
            {
                TRACE_DEBUG_FMT("external grammar = '%1%'.", path);
                //throw srgs::exception("external grammar is found");
                std::string save_path = ruleref_manager::instance()->grammar();
                ruleref_manager::instance()->grammar(path);
                gram = make_grammar(path);  // extend grammar
                ruleref_manager::instance()->grammar(save_path);
                // gram->dump(std::cout);
                
                const srgs::grammar::rules& ext_rules = gram->get_rules();
                for (size_t j = 0; j < ext_rules.size(); j++)
                {
                    theRule = ext_rules[j];
                    ext_ruleid = path + "#" + theRule->id();
                    theRule->ext_id(ext_ruleid);
                    main->add(theRule);     // append external rule
                }
            }
            //
            if (main == gram)
            {
                theRule = main->search(ref->str());
                if (theRule.get() == NULL)
                {
                    ERROR_LOG_FMT("Error: main rule '%1%' is undefined.",
                                  ref->str());
                    undefined_rule = true; // delay exception
                    main->dump(std::cout);
                }
            }
            else
            {
                theRule = gram->search(fragment);
                if (theRule.get() == NULL)
                {
                    ERROR_LOG_FMT("Error: external rule '%1%' is undefined.",
                                  fragment);
                    undefined_rule = true; // delay exception
                    gram->dump(std::cout);
                }
            }
            const_cast<srgs::ruleref *>(ref)->rule(theRule);
        }
        if (undefined_rule)
        {
            throw srgs::exception("undefined ruleref is found");
        }
    } while (rule_count < rulerefs.size());    
}


//
//
//
static grammar_manager _grammar_manager;

grammar_manager * grammar_manager::instance()
{
    return &_grammar_manager;
}

bool grammar::is_main() const
{
    bool result = false;
    std::string path = getopt_infile();
    if (path == this->name())
        result = true;
    return result;
}

}  // namespace

