/* $Id$ */

//=============================================================================
/**
 *  @file    element.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef SRGS_ELEMENT_HPP
#define SRGS_ELEMENT_HPP

#include <boost/tokenizer.hpp>
#include <boost/algorithm/string/trim.hpp>

namespace srgs
{

typedef boost::tokenizer<boost::escaped_list_separator<char> > Tok;

// class prototype
class rule_visitor;
class element;

typedef boost::shared_ptr<srgs::element>  shared_elm;

//
// element base class
//
class element
{
public:

    enum
    { 
        UNKNOWN, RULE, ITEM, ONE_OF, RULEREF, TAG, TOKEN, START, END,
        HEAD, TAIL, N_ENUM
    };

    element(int type = UNKNOWN) 
        : num_(0), type_(type), rule_(""), relation_() {}
    element(const char * v, int type = UNKNOWN)
        : num_(0), type_(type), rule_(""), relation_()
    {
        value_.push_back(std::string(v)); 
    }
    element(const std::string& v, int type = UNKNOWN)
        : num_(0), type_(type), rule_(""), relation_() { value_.push_back(v); }
    element(const element& rhs)
        : value_(rhs.value_), num_(rhs.num_), type_(rhs.type_), rule_(rhs.rule_),
          relation_(rhs.relation_) {}
    virtual ~element() {}
    
    void swap(element& rhs)
    {
        std::swap(value_, rhs.value_);
        std::swap(num_, rhs.num_);
        std::swap(type_, rhs.type_);
        std::swap(rule_, rhs.rule_);
        relation_.swap(rhs.relation_);
    }
    
    element& operator=(const element& rhs)
    {
        if (this == &rhs)
        {
            value_ = rhs.value_;
            num_ = rhs.num_;
            type_ = rhs.type_;
            rule_ = rhs.rule_;
            relation_ = rhs.relation_;
        }
        return *this;
    }
    void clear()
    {
        value_.clear();
        num_ = 0;
        type_ = element::UNKNOWN;
        rule_ = "";
        relation_.reset();
    }
    
    // Get/Set
    const element * relation() const { return relation_.get(); }
    const std::string& rule() const { return rule_; }
    int   number() const { return num_; }
    int   type()   const { return type_; }
    void  number(int num) { num_ = num; }
    void  rule(const std::string& rule) { rule_ = rule; }
    void  relation(const shared_elm& relation) { relation_ = relation; }
    static char * type_names[];
    const char * type_name() const
    {
        TRACE_FUNCTION("element::type_name");
        if (type_ > element::N_ENUM)
        {
            TRACE_DEBUG("invalid type number(%d)", type_);
            return "";
        }
        return element::type_names[type_];
    }

    int size() const { return value_.size(); }
    const std::string& str_at(int n) const
    {
        assert(value_.size() > n);
        return value_[n]; 
    }

    virtual const std::string& str() const
    {
        return (value_.empty() ? bee::empty_string : value_[0]); 
    }
    virtual void  str(const std::string& v)
    {
        //if (value_.size() > 0 && v.size() > 0)
        //    value_ += " ";
        //value_ += v;
        if (v.size() > 0)
            value_.push_back(v);
    }
    virtual element * clone() const = 0;

    int compare(const std::vector<std::string>& words,
                int current = 0) const
    {
        TRACE_FUNCTION("srgs::element::compare");
        if (this->str()[0] == '"')
        {
            if (this->str().size() >= 2 && this->str()[1] == '"' &&
                words.size() <= current)
                return current;
            return compare2(words, current);
        }
        else
        {
            if (this->str().empty() && words.size() <= current)
                return current;
            return compare1(words, current);
        }
    }
    
    int tokens(std::ostream& out) const
    {
        int n = 0;
        boost::escaped_list_separator<char> sep('\\', ' ', '\"');
        Tok tok(this->str(), sep);      
        for (Tok::iterator beg = tok.begin(); beg != tok.end(); ++beg)
        {
            std::string token(*beg);
            boost::trim(token);
            if (token.empty())
                continue ;
            if (n > 0)
                out << ",";
            out << "\"" << token << "\"";
            n++;
        }
        return n;
    }

    virtual int  accept(rule_visitor& vis) = 0;
    virtual void dump(std::ostream& out, int ident = 0) const
    {
        out << (value_.empty() ? "" : value_[0]);
    }

//
private:
    int compare1(const std::vector<std::string>& words,
                 int current = 0) const
    {
        TRACE_FUNCTION("srgs::element::compare1");
        int n = current;
        bool equal = true;
        boost::escaped_list_separator<char> sep('\\', ' ', '\"');
        Tok tok(this->str(), sep);      
        for (Tok::iterator beg = tok.begin(); beg != tok.end(); ++beg)
        {
            std::string token(*beg);
            boost::trim(token);
            if (token.empty())
                continue ;
            if (words[n] != token)
            {
                equal = false;
                break;
            }
            n++;
        }
        if ((! equal) || (n == current))
        {
            //TRACE_DEBUG_FMT("not equal(%1%)", this->str());
            return -1;
        }
        //TRACE_DEBUG("equal(%d)", n);
        return n;
    }

    int compare2(const std::vector<std::string>& words,
                 int current = 0) const
    {
        TRACE_FUNCTION("srgs::element::compare2");
        
        std::string ws = this->str();
        boost::trim_if(ws, boost::is_any_of("\""));
        int n = current;
        bool equal = true;
        boost::escaped_list_separator<char> sep('\\', ' ', '\"');
        Tok tok(ws, sep);      
        for (Tok::iterator beg = tok.begin(); beg != tok.end(); ++beg)
        {
            std::string token(*beg);
            boost::trim(token);
            if (token.empty())
                continue ;
            if (words[n] != token)
            {
                equal = false;
                break;
            }
            n++;
        }
        if ((! equal) || (n == current))
        {
            //TRACE_DEBUG_FMT("not equal(%1%)", this->str());
            return -1;
        }
        //TRACE_DEBUG("equal(%d)", n);
        return n;
    }

//        
private:
    std::vector<std::string> value_;
    int         num_;
    int         type_;
    std::string rule_;
    shared_elm  relation_;  // <tag> link point
};
 
//
// root(dummy) element
//
class root_element: public srgs::element
{
public:
    root_element(int type) : element(type) {}
    root_element(const root_element& rhs) : element(rhs) {}
    virtual ~root_element() {}
    
    root_element& operator=(const root_element& rhs)
    {
        root_element tmp(rhs);
        this->element::swap(tmp);
        return *this;
    }
    
    virtual element * clone() const { return new root_element(*this); }
    virtual int accept(rule_visitor& vis)
    {
        return 0;  // can't accept to rule_visitor
    }
       
private:
};

}  // *namespace* - srgs

#endif /* SRGS_ELEMENT_HPP */
