/* $Id$ */

//=============================================================================
/**
 *  @file    trace.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2004-2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef BEE_TRACE_H
#define BEE_TRACE_H

#include "bee/beeconf.hpp"
#include <boost/format.hpp>

namespace bee
{
//
//
//
class BEE_Export trace
{
public:
    enum
    {
        L_MUST = 0, L_HIGH, L_VIEW, L_MIDDLE, L_LOW, L_CONSTRUCT, L_ALL,
        L_IGNORE,
    };

    trace(const BCHAR * fname, int lv = L_CONSTRUCT);
    trace(const BCHAR * funcname, const BCHAR * filename, int lnum,
          int lv = L_CONSTRUCT);
    ~trace();

    const BCHAR * function_name() { return function_; }

    // Trace error message
    void error(const BCHAR * fmt, ...);
    void error(const boost::format& fmt);

    // Trace degug message
    void debug(const BCHAR * fmt, ...);
    void debug(const boost::format& fmt);

    // Trace info message
    void notice(const BCHAR * fmt, ...);
    void notice(const boost::format& fmt);

    static void init(int level = L_MIDDLE, void * = 0);
    static int  level();
    static std::ostream& dos();

private:
    trace() : function_(NULL), file_(NULL), linenum_(0),
              level_(L_IGNORE) {}

    const BCHAR * function_;
    const BCHAR * file_;
    int   linenum_;
    int   level_;
};

} // *namespace* - bee

#ifdef _NO_DEBUG
#define TRACE_FUNCTION0()
#define TRACE_FUNCTION(f)
#define TRACE_FUNCTION2(l,f)
#define TRACE_FUNCTION_(f)
#define TRACE_ERROR(X)
#define TRACE_NOTICE(X)
#define TRACE_DEBUG(X)
#define ERROR_LOG(X)
#define DEBUG_LOG(X)
#define TRACE_ERROR_FMT(F,X)
#define TRACE_DEBUG_FMT(F,X)
#define TRACE_NOTICE_FMT(F,X)
#define ERROR_LOG_FMT(X,...)
#define DEBUG_LOG_FMT(X,...)
#else
#define TRACE_FUNCTION0()    bee::trace __func(__FUNCTION__)
#define TRACE_FUNCTION(f)    bee::trace __func(f)
#define TRACE_FUNCTION2(l,f) bee::trace __func((f), (l))
#define TRACE_FUNCTION_(f)   bee::trace __func(f, bee::trace::L_IGNORE)

#define TRACE_ERROR(F,...)  do { __func.error(F, ##__VA_ARGS__); } while(0)
#define TRACE_NOTICE(F,...) do { __func.notice(F, ##__VA_ARGS__); } while(0)
#define TRACE_DEBUG(F,...)  do { __func.debug(F, ##__VA_ARGS__); } while(0)
#define ERROR_LOG(F,...) do { bee::trace ___f(__FUNCTION__); \
                              ___f.error(F, ##__VA_ARGS__); } while(0)
#define DEBUG_LOG(F,...) do { bee::trace ___f(__FUNCTION__); \
                              ___f.debug(F, ##__VA_ARGS__); } while(0)

#ifndef ARG_COUNT_CHECK
#define TRACE_ERROR_FMT(F,X)  do { __func.error(boost::format(F) % X); } while(0)
#define TRACE_DEBUG_FMT(F,X)  do { __func.debug(boost::format(F) % X); } while(0)
#define TRACE_NOTICE_FMT(F,X) do { __func.notice(boost::format(F)% X); } while(0)
#define ERROR_LOG_FMT(F,X)    do { bee::trace ___f(__FUNCTION__); \
                                   ___f.error(boost::format(F) % X); } while(0)
#define DEBUG_LOG_FMT(F,X)    do { bee::trace ___f(__FUNCTION__); \
                                   ___f.debug(boost::format(F) % X); } while(0)
#else
#define TRACE_ERROR_FMT(F,X)  do {                 \
	    try {                                      \
		    __func.error(boost::format(F) % X);    \
        } catch (boost::io::too_many_args& ex) {   \
            std::cerr << ex.what() << " [" << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        } catch (boost::io::too_few_args& ex) {    \
            std::cerr << ex.what() << "[ " << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        }                                          \
    } while(0)

#define TRACE_DEBUG_FMT(F,X)  do {                 \
	    try {                                      \
		    __func.debug(boost::format(F) % X);    \
        } catch (boost::io::too_many_args& ex) {   \
            std::cerr << ex.what() << " [" << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        } catch (boost::io::too_few_args& ex) {    \
            std::cerr << ex.what() << "[ " << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        }                                          \
    } while(0)

#define TRACE_NOTICE_FMT(F,X)  do {                \
	    try {                                      \
		    __func.notice(boost::format(F) % X);   \
        } catch (boost::io::too_many_args& ex) {   \
            std::cerr << ex.what() << " [" << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        } catch (boost::io::too_few_args& ex) {    \
            std::cerr << ex.what() << "[ " << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        }                                          \
    } while(0)

#define ERROR_LOG_FMT(F,X)     do {                \
        try {                                      \
            bee::trace ___f(__FUNCTION__);         \
            ___f.error(boost::format(F) % X);      \
        } catch (boost::io::too_many_args& ex) {   \
            std::cerr << ex.what() << " [" << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        } catch (boost::io::too_few_args& ex) {    \
            std::cerr << ex.what() << "[ " << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        }                                          \
    } while(0)

#define DEBUG_LOG_FMT(F,X)     do {                \
        try {                                      \
            bee::trace ___f(__FUNCTION__);         \
            ___f.debug(boost::format(F) % X);      \
        } catch (boost::io::too_many_args& ex) {   \
            std::cerr << ex.what() << " [" << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        } catch (boost::io::too_few_args& ex) {    \
            std::cerr << ex.what() << "[ " << __FILE__ << ": " \
                      << __LINE__ << "]\n"; \
        }                                          \
    } while(0)
#endif

#endif  /* _NO_DEBUG */

#endif  // BEE_TRACE_H
