/* $Id$ */

//=============================================================================
/**
 *  @file    lexicon.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef PLS_LEXICON_HPP
#define PLS_LEXICON_HPP

#include "bee/trace.hpp"
#include "bee/sequential_map.hpp"

namespace pls
{

struct exception: std::runtime_error
{
	exception(const std::string& msg) : std::runtime_error(msg) {}

};

//
// <grapheme> Element
//
class grapheme
{
public:
    grapheme() : graph_(""), orthography_("") {}
    grapheme(const grapheme& rhs) : graph_(rhs.graph_), orthography_(rhs.orthography_) {}
    ~grapheme() {}

    grapheme& operator=(const grapheme& rhs)
    {
        grapheme tmp(rhs);
        std::swap(graph_, tmp.graph_);
        std::swap(orthography_, tmp.orthography_);
        return *this;
    }

    // Get/Set
    const std::string& graph() const { return graph_; }
    const std::string& orthography() const { return orthography_; }
    void graph(const std::string& graph) { graph_ = graph; }
    void orthography(const std::string& orthography) { orthography_ = orthography; }

    void dump(std::ostream& out) const
    {
        out << graph_ ;
    }

private:
    std::string graph_;
    std::string orthography_;
};

//
// <phoneme> Element
//
class phoneme
{
public:
    phoneme() : phone_(""), prefer_(false), alphabet_("") {}
    phoneme(const phoneme& rhs)
        : phone_(rhs.phone_), prefer_(rhs.prefer_), alphabet_(rhs.alphabet_) {}
    ~phoneme() {}

    phoneme& operator=(const phoneme& rhs)
    {
        phoneme tmp(rhs);
        std::swap(phone_, tmp.phone_);
        std::swap(prefer_, tmp.prefer_);
        std::swap(alphabet_, tmp.alphabet_);
        return *this;
    }

    // Get/Set
    const std::string& phone() const { return phone_; }
    const std::string& alphabet() const { return alphabet_; }
    bool prefer() const { return prefer_; }
    void phone(const std::string& phone) { phone_ = phone; }
    void alphabet(const std::string& alphabet) { alphabet_ = alphabet; }
    void prefer(bool prefer) { prefer_ = prefer; }

    //
    void dump(std::ostream& out) const
    {
        out << phone_ ;
    }

private:
    std::string phone_;
    bool        prefer_;
    std::string alphabet_;
};

//
// <alias> Element
//
class alias
{
public:
    alias() : alias_(""), prefer_(false) {}
    alias(const alias& rhs) : alias_(rhs.alias_), prefer_(rhs.prefer_) {}
    ~alias() {}

    alias& operator=(const alias& rhs)
    {
        alias tmp(rhs);
        std::swap(alias_, tmp.alias_);
        std::swap(prefer_, tmp.prefer_);
        return *this;
    }

    // Get/Set
    const std::string& get_alias() const { return alias_; }
    bool prefer() const { return prefer_; }
    void set_alias(const std::string& alias) { alias_ = alias; }
    void prefer(bool prefer) { prefer_ = prefer; }

    void dump(std::ostream& out) const
    {
        out << alias_;
    }

private:
    std::string alias_;
    bool        prefer_;
};

//
// <lexeme> Element
//
class lexeme
{
public:
    lexeme() : id_("") {}
    lexeme(const lexeme& rhs)
        : id_(rhs.id_), role_(rhs.role_), graphes_(rhs.graphes_), phones_(rhs.phones_),
          alias_(rhs.alias_), examples_(rhs.examples_) {}
    ~lexeme() {}

    lexeme& operator=(const lexeme& rhs)
    {
        lexeme tmp(rhs);
        std::swap(id_, tmp.id_);
        std::swap(role_, tmp.role_);
        std::swap(graphes_, tmp.graphes_);
        std::swap(phones_, tmp.phones_);
        std::swap(alias_, tmp.alias_);
        std::swap(examples_, tmp.examples_);
        return *this;
    }

    void clear()
    {
        id_.clear();
        role_.clear();
        graphes_.clear();
        phones_.clear();
        alias_.clear();
        examples_.clear();
    }

    // Get/Set
    const std::string& id() const { return id_; }
    const std::string& role(size_t idx = 0) const
    {
        if (idx >= role_.size())
        {
            throw pls::exception("out of range");
        }
        return role_[idx]; 
    }
    const pls::grapheme& grapheme(size_t idx = 0) const
    {
        if (idx >= graphes_.size())
        {
            throw pls::exception("out of range");
        }
        return graphes_[idx]; 
    }
    const pls::phoneme& phoneme(size_t idx = 0) const
    {
        if (idx >= phones_.size())
        {
            throw pls::exception("out of range");
        }
        return phones_[idx]; 
    }
    const pls::alias& alias(size_t idx = 0) const
    {
        if (idx >= alias_.size())
        {
            throw pls::exception("out of range");
        }
        return alias_[idx]; 
    }
    const std::string& example(size_t idx = 0) const
    {
        if (idx >= examples_.size())
        {
            throw pls::exception("out of range");
        }
        return examples_[idx]; 
    }
    void id(const std::string& id) { id_ = id; }
    void role(const std::string& role) { role_.push_back(role); }
    void grapheme(const pls::grapheme& graph) { graphes_.push_back(graph); }
    void phoneme(const pls::phoneme& phone) { phones_.push_back(phone); }
    void alias(const pls::alias& alias) { alias_.push_back(alias); }
    void example(const std::string& example) { examples_.push_back(example); }   

    void dump(std::ostream& out) const
    {
        size_t i;
        if (id_.size() > 0)
            out << "[" << id_ << "] ";
        for (i = 0; i < role_.size(); i++)
        {
            if (i > 0)
                out << ",";
            out << role_[i];
        }
        out << ":";

        for (i = 0; i < graphes_.size(); i++)
        {
            if (i > 0)
                out << ",";
            graphes_[i].dump(out);
        }
        out << ":";
        for (i = 0; i < phones_.size(); i++)
        {
            if (i > 0)
                out << ",";
            phones_[i].dump(out);
        }
        out << ":";
        
        for (i = 0; i < alias_.size(); i++)
        {
            if (i > 0)
                out << ",";
            alias_[i].dump(out);
        }
        out << ":";

        for (i = 0; i < examples_.size(); i++)
        {
            if (i > 0)
                out << ",";
            out << examples_[i];
        }
        out << "\n";
    }

private:
    std::string id_;
    std::vector<std::string>   role_;
    std::vector<pls::grapheme> graphes_;
    std::vector<pls::phoneme>  phones_;
    std::vector<pls::alias>    alias_;
    std::vector<std::string>   examples_;
};

//
// <meta> element
//
class meta
{
public:
    meta() : name_(""), content_(""), http_equiv_("") {}
  	meta(const meta& rhs) : name_(rhs.name_), content_(rhs.content_),
  	                        http_equiv_(rhs.http_equiv_) {}
    ~meta() {}
    
    meta& operator=(const meta& rhs)
    {
    	pls::meta tmp(rhs);
    	name_ = tmp.name_;
    	content_ = tmp.content_;
    	http_equiv_ = tmp.http_equiv_;
    	return *this;
    }

    void clear()
    {
    	name_ = "";
    	content_ = "";
    	http_equiv_ = "";
    }

    // Get/Set
    const std::string& name() const { return name_; }
    const std::string& content() const { return content_; }
    const std::string& http_equiv() const { return http_equiv_; }
    void name(const std::string& name) { name_ = name; }
    void content(const std::string& content) { content_ = content; }
    void http_equiv(const std::string& http_equiv) { http_equiv_ = http_equiv; }

private:
  	std::string name_;
  	std::string content_;
  	std::string http_equiv_;
};

//
// <lexicon> Element
//
class lexicon
{
public:
    //typedef std::vector<pls::lexeme> lexis;
    typedef bee::sequential_map<pls::lexeme, std::string,
        std::less<std::string>, &pls::lexeme::id> lexis;

    lexicon(const std::string& path) : path_(path) {}

    // Get/Set
    const std::string& path() const { return path_; }
    pls::lexeme& back()
    {
        size_t last = lexis_.size() - 1;
        return *(lexis_[last]);
    }

    void meta(const pls::meta& meta)
    {
        meta_.push_back(meta);
    }
    void lexeme(const pls::lexeme& lex)
    {
        boost::shared_ptr<pls::lexeme> tmp(new pls::lexeme(lex));
        lexis_.add(tmp);
    }

    const pls::meta& meta(size_t idx)
    {
        if (meta_.size() <= idx)
        {
            throw pls::exception("out of range");
        }
        return meta_[idx];
    }
    pls::lexeme& find(const std::string& id)
    {
        pls::lexeme * lexp = lexis_.find(id);
        return *lexp;
        //lexis::iterator iter = lexis_.find(id);
        //if (iter == lexis_.end())
        //    return lexicon::null;
        //return iter.second;
    }

    //
    void dump(std::ostream& out) const
    {
        size_t i;
        for (i = 0; i < lexis_.size(); i++)
        {
            lexis_[i]->dump(out);
        }
    }

private:
    std::string path_;
    std::vector<pls::meta> meta_;
    lexis lexis_;
};

} // namespace - lexicon

#endif
