/* $Id$ */

//=============================================================================
/**
 *  @file    lexicon.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================


#include <iostream>
#include <algorithm>
#include <string>
#include "pls-parser.hxx"
#include "lexicon.hpp"
#include <boost/python/class.hpp>
#include <boost/python/module.hpp>
#include <boost/python/def.hpp>
#include <boost/shared_ptr.hpp>

using namespace boost;
using std::cerr;
using std::endl;

typedef ::pls::parser::version    version_type;
typedef ::xml_schema::parser::uri base_type;
typedef ::pls::parser::alphabet1  alphabet_type;
typedef ::namespace_::lang        lang_type;

///////////////////////////////////////////////////////////////////////////////
//
template<typename T, typename CONT, typename MF>
void item_tree(const T& elm)
{
  const xercesc::DOMNode * pnode = elm._node();
  if (pnode)
  {
    for (xercesc::DOMNode * n(pnode->getFirstChild());
         n != 0;
         n = n->getNextSibling ())
    {
      if (n->getNodeType() == xercesc::DOMNode::TEXT_NODE)
      {
        std::cerr << xsd::cxx::xml::transcode<char>(n->getTextContent()) << std::endl;
        break;
      }
    }
  }
  CONT cont(elm.MF());
  std::cerr << "    container: " << *cont << std::endl;
}

///////////////////////////////////////////////////////////////////////////////
//
void init_lexicon(int argc, char* argv[])
{
    xercesc::XMLPlatformUtils::Initialize();
}

void fini_lexicon()
{
    xercesc::XMLPlatformUtils::Terminate();
}

namespace parser = pls::parser;

pls::lexicon * make_lexicon(const std::string& fname)
{
    
    typedef pls::parser::prefer      prefer_base;
    typedef prefer_base::_xsd_prefer prefer_type;
    
    pls::lexicon * lexicon = NULL;
    xsd::cxx::tree::flags flags = (xsd::cxx::tree::flags::dont_validate |
                                   xsd::cxx::tree::flags::keep_dom);
    try
    {
        std::auto_ptr<pls::parser::lexicon> lex(pls::parser::lexicon_(fname, flags));

        lexicon = new pls::lexicon(fname);

        for (parser::lexicon::meta::const_iterator m(lex->meta().begin());
             m != lex->meta().end();
             ++m)
        {
            pls::meta _meta;
            _meta.name(m->name().get());
            _meta.content(m->content());
            _meta.http_equiv(m->http_equiv().get());
            lexicon->meta(_meta);
        }
    
        version_type verson = lex->version();
        base_type base("");
        if (lex->base() != NULL)
            base = lex->base().get();
        alphabet_type alphabet = lex->alphabet();
        lang_type lang = lex->lang();

        for (parser::lexicon::lexeme::const_iterator iter(lex->lexeme().begin());
             iter != lex->lexeme().end();
             ++iter)
        {
            pls::lexeme lexeme;
            lexicon->lexeme(lexeme);

            typedef ::xml_schema::parser::id id_type;
            id_type id("");
            if (iter->id() != NULL)
                id = iter->id().get();
            lexicon->back().id(id);

            std::string rolestr("");
            if (iter->role() != NULL)
            {
                const pls::parser::role_type& roletype = iter->role().get();
                //const xercesc::DOMNode * pnode = roletype.front();
                const xercesc::DOMNode * pnode = roletype._node();
                rolestr = xsd::cxx::xml::transcode<char>(pnode->getTextContent());
            }
            lexicon->back().role(rolestr);

            for (parser::lexeme::grapheme::const_iterator g(iter->grapheme().begin());
    	         g != iter->grapheme().end();
    	         ++g)
            {
                pls::grapheme _graph;
                _graph.graph(*g);
                lexicon->back().grapheme(_graph);
            }
            for (parser::lexeme::phoneme::const_iterator p(iter->phoneme().begin());
    	         p != iter->phoneme().end();
    	         ++p)
            {
                pls::phoneme _phone;
            
                const xercesc::DOMNode * pnode = p->_node();
                _phone.phone(xsd::cxx::xml::transcode<char>(pnode->getTextContent()));

                alphabet_type alphabet("");
                if (p->alphabet() != NULL)
                    alphabet = p->alphabet().get();
                _phone.alphabet(static_cast<std::string&>(alphabet));

                prefer_type _prefer;
                if (p->prefer() != NULL)
                    _prefer = (prefer_type)p->prefer().get();
                _phone.prefer((_prefer == prefer_base::true_) ? true : false);
 
                lexicon->back().phoneme(_phone);
            }
            for (parser::lexeme::alias::const_iterator a(iter->alias().begin());
    	         a != iter->alias().end();
    	         ++a)
            {
                pls::alias _alias;
                const xercesc::DOMNode * pnode = a->_node();
                _alias.set_alias(xsd::cxx::xml::transcode<char>(pnode->getTextContent()));

                prefer_type _prefer;
                if (a->prefer() != NULL)
                    _prefer = (prefer_type)a->prefer().get();
                _alias.prefer((_prefer == prefer_base::true_) ? true : false);

                lexicon->back().alias(_alias);
            }
            for (parser::lexeme::example::const_iterator e(iter->example().begin());
    	         e != iter->example().end();
    	         ++e)
            {
                lexicon->back().example(*e);
            }
        }
    }
    catch (const xml_schema::parser::exception& e)
    {
        cerr << e << endl;
        return NULL;
    }
    return lexicon;
}

namespace pls
{

using namespace boost::python;

struct pls_wrapper
{
    boost::shared_ptr<pls::lexicon> lexicon_;

    pls_wrapper() {}
    pls_wrapper(const pls_wrapper& rhs) : lexicon_(rhs.lexicon_) {}
    ~pls_wrapper() {}

    pls_wrapper& operator=(const pls_wrapper& rhs)
    {
        if (this != &rhs)
            lexicon_ = rhs.lexicon_;
        return *this; 
    }

    std::string find(const std::string& id)
    {
        if (lexicon_.get() == NULL)
            return "";

        std::stringstream sout("");
        try
        {
            pls::lexeme& lex = lexicon_->find(id);
            lex.dump(sout);
        }
        catch (pls::exception ex)
        {
            return "";
        }
        return sout.str(); 
    }

    std::string dump()
    {
        if (lexicon_.get() == NULL)
            return "";
        std::stringstream sout;
        lexicon_->dump(sout); 
        return sout.str(); 
    }

    static void wrap()
    {
        class_<pls::pls_wrapper>("lexicon", init<>())
            .def("find", &pls::pls_wrapper::find)
            .def("dump", &pls::pls_wrapper::dump)
            ;
    }
};

pls_wrapper make_pls(const std::string& fname) 
{
    pls_wrapper result;

    try
    {
        result.lexicon_ = boost::shared_ptr<pls::lexicon>(make_lexicon(fname));
    }
    catch (const pls::exception& e)
    {
        std::cerr << e.what() << std::endl;
    }
    catch (const xml_schema::parser::exception& e)
    {
        std::cerr << e << std::endl;
    }
    catch (const std::exception& e)
    {
        std::cerr << e.what() << std::endl;
    }
           
    return result;
}


int init(/* const std::string& opt */) 
{
    //xml_schema::init();
    //bee::trace::init(getopt_trace());
    return 0;
}

void init_module()
{
    def("init", init);
    def("make_lexicon", make_pls);
    pls_wrapper::wrap();
}

} // *namespace* - pls

BOOST_PYTHON_MODULE(pls)
{
    pls::init_module();
}



