/* $Id$ */

//=============================================================================
/**
 *  @file    uri.hpp
 *
 *  @author Fukasawa Mitsuo
 *
 *    Copyright (C) 2006 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef BEE_URI_H
#define BEE_URI_H

#include "bee/beeconf.hpp"
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/algorithm/string.hpp>

namespace bee
{

class uri
{
public:
    enum { FILE, HTTP, FTP };

    uri();
    uri(const char * uri) { this->parse(uri); }
    uri(const std::string& uri) { this->parse(uri); }
    uri(const uri& rhs) : uri_(rhs.uri_), dir_(rhs.dir_), name_(rhs.name_),
                          query_(rhs.query_), fragment_(rhs.fragment_),
                          schema_(uri::FILE), path_(rhs.path_) {}
    ~uri() {}
    
    uri& operator=(const uri& rhs)
    {
        uri tmp(rhs);
        this->swap(tmp);
        return *this;   
    }
    void swap(uri& rhs)
    {
        std::swap(uri_, rhs.uri_);
        std::swap(dir_, rhs.dir_);
        std::swap(name_, rhs.name_);
        std::swap(query_, rhs.query_);
        std::swap(fragment_, rhs.fragment_);
        std::swap(schema_, rhs.schema_);
        std::swap(path_, rhs.path_);
    }

    // Get/Set
    const std::string& name() const { return name_; } 
    const std::string& dir() const { return dir_; } 
    const std::string& fragment() const { return fragment_; } 
    const std::string& path() const { return path_.string(); } 
    void  set(const std::string& uri)
    {
        this->parse(uri);
    }
    bool is_directory() const { return fs::is_directory(path_); }
    bool exists() const { return fs::exists(path_); }
 
private:
    void parse(const std::string& uri)
    {
        unsigned int p;

        std::string tmp = uri;
        uri_ = uri;
        path_ = fs::initial_path();
        fragment_ = "";
        dir_ = "";
        name_ = "";
        query_ = "";

        if ((p = uri_.find('#')) == 0)
        {   // "#fragment"
            fragment_ = uri_.substr(p+1);
            schema_ = uri::FILE;
            return ;
        }
        else if (p != std::string::npos)
        {
            fragment_ = uri_.substr(p+1);
            tmp = uri_.substr(0, p);
        }

        if ((p = tmp.find("://")) == std::string::npos)
        {
            path_ = tmp;
            schema_ = uri::FILE;
        }
        else
        {
            std::string schema_name(tmp.substr(0, p));
            boost::to_lower(schema_name);
            if (schema_name == "file")
            {
                path_ = tmp.substr(p+3); // purge "file://"
                schema_ = uri::FILE;
            }
            else if (schema_name == "http")
                schema_ = uri::HTTP;
            else if (schema_name == "ftp")
                schema_ = uri::FTP;
            else
            {
                std::cerr << "Illegal schema(" << uri_ << ")\n";
            }
        }
        if (schema_ == uri::FILE)
        {
            if (fs::exists(path_) && fs::is_directory(path_))
            {
                dir_ = path_.string();
            }
            else
            {
                dir_ = path_.branch_path().string();
                name_ = path_.leaf();
            }
        }
    }
    	
private:
    std::string uri_;
    std::string dir_;
    std::string name_;
    std::string query_;
    std::string fragment_;
    int         schema_;
    fs::path    path_;   
};


}  // namespace * bee

#endif  // BEE_URI_H
