<?php
/**
 *
 * @package XCube
 * @version $Id: Xdbase_Property.class.php,v 1.3 2010/02/12 04:36:02 bluemelon Exp $
 * @copyright Copyright 2005-2007 XOOPS Cube Project  <http://xoopscube.sourceforge.net/>
 * @license http://xoopscube.sourceforge.net/bsd_licenses.txt Modified BSD license
 *
 */

/**
 * @public
 * @brief [Abstract] Defines a interface for the property class group.
 * 
 * Xdbase_PropertyInterface is designed to work in Xdbase_ActionForm or Xdbase_Service (in the near future).
 * Therefore only sub-classes of them should call constructors of Xdbase_Property classes.
 */
class Xdbase_PropertyInterface
{
	/**
	 * @public
	 * @brief Constructor.
	 * @param $name string - A name of this property.
	 */
	function Xdbase_PropertyInterface($name)
	{
	}

	/**
	 * @public
	 * @brief [Abstract] Sets $value as raw value to this property. And the value is casted by the property's type'.
	 * @param $value mixed
	 */	
	function set($value)
	{
	}
	
	/**
	 * @public
	 * @brief [Abstract] Gets the value of this property.
	 * @return mixed
	 */
	function get()
	{
	}

	/**
	 * @deprecated
	 */	
	function setValue($arg0 = null, $arg1 = null)
	{
		$this->set($arg0, $arg1);
	}
	
	/**
	 * @deprecated
	 */	
	function getValue($arg0 = null)
	{
		return $this->get($arg0);
	}
	
	/**
	 * @public
	 * @brief [Abstract] Gets a value indicating whether this object expresses Array.
	 * @return bool
	 */
	function isArray()
	{
	}
	
	/**
	 * @public
	 * @brief [Abstract] Gets a value indicating whether this object is null.
	 * @return bool
	 */
	function isNull()
	{
	}
	
	/**
	 * @public
	 * @brief [Abstract] Gets a value as integer.
	 * @return int
	 */
	function toNumber()
	{
	}
	
	/**
	 * @public
	 * @brief [Abstract] Gets a value as string.
	 * @return string
	 */
	function toString()
	{
	}

	/**
	 * @public
	 * @brief [Abstract] Gets a value as encoded HTML code.
	 * @return string - HTML
	 * @deprecated
	 */	
	function toHTML()
	{
	}
	
	/**
	 * @public
	 * @brief [Abstract] Gets a value indicating whether this object has a fetch control.
	 * @return bool
	 */
	function hasFetchControl()
	{
	}

	/**
	 * @public [Abstract] Fetches values.
	 * @param $form Xdbase_ActionForm
	 * @return void
	 */	
	function fetch(&$form)
	{
	}
}

/**
 * @public
 * @brief [Abstract] The base class which implements Xdbase_PropertyInterface, for all properties.
 */
class Xdbase_AbstractProperty extends Xdbase_PropertyInterface
{
	/**
	 * @protected
	 * @brief string
	 */
	var $mName = null;
	
	/**
	 * @protected
	 * @brief string
	 */
	var $mValue = null;

	/**
	 * @public
	 * @brief Constructor.
	 * @param $name string - A name of this property.
	 */
	function Xdbase_AbstractProperty($name)
	{
		parent::Xdbase_PropertyInterface($name);
		$this->mName = $name;
	}
	
	/**
	 * @public
	 * @brief Sets $value as raw value to this property. And the value is casted by the property's type'.
	 * @param $value mixed
	 */	
	function set($value)
	{
		$this->mValue = $value;
	}
	
	/**
	 * @public
	 * @brief Gets the value of this property.
	 * @return mixed
	 */
	function get($index = null)
	{
		return $this->mValue;
	}
	
	/**
	 * @public
	 * @brief Gets a value indicating whether this object expresses Array.
	 * @return bool
	 * 
	 * @remarks
	 *     This class is a base class for none-array properties, so a sub-class of this
	 *     does not override this method.
	 */
	function isArray()
	{
		return false;
	}
	
	/**
	 * @public
	 * @brief Gets a value indicating whether this object is null.
	 * @return bool
	 */
	function isNull()
	{
		return (strlen(trim($this->mValue)) == 0);
	}
	
	/**
	 * @public
	 * @brief Gets a value as integer.
	 * @return int
	 */
	function toNumber()
	{
		return $this->mValue;
	}
	
	/**
	 * @public
	 * @brief Gets a value as string.
	 * @return string
	 */
	function toString()
	{
		return $this->mValue;
	}
	
	/**
	 * @public
	 * @brief Gets a value as encoded HTML code.
	 * @return string - HTML
	 * @deprecated
	 */	
	function toHTML()
	{
		return htmlspecialchars($this->toString(), ENT_QUOTES);
	}
	
	/**
	 * @public
	 * @brief Gets a value indicating whether this object has a fetch control.
	 * @return bool
	 */
	function hasFetchControl()
	{
		return false;
	}
}

/**
 * @public
 * @brief [Abstract] Defines common array property class which implements Xdbase_PropertyInterface.
 * 
 * This class is a kind of template-class --- Xdbase_GenericArrayProperty<T>.
 * Developers should know about sub-classes of Xdbase_AbstractProperty.
 */
class Xdbase_GenericArrayProperty extends Xdbase_PropertyInterface
{
	/**
	 * @protected
	 * @brief string
	 */
	var $mName = null;

	/**
	 * @protected
	 * @brief Xdbase_AbstractProperty[] - std::map<mixed_key, mixed_value>
	 */
	var $mProperties = array();
	
	/**
	 * @protected
	 * @brief string - <T>
	 * 
	 * If this class is Xdbase_GenericArrayProperty<T>, mPropertyClassName is <T>.
	 */
	var $mPropertyClassName = null;
	
	/**
	 * @public
	 * @brief Constructor.
	 * @param $classname string - <T>
	 * @param $name string - A name of the property.
	 */
	function Xdbase_GenericArrayProperty($classname, $name)
	{
		$this->mPropertyClassName = $classname;
		$this->mName = $name;
	}
	
	/**
	 * @public
	 * @brief Sets a value. And the value is casted by the property's type'.
	 * 
	 *   This member function has two signatures.
	 * 
	 * \par set(something[] values);
	 *    Fetches values from the array.
	 * 
	 * \par set(mixed key, mixed value);
	 *    Set values with index 'key'.
	 */
	function set($arg1, $arg2 = null)
	{
		if (is_array($arg1) && $arg2 == null) {
			$this->reset();
			foreach ($arg1 as $t_key => $t_value) {
				$this->_set($t_key, $t_value);
			}
		}
		elseif ($arg1 !== null && $arg2 !== null) {
			$this->_set($arg1, $arg2);
		}
	}
	
	/**
	 * @internal
	 * @todo Research this method.
	 */
	function add($arg1, $arg2 = null)
	{
		if (is_array($arg1) && $arg2 == null) {
			foreach ($arg1 as $t_key => $t_value) {
				$this->_set($t_key, $t_value);
			}
		}
		elseif ($arg1 !== null && $arg2 !== null) {
			$this->_set($arg1, $arg2);
		}
	}
	
	/**
	 * @private
	 * @brief This member function helps set().
	 * @param string $index
	 * @param mixed $value
	 * @return void
	 */
	function _set($index, $value)
	{
		if (!isset($this->mProperties[$index])) {
			$this->mProperties[$index] =& new $this->mPropertyClassName($this->mName);
		}
		$this->mProperties[$index]->set($value);
	}
	
	/**
	 * @public
	 * @brief Gets values of this property.
	 * @param $index mixed - If $indes is null, gets array (std::map<mixed_key, mixed_value>).
	 * @return mixed
	 */
	function get($index = null)
	{
		if ($index === null) {
			$ret = array();
			
			foreach ($this->mProperties as $t_key => $t_value) {
				$ret[$t_key] = $t_value->get();
			}
			
			return $ret;
		}
		
		return isset($this->mProperties[$index]) ? $this->mProperties[$index]->get() : null;
	}
	
	/**
	 * @protected
	 * @brief Resets all properties of this.
	 */
	function reset()
	{
		unset($this->mProperties);
		$this->mProperties = array();
	}
	
	/**
	 * @public
	 * @brief Gets a value indicating whether this object expresses Array.
	 * @return bool
	 * 
	 * @remarks
	 *     This class is a base class for array properties, so a sub-class of this
	 *     does not override this method.
	 */
	function isArray()
	{
		return true;
	}
	
	/**
	 * @public
	 * @brief Gets a value indicating whether this object is null.
	 * @return bool
	 */
	function isNull()
	{
		return (count($this->mProperties) == 0);
	}
	
	/**
	 * @public
	 * @brief Gets a value as integer --- but, gets null always.
	 * @return int
	 */
	function toNumber()
	{
		return null;
	}
	
	/**
	 * @public
	 * @brief Gets a value as string --- but, gets 'Array' always.
	 * @return string
	 */
	function toString()
	{
		return 'Array';
	}
	
	/**
	 * @public
	 * @brief Gets a value as encoded HTML code --- but, gets 'Array' always.
	 * @return string - HTML
	 * @deprecated
	 */	
	function toHTML()
	{
		return htmlspecialchars($this->toString(), ENT_QUOTES);
	}
	
	/**
	 * @public
	 * @brief Gets a value indicating whether this object has a fetch control.
	 * @return bool
	 */
	function hasFetchControl()
	{
		return false;
	}
}

/**
 * @internal
 * @deprecated
 */
class Xdbase_AbstractArrayProperty extends Xdbase_GenericArrayProperty
{
	function Xdbase_AbstractArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty($this->mPropertyClassName, $name);
	}
}

/**
 * @public
 * @brief Represents bool property. 
 */
class Xdbase_BoolProperty extends Xdbase_AbstractProperty
{
	function set($value)
	{
		if (strlen(trim($value)) > 0) {
			$this->mValue = (intval($value) > 0) ? 1 : 0;
		}
		else {
			$this->mValue = 0;
		}
	}
}

/**
 * @public
 * @brief Represents bool[] property. Xdbase_GenericArrayProperty<Xdbase_BoolProperty>.
 * @see Xdbase_BoolProperty
 */
class Xdbase_BoolArrayProperty extends Xdbase_GenericArrayProperty
{
	function Xdbase_BoolArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty("Xdbase_BoolProperty", $name);
	}
}

/**
 * @public
 * @brief Represents int property. 
 */
class Xdbase_IntProperty extends Xdbase_AbstractProperty
{
	function set($value)
	{
		if (strlen(trim($value)) > 0) {
			$this->mValue = intval($value);
		}
		else {
			$this->mValue = null;
		}
	}
}

/**
 * @public
 * @brief Represents int[] property. Xdbase_GenericArrayProperty<Xdbase_IntProperty>.
 * @see Xdbase_IntProperty
 */
class Xdbase_IntArrayProperty extends Xdbase_GenericArrayProperty
{
	function Xdbase_IntArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty("Xdbase_IntProperty", $name);
	}
}

/**
 * @public
 * @brief Represents float property. 
 */
class Xdbase_FloatProperty extends Xdbase_AbstractProperty
{
	function set($value)
	{
		if (strlen(trim($value)) > 0) {
			$this->mValue = floatval($value);
		}
		else {
			$this->mValue = null;
		}
	}
}

/**
 * @public
 * @brief Represents float[] property. Xdbase_GenericArrayProperty<Xdbase_FloatProperty>.
 * @see Xdbase_FloatProperty
 */
class Xdbase_FloatArrayProperty extends Xdbase_GenericArrayProperty
{
	function Xdbase_FloatArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty("Xdbase_FloatProperty", $name);
	}
}

/**
 * @public
 * @brief Represents string property. 
 * 
 * This class shows the property of string. Check whether a request includes control
 * code. If it does, stop own process.
 */
class Xdbase_StringProperty extends Xdbase_AbstractProperty
{
	function set($value)
	{
		// if (preg_match_all("/[\\x00-\\x1f]/", $value, $matches, PREG_PATTERN_ORDER)) {
		// 	die("Get control code :" . ord($matches[0][0]));
		// }
		
		$this->mValue = preg_replace("/[\\x00-\\x1f]/", '' , $value);
	}
	
	function toNumber()
	{
		return intval($this->mValue);
	}
}

/**
 * @public
 * @brief Represents string[] property. Xdbase_GenericArrayProperty<Xdbase_StringProperty>.
 * @see Xdbase_StringProperty
 */
class Xdbase_StringArrayProperty extends Xdbase_GenericArrayProperty
{
	function Xdbase_StringArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty("Xdbase_StringProperty", $name);
	}
}

/**
 * @public
 * @brief Represents string property which allows CR and LF.
 *  
 *  This class shows the property of text. Check whether a request includes control
 * code. If it does, stop own process.
 */
class Xdbase_TextProperty extends Xdbase_AbstractProperty
{
	function set($value)
	{
		$matches = array();
		
		// if (preg_match_all("/[\\x00-\\x08]|[\\x0b-\\x0c]|[\\x0e-\\x1f]/", $value, $matches,PREG_PATTERN_ORDER)) {
		// 	die("Get control code :" . ord($matches[0][0]));
		// }

		$this->mValue = preg_replace("/[\\x00-\\x08]|[\\x0b-\\x0c]|[\\x0e-\\x1f]/", '', $value);
	}
	
	function toNumber()
	{
		return intval($this->mValue);
	}
}

/**
 * @public
 * @brief Represents string[] property which allows CR and LF. Xdbase_GenericArrayProperty<Xdbase_TextProperty>.
 * @see Xdbase_TextProperty
 */
class Xdbase_TextArrayProperty extends Xdbase_GenericArrayProperty
{
	function Xdbase_TextArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty("Xdbase_TextProperty", $name);
	}
}

/**
 * @public
 * @brief Represents the special property which handles uploaded file.
 * @see Xdbase_FormFile
 */
class Xdbase_FileProperty extends Xdbase_AbstractProperty
{
	/**
	 * @protected
	 * @brief mixed - ID for Xdbase_FileArrayProperty.
	 * 
	 * friend Xdbase_FileArrayProperty;
	 */
	var $mIndex = null;
	
	function Xdbase_FileProperty($name)
	{
		parent::Xdbase_AbstractProperty($name);
		$this->mValue =& new Xdbase_FormFile($name);
	}
	
	function hasFetchControl()
	{
		return true;
	}
	
	function fetch(&$form)
	{
		if (!is_object($this->mValue)) {
			return false;
		}
		
		if ($this->mIndex !== null) {
			$this->mValue->mKey = $this->mIndex;
		}
		
		$this->mValue->fetch();
		
		if (!$this->mValue->hasUploadFile()) {
			$this->mValue = null;
		}
	}
	
	function isNull()
	{
		if (!is_object($this->mValue)) {
			return true;
		}
		
		return !$this->mValue->hasUploadFile();
	}
	
	function toString()
	{
		return null;
	}
	
	function toNumber()
	{
		return null;
	}
}

/**
 * @public
 * @brief Represents the special property[] which handles uploaded file. Xdbase_GenericArrayProperty<Xdbase_FileProperty>.
 * @see Xdbase_FileProperty
 */
class Xdbase_FileArrayProperty extends Xdbase_GenericArrayProperty
{
	function Xdbase_FileArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty("Xdbase_FileProperty", $name);
	}
	
	function hasFetchControl()
	{
		return true;
	}
	
	function fetch(&$form)
	{
		unset($this->mProperties);
		$this->mProperties = array();
		if (isset($_FILES[$this->mName]) && is_array($_FILES[$this->mName]['name'])) {
			foreach ($_FILES[$this->mName]['name'] as $_key => $_val) {
				$this->mProperties[$_key] =& new $this->mPropertyClassName($this->mName);
				$this->mProperties[$_key]->mIndex = $_key;
				$this->mProperties[$_key]->fetch($form);
			}
		}
	}
}

/**
 * @public
 * @brief This is extended Xdbase_FileProperty and limits uploaded files by image files.
 * @see Xdbase_FormImageFile
 */
class Xdbase_ImageFileProperty extends Xdbase_FileProperty
{
	function Xdbase_ImageFileProperty($name)
	{
		parent::Xdbase_AbstractProperty($name);
		$this->mValue =& new Xdbase_FormImageFile($name);
	}
}

/**
 * @public
 * @brief  Xdbase_GenericArrayProperty<Xdbase_ImageFileProperty>.
 * @see Xdbase_ImageFileProperty
 */
class Xdbase_ImageFileArrayProperty extends Xdbase_FileArrayProperty
{
	function Xdbase_ImageFileArrayProperty($name)
	{
		parent::Xdbase_GenericArrayProperty("Xdbase_ImageFileProperty", $name);
	}
}

?>
