
require 'wx'
include Wx

# menu items
CLIENT_QUIT = 1000
CLIENT_ABOUT = 1002
CLIENT_OPEN = 1003
CLIENT_TEST1 = 1004
CLIENT_TEST2 = 1005
CLIENT_TEST3 = 1006
CLIENT_CLOSE = 1007
CLIENT_TESTURL = 1008
CLIENT_DGRAM = 1009

# id for socket
SOCKET_ID = 1010

class MyFrame < WxFrame
    def initialize
        super(nil, -1,
                     "WxSocket demo: Client",
                     WxDefaultPosition, WxSize.new(300, 200))

        # Give the frame an icon
        SetIcon(WxIcon.new("mondrian.xpm"))

        # Make menus
        @m_menuFile = WxMenu.new
        @m_menuFile.Append(CLIENT_ABOUT, "&About...\tCtrl-A", "Show about dialog")
        @m_menuFile.AppendSeparator()
        @m_menuFile.Append(CLIENT_QUIT, "E&xit\tAlt-X", "Quit client")

        @m_menuSocket = WxMenu.new
        @m_menuSocket.Append(CLIENT_OPEN, "&Open session", "Connect to server")
        @m_menuSocket.AppendSeparator()
        @m_menuSocket.Append(CLIENT_TEST1, "Test &1", "Test basic functionality")
        @m_menuSocket.Append(CLIENT_TEST2, "Test &2", "Test ReadMsg and WriteMsg")
        @m_menuSocket.Append(CLIENT_TEST3, "Test &3", "Test large data transfer")
        @m_menuSocket.AppendSeparator()
        @m_menuSocket.Append(CLIENT_CLOSE, "&Close session", "Close connection")

        @m_menuDatagramSocket = WxMenu.new
        @m_menuDatagramSocket.Append(CLIENT_DGRAM, "Send Datagram", "Test UDP sockets")

        @m_menuProtocols = WxMenu.new
        @m_menuProtocols.Append(CLIENT_TESTURL, "Test URL", "Get data from the specified URL")

        # Append menus to the menubar
        @m_menuBar = WxMenuBar.new
        @m_menuBar.Append(@m_menuFile, "&File")
        @m_menuBar.Append(@m_menuSocket, "&SocketClient")
        @m_menuBar.Append(@m_menuDatagramSocket, "&DatagramSocket")
        @m_menuBar.Append(@m_menuProtocols, "&Protocols")
        SetMenuBar(@m_menuBar)

        # Status bar
        CreateStatusBar(2)

        # Make a textctrl for logging
        @m_text  = WxTextCtrl.new(self, -1,
                               "Welcome to WxSocket demo: Client\nClient ready\n",
                               WxDefaultPosition, WxDefaultSize,
                               WxTE_MULTILINE | WxTE_READONLY)

        # Create the socket
        @m_sock = WxSocketClient.new

        # Setup the event handler and subscribe to most events
        @m_sock.SetEventHandler(self, SOCKET_ID)
        @m_sock.SetNotify(WxSOCKET_CONNECTION_FLAG |
                        WxSOCKET_INPUT_FLAG |
                        WxSOCKET_LOST_FLAG)
        @m_sock.Notify(TRUE)

        @m_busy = FALSE
        UpdateStatusBar()

        EVT_MENU(self,CLIENT_QUIT,     "OnQuit")
        EVT_MENU(self,CLIENT_ABOUT,    "OnAbout")
        EVT_MENU(self,CLIENT_OPEN,     "OnOpenConnection")
        EVT_MENU(self,CLIENT_TEST1,    "OnTest1")
        EVT_MENU(self,CLIENT_TEST2,    "OnTest2")
        EVT_MENU(self,CLIENT_TEST3,    "OnTest3")
        EVT_MENU(self,CLIENT_CLOSE,    "OnCloseConnection")
        EVT_MENU(self,CLIENT_DGRAM,    "OnDatagram")
        EVT_MENU(self,CLIENT_TESTURL,  "OnTestURL")
        EVT_SOCKET(self,SOCKET_ID,     "OnSocketEvent")

    end

    def OnQuit(event)
      # TRUE is to force the frame to close
      Close(TRUE)
    end

    def OnAbout(event)
      WxMessageBox("WxSocket demo: Client\n(c) 1999 Guillermo Rodriguez Garcia\n",
                   "About Client",
                   WxOK | WxICON_INFORMATION, self)
    end

    def OnOpenConnection(event)

      @m_menuSocket.Enable(CLIENT_OPEN, FALSE)
      @m_menuSocket.Enable(CLIENT_CLOSE, FALSE)

      # Ask user for server address
      hostname = WxGetTextFromUser(
        "Enter the address of the WxSocket demo server:",
        "Connect ...",
        "localhost")

      # Mini-tutorial for Connect() :-)
      # ---------------------------
      #
      # There are two ways to use Connect(): blocking and non-blocking,
      # depending on the value passed as the 'wait' (2nd) parameter.
      #
      # Connect(addr, TRUE) will wait until the connection completes,
      # returning TRUE on success and FALSE on failure. This call blocks
      # the GUI (self might be changed in future releases to honour the
      # WxSOCKET_BLOCK flag).
      #
      # Connect(addr, FALSE) will issue a nonblocking connection request
      # and return immediately. If the return value is TRUE, then the
      # connection has been already succesfully established. If it is
      # FALSE, you must wait for the request to complete, either with
      # WaitOnConnect() or by watching WxSOCKET_CONNECTION / LOST
      # events (please read the documentation).
      #
      # WaitOnConnect() itself never blocks the GUI (self might change
      # in the future to honour the WxSOCKET_BLOCK flag). This call will
      # return FALSE on timeout, or TRUE if the connection request
      # completes, which in turn might mean:
      #
      #   a) That the connection was successfully established
      #   b) That the connection request failed (for example, because
      #      it was refused by the peer.
      #
      # Use IsConnected() to distinguish between these two.
      #
      # So, in a brief, you should do one of the following things:
      #
      # For blocking Connect:
      #
      #   success = client.Connect(addr, TRUE)
      #
      # For nonblocking Connect:
      #
      #   client.Connect(addr, FALSE)
      #
      #   waitmore = TRUE
      #   while (! client.WaitOnConnect(seconds, millis) && waitmore )
      #
      #     # possibly give some feedback to the user,
      #     # update waitmore if needed.
      #   end
      #   success = client.IsConnected()
      #
      # And that's all :-)

      @m_text.AppendText("\nTrying to connect (timeout = 10 sec) ...\n")
      @m_sock.Connect(hostname,3000, FALSE)
      @m_sock.WaitOnConnect(10)

      if @m_sock.IsConnected()
        @m_text.AppendText("Succeeded ! Connection established\n")
      else
        @m_sock.Close()
        @m_text.AppendText("Failed ! Unable to connect\n")
        WxMessageBox("Can't connect to the specified host", "Alert !")
      end

      UpdateStatusBar()
    end

    def OnTest1(event)

      # Disable socket menu entries (exception: Close Session)
      @m_busy = TRUE
      UpdateStatusBar()

      @m_text.AppendText("\n=== Test 1 begins ===\n")

      # Tell the server which test we are running
      c = 0xBE.chr
      @m_sock.Write(c, 1)

      # Send some data and read it back. We know the size of the
      # buffer, so we can specify the exact number of bytes to be
      # sent or received and use the WxSOCKET_WAITALL flag. Also,
      # we have disabled menu entries which could interfere with
      # the test, so we can safely avoid the WxSOCKET_BLOCK flag.
      #
      # First we send a byte with the length of the string, then
      # we send the string itself (do NOT try to send any integral
      # value larger than a byte "as is" across the network, or
      # you might be in trouble! Ever heard about big and little
      # endian computers?)

      @m_sock.SetFlags(WxSOCKET_WAITALL)

      buf1 = "Test string (less than 256 chars!)"
      len = buf1.length
      buf2 = "\0" * len

      @m_text.AppendText("Sending a test buffer to the server ...")
      len2 = [len].pack("C")
      @m_sock.Write(len2, 1)
      @m_sock.Write(buf1, len)
      @m_text.AppendText(@m_sock.Error() ? "failed !\n" : "done\n")

      @m_text.AppendText("Receiving the buffer back from server ...")
      @m_sock.Read(buf2, len)
      @m_text.AppendText(@m_sock.Error() ? "failed !\n" : "done\n")

      @m_text.AppendText("Comparing the two buffers ...")
      if buf1[0,len] != buf2[0,len]
        @m_text.AppendText("failed!\n")
        @m_text.AppendText("Test 1 failed !\n")
      else
        @m_text.AppendText("done\n")
        @m_text.AppendText("Test 1 passed !\n")
      end
      @m_text.AppendText("=== Test 1 ends ===\n")

      @m_busy = FALSE
      UpdateStatusBar()
    end

    def OnTest2(event)
      # Disable socket menu entries (exception: Close Session)
      @m_busy = TRUE
      UpdateStatusBar()

      @m_text.AppendText("\n=== Test 2 begins ===\n")

      # Tell the server which test we are running
      c = 0xCE.chr
      @m_sock.Write(c, 1)

      # Here we use ReadMsg and WriteMsg to send messages with
      # a header with size information. Also, the reception is
      # event triggered, so we test input events as well.
      #
      # We need to set no flags here (ReadMsg and WriteMsg are
      # not affected by flags)

      @m_sock.SetFlags(WxSOCKET_WAITALL)

      s = WxGetTextFromUser(
        "Enter an arbitrary string to send to the server:",
        "Test 2 ...",
        "Yes I like WxWindows!")

      msg1 = s
      len = msg1.length
      msg2 = "\0" * len

      @m_text.AppendText("Sending the string with WriteMsg ...")
      @m_sock.WriteMsg(msg1, len)
      @m_text.AppendText(@m_sock.Error() ? "failed !\n" : "done\n")
      @m_text.AppendText("Waiting for an event (timeout = 2 sec)\n")

      # Wait until data available (will also return if the connection is lost)
      @m_sock.WaitForRead(2)

      if @m_sock.IsData()
        @m_text.AppendText("Reading the string back with ReadMsg ...")
        @m_sock.ReadMsg(msg2, len)
        @m_text.AppendText(@m_sock.Error() ? "failed !\n" : "done\n")
        @m_text.AppendText("Comparing the two buffers ...")
        if msg1[0,len] != msg2[0,len]
          @m_text.AppendText("failed!\n")
          @m_text.AppendText("Test 2 failed !\n")
        else
          @m_text.AppendText("done\n")
          @m_text.AppendText("Test 2 passed !\n")
        end
      else
        @m_text.AppendText("Timeout ! Test 2 failed.\n")
      end
      @m_text.AppendText("=== Test 2 ends ===\n")

      @m_busy = FALSE
      UpdateStatusBar()
    end

    def OnTest3(event)
      # Disable socket menu entries (exception: Close Session)
      @m_busy = TRUE
      UpdateStatusBar()

      @m_text.AppendText("\n=== Test 3 begins ===\n")

      # Tell the server which test we are running
      c = 0xDE.chr
      @m_sock.Write(c, 1)

      # This test also is similar to the first one but it sends a
      # large buffer so that WxSocket is actually forced to split
      # it into pieces and take care of sending everything before
      # returning.

      @m_sock.SetFlags(WxSOCKET_WAITALL)

      # Note that len is in kbytes here!
      len  = 32
      buf1 = "\0" * (len * 1024)
      buf2 = "\0" * (len * 1024)

      for i in 0 ... (len * 1024)
         buf1[i] = i % 256
      end

      @m_text.AppendText("Sending a large buffer (32K) to the server ...")
      len2 = [len].pack("C")
      @m_sock.Write(len2, 1)
      @m_sock.Write(buf1, len * 1024)
      @m_text.AppendText(@m_sock.Error() ? "failed !\n" : "done\n")

      @m_text.AppendText("Receiving the buffer back from server ...")
      @m_sock.Read(buf2, len * 1024)
      @m_text.AppendText(@m_sock.Error() ? "failed !\n" : "done\n")

      @m_text.AppendText("Comparing the two buffers ...")
      if buf1[0,len] != buf2[0,len]
        @m_text.AppendText("failed!\n")
        @m_text.AppendText("Test 3 failed !\n")
      else
        @m_text.AppendText("done\n")
        @m_text.AppendText("Test 3 passed !\n")
      end
      @m_text.AppendText("=== Test 3 ends ===\n")

      @m_busy = FALSE
      UpdateStatusBar()
    end

    def OnCloseConnection(event)
      @m_sock.Close()
      UpdateStatusBar()
    end

    def OnDatagram(event)
      @m_text.AppendText("\n=== Datagram test begins ===\n")
      @m_text.AppendText("Sorry, not implemented\n")
      @m_text.AppendText("=== Datagram test ends ===\n")
    end

    def OnTestURL(event)
      # Note that we are creating a new socket here, so self
      # won't mess with the client/server demo.

      # Ask for the URL
      @m_text.AppendText("\n=== URL test begins ===\n")
      urlname = WxGetTextFromUser("Enter an URL to get",
                                           "URL:",
                                           "http://localhost")

      # Parse the URL
      url = WxURL.new(urlname)
      if url.GetError() != WxURL_NOERR
        @m_text.AppendText("Error: couldn't parse URL\n")
        @m_text.AppendText("=== URL test ends ===\n")
        return nil
      end

      # Try to get the input stream (connects to the given URL)
      @m_text.AppendText("Trying to establish connection...\n")
      WxYield()
      data = url.GetInputStream()
      if !data
        @m_text.AppendText("Error: couldn't read from URL\n")
        @m_text.AppendText("=== URL test ends ===\n")
        return nil
      end

      # Print the contents type and file size
      s = sprintf("Contents type: %s\nStarting to download...\n",
                 url.GetProtocol().GetContentType())
      @m_text.AppendText(s)
      WxYield()

      # Get the data
      sout = WxFileOutputStream.new("test.url")
      if !sout.Ok()
        @m_text.AppendText("Error: couldn't open file for output\n")
        @m_text.AppendText("=== URL test ends ===\n")
        return nil
      end

      data.Read(sout)
      @m_text.AppendText("Results written to file: test.url\n")
      @m_text.AppendText("Done.\n")
      @m_text.AppendText("=== URL test ends ===\n")

      data.free
    end

    def OnSocketEvent(event)
      s = "OnSocketEvent: "

      case event.GetSocketEvent()
        when WxSOCKET_INPUT
            s << "WxSOCKET_INPUT\n"
        when WxSOCKET_LOST
            s << "WxSOCKET_LOST\n"
        when WxSOCKET_CONNECTION
            s << "WxSOCKET_CONNECTION\n"
        else
            s << "Unexpected event !\n"
      end

      @m_text.AppendText(s)
      UpdateStatusBar()
    end

    # convenience functions
    def UpdateStatusBar()
      s = ""
      if !@m_sock.IsConnected()
        s = "Not connected"
      else
        hostname,service = @m_sock.GetPeer()
        s = sprintf("%s : %d", hostname,service)
      end

      SetStatusText(s, 1)

      @m_menuSocket.Enable(CLIENT_OPEN, !@m_sock.IsConnected() && !@m_busy)
      @m_menuSocket.Enable(CLIENT_TEST1, @m_sock.IsConnected() && !@m_busy)
      @m_menuSocket.Enable(CLIENT_TEST2, @m_sock.IsConnected() && !@m_busy)
      @m_menuSocket.Enable(CLIENT_TEST3, @m_sock.IsConnected() && !@m_busy)
      @m_menuSocket.Enable(CLIENT_CLOSE, @m_sock.IsConnected())
    end

end

class MyApp < WxApp
    def OnInit
        # Create the main application window
        frame = MyFrame.new

        # Show it and tell the application that it's our main window
        frame.Show(TRUE)
        SetTopWindow(frame)
    end
end

a = MyApp.new
a.MainLoop()

