
require 'wx'
include Wx

# IDs for the menu commands
LIST_ABOUT,
LIST_QUIT,

LIST_LIST_VIEW,
LIST_ICON_VIEW,
LIST_ICON_TEXT_VIEW,
LIST_SMALL_ICON_VIEW,
LIST_SMALL_ICON_TEXT_VIEW,
LIST_REPORT_VIEW,
LIST_VIRTUAL_VIEW,

LIST_DESELECT_ALL,
LIST_SELECT_ALL,
LIST_DELETE_ALL,
LIST_DELETE,
LIST_ADD,
LIST_EDIT,
LIST_SORT,
LIST_SET_FG_COL,
LIST_SET_BG_COL,
LIST_TOGGLE_MULTI_SEL,
LIST_TOGGLE_FIRST,
LIST_SHOW_COL_INFO,
LIST_SHOW_SEL_INFO,
LIST_FOCUS_LAST,
LIST_FREEZE,
LIST_THAW = (0 .. 24).to_a

LIST_CTRL = 1000

# number of items in list/report view
NUM_ITEMS = 30

# number of items in icon/small icon view
NUM_ICONS = 9

class MyListCtrl < WxListCtrl
    def initialize(parent,id,pos,size,style)
        super(parent, id, pos, size, style)
        @m_attr = WxListItemAttr.new(WxBLUE, WxLIGHT_GREY, WxNullFont)

    EVT_LIST_BEGIN_DRAG(self,LIST_CTRL, "OnBeginDrag")
    EVT_LIST_BEGIN_RDRAG(self,LIST_CTRL, "OnBeginRDrag")
    EVT_LIST_BEGIN_LABEL_EDIT(self,LIST_CTRL, "OnBeginLabelEdit")
    EVT_LIST_END_LABEL_EDIT(self,LIST_CTRL, "OnEndLabelEdit")
    EVT_LIST_DELETE_ITEM(self,LIST_CTRL, "OnDeleteItem")
    EVT_LIST_DELETE_ALL_ITEMS(self,LIST_CTRL, "OnDeleteAllItems")
    EVT_LIST_GET_INFO(self,LIST_CTRL, "OnGetInfo")
    EVT_LIST_SET_INFO(self,LIST_CTRL, "OnSetInfo")
    EVT_LIST_ITEM_SELECTED(self,LIST_CTRL, "OnSelected")
    EVT_LIST_ITEM_DESELECTED(self,LIST_CTRL, "OnDeselected")
    EVT_LIST_KEY_DOWN(self,LIST_CTRL, "OnListKeyDown")
    EVT_LIST_ITEM_ACTIVATED(self,LIST_CTRL, "OnActivated")
    EVT_LIST_ITEM_FOCUSED(self,LIST_CTRL, "OnFocused")

    EVT_LIST_COL_CLICK(self,LIST_CTRL, "OnColClick")
    EVT_LIST_COL_RIGHT_CLICK(self,LIST_CTRL, "OnColRightClick")
    EVT_LIST_COL_BEGIN_DRAG(self,LIST_CTRL, "OnColBeginDrag")
    EVT_LIST_COL_DRAGGING(self,LIST_CTRL, "OnColDragging")
    EVT_LIST_COL_END_DRAG(self,LIST_CTRL, "OnColEndDrag")

    EVT_LIST_CACHE_HINT(self,LIST_CTRL, "OnCacheHint")

    EVT_CHAR(self,"OnChar")

    end

    def OnCacheHint(event)
        WxLogMessage( "OnCacheHint: cache items %d..%d",
                      event.GetCacheFrom(), event.GetCacheTo() )
    end

    def SetColumnImage(col,image)
        item = WxListItem.new
        item.SetMask(WxLIST_MASK_IMAGE)
        item.SetImage(image)
        SetColumn(col, item)
    end

    def OnColClick(event)
        col = event.GetColumn()
        SetColumnImage(col, 0)

        WxLogMessage( "OnColumnClick at %d.", col )
    end

    def OnColRightClick(event)
        col = event.GetColumn()
        if col != -1
            SetColumnImage(col, -1)
        end

        WxLogMessage( "OnColumnRightClick at %d.", event.GetColumn() )
    end

    def LogColEvent(event,name)
        col = event.GetColumn()

        WxLogMessage("%s: column %d (width = %d or %d).",
                     name,
                     col,
                     event.GetItem().GetWidth(),
                     GetColumnWidth(col))
    end

    def OnColBeginDrag(event)
        LogColEvent( event, "OnColBeginDrag" )
    end

    def OnColDragging(event)
        LogColEvent( event, "OnColDragging" )
    end

    def OnColEndDrag(event)
        LogColEvent( event, "OnColEndDrag" )
    end

    def OnBeginDrag(event)
        pt = event.GetPoint()
        i,flags = HitTest(pt)
        WxLogMessage( "OnBeginDrag at (%d, %d), item %d.",
                      pt.x, pt.y, i )
    end

    def OnBeginRDrag(event)
        WxLogMessage( "OnBeginRDrag at %d,%d.",
                      event.GetPoint().x, event.GetPoint().y )
    end

    def OnBeginLabelEdit(event)
        WxLogMessage( "OnBeginLabelEdit: %s", event.GetText())
    end

    def OnEndLabelEdit(event)
        WxLogMessage( "OnEndLabelEdit: %s", event.GetText())
    end

    def OnDeleteItem(event)
        LogEvent(event, "OnDeleteItem")
    end

    def OnDeleteAllItems(event)
        LogEvent(event, "OnDeleteAllItems")
    end

    def OnGetInfo(event)
        msg = ""
        msg << "OnGetInfo (" << event.GetItem().GetId() << ", " << event.GetItem().GetColumn() << ""
        if (event.GetMask() & WxLIST_MASK_STATE ) != 0
            msg << " WxLIST_MASK_STATE"
        end
        if ( event.GetMask() & WxLIST_MASK_TEXT ) != 0
            msg << " WxLIST_MASK_TEXT"
        end
        if ( event.GetMask() & WxLIST_MASK_IMAGE ) != 0
            msg << " WxLIST_MASK_IMAGE"
        end
        if ( event.GetMask() & WxLIST_MASK_DATA ) != 0
            msg << " WxLIST_MASK_DATA"
        end
        if ( event.GetMask() & WxLIST_SET_ITEM ) != 0
            msg << " WxLIST_SET_ITEM"
        end
        if ( event.GetMask() & WxLIST_MASK_WIDTH ) != 0
            msg << " WxLIST_MASK_WIDTH"
        end
        if ( event.GetMask() & WxLIST_MASK_FORMAT ) != 0
            msg << " WxLIST_MASK_WIDTH"
        end

        if ( event.GetMask() & WxLIST_MASK_TEXT ) != 0
            event.m_item.m_text = "My callback text"
        end

        WxLogMessage(msg)
    end

    def OnSetInfo(event)
        LogEvent(event, "OnSetInfo")
    end

    def OnSelected(event)
        LogEvent(event, "OnSelected")

        if ( GetWindowStyle() & WxLC_REPORT ) != 0
            info = WxListItem.new
            info.SetId(event.GetIndex())
            info.SetColumn(1)
            info.SetMask(WxLIST_MASK_TEXT)
            if GetItem(info)
                WxLogMessage("Value of the 2nd field of the selected item: %s",
                             info.GetText())
            else
                WxLogMessage("WxListCtrl::GetItem() failed")
            end
            info.free
        end
    end

    def OnDeselected(event)
        LogEvent(event, "OnDeselected")
    end

    def OnActivated(event)
        LogEvent(event, "OnActivated")
    end

    def OnFocused(event)
        LogEvent(event, "OnFocused")

        event.Skip()
    end

    def OnListKeyDown(event)
        case event.GetKeyCode()
            when ?c,?C
                    info = WxListItem.new
                    info.SetId(event.GetIndex())
                    GetItem(info)

                    attr = info.GetAttributes()
                    if !attr || !attr.HasTextColour()
                        info.SetTextColour(WxCYAN)

                        SetItem(info)

                        RefreshItem(info.GetId())
                    end
                    info.free
            when ?n,?N
                    item = GetNextItem(-1,
                                            WxLIST_NEXT_ALL, WxLIST_STATE_FOCUSED)
                    if item == GetItemCount() - 1
                        item = 0
                    else
                        item += 1
                    end

                    WxLogMessage("Focusing item %d", item)

                    SetItemState(item, WxLIST_STATE_FOCUSED, WxLIST_STATE_FOCUSED)
                    EnsureVisible(item)

            when WXK_DELETE
                    item = GetNextItem(-1,
                                            WxLIST_NEXT_ALL, WxLIST_STATE_SELECTED)
                    while item != -1
                        DeleteItem(item)

                        WxLogMessage("Item %d deleted", item)

                        # -1 because the indices were shifted by DeleteItem()
                        item = GetNextItem(item - 1,
                                           WxLIST_NEXT_ALL, WxLIST_STATE_SELECTED)
                    end
            when WXK_INSERT
                if ( GetWindowStyle() & WxLC_REPORT ) != 0
                    if ( GetWindowStyle() & WxLC_VIRTUAL ) != 0
                        SetItemCount(GetItemCount() + 1)
                    else # !virtual
                        InsertItemInReportView(event.GetIndex())
                    end
                end
            else
                LogEvent(event, "OnListKeyDown")
                event.Skip()
        end
    end

    def OnChar(event)
        WxLogMessage("Got char event.")

        case event.GetKeyCode()
            when ?n,?N,?c,?C
                ;
            else
                event.Skip()
        end
    end

    def LogEvent(event,eventName)
        WxLogMessage("Item %d: %s (item text = %s, data = %d)",
                     event.GetIndex(), eventName,
                     event.GetText(), event.GetData())
    end

    def OnGetItemText(item,column)
        return sprintf("Column %d of item %d", column, item)
    end

    def OnGetItemImage(item)
        return 0
    end

    def OnGetItemAttr(item)
        return ((item % 2) != 0 ? nil : @m_attr)
    end

    def InsertItemInReportView(i)
        buf = sprintf("This is item %d", i)
        tmp = InsertItem(i, buf, 0)
        SetItemData(tmp, i)

        buf = sprintf("Col 1, item %d", i)
        SetItem(i, 1, buf)

        buf = sprintf("Item %d in column 2", i)
        SetItem(i, 2, buf)
    end

end


# Define a new frame type
class MyFrame < WxFrame

    def initialize(title, x, y, w, h)
        super(nil, -1, title, WxPoint.new(x, y), WxSize.new(w, h))

        @m_listCtrl = nil
        @m_logWindow = nil

        # Give it an icon
        SetIcon( WxIcon.new("mondrian.xpm") )

        # Make an image list containing large icons
        @m_imageListNormal = WxImageList.new(32, 32, TRUE)
        @m_imageListSmall = WxImageList.new(16, 16, TRUE)

        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/toolbrai.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/toolchar.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/tooldata.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/toolnote.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/tooltodo.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/toolchec.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/toolgame.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/tooltime.xpm" ) )
        @m_imageListNormal.Add( WxBitmap.new( "bitmaps/toolword.xpm" ) )

        @m_imageListSmall.Add( WxBitmap.new( "bitmaps/small1.xpm") )

        # Make a menubar
        menuFile = WxMenu.new
        menuFile.Append(LIST_ABOUT, "&About")
        menuFile.AppendSeparator()
        menuFile.Append(LIST_QUIT, "E&xit\tAlt-X")

        menuView = WxMenu.new
        menuView.Append(LIST_LIST_VIEW, "&List view\tF1")
        menuView.Append(LIST_REPORT_VIEW, "&Report view\tF2")
        menuView.Append(LIST_ICON_VIEW, "&Icon view\tF3")
        menuView.Append(LIST_ICON_TEXT_VIEW, "Icon view with &text\tF4")
        menuView.Append(LIST_SMALL_ICON_VIEW, "&Small icon view\tF5")
        menuView.Append(LIST_SMALL_ICON_TEXT_VIEW, "Small icon &view with text\tF6")
        menuView.Append(LIST_VIRTUAL_VIEW, "Virtual view\tF7")

        menuList = WxMenu.new
        menuList.Append(LIST_FOCUS_LAST, "&Make last item current\tCtrl-L")
        menuList.Append(LIST_TOGGLE_FIRST, "To&ggle first item\tCtrl-G")
        menuList.Append(LIST_DESELECT_ALL, "&Deselect All\tCtrl-D")
        menuList.Append(LIST_SELECT_ALL, "S&elect All\tCtrl-A")
        menuList.AppendSeparator()
        menuList.Append(LIST_SHOW_COL_INFO, "Show &column info\tCtrl-C")
        menuList.Append(LIST_SHOW_SEL_INFO, "Show &selected items\tCtrl-S")
        menuList.AppendSeparator()
        menuList.Append(LIST_SORT, "&Sort\tCtrl-S")
        menuList.AppendSeparator()
        menuList.Append(LIST_ADD, "&Append an item\tCtrl-P")
        menuList.Append(LIST_EDIT, "&Edit the item\tCtrl-E")
        menuList.Append(LIST_DELETE, "&Delete first item\tCtrl-X")
        menuList.Append(LIST_DELETE_ALL, "Delete &all items")
        menuList.AppendSeparator()
        menuList.Append(LIST_FREEZE, "Free&ze\tCtrl-Z")
        menuList.Append(LIST_THAW, "Tha&w\tCtrl-W")
        menuList.AppendSeparator()
        menuList.Append(LIST_TOGGLE_MULTI_SEL, "&Multiple selection\tCtrl-M",
                "Toggle multiple selection", WxITEM_CHECK)

        menuCol = WxMenu.new
        menuCol.Append(LIST_SET_FG_COL, "&Foreground colour...")
        menuCol.Append(LIST_SET_BG_COL, "&Background colour...")

        menubar = WxMenuBar.new
        menubar.Append(menuFile, "&File")
        menubar.Append(menuView, "&View")
        menubar.Append(menuList, "&List")
        menubar.Append(menuCol, "&Colour")
        SetMenuBar(menubar)

        @m_panel = WxPanel.new(self, -1)
        @m_logWindow = WxTextCtrl.new(@m_panel, -1, WxEmptyString,
                                     WxDefaultPosition, WxDefaultSize,
                                     WxTE_MULTILINE | WxSUNKEN_BORDER)

        @m_logOld = WxLog::SetActiveTarget(WxLogTextCtrl.new(@m_logWindow))

        RecreateList(WxLC_REPORT | WxLC_SINGLE_SEL)

        CreateStatusBar(3)

        EVT_SIZE(self,"OnSize")

        EVT_MENU(self,LIST_QUIT, "OnQuit")
        EVT_MENU(self,LIST_ABOUT, "OnAbout")
        EVT_MENU(self,LIST_LIST_VIEW, "OnListView")
        EVT_MENU(self,LIST_REPORT_VIEW, "OnReportView")
        EVT_MENU(self,LIST_ICON_VIEW, "OnIconView")
        EVT_MENU(self,LIST_ICON_TEXT_VIEW, "OnIconTextView")
        EVT_MENU(self,LIST_SMALL_ICON_VIEW, "OnSmallIconView")
        EVT_MENU(self,LIST_SMALL_ICON_TEXT_VIEW, "OnSmallIconTextView")
        EVT_MENU(self,LIST_VIRTUAL_VIEW, "OnVirtualView")

        EVT_MENU(self,LIST_FOCUS_LAST, "OnFocusLast")
        EVT_MENU(self,LIST_TOGGLE_FIRST, "OnToggleFirstSel")
        EVT_MENU(self,LIST_DESELECT_ALL, "OnDeselectAll")
        EVT_MENU(self,LIST_SELECT_ALL, "OnSelectAll")
        EVT_MENU(self,LIST_DELETE, "OnDelete")
        EVT_MENU(self,LIST_ADD, "OnAdd")
        EVT_MENU(self,LIST_EDIT, "OnEdit")
        EVT_MENU(self,LIST_DELETE_ALL, "OnDeleteAll")
        EVT_MENU(self,LIST_SORT, "OnSort")
        EVT_MENU(self,LIST_SET_FG_COL, "OnSetFgColour")
        EVT_MENU(self,LIST_SET_BG_COL, "OnSetBgColour")
        EVT_MENU(self,LIST_TOGGLE_MULTI_SEL, "OnToggleMultiSel")
        EVT_MENU(self,LIST_SHOW_COL_INFO, "OnShowColInfo")
        EVT_MENU(self,LIST_SHOW_SEL_INFO, "OnShowSelInfo")
        EVT_MENU(self,LIST_FREEZE, "OnFreeze")
        EVT_MENU(self,LIST_THAW, "OnThaw")

        EVT_UPDATE_UI(self,LIST_SHOW_COL_INFO, "OnUpdateShowColInfo")

	EVT_CLOSE(self,"OnClose")

    end

    def OnSize(event=nil)
        if !@m_logWindow
            return Qnil
        end

        size = GetClientSize()
        y = (2*size.y)/3
        @m_listCtrl.SetSize(0, 0, size.x, y)
        @m_logWindow.SetSize(0, y + 1, size.x, size.y - y)

        event.Skip() if event
    end

    def OnClose(event)
    	WxLog::SetActiveTarget(@m_logOld)
        Destroy()
    end

    def OnQuit(event)
        Close(TRUE)
    end

    def OnAbout(event)
        dialog = WxMessageDialog.new(self, "List test sample\nJulian Smart (c) 1997",
                "About list test", WxOK|WxCANCEL)

        dialog.ShowModal()
    end

    def OnFreeze(event)
        WxLogMessage("Freezing the control")
        @m_listCtrl.Freeze()
    end

    def OnThaw(event)
        WxLogMessage("Thawing the control")

        @m_listCtrl.Thaw()
    end

    def OnFocusLast(event)
        index = @m_listCtrl.GetItemCount() - 1
        if index == -1
            return Qnil
        end

        @m_listCtrl.SetItemState(index, WxLIST_STATE_FOCUSED, WxLIST_STATE_FOCUSED)
        @m_listCtrl.EnsureVisible(index)
    end

    def OnToggleFirstSel(event)
        @m_listCtrl.SetItemState(0, WxLIST_STATE_SELECTED, WxLIST_STATE_SELECTED)
    end

    def OnDeselectAll(event)
        n = @m_listCtrl.GetItemCount()
        for i in 0 ... n
            @m_listCtrl.SetItemState(i,0,WxLIST_STATE_SELECTED)
        end
    end

    def OnSelectAll(event)
        n = @m_listCtrl.GetItemCount()
        for i in 0 ... n
            @m_listCtrl.SetItemState(i,WxLIST_STATE_SELECTED, WxLIST_STATE_SELECTED)
        end
    end

    # ----------------------------------------------------------------------------
    # changing listctrl modes
    # ----------------------------------------------------------------------------

    def RecreateList(flags,withText = TRUE)

        # we could avoid recreating it if we don't set/clear the WxLC_VIRTUAL
        # style, but it is more trouble to do it than not

        @m_listCtrl.free if @m_listCtrl

        @m_listCtrl = MyListCtrl.new(@m_panel, LIST_CTRL,
                                    WxDefaultPosition, WxDefaultSize,
                                    flags |
                                    WxSUNKEN_BORDER | WxLC_EDIT_LABELS)

        case flags & WxLC_MASK_TYPE
            when WxLC_LIST
                InitWithListItems()
            when WxLC_ICON
                InitWithIconItems(withText)
            when WxLC_SMALL_ICON
                InitWithIconItems(withText, TRUE)
            when WxLC_REPORT
                if (flags & WxLC_VIRTUAL) != 0
                    InitWithVirtualItems()
                else
                    InitWithReportItems()
                end
            else
                WxLogMessage( "unknown listctrl mode" )
        end

        OnSize(nil)

        @m_logWindow.Clear()
    end

    def OnListView(event)
        RecreateList(WxLC_LIST)
    end

    def InitWithListItems()
        for i in 0 ... NUM_ITEMS
            @m_listCtrl.InsertItem(i, sprintf("Item %d", i))
        end
    end

    def OnReportView(event)
        RecreateList(WxLC_REPORT)
    end

    def InitWithReportItems()
        @m_listCtrl.SetImageList(@m_imageListSmall, WxIMAGE_LIST_SMALL)

        # under MSW for SetColumnWidth() to work we need to create the items with
        # images initially
        itemCol = WxListItem.new
        itemCol.SetMask(WxLIST_MASK_TEXT | WxLIST_MASK_IMAGE)
        itemCol.SetText("Column 1")
        itemCol.SetImage(-1)
        @m_listCtrl.InsertColumn(0, itemCol)
        itemCol.SetText("Column 2")
        @m_listCtrl.InsertColumn(1, itemCol)
        itemCol.SetText("Column 3")
        @m_listCtrl.InsertColumn(2, itemCol)

#        m_listCtrl.InsertColumn(0, "Column 1") # , WxLIST_FORMAT_LEFT, 140)
#        m_listCtrl.InsertColumn(1, "Column 2") # , WxLIST_FORMAT_LEFT, 140)
#        m_listCtrl.InsertColumn(2, "One More Column (2)") # , WxLIST_FORMAT_LEFT, 140)

        # to speed up inserting we hide the control temporarily
        @m_listCtrl.Hide()

        sw = WxStopWatch.new

        for i in 0 ... NUM_ITEMS
            @m_listCtrl.InsertItemInReportView(i)
        end

        @m_logWindow.WriteText(sprintf("%d items inserted in %dms\n",
                                                NUM_ITEMS, sw.Time()))
        sw.free
        @m_listCtrl.Show()

        # we leave all mask fields to 0 and only change the colour
        item = WxListItem.new
        item.SetId(0)
        item.SetTextColour(WxRED)
        @m_listCtrl.SetItem( item )

        item.SetId(2)
        item.SetTextColour(WxGREEN)
        @m_listCtrl.SetItem( item )
        item.SetId(4)
        item.SetTextColour(WxLIGHT_GREY)
        item.SetFont(WxITALIC_FONT)
        item.SetBackgroundColour(WxRED)
        @m_listCtrl.SetItem( item )

        @m_listCtrl.SetTextColour(WxBLUE)
        @m_listCtrl.SetBackgroundColour(WxLIGHT_GREY)

        @m_listCtrl.SetColumnWidth( 0, WxLIST_AUTOSIZE )
        @m_listCtrl.SetColumnWidth( 1, WxLIST_AUTOSIZE )
        @m_listCtrl.SetColumnWidth( 2, WxLIST_AUTOSIZE )
    end

    def InitWithIconItems(withText,sameIcon = FALSE)
        @m_listCtrl.SetImageList(@m_imageListNormal, WxIMAGE_LIST_NORMAL)
        @m_listCtrl.SetImageList(@m_imageListSmall, WxIMAGE_LIST_SMALL)

        for i in 0 ... NUM_ICONS
            image = sameIcon ? 0 : i

            if withText
                @m_listCtrl.InsertItem(i, sprintf("Label %d", i),
                                       image)
            else
                @m_listCtrl.InsertItem(i, image)
            end
        end
    end

    def OnIconView(event)
        RecreateList(WxLC_ICON, FALSE)
    end

    def OnIconTextView(event)
        RecreateList(WxLC_ICON)
    end

    def OnSmallIconView(event)
        RecreateList(WxLC_SMALL_ICON, FALSE)
    end

    def OnSmallIconTextView(event)
        RecreateList(WxLC_SMALL_ICON)
    end

    def OnVirtualView(event)
        RecreateList(WxLC_REPORT | WxLC_VIRTUAL)
    end

    def InitWithVirtualItems()
        @m_listCtrl.SetImageList(@m_imageListSmall, WxIMAGE_LIST_SMALL)

        @m_listCtrl.InsertColumn(0, "First Column")
        @m_listCtrl.InsertColumn(1, "Second Column")
        @m_listCtrl.SetColumnWidth(0, 150)
        @m_listCtrl.SetColumnWidth(1, 150)

        @m_listCtrl.SetItemCount(1000000)
    end

    def MyCompareFunction(item1,item2)

        # inverse the order
        return -1 if item1 < item2
        return 1 if item1 > item2
        return 0
    end

    def OnSort(event)
        sw = WxStopWatch.new
        @m_listCtrl.SortItems(self,"MyCompareFunction")

        @m_logWindow.WriteText(sprintf("Sorting %d items took %d ms\n",
                                                @m_listCtrl.GetItemCount(),
                                                sw.Time()))
        sw.free
    end

    def OnShowSelInfo(event)
        selCount = @m_listCtrl.GetSelectedItemCount()
        WxLogMessage("%d items selected:", selCount)

        # don't show too many items
        shownCount = 0

        item = @m_listCtrl.GetNextItem(-1, WxLIST_NEXT_ALL,
                                            WxLIST_STATE_SELECTED)
        while item != -1
            WxLogMessage("\t%d (%s)",
                         item, @m_listCtrl.GetItemText(item))

            shownCount += 1
            if shownCount > 10
                WxLogMessage("\t... more selected items snipped...")
                break
            end

            item = @m_listCtrl.GetNextItem(item, WxLIST_NEXT_ALL,
                                           WxLIST_STATE_SELECTED)
        end
    end

    def OnShowColInfo(event)
        count = @m_listCtrl.GetColumnCount()
        WxLogMessage("%d columns:", count)
        for c in 0 ... count
            WxLogMessage("\tcolumn %d has width %d", c,
                         @m_listCtrl.GetColumnWidth(c))
        end
    end

    def OnUpdateShowColInfo(event)
        event.Enable( (@m_listCtrl.GetWindowStyleFlag() & WxLC_REPORT) != 0 )
    end

    def OnToggleMultiSel(event)
        flags = @m_listCtrl.GetWindowStyleFlag()
        if (flags & WxLC_SINGLE_SEL) != 0
            flags &= ~WxLC_SINGLE_SEL
        else
            flags |= WxLC_SINGLE_SEL
        end

        @m_logWindow.WriteText(sprintf("Current selection mode: %sle\n",
                               (flags & WxLC_SINGLE_SEL) != 0 ? "sing" : "multip"))

        RecreateList(flags)
    end

    def OnSetFgColour(event)
        @m_listCtrl.SetForegroundColour(WxGetColourFromUser(self))
        @m_listCtrl.Refresh()
    end

    def OnSetBgColour(event)
        @m_listCtrl.SetBackgroundColour(WxGetColourFromUser(self))
        @m_listCtrl.Refresh()
    end

    def OnAdd(event)
        @m_listCtrl.InsertItem(@m_listCtrl.GetItemCount(), "Appended item")
    end

    def OnEdit(event)
        itemCur = @m_listCtrl.GetNextItem(-1, WxLIST_NEXT_ALL,
                                               WxLIST_STATE_FOCUSED)

        if itemCur != -1
            @m_listCtrl.EditLabel(itemCur)
        else
            @m_logWindow.WriteText("No item to edit")
        end
    end

    def OnDelete(event)
        if @m_listCtrl.GetItemCount() != 0
            @m_listCtrl.DeleteItem(0)
        else
            @m_logWindow.WriteText("Nothing to delete")
        end
    end

    def OnDeleteAll(event)
        sw = WxStopWatch.new

        @m_listCtrl.DeleteAllItems()

        @m_logWindow.WriteText(sprintf("Deleting %d items took %d ms\n",
                                                @m_listCtrl.GetItemCount(),
                                                sw.Time()))
        sw.free
    end

end


class MyApp < WxApp
    def OnInit
        # Create the main frame window
        frame = MyFrame.new("WxListCtrl Test", 50, 50, 450, 340)

        # Show the frame
        frame.Show(TRUE)

        SetTopWindow(frame)

    end
end


a = MyApp.new
a.MainLoop()


